//
//  IFC97_h_vap_2c3b_s.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 12/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_h_vap_2c3b_s.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */

// Boundary equation for the saturated vapor line in Region 2 with entropy input
double IFC97_h_vap_2c3b_s(double s) {
    /* Provides the backward equation for determining the saturated vapor enthalpy based on entropy in Region 2.
     
     Equations are based on Equation (6) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        h   = Specific enthalpy, kJ/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 17.
    int I[] = { 0,  0,  0,  0,  1,  1,  5,  6,  7,  8,  8,
                   12, 16, 22, 22, 24, 36   };
    
    int J[] = { 0,  0,  3,  4,  0, 12, 36, 12, 16,  2, 20,
                   32, 36,  2, 32,  7, 20 };
    
    double n[] = {   0.0,
                     0.104351280732769e+01,
                    -0.227807912708513e+01,
                     0.180535256723202e+01,
                     0.420440834792042e+00,
                    -0.105721244834660e+06,
                     0.436911607493884e+25,
                    -0.328032702839753e+12,
                    -0.678686760804270e+16,
                     0.743957464645363e+04,
                    -0.356896445355761e+20,
                     0.167590585186801e+32,
                    -0.355028625419105e+38,
                     0.396611982166538e+12,
                    -0.414716268484468e+41,
                     0.359080103867382e+19,
                    -0.116994334851995e+41 };
    
    // Reference Values
    double hstar = 2800.0;  // Units of kJ/kg
    double sstar = 5.9; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<17; i++) {
        hold += n[i] * pow(Sigma-1.02,I[i]) * pow(Sigma-0.726,J[i]);
    };
    
    return pow(hold,4.0) * hstar;
    
};
