//
//  IFC97_h_vap_2ab_s.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 12/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_h_vap_2ab_s.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */

// Boundary equation for the saturated vapor line in Region 2 with entropy input
double IFC97_h_vap_2ab_s(double s) {
    /* Provides the backward equation for determining the saturated vapor enthalpy based on entropy in Region 2.
     
     Equations are based on Equation (5) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        h   = Specific enthalpy, kJ/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 16.
    int I[] = { 0,  1,  1,  2,  2,  4,  4,  7,  8,  8, 10,
                   12, 12, 18, 20, 24, 28, 28, 28, 28, 28,
                   32, 32, 32, 32, 32, 36, 36, 36, 36, 36 };
    
    int J[] = { 0,  8, 24,  4, 32,  1,  2,  7,  5, 12,  1,
                    0,  7, 10, 12, 32,  8, 12, 20, 22, 24,
                    2,  7, 12, 14, 24, 10, 12, 20, 22, 28  };
    
    double n[] = {   0.0,
                    -0.524581170928788e+03,
                    -0.926947218142218e+07,
                    -0.237385107491666e+03,
                     0.210770155812776e+11,
                    -0.239494562010986e+02,
                     0.221802480294197e+03,
                    -0.510472533393438e+07,
                     0.124981396109147e+07,
                     0.200008436996201e+10,
                    -0.815158509791035e+03,
                    -0.157612685637523e+03,
                    -0.114200422332791e+11,
                     0.662364680776872e+16,
                    -0.227622818296144e+19,
                    -0.171048081348406e+32,
                     0.660788766938091e+16,
                     0.166320055886021e+23,
                    -0.218003784381501e+30,
                    -0.787276140295618e+30,
                     0.151062329700346e+32,
                     0.795732170300541e+07,
                     0.131957647355347e+16,
                    -0.325097068299140e+24,
                    -0.418600611419248e+26,
                     0.297478906557467e+35,
                    -0.953588761745473e+20,
                     0.166957699620939e+25,
                    -0.175407764869978e+33,
                     0.347581490626396e+35,
                    -0.710971318427851e+39  };
    
    // Reference Values
    double hstar  = 2800.0;  // Units of kJ/kg
    double sstar1 = 5.21; // Units of kJ/(kg K)
    double sstar2 = 9.2; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Sigma1 = s/sstar1;
    double Sigma2 = s/sstar2;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<31; i++) {
        hold += n[i] * pow((1.0/Sigma1)-0.513,I[i]) * pow(Sigma2-0.524,J[i]);
    };
    
    return exp(hold) * hstar;
};
