//
//  IFC97_h_liq_3a_s.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 11/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_h_liq_3a_s.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */

// Boundary equation for the saturated liquid line in Region 3a with entropy input
double IFC97_h_liq_3a_s(double s) {
    /* Provides the backward equation for determining the saturated liquid enthalpy based on entropy in Region 3a.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        h   = Specific enthalpy, kJ/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 10.
    int I[] = { 0,  0,  0,  0,  0,  2,  3,  4,  4,  5,  5,
                    6,  7,  7,  7, 10, 10, 10, 32, 32  };
    
    int J[] = { 0,  1,  4, 10, 16,  1, 36,  3, 16, 20, 36,
                    4,  2, 28, 32, 14, 32, 36,  0,  6  };
    
    double n[] = {   0.0,
                     0.822673364673336e+00,
                     0.181977213534479e+00,
                    -0.112000260313624e-01,
                    -0.746778287048033e-03,
                    -0.179046263257381e+00,
                     0.424220110836657e-01,
                    -0.341355823438768e+00,
                    -0.209881740853565e+01,
                    -0.822477343323596e+01,
                    -0.499684082076008e+01,
                     0.191413958471069e+00,
                     0.581062241093136e-01,
                    -0.165505498701029e+04,
                     0.158870443421201e+04,
                    -0.850623535172818e+02,
                    -0.317714386511207e+05,
                    -0.945890406632871e+05,
                    -0.139273847088690e-05,
                     0.631052532240980e+00   };
    
    // Reference Values
    double hstar = 1700.0;  // Units of kJ/kg
    double sstar = 3.8; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<20; i++) {
        hold += n[i] * pow(Sigma-1.09,I[i]) * pow(Sigma+0.366e-04,J[i]);
    };
    
    return hold * hstar;
    
};
