//
//  IFC97_h_liq_1_s.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 11/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_h_liq_1_s.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */


// Boundary equation for the saturated liquid line in Region 1 with entropy input
double IFC97_h_liq_1_s(double s) {
    /* Provides the backward equation for determining the saturated liquid enthalpy based on entropy in Region 1.
     
     Equations are based on Equation (3) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        h   = Specific enthalpy, kJ/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 9.
    int I[] = { 0,  0,  0,  1,  1,  2,  2,  3,  3,  4,  4,
                    4,  5,  5,  7,  8, 12, 12, 14, 14, 16,
                   20, 20, 22, 24, 28, 32, 32  };
    
    int J[] = { 0, 14, 36,  3, 16,  0,  5,  4, 36,  4, 16,
                   24, 18, 24,  1,  4,  2,  4,  1, 22, 10,
                   12, 28,  8,  3,  0,  6,  8  };
    
    double n[] = {   0.0,
                     0.332171191705237e+00,
                     0.611217706323496e-03,
                    -0.882092478906822e+01,
                    -0.455628192543250e+00,
                    -0.263483840850452e-04,
                    -0.223949661148062e+02,
                    -0.428398660164013e+01,
                    -0.616679338856916e+00,
                    -0.146823031104040e+02,
                     0.284523138727299e+03,
                    -0.113398503195444e+03,
                     0.115671380760859e+04,
                     0.395551267359325e+03,
                    -0.154891257229285e+01,
                     0.194486637751291e+02,
                    -0.357915139457043e+01,
                    -0.335369414148819e+01,
                    -0.664426796332460e+00,
                     0.323321885383934e+05,
                     0.331766744667084e+04,
                    -0.223501257931087e+05,
                     0.573953875852936e+07,
                     0.173226193407919e+03,
                    -0.363968822121321e-01,
                     0.834596332878346e-06,
                     0.503611916682674e+01,
                     0.655444787064505e+02 };
    
    // Reference Values
    double hstar = 1700.0;  // Units of kJ/kg
    double sstar = 3.8; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<28; i++) {
        hold += n[i] * pow(Sigma-1.09,I[i]) * pow(Sigma+0.366e-04,J[i]);
    };
    
    return hold * hstar;
    
};
