//
//  IFC97_hB13_s.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 12/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_hB13_s.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */

// Boundary equation for enthalpy the boundary between Regions 1 and 3 with entropy input
double IFC97_hB13_s(double s) {
    /* Provides the backward equation for determining the boundary enthalpy between Regions 1 and 3 with entropy input
     
     Equations are based on Equation (7) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        h   = Specific enthalpy, kJ/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 23.
    int I[] = { 0,  0,  1,  1,   3,  5,  6  };
    
    int J[] = { 0,  0, -2,  2, -12, -4, -3 };
    
    double n[] = {   0.0,
                     0.913965547600543e+00,
                    -0.430944856041991e-04,
                     0.603235694765419e+02,
                     0.117518273082168e-17,
                     0.220000904781292e+00,
                    -0.690815545851641e+02 };
    
    // Reference Values
    double hstar = 1700.0;  // Units of kJ/kg
    double sstar = 3.8; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<7; i++) {
        hold += n[i] * pow(Sigma-0.884,I[i]) * pow(Sigma-0.864,J[i]);
    };
    
    return hold * hstar;
    
};
