//
//  IFC97_Tsat_hs.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 12/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_Tsat_hs.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */

// Region 4 Saturation temperature based on h,s for s>s''(623.15K)
double IFC97_Tsat_hs(double h, double s) {
    /* Provides the Region 4 Saturation temperature based on h,s for s>s''(623.15K)
     
     Equations are based on Equation (9) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Saturation temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 28.
    int I[] = { 0, 0,  0,  0,  1,  1,  1,  1,  2,  2,  2,
                   3,  3,  3,  3,  4,  4,  5,  5,  5,  5,
                   6,  6,  6,  8, 10, 10, 12, 14, 14, 16,
                  16, 18, 18, 18, 20, 28  };
    
    int J[] = { 0,  0,  3, 12,  0,  1,  2,  5,  0,  5,  8,
                    0,  2,  3,  4,  0,  1,  1,  2,  4, 16,
                    6,  8, 22,  1, 20, 36, 24,  1, 28, 12,
                   32, 14, 22, 36, 24, 36  };
    
    double n[] = {   0.0,
                     0.179882673606601e+00,
                    -0.267507455199603e+00,
                     0.116276722612600e+01,
                     0.147545428713616e+00,
                    -0.512871635973248e+00,
                     0.421333567697984e+00,
                     0.563749522189870e+00,
                     0.429274443819153e+00,
                    -0.335704552142140e+01,
                     0.108890916499278e+02,
                    -0.248483390456012e+00,
                     0.304153221906390e+00,
                    -0.494819763939905e+00,
                     0.107551674933261e+01,
                     0.733888415457688e-01,
                     0.140170545411085e-01,
                    -0.106110975998808e+00,
                     0.168324361811875e-01,
                     0.125028363714877e+01,
                     0.101316840309509e+04,
                    -0.151791558000712e+01,
                     0.524277865990866e+02,
                     0.230495545563912e+05,
                     0.249459806365456e-01,
                     0.210796467412137e+07,
                     0.366836848613065e+09,
                    -0.144814105365163e+09,
                    -0.179276373003590e-02,
                     0.489955602100459e+10,
                     0.471262212070518e+03,
                    -0.829294390198652e+11,
                    -0.171545662263191e+04,
                     0.355777682973575e+07,
                     0.586062760258436e+12,
                    -0.129887635078195e+08,
                     0.317247449371057e+11 };
    
    // Reference Values
    double Tstar = 550.0;  // Units of K
    double hstar = 2800.0;  // Units of kJ/kg
    double sstar = 9.2; // Units of kJ/ (kg K)
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<37; i++) {
        hold += n[i] * pow(Eta-0.119,I[i]) * pow(Sigma-1.07,J[i]);
    };
    
    return hold * Tstar;
};
