//
//  IFC97_T_B23_hs.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 12/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_T_B23_hs.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */

// Boundary equation between Region 2 and 3 for temperature based on h,s
double IFC97_T_B23_hs(double h, double s) {
    /* Provides the backward Boundary equation between Region 2 and 3 for temperature based on h,s
     
     Equations are based on Equation (8) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = B23 boundary temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 25.
    int I[] = { 0, -12, -10,  -8,  -4,  -3,  -2, -2, -2, -2,  0,
                     1,   1,   1,   3,   3,   5,  6,  6,  8,  8,
                     8,  12,  12,  14,  14  };
    
    int J[] = { 0, 10,   8,  3,   4,  3, -6,  2,  3,  4,  0,
                   -3,  -2, 10,  -2, -1, -5, -6, -3, -8, -2,
                   -1, -12, -1, -12,  1  };
    
    double n[] = {   0.0,
                     0.629096260829810e-03,
                    -0.823453502583165e-03,
                     0.515446951519474e-07,
                    -0.117565945784945e+01,
                     0.348519684726192e+01,
                    -0.507837382408313e-11,
                    -0.284637670005479e+01,
                    -0.236092263939673e+01,
                     0.601492324973779e+01,
                     0.148039650824546e+01,
                     0.360075182221907e-03,
                    -0.126700045009952e-01,
                    -0.122184332521413e+07,
                     0.149276502463272e+00,
                     0.698733471798484e+00,
                    -0.252207040114321e-01,
                     0.147151930985213e-01,
                    -0.108618917681849e+01,
                    -0.936875039816322e-03,
                     0.819877897570217e+02,
                    -0.182041861521835e+03,
                     0.261907376402688e-05,
                    -0.291626417025961e+05,
                     0.140660774926165e-04,
                     0.783237062349385e+07 };
    
    // Reference Values
    double Tstar = 900.0;  // Units of K
    double hstar = 3000.0;  // Units of kJ/kg
    double sstar = 5.3; // Units of kJ/ (kg K)
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<26; i++) {
        hold += n[i] * pow(Eta-0.727,I[i]) * pow(Sigma-0.864,J[i]);
    };
    
    return hold * Tstar;
};
