//
//  IFC97_Single_Iteration.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 19/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_Single_Iteration.h"
#include <math.h>

int SingleIteration(FUNC Function, double *InVar0, double *InVar1, int ChangeVariable, double SolutionTolerance, double Solution, int MaxIterations) {
/*
    Function performs Newton's Method on the provided two variable function to find variable to make the Function equal Solution. The required variable is returned in provided input variable.
 
    Inputs:
        Function:           Two-variable (each a double) function that returns a double
        Var0:               First variable of Function. If iterated variable, this is to be the initial guess.
        Var1:               Second variable of Function. If iterated variable, this is to be the initial guess.
        ChangeVariable:     Indicates which function variable to use as the iteration variable.
                                - 0 = Var0
                                - 1 = Var1
        SolutionTolerance:  Absolute tolerance to use to accept iterated solution. In the same units as Solution units
        Solution:           Required value of Function
        MaxIterations:      Maximum allowed iterations for solution
 
    Outputs:
        return:             Number of iterations required for solution
 
    Errors:
        -1.  Iterations exceeded
 */
    // Transfer input information to internal variables
    double Var0 = *InVar0;
    double Var1 = *InVar1;
    
    // Establish ChangeVariable iteration step for determination of derivative. Use 0.5% change as basis.
    double DeltaVariable; // Delcare the step value
    double Percent = 0.5; // Percent of variable to utilize as the step value for the derivative
    
    if ( ChangeVariable == 0 ) DeltaVariable = Percent*Var0/100.0;
    else DeltaVariable = Percent*Var1/100.0;
   
    double Error = Solution - Function(Var0, Var1);  // Define the Error = Solution - Function
    double Slope = 0.0; // Declare the estimated slope of the function
    
    // Perform Newton's Method iteration
    for (int Count=0; Count<MaxIterations; Count++) {

        if (fabs(Error) < SolutionTolerance) { // If error is within allowable tolerance, exit
            // Transfer Solution to Input Variables
            *InVar0 = Var0;
            *InVar1 = Var1;
            
            // Return the number of iterations.
            return Count+1;
        };
        
        // Estimate the slope at the current location
        if (ChangeVariable == 0) {
            Slope = (Error - Solution + Function(Var0-DeltaVariable,Var1)) / DeltaVariable;
            Var0  -= Error/Slope; // Set new iterated variable
            if ( Var0 < 0.0) {
                Var0 = DeltaVariable;
            };
            DeltaVariable = Percent*Var0/100.0;
        }
        else {
            Slope = (Error - Solution + Function(Var0,Var1-DeltaVariable)) / DeltaVariable;
            Var1  -= Error/Slope;  // Set new iterated variable
            if ( Var1 < 0.0) {
                Var1 = DeltaVariable;
            };
            DeltaVariable = Percent*Var1/100.0;
        };
        
        Error = Solution - Function(Var0, Var1);  // Determine new error based on updated iteration
    };
    
    return -1; // Return error based on too many iterations.
};