//
//  IFC97_Saturation_Curve.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 18/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_Saturation_Curve.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

double IFC97_SaturationCurve(int iType, double P, double T) {
    /* Provides the Basic equations for the Saturation Curve for Region 4 determination.
     
     Equations are based on Equation (30) and (31) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iType = Indicates if Pressure or Temperature is to be determined based on inputs
        P     = Saturation Pressure, MPa
        T     = Saturation Temperature, K
     
     Return:
        For iType 0, Saturation Pressure, MPa
        For iType 1, Saturation Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Coefficients from Table 34.
    double n[] = {   0.0,
                     0.11670521452767e+04,
                    -0.72421316703206e+06,
                    -0.17073846940092e+02,
                     0.12020824702470e+05,
                    -0.32325550322333e+07,
                     0.14915108613530e+02,
                    -0.48232657361591e+04,
                     0.40511340542057e+06,
                    -0.23855557567849e+00,
                     0.65017534844798e+03    };
    
    //Intermediate Values
    double Pstar = 1.0;
    double Tstar = 1.0;
    
    if ( iType == 0 ) {
        double theta = (T/Tstar) + ( n[9]/((T/Tstar) - n[10]) );
        
        double A =      pow(theta,2.0) + n[1]*theta + n[2];
        double B = n[3]*pow(theta,2.0) + n[4]*theta + n[5];
        double C = n[6]*pow(theta,2.0) + n[7]*theta + n[8];
        
        double hold = pow(B,2.0) - 4.0*A*C;
        
        return pow((2.0*C) / (-B + sqrt(hold)), 4.0) * Pstar;
    }
    else {
        double beta = pow(P/Pstar, 0.25);// See Equation 29a.
        
        double E =      pow(beta,2.0) + n[3]*beta + n[6];
        double F = n[1]*pow(beta,2.0) + n[4]*beta + n[7];
        double G = n[2]*pow(beta,2.0) + n[5]*beta + n[8];
        
        double hold = pow(F,2.0) - 4.0*E*G;
        
        double D = (2.0*G) / (-F - sqrt(hold));
        
        //Values for Solution
        double num1 = pow(n[10]+D, 2.0) - 4.0*(n[9]+(n[10]*D));
        
        double T_ratio = 0.5 * (n[10] + D - sqrt(num1));// This is T_sat/T_star (see Equation 31)
        
        return T_ratio * Tstar;
    };
};

// Get saturation temperature (K) from input pressure (MPa). Calls IFC97_SaturationCurve
double IFC97_Tsat(double P) {
    /* Returns the saturation temperature given the saturation pressure.
     
     Equations are based on Equation (30) and (31) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Saturation Pressure, MPa
     
     Return:
        T = Saturation Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_SaturationCurve(1, P, 0.0);

};

// Get saturation pressure (MPa) from input temperature (K). Calls IFC97_SaturationCurve
double IFC97_Psat(double T) {
    /* Returns the saturation pressure given the saturation temperature.
     
     Equations are based on Equation (30) and (31) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        T = Saturation Temperature, K
     
     Return:
        P = Saturation Pressure, MPa
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_SaturationCurve(0, 0.0, T);
    
};
