//
//  IFC97_R3_T_Boundaries.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 15/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_T_Boundaries.h"
#include <math.h>

/*
Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations for Specific Volume a a Function of Pressure and temperature v(P,T) for Region 2 of the IAWPS Industrial Formaulation 1997 for the Thermodynamic Properties of Water and Steam. Santorini, Greece, July 2005.
*/

// Backward Equations outside the critical region. Based on Equation (2)
double IFC97_R3_T3ab_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3a and 3b in Region 3.
     
     Equations are based on Equation (2) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, -1, -2 };
    
    double n[] = {   0.0,
                     0.154793642129415e+04,
                    -0.187661219490113e+03,
                     0.213144632222113e+02,
                    -0.191887498864292e+04,
                     0.918419702359447e+03 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<6; i++) {
        hold += n[i] * pow(log(Pi),I[i]);
    };
    
    return hold * Tstar;
};


double IFC97_R3_T3op_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3o and 3p in Region 3.
     
     Equations are based on Equation (2) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, -1, -2 };
    
    double n[] = {   0.0,
                     0.969461372400213e+03,
                    -0.332500170441278e+03,
                     0.642859598466067e+02,
                     0.773845935768222e+03,
                    -0.152313732937084e+04 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<6; i++) {
        hold += n[i] * pow(log(Pi),I[i]);
    };
    
    return hold * Tstar;
};

// Backward Equations outside the critical region. Based on Equation (1)
double IFC97_R3_T3cd_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3c and 3d in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, 3 };
    
    double n[] = {   0.0,
                     0.585276966696349e+03,
                     0.278233532206915e+01,
                    -0.127283549295878e-01,
                     0.159090746562729e-03 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<5; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

double IFC97_R3_T3gh_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3g and 3h in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, 3, 4 };
    
    double n[] = {   0.0,
                    -0.249284240900418e+05,
                     0.428143584791546e+04,
                    -0.269029173140130e+03,
                     0.751608051114157e+01,
                    -0.787105249910383e-01 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<6; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

double IFC97_R3_T3ij_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3i and 3j in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, 3, 4 };
    
    double n[] = {   0.0,
                     0.584814781649163e+03,
                    -0.616179320924617e+00,
                     0.260763050899562e+00,
                    -0.587071076864459e-02,
                     0.515308185433082e-04 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<6; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

double IFC97_R3_T3jk_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3j and 3k in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, 3, 4 };
    
    double n[] = {   0.0,
                     0.617229772068439e+03,
                    -0.770600270141675e+01,
                     0.697072596851896e+00,
                    -0.157391839848015e-01,
                     0.137897492684194e-03 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<6; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

double IFC97_R3_T3mn_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3m and 3n in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, 3 };
    
    double n[] = {   0.0,
                     0.535339483742384e+03,
                     0.761978122720128e+01,
                    -0.158365725441648e+00,
                     0.192871054508108e-02 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<5; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

double IFC97_R3_T3qu_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3q and 3u in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, 3 };
    
    double n[] = {   0.0,
                     0.565603648239126e+03,
                     0.529062258221222e+01,
                    -0.102020639611016e+00,
                     0.122240301070145e-02 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<5; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

double IFC97_R3_T3rx_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3r and 3x in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 1.
    int I[] =   { 0, 0, 1, 2, 3 };
    
    double n[] = {   0.0,
                     0.584561202520006e+03,
                    -0.102961025163669e+01,
                     0.243293362700452e+00,
                    -0.294905044740799e-02 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<5; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

// Backward Equations outside the critical region. Based on Equation (3)
double IFC97_R3_T3ef_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3e and 3f in Region 3.
     
     Equations are based on Equation (3) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K

    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Non-dimensional values
    double Factor = 3.727888004; // Value of dTheta.sat / dPi
    
    return (Factor*(Pi-22.064) + 647.096) * Tstar;
};

// Auxiliary Equations near the critical region. Based on Equation (1)
double IFC97_R3_T3uv_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3u and 3v in Region 3.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 9.
    int I[] =   { 0, 0, 1, 2, 3 };
    
    double n[] = {   0.0,
                     0.528199646263062e+03,
                     0.890579602135307e+01,
                    -0.222814134903755e+00,
                     0.286791682263697e-02  };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<5; i++) {
        hold += n[i] * pow(Pi,I[i]);
    };
    
    return hold * Tstar;
};

// Auxiliary Equations near the critical region. Based on Equation (2)
double IFC97_R3_T3wx_P(double P) {
    /* Provides the equation for the temperature boundary between Regions 3w and 3x in Region 3.
     
     Equations are based on Equation (2) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
     
     Return:
        T   = Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 9.
    int I[] =   { 0, 0, 1, 2, -1, -2 };
    
    double n[] = {   0.0,
                     0.728052609145380e+01,
                     0.973505869861952e+02,
                     0.147370491183191e+02,
                     0.329196213998375e+03,
                     0.873371668682417e+03 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    
    // Non-dimensional values
    double Pi = P/Pstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<6; i++) {
        hold += n[i] * pow(log(Pi),I[i]);
    };
    
    return hold * Tstar;
};
