//
//  IFC97_R3_Saturation_V.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 23/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Saturation_V.h"

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_GetRegion_Ph.h"
#include "IFC97_GetProperty_Ph.h"
#include "IFC97_GetProperty_Ph_Region4.h"
#include "IFC97_GetRegion_Ps.h"
#include "IFC97_GetProperty_Ps.h"
#include "IFC97_GetProperty_Ps_Region4.h"
#include "IFC97_GetRegion_hs.h"
#include "IFC97_Bhmax_s.h"
#include "IFC97_Bhmin_s.h"

/*
 Function determines the specific volume at the indicated saturation conditions in Region 3
 */

double IFC97_R3_Saturation_V(double Psat, double Tsat, int LiqVap) {
    /* Provides the saturated specific volume (either liquid or vapor) in Region 3.  Either the Saturation Pressure (Psat) or the Saturation Temperature (Tsat) is to be provided, not both.  If both are provided, the calculations are based on the given Saturation Pressure.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         Tsat   = Saturation temperature, K
         Psat   = Saturation pressure, MPa
         LiqVap = Indication of which saturation condition to provided.
                    0 = Saturated Liquid
                    1 = Saturated Vapor
     Return:
         SV     = Saturated specific volume at the indicated conditions
     
     Errors:
        -9990.0 = Error in Region Determination
        -9999.0 = Iterations exceeded
     */
    // Set used Pressure and Temperature values
    double P = 0.0; // Used pressure in the calculation
    double T = 0.0; // Used temperature in the calculation
    
    if ( Psat == 0.0 ) {
        P = IFC97_Psat(Tsat);
        T = Tsat;
    }
    else {
        P = Psat;
        T = IFC97_Tsat(Psat);
    };
    
    // Obtain the proper region/subregion information
    int Region    = -1;
    int SubRegion = -1;
    
    // Placeholder for error in GetRegion
    int RegionCheck = -1;
    
    // Bump the temperature up or down to ensure getting the proper SubRegion for either liquid or vapor conditions
    if ( LiqVap == 0) RegionCheck = IFC97_GetRegion_PT(P, T-0.001, &Region, &SubRegion); // Bump down to get liquid
    else              RegionCheck = IFC97_GetRegion_PT(P, T+0.001, &Region, &SubRegion); // Bump up to get vapor conditions
   
    // Exit on error in getting region
    if ( RegionCheck < 0 ) return -9990.0;
    
    // If no errors in region determination, get saturated specific volume
    double RHO = 1.0/IFC97_R3_V_pt(SubRegion, P, T);
    
    // Establish allowable tolerances on solutions to iterations
    double Tol_P = 0.00000000037; // Tolerance on pressure iterations, MPa
    
    // Set the maximum allowed number of iterations
    int MaxIterations  = 500;
    
    // Perform iteration to obtain specific volume
    if ( SingleIteration(IFC97_R3_P_dt, &RHO, &T, 0, Tol_P, P, MaxIterations) < 0 ) return -9999.0; // Get actual density or iteration error
    
    // Return the provided specific volume
    return 1.0/RHO;

};