//
//  IFC97_R3_Basic.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 24/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Basic.h"
#include "IAPWS_Surface_Tension.h"
#include "IAPWS_DynamicViscosity_Td.h"
#include "IAPWS_ThermalConductivity_TP.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

double IFC97_R3_Basic(int Property, double RHO, double T) {
    /* Provides the Basic equation for the Region 3 Hemholtz free energy function.
     
     Equations are based on Equation (28) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Property = Indicates property to be returned
        RHO      = Density, kg/m^3
        T        = Saturation Temperature, K
     
     Return:
         For Property = 0, Pressure, MPa
                        1, Temperature, K
                        2, Specific volume, m^3/kg
                        3, Specific internal energy, kJ/kg
                        4, Specific entropy, kJ/(kg K)
                        5, Specific enthalpy, kJ/kg
                        6, Specific isobaric heat capacity, kJ/(kg K)
                        7, Specific isochoric heat capacity, kJ/(kg K)
                        8, Density, kg/m^3
                        9, Speed of sound, m/s
                        10, Gibbs free energy, kJ/kg
                        11, Hemholtz free energy, kJ/kg
                        12, Quality, dimensionless
                        13, Ratio of specific heats (Cp/Cv), dimensionless
                        14, Surface Tension, N/m
                        15. Dynamic Viscosity, Pa*s
                        16. Thermal Conductivity, W/(K*m)
                        17. Thermal diffusivity, m^2/s
                        18. Kinematic Viscosity, m^2/s
                        19. Prandtl Number, dimensionless
                        20. Drho_DP_T, kg/(m^3 MPa), for thermal conductivity calculation
     
     Errors:
        -10000.0 = Incorrect iType input
        -90000.0 = Unknown error
        -80000.0 = Invalid Property Type
     */
    
    // Numerical coefficients from Table 30.
    int I[] = { 0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,
                    1,  1,  2,  2,  2,  2,  2,  2,  3,  3,
                    3,  3,  3,  4,  4,  4,  4,  5,  5,  5,
                    6,  6,  6,  7,  8,  9,  9, 10, 10, 11  };
    
    int J[] = { 0,  0,   0,  1,  2,  7, 10, 12, 23,  2,  6,
                    15, 17,  0,  2,  6,  7, 22, 26,  0,  2,
                     4, 16, 26,  0,  2,  4, 26,  1,  3, 26,
                     0,  2, 26,  2, 26,  2, 26,  0,  1, 26 };
    
    double n[] = {   0.0,
                     0.10658070028513e+01,
                    -0.15732845290239e+02,
                     0.20944396974307e+02,
                    -0.76867707878716e+01,
                     0.26185947787954e+01,
                    -0.28080781148620e+01,
                     0.12053369696517e+01,
                    -0.84566812812502e-02,
                    -0.12654315477714e+01,
                    -0.11524407806681e+01,
                     0.88521043984318e+00,
                    -0.64207765181607e+00,
                     0.38493460186671e+00,
                    -0.85214708824206e+00,
                     0.48972281541877e+01,
                    -0.30502617256965e+01,
                     0.39420536879154e-01,
                     0.12558408424308e+00,
                    -0.27999329698710e+00,
                     0.13899799569460e+01,
                    -0.20189915023570e+01,
                    -0.82147637173963e-02,
                    -0.47596035734923e+00,
                     0.43984074473500e-01,
                    -0.44476435428739e+00,
                     0.90572070719733e+00,
                     0.70522450087967e+00,
                     0.10770512626332e+00,
                    -0.32913623258954e+00,
                    -0.50871062041158e+00,
                    -0.22175400873096e-01,
                     0.94260751665092e-01,
                     0.16436278447961e+00,
                    -0.13503372241348e-01,
                    -0.14834345352472e-01,
                     0.57922953628084e-03,
                     0.32308904703711e-02,
                     0.80964802996215e-04,
                    -0.16557679795037e-03,
                    -0.44923899061815e-04 };
       
    
    // Reference Values
    double RHOstar = 322.0;    // Units of kg/m^3, critical density
    double Tstar   = 647.096;  // Units of K, cricital temperature
    double R       = 0.461526; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Delta  = RHO/RHOstar;
    double Tau    = Tstar/T;
    
    // Temporary Variables
    double hold  = 0.0;
    double denom = 0.0;
    double hold1 = 0.0;
    double hold2 = 0.0;
    
    // Return requested property
    switch (Property) {
        case 0: // return Pressure
            return Delta * IFC97_R3_Get_Hr(Delta, Tau, n, I, J) * RHO * R * T / 1000.0;
            
        case 1: // return Temperature
            return T;
            
        case 2: // return specific volume
            return 1.0/RHO;
        
        case 3: // return specific internal energy
            return Tau*IFC97_R3_Get_Ht(Delta, Tau, n, I, J) * R * T;
            
        case 4: // return specific entropy
            return (Tau*IFC97_R3_Get_Ht(Delta, Tau, n, I, J)-IFC97_R3_Hemholtz_ND(Delta, Tau, n, I, J)) * R;
            
        case 5: // return specific enthalpy
            return (Tau*IFC97_R3_Get_Ht(Delta, Tau, n, I, J)+Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J)) * R * T;
            
        case 6: // return specific isobaric heat capacity
            hold  = pow(Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J)-Delta*Tau*IFC97_R3_Get_Hrt(Delta, Tau, n, I, J), 2.0);
            denom = 2.0*Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J) + pow(Delta,2.0)*IFC97_R3_Get_Hrr(Delta, Tau, n, I, J);
            return (-pow(Tau,2.0)*IFC97_R3_Get_Htt(Delta, Tau, n, I, J) + hold/denom) * R;
            
        case 7: // return specific isochoric heat capacity
            return -pow(Tau,2.0) * IFC97_R3_Get_Htt(Delta, Tau, n, I, J) * R;
            
        case 8: // return density
            return RHO;
        
        case 9: // return speed of sound
            hold  = pow(Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J)-Delta*Tau*IFC97_R3_Get_Hrt(Delta, Tau, n, I, J), 2.0);
            denom = pow(Tau,2.0) * IFC97_R3_Get_Htt(Delta, Tau, n, I, J);
            return sqrt( (2.0*Delta*IFC97_R3_Get_Hr(Delta,Tau,n,I,J)+pow(Delta,2.0)*IFC97_R3_Get_Hrr(Delta,Tau,n,I,J)-hold/denom)*1000.0*R*T );
            
        case 10: // return Gibbs free energy
            return (IFC97_R3_Hemholtz_ND(Delta, Tau, n, I, J) +  Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J)) * R * T;
            
        case 11: // return Hemholtz free energy
            return IFC97_R3_Hemholtz_ND(Delta, Tau, n, I, J) * R * T;
            
        case 12: // Quality.  Is less than T_critical use liquid, otherwise use vapor
            if ( T <= 647.096) return 0.0;
            else               return 1.0;
            
        case 13: // return ratio of specific heats (Cp/Cv)
            hold  = pow(Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J)-Delta*Tau*IFC97_R3_Get_Hrt(Delta, Tau, n, I, J), 2.0);
            denom = 2.0*Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J) + pow(Delta,2.0)*IFC97_R3_Get_Hrr(Delta, Tau, n, I, J);
            return ( (-pow(Tau,2.0)*IFC97_R3_Get_Htt(Delta, Tau, n, I, J) + hold/denom) * R ) /  // Cp
                   ( -pow(Tau,2.0) * IFC97_R3_Get_Htt(Delta, Tau, n, I, J) * R              );   // Cv
            
        case 14: // return Surface Tension, N/m
            if ( T <= Tstar ) return IAPWS_SurfaceTension(T);
            else              return -80000.0;  // Not valid over T_critical
            
        case 15: // return Dynamic Viscosity, Pa*s
            return IAPWS_DynamicViscosity_Td(T, RHO);
            
        case 16: // return Thermal Conductivity, W/(K*m)
            hold = Delta * IFC97_R3_Get_Hr(Delta, Tau, n, I, J) * RHO * R * T / 1000.0;
            return IAPWS_ThermalConductivity_TP(T, hold);
   
        case 17:  // Thermal diffusivity, m^2/sec
            hold  = Delta * IFC97_R3_Get_Hr(Delta, Tau, n, I, J) * RHO * R * T / 1000.0; // Pressure
            
            hold1 = pow(Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J)-Delta*Tau*IFC97_R3_Get_Hrt(Delta, Tau, n, I, J), 2.0);
            denom = 2.0*Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J) + pow(Delta,2.0)*IFC97_R3_Get_Hrr(Delta, Tau, n, I, J);
            hold2 = (-pow(Tau,2.0)*IFC97_R3_Get_Htt(Delta, Tau, n, I, J) + hold1/denom) * R;  // Cp
            
            return IAPWS_ThermalConductivity_TP(T, hold) / (RHO * hold2 * 1000.0 );
            
        case 18: // Kinematic viscosity, m^2/sec
            return IAPWS_DynamicViscosity_Td(T, RHO) / RHO;
            
        case 19: // Prandtl Number, dimensionless
            hold  = Delta * IFC97_R3_Get_Hr(Delta, Tau, n, I, J) * RHO * R * T / 1000.0; // Pressure
            
            hold1 = pow(Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J)-Delta*Tau*IFC97_R3_Get_Hrt(Delta, Tau, n, I, J), 2.0);
            denom = 2.0*Delta*IFC97_R3_Get_Hr(Delta, Tau, n, I, J) + pow(Delta,2.0)*IFC97_R3_Get_Hrr(Delta, Tau, n, I, J);
            hold2 = (-pow(Tau,2.0)*IFC97_R3_Get_Htt(Delta, Tau, n, I, J) + hold1/denom) * R;  // Cp

            return IAPWS_DynamicViscosity_Td(T, RHO)*hold2*1000.0/IAPWS_ThermalConductivity_TP(T, hold);
            
        case 20: // Drho_DP_T, kg/(m^3 MPa) for thermal conductivity calculation
            return (1000.0)/(Delta*R*T*((Delta*IFC97_R3_Get_Hrr(Delta, Tau, n, I, J) + 2.0*IFC97_R3_Get_Hr(Delta, Tau, n, I, J))));
            
        default:
            return -10000.0;
    };
    
    return -90000.0;
};

// Non-Dimensional Hemholtz free energy. Equation (28)
double IFC97_R3_Hemholtz_ND(double Delta, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional Hemholtz free energy for Region 3.
     
     Equations are based on Equation (28) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Delta  = Non-dimensional density for Region 3
        Tau    = Non-dimensional temperature for Region 3
        n      = Table 30 n-coefficient array
        I      = Table 30 I-coefficient array
        J      = Table 30 J-coefficient array
     
     Return:
        Non-dimensional Hemholtz free energy
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    double hold = n[1]*log(Delta); // summer variable
    
    for (int i=2; i<41; i++) {
        hold += n[i] * pow(Delta,I[i]) * pow(Tau,J[i]);  // Equation (7)
    };
    
    return hold;
};


// Derivatives for the Hemholtz free energy per Table 32.
double IFC97_R3_Get_Hr(double Delta, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the Hemholtz free energy wrt density (rho)
     
     Equations are based on Table 32 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         Delta  = Non-dimensional density for Region 3
         Tau    = Non-dimensional temperature for Region 3
         n      = Table 30 n-coefficient array
         I      = Table 30 I-coefficient array
         J      = Table 30 J-coefficient array
     
     Return:
        Non-dimensional derivative of H wrt rho
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = n[1]/Delta; // summer variable
    
    for (int i=2; i<41; i++) {
        hold += n[i] * I[i] * pow(Delta,I[i]-1) * pow(Tau,J[i]);
    };
    
    return hold;
};


double IFC97_R3_Get_Hrr(double Delta, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the Hemholtz free energy wrt density (rho), density (rho)
     
     Equations are based on Table 32 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         Delta  = Non-dimensional density for Region 3
         Tau    = Non-dimensional temperature for Region 3
         n      = Table 30 n-coefficient array
         I      = Table 30 I-coefficient array
         J      = Table 30 J-coefficient array
     
     Return:
        Non-dimensional derivative of H wrt rho, rho
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = -n[1]/pow(Delta,2.0); // summer variable
    
    for (int i=2; i<41; i++) {
        hold += n[i] * I[i] * (I[i]-1) * pow(Delta,I[i]-2) * pow(Tau,J[i]);
    };
    
    return hold;
};


double IFC97_R3_Get_Ht(double Delta, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the Hemholtz free energy wrt T
     
     Equations are based on Table 32 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         Delta  = Non-dimensional density for Region 3
         Tau    = Non-dimensional temperature for Region 3
         n      = Table 30 n-coefficient array
         I      = Table 30 I-coefficient array
         J      = Table 30 J-coefficient array
     
     Return:
        Non-dimensional derivative of H wrt T
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=2; i<41; i++) {
        hold += n[i] * pow(Delta,I[i]) * J[i] * pow(Tau,J[i]-1);
    };
    
    return hold;
};

double IFC97_R3_Get_Htt(double Delta, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the Hemholtz free energy wrt T, T
     
     Equations are based on Table 32 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         Delta  = Non-dimensional density for Region 3
         Tau    = Non-dimensional temperature for Region 3
         n      = Table 30 n-coefficient array
         I      = Table 30 I-coefficient array
         J      = Table 30 J-coefficient array
         
     Return:
        Non-dimensional derivative of H wrt T, T
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=2; i<41; i++) {
        hold += n[i] * pow(Delta,I[i]) * J[i] * (J[i]-1) * pow(Tau,J[i]-2);
    };
    
    return hold;
};

double IFC97_R3_Get_Hrt(double Delta, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the Hemholtz free energy wrt rho, T
     
     Equations are based on Table 32 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         Delta  = Non-dimensional density for Region 3
         Tau    = Non-dimensional temperature for Region 3
         n      = Table 30 n-coefficient array
         I      = Table 30 I-coefficient array
         J      = Table 30 J-coefficient array
         
     Return:
        Non-dimensional derivative of H wrt rho, T
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=2; i<41; i++) {
        hold += n[i] * I[i] * pow(Delta,I[i]-1) * J[i] * pow(Tau,J[i]-1);
    };
    
    return hold;
};

// Interface functions to obtain specific properties
double IFC97_R3_P_dt(double RHO, double T) {
    /* Provides the Region 3 Pressure given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        RHO = Density, kg/m^3
        T   = Temperature, K
     
     Return:
        Pressure, MPa
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(0, RHO, T);
};

double IFC97_R3_U_dt(double RHO, double T) {
    /* Provides the Region 3 specific internal energy given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        RHO = Density, kg/m^3
        T   = Temperature, K
     
     Return:
        Specific internal energy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(3, RHO, T);
};


double IFC97_R3_S_dt(double RHO, double T) {
    /* Provides the Region 3 specific entropy given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         RHO = Density, kg/m^3
         T   = Temperature, K
         
     Return:
        Specific entropy, kJ/(kg K)
     
     Errors:
     None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(4, RHO, T);
};


double IFC97_R3_H_dt(double RHO, double T) {
    /* Provides the Region 3 specific enthalpy given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         RHO = Density, kg/m^3
         T   = Temperature, K
         
     Return:
         Specific enthalpy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(5, RHO, T);

};


double IFC97_R3_Cp_dt(double RHO, double T) {
    /* Provides the Region 3 specific isobaric heat capacity given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         RHO = Density, kg/m^3
         T   = Temperature, K
     
     Return:
        Specific isobaric heat capacity, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(6, RHO, T);
};


double IFC97_R3_Cv_dt(double RHO, double T) {
    /* Provides the Region 3 specific isochoric heat capacity given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        RHO = Density, kg/m^3
        T   = Temperature, K
     
     Return:
        Specific isochoric heat capacity, kJ/(kg K)
     
     Errors:
     None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(7, RHO, T);
};


double IFC97_R3_SOS_dt(double RHO, double T) {
    /* Provides the Region 3 speed of sound given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         RHO = Density, kg/m^3
         T   = Temperature, K
     
     Return:
        Speed of sound, m/s
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(8, RHO, T);
};


double IFC97_R3_Gibbs_dt(double RHO, double T) {
    /* Provides the Region 3 Gibbs free energy given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         RHO = Density, kg/m^3
         T   = Temperature, K
     
     Return:
        Gibbs free energy, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(9, RHO, T);
};


double IFC97_R3_Hemholtz_dt(double RHO, double T) {
    /* Provides the Region 3 Hemholtz free energy given density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
         RHO = Density, kg/m^3
         T   = Temperature, K
     
     Return:
        Hemholtz free energy, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Basic(10, RHO, T);
};


double IFC97_R3_k_Ratio_dt(double RHO, double T) {
    /* Provides the Region 3 ratio of specific heats given density and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        RHO = Density, kg/m^3
        T   = Temperature, K
     
     Return:
        Ratio of specific heats, k, dimensionless
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R3_Cp_dt(RHO, T) / IFC97_R3_Cv_dt(RHO, T);
};


double IFC97_R3_SV_dt(double RHO, double T) {
    /* Provides the Region 3 specific volume density and temperature
     
     Functions calls the basic Region 3 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        RHO = Density, kg/m^3
        T   = Temperature, K
     
     Return:
        Specific volume, m^3/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return 1.0/RHO;
};
