//
//  IFC97_R3_Backward_Vpt.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 16/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Backward_Vpt.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations for Specific Volume a a Function of Pressure and temperature v(P,T) for Region 2 of the IAWPS Industrial Formaulation 1997 for the Thermodynamic Properties of Water and Steam. Santorini, Greece, July 2005.
 */

// Coefficient values for Backward and Auxiliary Equations as noted.  From Table 4 and Table 12
double Coefficients[26][9] = {  {0.0024, 100.0, 760.0, 30.0, 0.085, 0.817, 1.0,  1.0, 1.0 },  // Sub-region 3a parameters
                                {0.0041, 100.0, 860.0, 32.0, 0.280, 0.779, 1.0,  1.0, 1.0 },  // Sub-region 3b parameters
                                {0.0022,  40.0, 690.0, 35.0, 0.259, 0.903, 1.0,  1.0, 1.0 },  // Sub-region 3c parameters
                                {0.0029,  40.0, 690.0, 38.0, 0.559, 0.939, 1.0,  1.0, 4.0 },  // Sub-region 3d parameters
                                {0.0032,  40.0, 710.0, 29.0, 0.587, 0.918, 1.0,  1.0, 1.0 },  // Sub-region 3e parameters
                                {0.0064,  40.0, 730.0, 42.0, 0.587, 0.891, 0.5,  1.0, 4.0 },  // Sub-region 3f parameters
                                {0.0027,  25.0, 660.0, 38.0, 0.872, 0.971, 1.0,  1.0, 4.0 },  // Sub-region 3g parameters
                                {0.0032,  25.0, 660.0, 29.0, 0.898, 0.983, 1.0,  1.0, 4.0 },  // Sub-region 3h parameters
                                {0.0041,  25.0, 660.0, 42.0, 0.910, 0.984, 0.5,  1.0, 4.0 },  // Sub-region 3i parameters
                                {0.0054,  25.0, 670.0, 29.0, 0.875, 0.964, 0.5,  1.0, 4.0 },  // Sub-region 3j parameters
                                {0.0077,  25.0, 680.0, 34.0, 0.802, 0.935, 1.0,  1.0, 1.0 },  // Sub-region 3k parameters
                                {0.0026,  24.0, 650.0, 43.0, 0.908, 0.989, 1.0,  1.0, 4.0 },  // Sub-region 3l parameters
                                {0.0028,  23.0, 650.0, 40.0, 1.000, 0.997, 1.0, 0.25, 1.0 },  // Sub-region 3m parameters
                                {0.0031,  23.0, 650.0, 39.0, 0.976, 0.997, 0.0,  0.0, 0.0 },  // Sub-region 3n parameters
                                {0.0034,  23.0, 650.0, 24.0, 0.974, 0.996, 0.5,  1.0, 1.0 },  // Sub-region 3o parameters
                                {0.0041,  23.0, 650.0, 27.0, 0.972, 0.997, 0.5,  1.0, 1.0 },  // Sub-region 3p parameters
                                {0.0022,  23.0, 650.0, 24.0, 0.848, 0.983, 1.0,  1.0, 4.0 },  // Sub-region 3q parameters
                                {0.0054,  23.0, 650.0, 27.0, 0.874, 0.982, 1.0,  1.0, 1.0 },  // Sub-region 3r parameters
                                {0.0022,  21.0, 640.0, 29.0, 0.886, 0.990, 1.0,  1.0, 4.0 },  // Sub-region 3s parameters
                                {0.0088,  20.0, 650.0, 33.0, 0.803, 1.020, 1.0,  1.0, 1.0 },  // Sub-region 3t parameters
                                {0.0026,  23.0, 650.0, 38.0, 0.902, 0.988, 1.0,  1.0, 1.0 },  // Sub-region 3u parameters (auxiliary equations)
                                {0.0031,  23.0, 650.0, 39.0, 0.960, 0.995, 1.0,  1.0, 1.0 },  // Sub-region 3v parameters (auxiliary equations)
                                {0.0039,  23.0, 650.0, 35.0, 0.959, 0.995, 1.0,  1.0, 4.0 },  // Sub-region 3w parameters (auxiliary equations)
                                {0.0049,  23.0, 650.0, 36.0, 0.910, 0.988, 1.0,  1.0, 1.0 },  // Sub-region 3x parameters (auxiliary equations)
                                {0.0031,  22.0, 650.0, 20.0, 0.996, 0.994, 1.0,  1.0, 4.0 },  // Sub-region 3y parameters (auxiliary equations)
                                {0.0038,  22.0, 650.0, 23.0, 0.993, 0.994, 1.0,  1.0, 4.0 }   // Sub-region 3z parameters (auxiliary equations)
                            };



// Backward Equations outside the critical region based on Equation (4)
double IFC97_R3_V_pt_3a(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3a.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 0;
    
    // Numerical coefficients from Table A1.1.
    int I[] = { 0, -12, -12, -12, -10, -10, -10,  -8, -8, -8, -6,
                    -5,  -5,  -5,  -4,  -3,  -3,  -3, -3, -2, -2,
                    -2,  -1,  -1,  -1,   0,   0,   1,  1,  2,  2  };
    
    int J[] = { 0,  5, 10, 12,  5, 10, 12,  5,  8, 10,  1,
                    1,  5, 10,  8,  0,  1,  3,  6,  0,  2,
                    3,  0,  1,  2,  0,  1,  0,  2,  0,  2   };
    
    double n[] = {   0.0,
                     0.110879558823853e-02,
                     0.572616740810616e+03,
                    -0.767051948380852e+05,
                    -0.253321069529674e-01,
                     0.628008049345689e+04,
                     0.234105654131876e+06,
                     0.216867826045856e+00,
                    -0.156237904341963e+03,
                    -0.269893956176613e+05,
                    -0.180407100085505e-03,
                     0.116732227668261e-02,
                     0.266987040856040e+02,
                     0.282776617243286e+05,
                    -0.242431520029523e+04,
                     0.435217323022733e-03,
                    -0.122494831378441e-01,
                     0.179357604019989e+01,
                     0.442729521058314e+02,
                    -0.593223489018342e-02,
                     0.453186261685774e+00,
                     0.135825703129140e+01,
                     0.408748415856745e-01,
                     0.474686397863312e+00,
                     0.118646814997915e+01,
                     0.546987265727549e+00,
                     0.195266770452643e+00,
                    -0.502268790869663e-01,
                    -0.369645308193377e+00,
                     0.633828037528420e-02,
                     0.797441793901017e-01 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};


double IFC97_R3_V_pt_3b(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3b.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 1;
    
    // Numerical coefficients from Table A1.2.
    int I[] = { 0, -12, -12, -10, -10,  -8,  -6,  -6, -6, -5, -5,
                    -5,  -4,  -4,  -4,  -3,  -3,  -3, -3, -3, -2,
                    -2,  -2,  -1,  -1,   0,   0,   1,  1,  2,  3,
                     4,   4};
    
    int J[] = { 0, 10, 12,  8, 14,  8,  5,  6,  8,  5,  8,
                   10,  2,  4,  5,  0,  1,  2,  3,  5,  0,
                    2,  5,  0,  2,  0,  1,  0,  2,  0,  2,
                    0,  1 };
    
    double n[] = {   0.0,
                    -0.827670470003621e-01,
                     0.416887126010565e+02,
                     0.483651982197059e-01,
                    -0.291032084950276e+05,
                    -0.111422582236948e+03,
                    -0.202300083904014e-01,
                     0.294002509338515e+03,
                     0.140244997609658e+03,
                    -0.344384158811459e+03,
                     0.361182452612149e+03,
                    -0.140699677420738e+04,
                    -0.202023902676481e-02,
                     0.171346792457471e+03,
                    -0.425597804058632e+01,
                     0.691346085000334e-05,
                     0.151140509678925e-02,
                    -0.416375290166236e-01,
                    -0.413754957011042e+02,
                    -0.506673295721637e+02,
                    -0.572212965569023e-03,
                     0.608817368401785e+01,
                     0.239600660256161e+02,
                     0.122261479925384e-01,
                     0.216356057692938e+01,
                     0.398198903368642e+00,
                    -0.116892827834085e+00,
                    -0.102845919373532e+00,
                    -0.492676637589284e+00,
                     0.655540456406790e-01,
                    -0.240462535078530e+00,
                    -0.269798180310075e-01,
                     0.128369435967012e+00 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3c(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3c.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 2;
    
    // Numerical coefficients from Table A1.3.
    int I[] = { 0, -12, -12, -12, -10, -10, -10,  -8, -8, -8, -6,
                    -5,  -5,  -5,  -4,  -4,  -3,  -3, -2, -2, -2,
                    -1,  -1,  -1,   0,   0,   0,   1,  1,  2,  2,
                     2,   2,   3,   3,   8};
    
    int J[] = { 0,  6,  8, 10,  6,  8, 10,  5,  6,  7,  8,
                    1,  4,  7,  2,  8,  0,  3,  0,  4,  5,
                    0,  1,  2,  0,  1,  2,  0,  2,  0,  1,
                    3,  7,  0,  7,  1 };
    
    double n[] = {   0.0,
                     0.311967788763030e+01,
                     0.276713458847564e+05,
                     0.322583103403269e+08,
                    -0.342416065095363e+03,
                    -0.899732529907377e+06,
                    -0.793892049821251e+08,
                     0.953193003217388e+02,
                     0.229784742345072e+04,
                     0.175336675322499e+06,
                     0.791214365222792e+07,
                     0.319933345844209e-04,
                    -0.659508863555767e+02,
                    -0.833426563212851e+06,
                     0.645734680583292e-01,
                    -0.382031020570813e+07,
                     0.406398848470079e-04,
                     0.310327498492008e+02,
                    -0.892996718483724e-03,
                     0.234604891591616e+03,
                     0.377515668966951e+04,
                     0.158646812591361e-01,
                     0.707906336241843e+00,
                     0.126016225146570e+02,
                     0.736143655772152e+00,
                     0.676544268999101e+00,
                    -0.178100588189137e+02,
                    -0.156531975531713e+00,
                     0.117707430048158e+02,
                     0.840143653860447e-01,
                    -0.186442467471949e+00,
                    -0.440170203949645e+02,
                     0.123290423502494e+07,
                    -0.240650039730845e-01,
                    -0.107077716660869e+07,
                     0.438319858566475e-01 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};


double IFC97_R3_V_pt_3d(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3d.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 3;
    
    // Numerical coefficients from Table A1.4.
    int I[] = { 0, -12, -12, -12, -12, -12, -12, -10, -10, -10, -10,
                   -10, -10, -10,  -8,  -8,  -8,  -8,  -6,  -6,  -5,
                    -5,  -5,  -5,  -4,  -4,  -4,  -3,  -3,  -2,  -2,
                    -1,  -1,  -1,   0,   0,   1,   1,   3   };
    
    int J[] = { 0,  4,  6,  7, 10, 12, 16,  0,  2,  4,  6,
                    8, 10, 14,  3,  7,  8, 10,  6,  8,  1,
                    2,  5,  7,  0,  1,  7,  2,  4,  0,  1,
                    0,  1,  5,  0,  2,  0,  6,  0   };
    
    double n[] = {   0.0,
                    -0.452484847171645e-09,
                     0.315210389538801e-04,
                    -0.214991352047545e-02,
                     0.508058874808345e+03,
                    -0.127123036845932e+08,
                     0.115371133120497e+13,
                    -0.197805728776273e-15,
                     0.241554806033972e-10,
                    -0.156481703640525e-05,
                     0.277211346836625e-02,
                    -0.203578994462286e+02,
                     0.144369489909053e+07,
                    -0.411254217946539e+11,
                     0.623449786243773e-05,
                    -0.221774281146038e+02,
                    -0.689315087933158e+05,
                    -0.195419525060713e+08,
                     0.316373510564015e+04,
                     0.224040754426988e+07,
                    -0.436701347922356e-05,
                    -0.404213852833996e-03,
                    -0.348153203414663e+03,
                    -0.385294213555289e+06,
                     0.135203700099403e-06,
                     0.134648383271089e-03,
                     0.125031835351736e+06,
                     0.968123678455841e-01,
                     0.225660517512438e+03,
                    -0.190102435341872e-03,
                    -0.299628410819229e-01,
                     0.500833915372121e-02,
                     0.387842482998411e+00,
                    -0.138535367777182e+04,
                     0.870745245971773e+00,
                     0.171946252068742e+01,
                    -0.326650121426383e-01,
                     0.498044171727877e+04,
                     0.551478022765087e-02 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};


double IFC97_R3_V_pt_3e(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3e.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 4;
    
    // Numerical coefficients from Table A1.5.
    int I[] = { 0, -12, -12, -10, -10, -10, -10, -10,  -8, -8, -8,
                    -6,  -5,  -4,  -4,  -3,  -3,  -3,  -2, -2, -2,
                    -2,  -1,   0,   0,   1,   1,   1,   2,  2  };
    
    int J[] = { 0, 14, 16,  3,  6, 10, 14, 16,  7,  8, 10,
                    6,  6,  2,  4,  2,  6,  7,  0,  1,  3,
                    4,  0,  0,  1,  0,  4,  6,  0,  2  };
    
    double n[] = {   0.0,
                     0.715815808404721e+09,
                    -0.114328360753449e+12,
                     0.376531002015720e-11,
                    -0.903983668691157e-04,
                     0.665695908836252e+06,
                     0.535364174960127e+10,
                     0.794977402335603e+11,
                     0.922230563421437e+02,
                    -0.142586073991215e+06,
                    -0.111796381424162e+07,
                     0.896121629640760e+04,
                    -0.669989239070491e+04,
                     0.451242538486834e-02,
                    -0.339731325977713e+02,
                    -0.120523111552278e+01,
                     0.475992667717124e+05,
                    -0.266627750390341e+06,
                    -0.153314954386524e-03,
                     0.305638404828265e+00,
                     0.123654999499486e+03,
                    -0.104390794213011e+04,
                    -0.157496516174308e-01,
                     0.685331118940253e+00,
                     0.178373462873903e+01,
                    -0.544674124878910e+00,
                     0.204529931318843e+04,
                    -0.228342359328752e+05,
                     0.413197481515899e+00,
                    -0.341931835910405e+02 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3f(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3f.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 5;
    
    // Numerical coefficients from Table A1.6.
    int I[] = { 0, 0,  0,  0,  0,  0,  0,  1,  1,  1,  1,
                   2,  2,  3,  3,  3,  4,  5,  5,  6,  7,
                   7, 10, 12, 12, 12, 14, 14, 14, 14, 14,
                  16, 16, 18, 18, 20, 20, 20, 22, 24, 24,
                  28, 32 };
    
    int J[] = { 0,  -3,  -2,  -1,   0,   1,   2,  -1,   1,   2,  3,
                     0,   1,  -5,  -2,   0,  -3,  -8,   1,  -6, -4,
                     1,  -6, -10,  -8,  -4, -12, -10,  -8,  -6, -4,
                   -10,  -8, -12, -10, -12, -10,  -6, -12, -12, -4,
                   -12, -12};
    
    double n[] = {   0.0,
                    -0.251756547792325e-07,
                     0.601307193668763e-05,
                    -0.100615977450049e-02,
                     0.999969140252192e+00,
                     0.214107759236486e+01,
                    -0.165175571959086e+02,
                    -0.141987303638727e-02,
                     0.269251915156554e+01,
                     0.349741815858722e+02,
                    -0.300208695771783e+02,
                    -0.131546288252539e+01,
                    -0.839091277286169e+01,
                     0.181545608337015e-09,
                    -0.591099206478909e-03,
                     0.152115067087106e+01,
                     0.252956470663225e-04,
                     0.100726265203786e-14,
                    -0.149774533860650e+01,
                    -0.793940970562969e-09,
                    -0.150290891264717e-03,
                     0.151205531275133e+01,
                     0.470942606221652e-05,
                     0.195049710391712e-12,
                    -0.911627886266077e-08,
                     0.604374640201265e-03,
                    -0.225132933900136e-15,
                     0.610916973582981e-11,
                    -0.303063908043404e-06,
                    -0.137796070798409e-04,
                    -0.919296736666106e-03,
                     0.639288223132545e-09,
                     0.753259479898699e-06,
                    -0.400321478682929e-12,
                     0.756140294351614e-08,
                    -0.912082054034891e-11,
                    -0.237612381140539e-07,
                     0.269586010591874e-04,
                    -0.732828135157839e-10,
                     0.241995578306660e-09,
                    -0.405735532730322e-03,
                     0.189424143498011e-09,
                    -0.486632965074563e-09 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3g(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3g.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 6;
    
    // Numerical coefficients from Table A1.7.
    int I[] = { 0, -12, -12, -12, -12, -12, -12, -10, -10, -10, -8,
                    -8,  -8,  -8,  -6,  -6,  -5,  -5,  -4, -3,  -2,
                    -2,  -2,  -2,  -1,  -1,  -1,   0,   0,  0,   1,
                     1,   1,   3,   5,   6,   8,  10,  10  };
    
    int J[] = { 0,  7, 12, 14, 18, 22, 24, 14, 20, 24,  7,
                    8, 10, 12,  8, 22,  7, 20, 22,  7,  3,
                    5, 14, 24,  2,  8, 18,  0,  1,  2,  0,
                    1,  3, 24, 22, 12,  3,  0,  6  };
    
    double n[] = {   0.0,
                     0.412209020652996e-04,
                    -0.114987238280587e+07,
                     0.948180885032080e+10,
                    -0.195788865718971e+18,
                     0.496250704871300e+25,
                    -0.105549884548496e+29,
                    -0.758642165988278e+12,
                    -0.922172769596101e+23,
                     0.725379072059348e+30,
                    -0.617718249205859e+02,
                     0.107555033344858e+05,
                    -0.379545802336487e+08,
                     0.228646846221831e+12,
                    -0.499741093010619e+07,
                    -0.280214310054101e+31,
                     0.104915406769586e+07,
                     0.613754229168619e+28,
                     0.802056715528378e+32,
                    -0.298617819828065e+08,
                    -0.910782540134681e+02,
                     0.135033227281565e+06,
                    -0.712949383408211e+19,
                    -0.104578785289542e+37,
                     0.304331584444093e+02,
                     0.593250797959445e+10,
                    -0.364174062110798e+28,
                     0.921791403532461e+00,
                    -0.337693609657471e+00,
                    -0.724644143758508e+02,
                    -0.110480239272601e+00,
                     0.536516031875059e+01,
                    -0.291441872156205e+04,
                     0.616338176535305e+40,
                    -0.120889175861180e+39,
                     0.818396024524612e+23,
                     0.940781944835829e+09,
                    -0.367279669545448e+05,
                    -0.837513931798655e+16 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3h(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3h.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 7;
    
    // Numerical coefficients from Table A1.8.
    int I[] = { 0, -12, -12, -10, -10, -10, -10, -10, -10, -8, -8,
                    -8,  -8,  -8,  -6,  -6,  -6,  -5,  -5, -5, -4,
                    -4,  -3,  -3,  -2,  -1,  -1,   0,   1,  1  };
    
    int J[] = { 0,  8, 12,  4,  6,  8, 10, 14, 16,  0,  1,
                    6,  7,  8,  4,  6,  8,  2,  3,  4,  2,
                    4,  1,  2,  0,  0,  2,  0,  0,  2  };
    
    double n[] = {   0.0,
                     0.561379678887577e-01,
                     0.774135421587083e+10,
                     0.111482975877938e-08,
                    -0.143987128208183e-02,
                     0.193696558764920e+04,
                    -0.605971823585005e+09,
                     0.171951568124337e+14,
                    -0.185461154985145e+17,
                     0.387851168078010e-16,
                    -0.395464327846105e-13,
                    -0.170875935679023e+03,
                    -0.212010620701220e+04,
                     0.177683337348191e+08,
                     0.110177443629575e+02,
                    -0.234396091693313e+06,
                    -0.656174421999594e+07,
                     0.156362212977396e-04,
                    -0.212946257021400e+01,
                     0.135249306374858e+02,
                     0.177189164145813e+00,
                     0.139499167345464e+04,
                    -0.703670932036388e-02,
                    -0.152011044389648e+00,
                     0.981916922991113e-04,
                     0.147199658618076e-02,
                     0.202618487025578e+02,
                     0.899345518944240e+00,
                    -0.211346402240858e+00,
                     0.249971752957491e+02 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3i(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3i.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 8;
    
    // Numerical coefficients from Table A1.9.
    int I[] = { 0, 0,  0,  0,  1,  1,  1,  1,  2,  3,  3,
                   4,  4,  4,  5,  5,  5,  7,  7,  8,  8,
                  10, 12, 12, 12, 14, 14, 14, 14, 18, 18,
                  18, 18, 18, 20, 20, 22, 24, 24, 32, 32,
                  36, 36  };
    
    int J[] = { 0,  0,   1, 10,  -4,  -2,  -1,   0,   0,  -5,   0,
                   -3,  -2, -1,  -6,  -1,  12,  -4,  -3,  -6,  10,
                   -8, -12, -6,  -4, -10,  -8,  -4,   5, -12, -10,
                   -8,  -6,  2, -12, -10, -12, -12,  -8, -10,  -5,
                  -10,  -8 };
    
    double n[] = {   0.0,
                     0.106905684359136e+01,
                    -0.148620857922333e+01,
                     0.259862256980408e+15,
                    -0.446352055678749e-11,
                    -0.566620757170032e-06,
                    -0.235302885736849e-02,
                    -0.269226321968839e+00,
                     0.922024992944392e+01,
                     0.357633505503772e-11,
                    -0.173942565562222e+02,
                     0.700681785556229e-05,
                    -0.267050351075768e-03,
                    -0.231779669675624e+01,
                    -0.753533046979752e-12,
                     0.481337131452891e+01,
                    -0.223286270422356e+22,
                    -0.118746004987383e-04,
                     0.646412934136496e-02,
                    -0.410588536330937e-09,
                     0.422739537057241e+20,
                     0.313698180473812e-12,
                     0.164395334345040e-23,
                    -0.339823323754373e-05,
                    -0.135268639905021e-01,
                    -0.723252514211625e-14,
                     0.184386437538366e-08,
                    -0.463959533752385e-01,
                    -0.992263100376750e+14,
                     0.688169154439335e-16,
                    -0.222620998452197e-10,
                    -0.540843018624083e-07,
                     0.345570606200257e-02,
                     0.422275800304086e+11,
                    -0.126974478770487e-14,
                     0.927237985153679e-09,
                     0.612670812016489e-13,
                    -0.722693924063497e-11,
                    -0.383669502636822e-03,
                     0.374684572410204e-03,
                    -0.931976897511086e+05,
                    -0.247690616026922e-01,
                     0.658110546759474e+02 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3j(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3j.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 9;
    
    // Numerical coefficients from Table A1.10.
    int I[] = { 0,  0,  0,  0,  1,  1,  1,  2,  2,  3,  4,
                    4,  5,  5,  5,  6, 10, 12, 12, 14, 14,
                   14, 16, 18, 20, 20, 24, 24, 28, 28 };
    
    int J[] = { 0, -1,   0,   1, -2,   -1,   1, -1,   1,  -2,  -2,
                    2,  -3,  -2,  0,    3,  -6, -8,  -3, -10,  -8,
                   -5, -10, -12, -12, -10, -12, -6, -12,  -5 };
    
    double n[] = {   0.0,
                    -0.111371317395540e-03,
                     0.100342892423685e+01,
                     0.530615581928979e+01,
                     0.179058760078792e-05,
                    -0.728541958464774e-03,
                    -0.187576133371704e+02,
                     0.199060874071849e-02,
                     0.243574755377290e+02,
                    -0.177040785499444e-03,
                    -0.259680385227130e-02,
                    -0.198704578406823e+03,
                     0.738627790224287e-04,
                    -0.236264692844138e-02,
                    -0.161023121314333e+01,
                     0.622322971786473e+04,
                    -0.960754116701669e-08,
                    -0.510572269720488e-10,
                     0.767373781404211e-02,
                     0.663855469485254e-14,
                    -0.717590735526745e-09,
                     0.146564542926508e-04,
                     0.309029474277013e-11,
                    -0.464216300971708e-15,
                    -0.390499637961161e-13,
                    -0.236716126781431e-09,
                     0.454652854268717e-11,
                    -0.422271787482497e-02,
                     0.283911742354706e-10,
                     0.270929002720228e+01 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3k(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3k.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 10;
    
    // Numerical coefficients from Table A1.11.
    int I[] = { 0, -2, -2, -1, -1,  0, 0, 0, 0, 0, 0,
                    0,  0,  0,  1,  1, 1, 1, 1, 2, 2,
                    2,  2,  2,  2,  5, 5, 5, 6, 6, 6,
                    6,  8, 10, 12   };
    
    int J[] = { 0, 10, 12,   -5,   6, -12, -6, -2,  -1,   0,  1,
                    2,  3,   14,  -3,  -2,  0,  1,   2,  -8, -6,
                   -3, -2,    0,   4, -12, -6, -3, -12, -10, -8,
                   -5, -12, -12, -10   };
    
    double n[] = {   0.0,
                    -0.401215699576099e+09,
                     0.484501478318406e+11,
                     0.394721471363678e-14,
                     0.372629967374147e+05,
                    -0.369794374168666e-29,
                    -0.380436407012452e-14,
                     0.475361629970233e-06,
                    -0.879148916140706e-03,
                     0.844317863844331e+00,
                     0.122433162656600e+02,
                    -0.104529634830279e+03,
                     0.589702771277429e+03,
                    -0.291026851164444e+14,
                     0.170343072841850e-05,
                    -0.277617606975748e-03,
                    -0.344709605486686e+01,
                     0.221333862447095e+02,
                    -0.194646110037079e+03,
                     0.808354639772825e-15,
                    -0.180845209145470e-10,
                    -0.696664158132412e-05,
                    -0.181057560300994e-02,
                     0.255830298579027e+01,
                     0.328913873658481e+04,
                    -0.173270241249904e-18,
                    -0.661876792558034e-06,
                    -0.395688923421250e-02,
                     0.604203299819132e-17,
                    -0.400879935920517e-13,
                     0.160751107464958e-08,
                     0.383719409025556e-04,
                    -0.649565446702457e-14,
                    -0.149095328506000e-11,
                     0.541449377329581e-08 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3l(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3l.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 11;
    
    // Numerical coefficients from Table A1.12.
    int I[] = { 0, -12, -12, -12, -12, -12, -10, -10, -8, -8, -8,
                    -8,  -8,  -8,  -8,  -6,  -5,  -5, -4, -4, -3,
                    -3,  -3,  -3,  -2,  -2,  -2,  -1, -1, -1,  0,
                     0,   0,   0,   1,   1,   2,   4,  5,  5,  6,
                    10,  10,  14  };
    
    int J[] = { 0, 14, 16, 18, 20, 22, 14, 24,  6, 10, 12,
                   14, 18, 24, 36,  8,  4,  5,  7, 16,  1,
                    3, 18, 20,  2,  3, 10,  0,  1,  3,  0,
                    1,  2, 12,  0, 16,  1,  0,  0,  1, 14,
                    4, 12, 10 };
    
    double n[] = {   0.0,
                     0.260702058647537e+10,
                    -0.188277213604704e+15,
                     0.554923870289667e+19,
                    -0.758966946387758e+23,
                     0.413865186848908e+27,
                    -0.815038000738060e+12,
                    -0.381458260489955e+33,
                    -0.123239564600519e-01,
                     0.226095631437174e+08,
                    -0.495017809506720e+12,
                     0.529482996422863e+16,
                    -0.444359478746295e+23,
                     0.521635864527315e+35,
                    -0.487095672740742e+55,
                    -0.714430209937547e+06,
                     0.127868634615495e+00,
                    -0.100752127917598e+02,
                     0.777451437960990e+07,
                    -0.108105480796471e+25,
                    -0.357578581169659e-05,
                    -0.212857169423484e+01,
                     0.270706111085238e+30,
                    -0.695953622348829e+33,
                     0.110609027472280e+00,
                     0.721559163361354e+02,
                    -0.306367307532219e+15,
                     0.265839618885530e-04,
                     0.253392392889754e-01,
                    -0.214443041836579e+03,
                     0.937846601489667e+00,
                     0.223184043101700e+01,
                     0.338401222509191e+02,
                     0.494237237179718e+21,
                    -0.198068404154428e+00,
                    -0.141415349881140e+31,
                    -0.993862421613651e+02,
                     0.125070534142731e+03,
                    -0.996473529004439e+03,
                     0.473137909872765e+05,
                     0.116662121219322e+33,
                    -0.315874976271533e+16,
                    -0.445703369196945e+33,
                     0.642794932373694e+33 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3m(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3m.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 12;
    
    // Numerical coefficients from Table A1.13.
    int I[] = { 0,  0, 3,  8, 20, 1, 3, 4, 5,  1,  6,
                    2, 4, 14,  2, 5, 3, 0, 1,  1,  1,
                   28, 2, 16,  0, 5, 0, 3, 4, 12, 16,
                    1, 8, 14,  0, 2, 3, 4, 8, 14, 24 };
    
    int J[] = { 0,  0,  0,  0,  2,  5,  5,  5,  5,  6,  6,
                    7,  8,  8, 10, 10, 12, 14, 14, 18, 20,
                   20, 22, 22, 24, 24, 28, 28, 28, 28, 28,
                   32, 32, 32, 36, 36, 36, 36, 36, 36, 36  };
    
    double n[] = {   0.0,
                     0.811384363481847e+00,
                    -0.568199310990094e+04,
                    -0.178657198172556e+11,
                     0.795537657613427e+32,
                    -0.814568209346872e+05,
                    -0.659774567602874e+08,
                    -0.152861148659302e+11,
                    -0.560165667510446e+12,
                     0.458384828593949e+06,
                    -0.385754000383848e+14,
                     0.453735800004273e+08,
                     0.939454935735563e+12,
                     0.266572856432938e+28,
                    -0.547578313899097e+10,
                     0.200725701112386e+15,
                     0.185007245563239e+13,
                     0.185135446828337e+09,
                    -0.170451090076385e+12,
                     0.157890366037614e+15,
                    -0.202530509748774e+16,
                     0.368193926183570e+60,
                     0.170215539458936e+18,
                     0.639234909918741e+42,
                    -0.821698160721956e+15,
                    -0.795260241872306e+24,
                     0.233415869478510e+18,
                    -0.600079934586803e+23,
                     0.594584382273384e+25,
                     0.189461279349492e+40,
                    -0.810093428842645e+46,
                     0.188813911076809e+22,
                     0.111052244098768e+36,
                     0.291133958602503e+46,
                    -0.329421923951460e+22,
                    -0.137570282536696e+26,
                     0.181508996303902e+28,
                    -0.346865122768353e+30,
                    -0.211961148774260e+38,
                    -0.128617899887675e+49,
                     0.479817895699239e+65 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3n(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3n.
     
     Equations are based on Equation (5) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 13;
    
    // Numerical coefficients from Table A1.14.
    int I[] = { 0, 0, 3, 4, 6,  7, 10, 12, 14, 18, 0,
                   3, 5, 6, 8, 12,  0,  3,  7, 12, 2,
                   3, 4, 2, 4,  7,  4,  3,  5,  6, 0,
                   0, 3, 1, 0,  1,  0,  1,  0,  1     };
    
    int J[] = { 0, -12, -12, -12, -12, -12, -12, -12, -12, -12, -10,
                   -10, -10, -10, -10, -10,  -8,  -8,  -8,  -8,  -6,
                    -6,  -6,  -5,  -5,  -5,  -4,  -3,  -3,  -3,  -2,
                    -1,  -1,   0,   1,   1,   2,   4,   5,   6   };
    
    double n[] = {   0.0,
                     0.280967799943151e-38,
                     0.614869006573609e-30,
                     0.582238667048942e-27,
                     0.390628369238462e-22,
                     0.821445758255119e-20,
                     0.402137961842776e-14,
                     0.651718171878301e-12,
                    -0.211773355803058e-07,
                     0.264953354380072e-02,
                    -0.135031446451331e-31,
                    -0.607246643970893e-23,
                    -0.402352115234494e-18,
                    -0.744938506925544e-16,
                     0.189917206526237e-12,
                     0.364975183508473e-05,
                     0.177274872361946e-25,
                    -0.334952758812999e-18,
                    -0.421537726098389e-08,
                    -0.391048167929649e-01,
                     0.541276911564176e-13,
                     0.705412100773699e-11,
                     0.258585887897486e-08,
                    -0.493111362030162e-10,
                    -0.158649699894543e-05,
                    -0.525037427886100e+00,
                     0.220019901729615e-02,
                    -0.643064132636925e-02,
                     0.629154149015048e+02,
                     0.135147318617061e+03,
                     0.240560808321713e-06,
                    -0.890763306701305e-03,
                    -0.440209599407714e+04,
                    -0.302807107747776e+03,
                     0.159158748314599e+04,
                     0.232534272709876e+06,
                    -0.792681207132600e+06,
                    -0.869871364662769e+11,
                     0.354542769185671e+12,
                     0.400849240129329e+15 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
//    double c = Coefficients[SubRegion][6];  Not Used for 3n
//    double d = Coefficients[SubRegion][7];  Not Used for 3n
//    double e = Coefficients[SubRegion][8];  Not Used for 3n
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(Pi-a,I[i]) * pow(Theta-b,J[i]);
    };
    
    return exp(hold) * vstar;
};

double IFC97_R3_V_pt_3o(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3o.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 14;
    
    // Numerical coefficients from Table A1.15.
    int I[] = { 0, 0,  0,  0,  2, 3, 4, 4,  4,  4,  4,
                   5,  5,  6,  7, 8, 8, 8, 10, 10, 14,
                  14, 20, 20, 24  };
    
    int J[] = { 0, -12,  -4,  -1,  -1, -10, -12, -8,  -5, -4,  -1,
                    -4,  -3,  -8, -12, -10,  -8, -4, -12, -8, -12,
                    -8, -12, -10, -12 };
    
    double n[] = {   0.0,
                     0.128746023979718e-34,
                    -0.735234770382342e-11,
                     0.289078692149150e-02,
                     0.244482731907223e+00,
                     0.141733492030985e-23,
                    -0.354533853059476e-28,
                    -0.594539202901431e-17,
                    -0.585188401782779e-08,
                     0.201377325411803e-05,
                     0.138647388209306e+01,
                    -0.173959365084772e-04,
                     0.137680878349369e-02,
                     0.814897605805513e-14,
                     0.425596631351839e-25,
                    -0.387449113787755e-17,
                     0.139814747930240e-12,
                    -0.171849638951521e-02,
                     0.641890529513296e-21,
                     0.118960578072018e-10,
                    -0.155282762571611e-17,
                     0.233907907347507e-07,
                    -0.174093247766213e-12,
                     0.377682649089149e-08,
                    -0.516720236575302e-10 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3p(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3p.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 15;
    
    // Numerical coefficients from Table A1.16.
    int I[] = { 0,  0,  0,  0,  0,  1,  2,  3,  3,  4,  6,
                    7,  7,  8, 10, 12, 12, 12, 14, 14, 14,
                   16, 18, 20, 22, 24, 24, 36  };
    
    int J[] = { 0, -1,  0,   1,   2,   1, -1,  -3,   0, -2, -2,
                   -5, -4,  -2,  -3, -12, -6,  -5, -10, -8, -3,
                   -8, -8, -10, -10, -12, -8, -12 };
    
    double n[] = {   0.0,
                    -0.982825342010366e-04,
                     0.105145700850612e+01,
                     0.116033094095084e+03,
                     0.324664750281543e+04,
                    -0.123592348610137e+04,
                    -0.561403450013495e-01,
                     0.856677401640869e-07,
                     0.236313425393924e+03,
                     0.972503292350109e-02,
                    -0.103001994531927e+01,
                    -0.149653706199162e-08,
                    -0.215743778861592e-04,
                    -0.834452198291445e+01,
                     0.586602660564988e+00,
                     0.343480022104968e-25,
                     0.816256095947021e-05,
                     0.294985697916798e-02,
                     0.711730466276584e-16,
                     0.400954763806941e-09,
                     0.107766027032853e+02,
                    -0.409449599138182e-06,
                    -0.729121307758902e-05,
                     0.677107970938909e-08,
                     0.602745973022975e-07,
                    -0.382323011855257e-10,
                     0.179946628317437e-02,
                    -0.345042834640005e-03 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3q(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3q.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 16;
    
    // Numerical coefficients from Table A1.17.
    int I[] = { 0, -12, -12, -10, -10, -10, -10, -8, -6, -5, -5,
                    -4,  -4,  -3,  -2,  -2,  -2, -2, -1, -1, -1,
                     0,   1,   1,   1  };
    
    int J[] = { 0, 10, 12,  6,  7,  8, 10, 8, 6, 2, 5,
                    3,  4,  3,  0,  1,  2, 4, 0, 1, 2,
                    0,  0,  1,  3 };
    
    double n[] = {   0.0,
                    -0.820433843259950e+05,
                     0.473271518461586e+11,
                    -0.805950021005413e-01,
                     0.328600025435980e+02,
                    -0.356617029982490e+04,
                    -0.172985781433335e+10,
                     0.351769232729192e+08,
                    -0.775489259985144e+06,
                     0.710346691966018e-04,
                     0.993499883820274e+05,
                    -0.642094171904570e+00,
                    -0.612842816820083e+04,
                     0.232808472983776e+03,
                    -0.142808220416837e-04,
                    -0.643596060678456e-02,
                    -0.428577227475614e+01,
                     0.225689939161918e+04,
                     0.100355651721510e-02,
                     0.333491455143516e+00,
                     0.109697576888873e+01,
                     0.961917379376452e+00,
                    -0.838165632204598e-01,
                     0.247795908411492e+01,
                    -0.319114969006533e+04 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3r(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3r.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 17;
    
    // Numerical coefficients from Table A1.18.
    int I[] = { 0, -8, -8, -3, -3, -3, -3, -3,  0,  0,  0,
                    0,  3,  3,  8,  8,  8,  8, 10, 10, 10,
                   10, 10, 10, 10, 10, 12, 14 };
    
    int J[] = { 0, 6, 14, -3,   3,   4,   5,   8,  -1,   0,  1,
                   5, -6, -2, -12, -10,  -8,  -5, -12, -10, -8,
                  -6, -5, -4,  -3,  -2, -12, -12  };
    
    double n[] = {   0.0,
                     0.144165955660863e-02,
                    -0.701438599628258e+13,
                    -0.830946716459219e-16,
                     0.261975135368109e+00,
                     0.393097214706245e+03,
                    -0.104334030654021e+05,
                     0.490112654154211e+09,
                    -0.147104222772069e-03,
                     0.103602748043408e+01,
                     0.305308890065089e+01,
                    -0.399745276971264e+07,
                     0.569233719593750e-11,
                    -0.464923504407778e-01,
                    -0.535400396512906e-17,
                     0.399988795693162e-12,
                    -0.536479560201811e-06,
                     0.159536722411202e-01,
                     0.270303248860217e-14,
                     0.244247453858506e-07,
                    -0.983430636716454e-05,
                     0.663513144224454e-01,
                    -0.993456957845006e+01,
                     0.546491323528491e+03,
                    -0.143365406393758e+05,
                     0.150764974125511e+06,
                    -0.337209709340105e-09,
                     0.377501980025469e-08 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3s(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3s.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 18;
    
    // Numerical coefficients from Table A1.19.
    int I[] = { 0, -12, -12, -10, -8, -6, -5, -5, -4, -4, -3,
                    -3,  -2,  -1, -1, -1,  0,  0,  0,  0,  1,
                     1,   3,   3,  3,  4,  4,  4,  5, 14  };
    
    int J[] = { 0, 20, 24, 22, 14, 36,  8, 16,  6, 32,  3,
                    8,  4,  1,  2,  3,  0,  1,  4, 28,  0,
                   32,  0,  1,  2,  3, 18, 24,  4, 24  };
    
    double n[] = {   0.0,
                    -0.532466612140254e+23,
                     0.100415480000824e+32,
                    -0.191540001821367e+30,
                     0.105618377808847e+17,
                     0.202281884477061e+59,
                     0.884585472596134e+08,
                     0.166540181638363e+23,
                    -0.313563197669111e+06,
                    -0.185662327545324e+54,
                    -0.624942093918942e-01,
                    -0.504160724132590e+10,
                     0.187514491833092e+05,
                     0.121399979993217e-02,
                     0.188317043049455e+01,
                    -0.167073503962060e+04,
                     0.965961650599775e+00,
                     0.294885696802488e+01,
                    -0.653915627346115e+05,
                     0.604012200163444e+50,
                    -0.198339358557937e+00,
                    -0.175984090163501e+58,
                     0.356314881403987e+01,
                    -0.575991255144384e+03,
                     0.456213415338071e+05,
                    -0.109174044987829e+08,
                     0.437796099975134e+34,
                    -0.616552611135792e+46,
                     0.193568768917797e+10,
                     0.950898170425042e+54 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3t(double P, double T) {
    /* Provides the backward equation for determining SV based on P,T in Region 3t.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 19;
    
    // Numerical coefficients from Table A1.20.
    int I[] = { 0,  0,  0,  0,  0,  1,  1,  2,  2,  2,  3,
                    3,  4,  4,  7,  7,  7,  7,  7, 10, 10,
                   10, 10, 10, 18, 20, 22, 22, 24, 28, 32,
                   32, 32, 36  };
    
    int J[] = { 0,  0,  1,  4, 12,  0, 10,  0,  6, 14,  3,
                    8,  0, 10,  3,  4,  7, 20, 36, 10, 12,
                   14, 16, 22, 18, 32, 22, 36, 24, 28, 22,
                   32, 36, 36 };
    
    double n[] = {   0.0,
                     0.155287249586268e+01,
                     0.664235115009031e+01,
                    -0.289366236727210e+04,
                    -0.385923202309848e+13,
                    -0.291002915783761e+01,
                    -0.829088246858083e+12,
                     0.176814899675218e+01,
                    -0.534686695713469e+09,
                     0.160464608687834e+18,
                     0.196435366560186e+06,
                     0.156637427541729e+13,
                    -0.178154560260006e+01,
                    -0.229746237623692e+16,
                     0.385659001648006e+08,
                     0.110554446790543e+10,
                    -0.677073830687349e+14,
                    -0.327910592086523e+31,
                    -0.341552040860644e+51,
                    -0.527251339709047e+21,
                     0.245375640937055e+24,
                    -0.168776617209269e+27,
                     0.358958955867578e+29,
                    -0.656475280339411e+36,
                     0.355286045512301e+39,
                     0.569021454413270e+58,
                    -0.700584546433113e+48,
                    -0.705772623326374e+65,
                     0.166861176200148e+53,
                    -0.300475129680486e+61,
                    -0.668481295196808e+51,
                     0.428432338620678e+69,
                    -0.444227367758304e+72,
                    -0.281396013562745e+77  };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

// Auxiliary Equations near the critical region based on Equation (4)
double IFC97_R3_V_pt_3u(double P, double T) {
    /* Provides the auxiliary equation for determining SV based on P,T in Region 3u.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 20;
    
    // Numerical coefficients from Table A2.1.
    int I[] = { 0, -12, -10, -10, -10, -8, -8, -8, -6, -6, -5,
                    -5,  -5,  -3,  -1, -1, -1, -1,  0,  0,  1,
                     2,   2,   3,   5,  5,  5,  6,  6,  8,  8,
                    10,  12,  12,  12, 14, 14, 14, 14  };
    
    int J[] = { 0, 14,  10, 12, 14,  10,  12, 14,  8, 12,  4,
                    8,  12,  2, -1,   1,  12, 14, -3,  1, -2,
                    5,  10, -5, -4,   2,   3, -5,  2, -8,  8,
                   -4, -12, -4,  4, -12, -10, -6,  6   };
    
    double n[] = {   0.0,
                     0.122088349258355e+18,
                     0.104216468608488e+10,
                    -0.882666931564652e+16,
                     0.259929510849499e+20,
                     0.222612779142211e+15,
                    -0.878473585050085e+18,
                    -0.314432577551552e+22,
                    -0.216934916996285e+13,
                     0.159079648196849e+21,
                    -0.339567617303423e+03,
                     0.884387651337836e+13,
                    -0.843405926846418e+21,
                     0.114178193518022e+02,
                    -0.122708229235641e-03,
                    -0.106201671767107e+03,
                     0.903443213959313e+25,
                    -0.693996270370852e+28,
                     0.648916718965575e-08,
                     0.718957567127851e+04,
                     0.105581745346187e-02,
                    -0.651903203602581e+15,
                    -0.160116813274676e+25,
                    -0.510254294237837e-08,
                    -0.152355388953402e+00,
                     0.677143292290144e+12,
                     0.276378438378930e+15,
                     0.116862983141686e-01,
                    -0.301426947980171e+14,
                     0.169719813884840e-07,
                     0.104674840020929e+27,
                    -0.108016904560140e+05,
                    -0.990623601934295e-12,
                     0.536116483602738e+07,
                     0.226145963747881e+22,
                    -0.488731565776210e-09,
                     0.151001548880670e-04,
                    -0.227700464643920e+05,
                    -0.781754507698846e+28 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3v(double P, double T) {
    /* Provides the auxiliary equation for determining SV based on P,T in Region 3v.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 21;
    
    // Numerical coefficients from Table A2.2.
    int I[] = { 0, -10, -8, -6, -6, -6, -6, -6, -6, -5, -5,
                    -5, -5, -5, -5, -4, -4, -4, -4, -3, -3,
                    -3, -2, -2, -1, -1,  0,  0,  0,  1,  1,
                     3,  4,  4,  4,  5,  8, 10, 12, 14 };
    
    int J[] = { 0, -8, -12, -12, -3,   5,   6,  8,  10,   1,  2,
                    6,   8,  10, 14, -12, -10, -6,  10,  -3,  10,
                    12,  2,   4, -2,   0,  -2,  6,  10, -12, -10,
                     3, -6,   3, 10,   2, -12, -2,  -3,   1  };
    
    double n[] = {   0.0,
                    -0.415652812061591e-54,
                     0.177441742924043e-60,
                    -0.357078668203377e-54,
                     0.359252213604114e-25,
                    -0.259123736380269e+02,
                     0.594619766193460e+05,
                    -0.624184007103158e+11,
                     0.313080299915944e+17,
                     0.105006446192036e-08,
                    -0.192824336984852e-05,
                     0.654144373749937e+06,
                     0.513117462865044e+13,
                    -0.697595750347391e+19,
                    -0.103977184454767e+29,
                     0.119563135540666e-47,
                    -0.436677034051655e-41,
                     0.926990036530639e-29,
                     0.587793105620748e+21,
                     0.280375725094731e-17,
                    -0.192359972440634e+23,
                     0.742705723302738e+27,
                    -0.517429682450605e+02,
                     0.820612048645469e+07,
                    -0.188214882341448e-08,
                     0.184587261114837e-01,
                    -0.135830407782663e-05,
                    -0.723681885626348e+17,
                    -0.223449194054124e+27,
                    -0.111526741826431e-34,
                     0.276032601145151e-28,
                     0.134856491567853e+15,
                     0.652440293345860e-09,
                     0.510655119774360e+17,
                    -0.468138358908732e+32,
                    -0.760667491183279e+16,
                    -0.417247986986821e-18,
                     0.312545677756104e+14,
                    -0.100375333864186e+15,
                     0.247761392329058e+27 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3w(double P, double T) {
    /* Provides the auxiliary equation for determining SV based on P,T in Region 3w.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 22;
    
    // Numerical coefficients from Table A2.3.
    int I[] = { 0, -12, -12, -10, -10, -8, -8, -8, -6, -6, -6,
                    -6,  -5,  -4,  -4, -3, -3, -2, -2, -1, -1,
                    -1,   0,   0,   1,  2,  2,  3,  3,  5,  5,
                     5,   8,   8,  10, 10 };
    
    int J[] = { 0,  8,  14,  -1,   8,   6, 8,  14, -4, -3,   2,
                    8, -10,  -1,   3, -10, 3,   1,  2, -8,  -4,
                    1, -12,   1,  -1,  -1, 2, -12, -5, -10, -8,
                   -6, -12, -10, -12,  -8   };
    
    double n[] = {   0.0,
                    -0.586219133817016e-07,
                    -0.894460355005526e+11,
                     0.531168037519774e-30,
                     0.109892402329239e+00,
                    -0.575368389425212e-01,
                     0.228276853990249e+05,
                    -0.158548609655002e+19,
                     0.329865748576503e-27,
                    -0.634987981190669e-24,
                     0.615762068640611e-08,
                    -0.961109240985747e+08,
                    -0.406274286652625e-44,
                    -0.471103725498077e-12,
                     0.725937724828145e+00,
                     0.187768525763682e-38,
                    -0.103308436323771e+04,
                    -0.662552816342168e-01,
                     0.579514041765710e+03,
                     0.237416732616644e-26,
                     0.271700235739893e-14,
                    -0.907886213483600e+02,
                    -0.171242509570207e-36,
                     0.156792067854621e+03,
                     0.923261357901470e+00,
                    -0.597865988422577e+01,
                     0.321988767636389e+07,
                    -0.399441390042203e-29,
                     0.493429086046981e-07,
                     0.812036983370565e-19,
                    -0.207610284654137e-11,
                    -0.340821291419719e-06,
                     0.542000573372233e-17,
                    -0.856711586510214e-12,
                     0.266170454405981e-13,
                     0.858133791857099e-05 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3x(double P, double T) {
    /* Provides the auxiliary equation for determining SV based on P,T in Region 3x.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 23;
    
    // Numerical coefficients from Table A2.4.
    int I[] = { 0, -8, -6, -5, -4, -4, -4, -3, -3, -1,  0,
                    0,  0,  1,  1,  2,  3,  3,  3,  4,  5,
                    5,  5,  6,  8,  8,  8,  8, 10, 12, 12,
                   12, 12, 14, 14, 14, 14 };
    
    int J[] = { 0, 14, 10,  10,   1,  2,  14, -2, 12,   5,  0,
                    4, 10, -10,  -1,  6, -12,  0,  8,   3, -6,
                   -2,  1,   1,  -6, -3,   1,  8, -8, -10, -8,
                   -5, -4, -12, -10, -8,  -6  };
    
    double n[] = {   0.0,
                     0.377373741298151e+19,
                    -0.507100883722913e+13,
                    -0.103363225598860e+16,
                     0.184790814320773e-05,
                    -0.924729378390945e-03,
                    -0.425999562292738e+24,
                    -0.462307771873973e-12,
                     0.107319065855767e+22,
                     0.648662492280682e+11,
                     0.244200600688281e+01,
                    -0.851535733484258e+10,
                     0.169894481433592e+22,
                     0.215780222509020e-26,
                    -0.320850551367334e+00,
                    -0.382642448458610e+17,
                    -0.275386077674421e-28,
                    -0.563199253391666e+06,
                    -0.326068646279314e+21,
                     0.397949001553184e+14,
                     0.100824008584757e-06,
                     0.162234569738433e+05,
                    -0.432355225319745e+11,
                    -0.592874245598610e+12,
                     0.133061647281106e+01,
                     0.157338197797544e+07,
                     0.258189614270853e+14,
                     0.262413209706358e+25,
                    -0.920011937431142e-01,
                     0.220213765905426e-02,
                    -0.110433759109547e+02,
                     0.847004870612087e+07,
                    -0.592910695762536e+09,
                    -0.183027173269660e-04,
                     0.181339603516302e+00,
                    -0.119228759669889e+04,
                     0.430867658061468e+07 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3y(double P, double T) {
    /* Provides the auxiliary equation for determining SV based on P,T in Region 3y.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 24;
    
    // Numerical coefficients from Table A2.5.
    int I[] = { 0, 0, 0, 0, 0, 1, 2, 2, 2,  2,  3,
                   3, 3, 4, 4, 5, 5, 8, 8, 10, 12 };
    
    int J[] = { 0, -3, 1,  5, 8,  8, -4, -1,  4,  5, -8,
                    4, 8, -6, 6, -2,  1, -8, -2, -5, -8 };
    
    double n[] = {   0.0,
                    -0.525597995024633e-09,
                     0.583441305228407e+04,
                    -0.134778968457925e+17,
                     0.118973500934212e+26,
                    -0.159096490904708e+27,
                    -0.315839902302021e-06,
                     0.496212197158239e+03,
                     0.327777227273171e+19,
                    -0.527114657850696e+22,
                     0.210017506281863e-16,
                     0.705106224399834e+21,
                    -0.266713136106469e+31,
                    -0.145370512554562e-07,
                     0.149333917053130e+28,
                    -0.149795620287641e+08,
                    -0.381881906271100e+16,
                     0.724660165585797e-04,
                    -0.937808169550193e+14,
                     0.514411468376383e+10,
                    -0.828198594040141e+05 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

double IFC97_R3_V_pt_3z(double P, double T) {
    /* Provides the auxiliary equation for determining SV based on P,T in Region 3z.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    // Define Region Number for Coefficient evaluation
    int SubRegion = 25;
    
    // Numerical coefficients from Table A2.6.
    int I[] = { 0, -8, -6, -5, -5, -4, -4, -4, -3, -3, -3,
                   -2, -1,  0,  1,  2,  3,  3,  6,  6,  6,
                    6,  8,  8 };
    
    int J[] = { 0,  3,  6,  6, 8, 5,  6,  8, -2,  5,  6,
                    2, -6,  3, 1, 6, -6, -2, -6, -5, -4,
                   -1, -8, -4  };
    
    double n[] = {   0.0,
                     0.244007892290650e-10,
                    -0.463057430331242e+07,
                     0.728803274777712e+10,
                     0.327776302858856e+16,
                    -0.110598170118409e+10,
                    -0.323899915729957e+13,
                     0.923814007023245e+16,
                     0.842250080413712e-12,
                     0.663221436245506e+12,
                    -0.167170186672139e+15,
                     0.253749358701391e+04,
                    -0.819731559610523e-20,
                     0.328380587890663e+12,
                    -0.625004791171543e+08,
                     0.803197957462023e+21,
                    -0.204397011338353e-10,
                    -0.378391047055938e+04,
                     0.972876545938620e-02,
                     0.154355721681459e+02,
                    -0.373962862928643e+04,
                    -0.682859011374572e+11,
                    -0.248488015614543e-03,
                     0.394536049497068e+07 };
    
    // Reference Values
    double vstar = Coefficients[SubRegion][0];  // Units of m^3/kg
    double Pstar = Coefficients[SubRegion][1];  // Units of MPa
    double Tstar = Coefficients[SubRegion][2];  // Units of K
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Theta = T/Tstar;
    
    // Coefficient Values from Matrix above
    double a = Coefficients[SubRegion][4];
    double b = Coefficients[SubRegion][5];
    double c = Coefficients[SubRegion][6];
    double d = Coefficients[SubRegion][7];
    double e = Coefficients[SubRegion][8];
    
    // Solution summer variable
    double hold = 0.0;
    
    // Iterations End Value
    int iEnd = (int)Coefficients[SubRegion][3] + 1;
    
    for (int i=1; i<iEnd; i++) {
        hold += n[i] * pow(pow(Pi-a,c),I[i]) * pow(pow(Theta-b,d),J[i]);
    };
    
    return pow(hold,e) * vstar;
};

// User interface function.  Calls appropriate sub-region function
double IFC97_R3_V_pt(int SubRegion, double P, double T) {
    /* Provides the backward and auxliary equation for determining V based on P,T in Region 3.
     
     Equations are based on Equation (4) and (5) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator.
                     0=3a,
                     1=3b,
                     2=3c,
                     3=3d,
                     4=3e,
                     5=3f,
                     6=3g,
                     7=3h,
                     8=3i,
                     9=3j,
                     10=3k,
                     11=3l,
                     12=3m,
                     13=3n,
                     14=3o,
                     15=3p,
                     16=3q,
                     17=3r,
                     18=3s,
                     19=3t,
                     20=3u,
                     21=3v,
                     22=3w,
                     23=3x,
                     24=3y,
                     25=3z
        P   = Pressure, MPa
        T   = Temperature, K
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    switch ( SubRegion) {
        case 0: // Utilize Sub-region 3a
            return IFC97_R3_V_pt_3a(P,T);
            
        case 1: // Utilize Sub-region 3b
            return IFC97_R3_V_pt_3b(P,T);
            
        case 2: // Utilize Sub-region 3c
            return IFC97_R3_V_pt_3c(P,T);
            
        case 3: // Utilize Sub-region 3d
            return IFC97_R3_V_pt_3d(P,T);
            
        case 4: // Utilize Sub-region 3e
            return IFC97_R3_V_pt_3e(P,T);
            
        case 5: // Utilize Sub-region 3f
            return IFC97_R3_V_pt_3f(P,T);
            
        case 6: // Utilize Sub-region 3g
            return IFC97_R3_V_pt_3g(P,T);
            
        case 7: // Utilize Sub-region 3h
            return IFC97_R3_V_pt_3h(P,T);
            
        case 8: // Utilize Sub-region 3i
            return IFC97_R3_V_pt_3i(P,T);
            
        case 9: // Utilize Sub-region 3j
            return IFC97_R3_V_pt_3j(P,T);
            
        case 10: // Utilize Sub-region 3k
            return IFC97_R3_V_pt_3k(P,T);
            
        case 11: // Utilize Sub-region 3l
            return IFC97_R3_V_pt_3l(P,T);
            
        case 12: // Utilize Sub-region 3m
            return IFC97_R3_V_pt_3m(P,T);
            
        case 13: // Utilize Sub-region 3n
            return IFC97_R3_V_pt_3n(P,T);
            
        case 14: // Utilize Sub-region 3o
            return IFC97_R3_V_pt_3o(P,T);
            
        case 15: // Utilize Sub-region 3p
            return IFC97_R3_V_pt_3p(P,T);
            
        case 16: // Utilize Sub-region 3q
            return IFC97_R3_V_pt_3q(P,T);
            
        case 17: // Utilize Sub-region 3r
            return IFC97_R3_V_pt_3r(P,T);
            
        case 18: // Utilize Sub-region 3s
            return IFC97_R3_V_pt_3s(P,T);
            
        case 19: // Utilize Sub-region 3t
            return IFC97_R3_V_pt_3t(P,T);
            
        case 20: // Utilize Sub-region 3u
            return IFC97_R3_V_pt_3u(P,T);
            
        case 21: // Utilize Sub-region 3v
            return IFC97_R3_V_pt_3v(P,T);
            
        case 22: // Utilize Sub-region 3w
            return IFC97_R3_V_pt_3w(P,T);
            
        case 23: // Utilize Sub-region 3x
            return IFC97_R3_V_pt_3x(P,T);
            
        case 24: // Utilize Sub-region 3y
            return IFC97_R3_V_pt_3y(P,T);
            
        case 25: // Utilize Sub-region 3z
            return IFC97_R3_V_pt_3z(P,T);
                
        default:
            return -10000.0; // Incorrect sub-region input
    };
    
    return -20000.0;  // Unknown error
};
