//
//  IFC97_R3_Backward_Vps.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 07/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Backward_Vps.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Supplementary Release on Backward Equations for the Functions T(P,h), v(P,h) and T(P,s), v(P,s) for Region 3 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properites of Water and Steam. Kyoto, Japan, September 2004
 */

// Sub-region 3a specific Equation (8) from above referenced IAPWS document
double IFC97_R3_V_ps_3a(double P, double s) {
    /* Provides the backward equation for determining SV based on P,s in Region 3a.
     
     Equations are based on Equation (8) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 13.
    int I[] = { 0, -12, -12, -12, -10, -10, -10, -10, -8, -8, -8,
                    -8,  -6,  -5,  -4,  -3,  -3,  -2, -2, -1, -1,
                     0,   0,   0,   1,   2,   4,   5,  6 };
    
    int J[] = { 0, 10, 12, 14,  4,  8, 10, 20,  5,  6, 14,
                   16, 28,  1,  5,  2,  4,  3,  8,  1,  2,
                    0,  1,  3,  0,  0,  2,  2,  0   };
    
    double n[] = {   0.0,
                     0.795544074093975e+02,
                    -0.238261242984590e+04,
                     0.176813100617787e+05,
                    -0.110524727080379e-02,
                    -0.153213833655326e+02,
                     0.297544599376982e+03,
                    -0.350315206871242e+08,
                     0.277513761062119e+00,
                    -0.523964271036888e+00,
                    -0.148011182995403e+06,
                     0.160014899374266e+07,
                     0.170802322663427e+13,
                     0.246866996006494e-03,
                     0.165326084797980e+01,
                    -0.118008384666987e+00,
                     0.253798642355900e+01,
                     0.965127704669424e+00,
                    -0.282172420532826e+02,
                     0.203224612353823e+00,
                     0.110648186063513e+01,
                     0.526127948451280e+00,
                     0.277000018736321e+00,
                     0.108153340501132e+01,
                    -0.744127885357893e-01,
                     0.164094443541384e-01,
                    -0.680468275301065e-01,
                     0.257988576101640e-01,
                    -0.145749861944416e-03  };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double vstar = 0.0028;  // Units of m^3/kg
    double sstar = 4.4; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<29; i++) {
        hold += n[i] * pow(Pi+0.187,I[i]) * pow(Sigma-0.755,J[i]);
    };
    
    return hold * vstar;
};


// Sub-region 3b specific Equation (9) from above referenced IAPWS document
double IFC97_R3_V_ps_3b(double P, double s) {
    /* Provides the backward equation for determining SV based on P,s in Region 3b.
     
     Equations are based on Equation (8) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 14.
    int I[] = { 0, -12, -12,  -12, -12, -12, -12, -10, -10, -10, -10,
                    -8,  -5,  -5, -5, -4, -4, -4, -4, -3, -2,
                    -2,  -2,  -2, -2, -2,  0,  0,  0,  1,  1,
                     2  };
    
    int J[] = { 0,  0,  1, 2,  3,  5, 6, 0, 1, 2, 4,
                    0,  1, 2,  3,  0, 1, 2, 3, 1, 0,
                    1,  2, 3,  4, 12, 0, 1, 2, 0, 2,
                    2   };
    
    double n[] = {   0.0,
                     0.591599780322238e-04,
                    -0.185465997137856e-02,
                     0.104190510480013e-01,
                     0.598647302038590e-02,
                    -0.771391189901699e+00,
                     0.172549765557036e+01,
                    -0.467076079846526e-03,
                     0.134533823384439e-01,
                    -0.808094336805495e-01,
                     0.508139374365767e+00,
                     0.128584643361683e-02,
                    -0.163899353915435e+01,
                     0.586938199318063e+01,
                    -0.292466667918613e+01,
                    -0.614076301499537e-02,
                     0.576199014049172e+01,
                    -0.121613320606788e+02,
                     0.167637540957944e+01,
                    -0.744135838773463e+01,
                     0.378168091437659e-01,
                     0.401432203027688e+01,
                     0.160279837479185e+02,
                     0.317848779347728e+01,
                    -0.358362310304853e+01,
                    -0.115995260446827e+07,
                     0.199256573577909e+00,
                    -0.122270624794624e+00,
                    -0.191449143716586e+02,
                    -0.150448002905284e-01,
                     0.146407900162154e+02,
                    -0.327477787188230e+01 };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double vstar = 0.0088; // Units of m^3/kg
    double sstar = 5.3; // Units of kJ/ (kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<32; i++) {
        hold += n[i] * pow(Pi+0.298,I[i]) * pow(Sigma-0.816,J[i]);
    };
    
    return hold * vstar;
};


// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R3_V_ps(int iRegion, double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 3.
     
     Equations are based on Equation (8) and (9) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=3a, 1=3b
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 3a
            return IFC97_R3_V_ps_3a(P, s);
            
        case 1: // Utilize Sub-region 3b
            return IFC97_R3_V_ps_3b(P, s);
            
        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};