//
//  IFC97_R3_Backward_Vph.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 03/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Backward_Vph.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Supplementary Release on Backward Equations for the Functions T(P,h), v(P,h) and T(P,s), v(P,s) for Region 3 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properites of Water and Steam. Kyoto, Japan, September 2004
 */

// Sub-region 3a specific Equation (4) from above referenced IAPWS document
double IFC97_R3_V_ph_3a(double P, double h) {
    /* Provides the backward equation for determining SV based on P,h in Region 3a.
     
     Equations are based on Equation (4) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 6.
    int I[] = { 0, -12, -12, -12, -12, -10, -10, -10, -8, -8, -6,
                    -6,  -6,  -4,  -4,  -3,  -2,  -2, -1, -1, -1,
                    -1,   0,   0,   1,   1,   1,   2,  2,  3,  4,
                     5,   8 };
    
    int J[] = { 0,  6,  8, 12, 18,  4,  7, 10,  5, 12,  3,
                    4, 22,  2,  3,  7,  3, 16,  0,  1,  2,
                    3,  0,  1,  0,  1,  2,  0,  2,  0,  2,
                    2,  2  };
    
    double n[] = {   0.0,
                     0.529944062966028e-02,
                    -0.170099690234461e+00,
                     0.111323814312927e+02,
                    -0.217898123145125e+04,
                    -0.506061827980875e-03,
                     0.556495239685324e+00,
                    -0.943672726094016e+01,
                    -0.297856807561527e+00,
                     0.939353943717186e+02,
                     0.192944939465981e-01,
                     0.421740664704763e+00,
                    -0.368914126282330e+07,
                    -0.737566847600639e-02,
                    -0.354753242424366e+00,
                    -0.199768169338727e+01,
                     0.115456297059049e+01,
                     0.568366875815960e+04,
                     0.808169540124668e-02,
                     0.172416341519307e+00,
                     0.104270175292927e+01,
                    -0.297691372792847e+00,
                     0.560394465163593e+00,
                     0.275234661176914e+00,
                    -0.148347894866012e+00,
                    -0.651142513478515e-01,
                    -0.292468715386302e+01,
                     0.664876096952665e-01,
                     0.352335014263844e+01,
                    -0.146340792313332e-01,
                    -0.224503486668184e+01,
                     0.110533464706142e+01,
                    -0.408757344495612e-01  };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double vstar = 0.0028;  // Units of m^3/kg
    double hstar = 2100.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<33; i++) {
        hold += n[i] * pow(Pi+0.128,I[i]) * pow(Eta-0.727,J[i]);
    };
    
    return hold * vstar;
};


// Sub-region 3b specific Equation (7) from above referenced IAPWS document
double IFC97_R3_V_ph_3b(double P, double h) {
    /* Provides the backward equation for determining SV based on P,h in Region 3b.
     
     Equations are based on Equation (7) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 7.
    int I[] = { 0, -12, -12,  -8, -8, -8, -8, -8, -8, -6, -6,
                    -6,  -6,  -6, -6, -4, -4, -4, -3, -3, -2,
                    -2,  -1,  -1, -1, -1,  0,  1,  1,  2,  2 };
    
    int J[] = { 0,  0,  1, 0,  1, 3, 6,  7, 8, 0, 1,
                    2,  5, 6, 10, 3, 6, 10, 0, 2, 1,
                    2,  0, 1,  4, 5, 0,  0, 1, 2, 6  };
    
    double n[] = {   0.0,
                    -0.225196934336318e-08,
                     0.140674363313486e-07,
                     0.233784085280560e-05,
                    -0.331833715229001e-04,
                     0.107956778514318e-02,
                    -0.271382067378863e+00,
                     0.107202262490333e+01,
                    -0.853821329075382e+00,
                    -0.215214194340526e-04,
                     0.769656088222730e-03,
                    -0.431136580433864e-02,
                     0.453342167309331e+00,
                    -0.507749535873652e+00,
                    -0.100475154528389e+03,
                    -0.219201924648793e+00,
                    -0.321087965668917e+01,
                     0.607567815637771e+03,
                     0.557686450685932e-03,
                     0.187499040029550e+00,
                     0.905368030448107e-02,
                     0.285417173048685e+00,
                     0.329924030996098e-01,
                     0.239897419685483e+00,
                     0.482754995951394e+01,
                    -0.118035753702231e+02,
                     0.169490044091791e+00,
                    -0.179967222507787e-01,
                     0.371810116332674e-01,
                    -0.536288335065096e-01,
                     0.160697101092520e+01 };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double vstar = 0.0088; // Units of m^3/kg
    double hstar = 2800.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<31; i++) {
        hold += n[i] * pow(Pi+0.0661,I[i]) * pow(Eta-0.720,J[i]);
    };
    
    return hold * vstar;
};


// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R3_V_ph(int iRegion, double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 3.
     
     Equations are based on Equation (2) and (3) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=3a, 1=3b
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        SV  = Region 3 specific volume, m^3/kg
     
     Errors:
     -1000.0 = Incorrect iType input
     -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 3a
            return IFC97_R3_V_ph_3a(P, h);
            
        case 1: // Utilize Sub-region 3b
            return IFC97_R3_V_ph_3b(P, h);
            
        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};