//
//  IFC97_R3_Backward_Tps.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 07/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Backward_Tps.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Supplementary Release on Backward Equations for the Functions T(P,h), v(P,h) and T(P,s), v(P,s) for Region 3 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properites of Water and Steam. Kyoto, Japan, September 2004
 */

// Sub-region 3a specific Equation (6) from above referenced IAPWS document
double IFC97_R3_T_ps_3a(double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 3a.
     
     Equations are based on Equation (6) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 3 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 10.
    int I[] = { 0, -12, -12, -10, -10, -10, -10, -8, -8, -8, -8,
                    -6,  -6,  -6,  -5,  -5,  -5, -4, -4, -4, -2,
                    -2,  -1,  -1,   0,   0,   0,  1,  2,  2,  3,
                     8,   8,  10  };
    
    int J[] = { 0, 28, 32,  4, 10, 12, 14,  5,  7,  8, 28,
                    2,  6, 32,  0, 14, 32,  6, 10, 36,  1,
                    4,  1,  6,  0,  1,  4,  0,  0,  3,  2,
                    0,  1,  2  };
    
    double n[] = {   0.0,
                     0.150042008263875e+10,
                    -0.159397258480424e+12,
                     0.502181140217975e-03,
                    -0.672057767855466e+02,
                     0.145058545404456e+04,
                    -0.823889534888890e+04,
                    -0.154852214233853e+00,
                     0.112305046746695e+02,
                    -0.297000213482822e+02,
                     0.438565132635495e+11,
                     0.137837838635464e-02,
                    -0.297478527157462e+01,
                     0.971777947349413e+13,
                    -0.571527767052398e-04,
                     0.288307949778420e+05,
                    -0.744428289262703e+14,
                     0.128017324848921e+02,
                    -0.368275545889071e+03,
                     0.664768904779177e+16,
                     0.449359251958880e-01,
                    -0.422897836099655e+01,
                    -0.240614376434179e+00,
                    -0.474341365254924e+01,
                     0.724093999126110e+00,
                     0.923874349695897e+00,
                     0.399043655281015e+01,
                     0.384066651868009e-01,
                    -0.359344365571848e-02,
                    -0.735196448821653e+00,
                     0.188367048396131e+00,
                     0.141064266818704e-03,
                    -0.257418501496337e-02,
                     0.123220024851555e-02 };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double Tstar = 760.0;  // Units of K
    double sstar = 4.4; // Units of kJ/ (kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<34; i++) {
        hold += n[i] * pow(Pi+0.240,I[i]) * pow(Sigma-0.703,J[i]);
    };
    
    return hold * Tstar;
};


// Sub-region 3b specific Equation (7) from above referenced IAPWS document
double IFC97_R3_T_ps_3b(double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 3b.
     
     Equations are based on Equation (7) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 3 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 11.
    int I[] = { 0, -12, -12, -12, -12, -8, -8, -8, -6, -6, -6,
                    -5,  -5,  -5,  -5, -5, -4, -3, -3, -2,  0,
                     2,   3,   4,   5,  6,  8, 12, 14 };
       
    
    int J[] = { 0,  1,  3, 4,  7, 0,  1, 3, 0, 2, 4,
                    0,  1, 2,  4, 6, 12, 1, 6, 2, 0,
                    1,  1, 0, 24, 0,  3, 1, 2 };

    
    double n[] = {   0.0,
                     0.527111701601660e+00,
                    -0.401317830052742e+02,
                     0.153020073134484e+03,
                    -0.224799398218827e+04,
                    -0.193993484669048e+00,
                    -0.140467557893768e+01,
                     0.426799878114024e+02,
                     0.752810643416743e+00,
                     0.226657238616417e+02,
                    -0.622873556909932e+03,
                    -0.660823667935396e+00,
                     0.841267087271658e+00,
                    -0.253717501764397e+02,
                     0.485708963532948e+03,
                     0.880531517490555e+03,
                     0.265015592794626e+07,
                    -0.359287150025783e+00,
                    -0.656991567673753e+03,
                     0.241768149185367e+01,
                     0.856873461222588e+00,
                     0.655143675313458e+00,
                    -0.213535213206406e+00,
                     0.562974957606348e-02,
                    -0.316955725450471e+15,
                    -0.699997000152457e-03,
                     0.119845803210767e-01,
                     0.193848122022095e-04,
                    -0.215095749182309e-04 };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double Tstar = 860.0;  // Units of K
    double sstar = 5.3; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<29; i++) {
        hold += n[i] * pow(Pi+0.760,I[i]) * pow(Sigma-0.818,J[i]);
    };
    
    return hold * Tstar;
};


// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R3_T_ps(int iRegion, double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 3.
     
     Equations are based on Equation (6) and (7) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=3a, 1=3b
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 3 temperature, K
     
     Errors:
     -1000.0 = Incorrect iType input
     -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 3a
            return IFC97_R3_T_ps_3a(P, s);
            
        case 1: // Utilize Sub-region 3b
            return IFC97_R3_T_ps_3b(P, s);
            
        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};