//
//  IFC97_R3_Backward_Tph.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 03/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Backward_Tph.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Supplementary Release on Backward Equations for the Functions T(P,h), v(P,h) and T(P,s), v(P,s) for Region 3 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properites of Water and Steam. Kyoto, Japan, September 2004
 */

// Sub-region 3a specific Equation (2) from above referenced IAPWS document
double IFC97_R3_T_ph_3a(double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 3a.
     
     Equations are based on Equation (2) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 3 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 3.
    int I[] = { 0, -12, -12, -12, -12, -12, -12, -12, -12, -10, -10,
                   -10,  -8,  -8,  -8,  -8,  -5,  -3,  -2,  -2,  -2,
                    -1,  -1,   0,   0,   1,   3,   3,   4,   4,  10,
                    12 };
    
    int J[] = { 0,  0, 1,  2,  6, 14, 16, 20, 22,  1,  5,
                   12, 0,  2,  4, 10,  2,  0,  1,  3,  4,
                    0, 2,  0,  1,  1,  0,  1,  0,  3,  4,
                    5 };
    
    double n[] = {   0.0,
                    -0.133645667811215e-06,
                     0.455912656802978e-05,
                    -0.146294640700979e-04,
                     0.639341312970080e-02,
                     0.372783927268847e+03,
                    -0.718654377460447e+04,
                     0.573494752103400e+06,
                    -0.267569329111439e+07,
                    -0.334066283302614e-04,
                    -0.245479214069597e-01,
                     0.478087847764996e+02,
                     0.764664131818904e-05,
                     0.128350627676972e-02,
                     0.171219081377331e-01,
                    -0.851007304583213e+01,
                    -0.136513461629781e-01,
                    -0.384460997596657e-05,
                     0.337423807911655e-02,
                    -0.551624873066791e+00,
                     0.729202277107470e+00,
                    -0.992522757376041e-02,
                    -0.119308831407288e+00,
                     0.793929190615421e+00,
                     0.454270731799386e+00,
                     0.209998591259910e+00,
                    -0.642109823904738e-02,
                    -0.235155868604540e-01,
                     0.252233108341612e-02,
                    -0.764885133368119e-02,
                     0.136176427574291e-01,
                    -0.133027883575669e-01  };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double Tstar = 760.0;  // Units of K
    double hstar = 2300.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<32; i++) {
        hold += n[i] * pow(Pi+0.240,I[i]) * pow(Eta-0.615,J[i]);
    };
    
    return hold * Tstar;
};


// Sub-region 3b specific Equation (3) from above referenced IAPWS document
double IFC97_R3_T_ph_3b(double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 3b.
     
     Equations are based on Equation (3) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 3 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 4.
    int I[] = { 0, -12, -12, -10, -10, -10, -10, -10, -8, -8, -8,
                    -8,  -8,  -6,  -6,  -6,  -4,  -4, -3, -2, -2,
                    -1,  -1,  -1,  -1,  -1,  -1,   0,  0,  1,  3,
                     5,   6,   8 };
    
    int J[] = { 0,  0,  1, 0,  1, 5,  10, 12, 0, 1, 2,
                    4, 10, 0,  1, 2,   0,  1, 5, 0, 4,
                    2,  4, 6, 10, 14, 16,  0, 2, 1, 1,
                    1,  1, 1 };
    
    double n[] = {   0.0,
                     0.323254573644920e-04,
                    -0.127575556587181e-03,
                    -0.475851877356068e-03,
                     0.156183014181602e-02,
                     0.105724860113781e+00,
                    -0.858514221132534e+02,
                     0.724140095480911e+03,
                     0.296475810273257e-02,
                    -0.592721983365988e-02,
                    -0.126305422818666e-01,
                    -0.115716196364853e+00,
                     0.849000969739595e+02,
                    -0.108602260086615e-01,
                     0.154304475328851e-01,
                     0.750455441524466e-01,
                     0.252520973612982e-01,
                    -0.602507901232996e-01,
                    -0.307622221350501e+01,
                    -0.574011959864879e-01,
                     0.503471360939849e+01,
                    -0.925081888584834e+00,
                     0.391733882917546e+01,
                    -0.773146007130190e+02,
                     0.949308762098587e+04,
                    -0.141043719679409e+07,
                     0.849166230819026e+07,
                     0.861095729446704e+00,
                     0.323346442811720e+00,
                     0.873281936020439e+00,
                    -0.436653048526683e+00,
                     0.286596714529479e+00,
                    -0.131778331276228e+00,
                     0.676682064330275e-02 };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double Tstar = 860.0;  // Units of K
    double hstar = 2800.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<34; i++) {
        hold += n[i] * pow(Pi+0.298,I[i]) * pow(Eta-0.720,J[i]);
    };
    
    return hold * Tstar;
};


// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R3_T_ph(int iRegion, double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 3.
     
     Equations are based on Equation (2) and (3) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=3a, 1=3b
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 3 temperature, K
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 3a
            return IFC97_R3_T_ph_3a(P, h);
            
        case 1: // Utilize Sub-region 3b
            return IFC97_R3_T_ph_3b(P, h);
            
        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};