//
//  IFC97_R3_Backward_Phs.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 10/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R3_Backward_Phs.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations P(h,s) for Region 3, Equations as a Function of h and s for the Redion Boundaries, and an Equation Tsat(h,s) for Region 4 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properties of Water and Steam. Kyoto, Japan, September 2004
 */

// Sub-region specific equations from above referenced IAPWS document
double IFC97_R3_P_hs_3a(double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 3a.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Region 1 pressure, MPa
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 3.
    int I[] = { 0,  0,  0,  0,  1,  1,  1,  1,  1,  2,  2,
                    3,  3,  3,  4,  4,  4,  4,  5,  6,  7,
                    8, 10, 10, 14, 18, 20, 22, 22, 24, 28,
                   28, 32, 32  };
    
    int J[] = { 0,  0,  1,  5,  0,  3,  4,  8, 14,  6, 16,
                    0,  2,  3,  0,  1,  4,  5, 28, 28, 24,
                    1, 32, 36, 22, 28, 36, 16, 28, 36, 16,
                   36, 10, 28  };
    
    double n[] = {   0.0,
                     0.770889828326934e+01,
                    -0.260835009128688e+02,
                     0.267416218930389e+03,
                     0.172221089496844e+02,
                    -0.293542332145970e+03,
                     0.614135601882478e+03,
                    -0.610562757725674e+05,
                    -0.651272251118219e+08,
                     0.735919313521937e+05,
                    -0.116646505914191e+11,
                     0.355267086434461e+02,
                    -0.596144543825955e+03,
                    -0.475842430145708e+03,
                     0.696781965359503e+02,
                     0.335674250377312e+03,
                     0.250526809130882e+05,
                     0.146997380630766e+06,
                     0.538069315091534e+20,
                     0.143619827291346e+22,
                     0.364985866165994e+20,
                    -0.254741561156775e+04,
                     0.240120197096563e+28,
                    -0.393847464679496e+30,
                     0.147073407024852e+25,
                    -0.426391250432059e+32,
                     0.194509340621077e+39,
                     0.666212132114896e+24,
                     0.706777016552858e+34,
                     0.175563621975576e+42,
                     0.108408607429124e+29,
                     0.730872705175151e+44,
                     0.159145847398870e+25,
                     0.377121605943324e+41 };
    
    // Reference Values
    double Pstar = 99.0;  // Units of MPa
    double hstar = 2300.0;  // Units of kJ/kg
    double sstar = 4.4; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<34; i++) {
        hold += n[i] * pow(Eta-1.01,I[i]) * pow(Sigma-0.750,J[i]);
    };
    
    return hold * Pstar;
    

};


double IFC97_R3_P_hs_3b(double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 3b.
     
     Equations are based on Equation (2) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Region 1 pressure, MPa
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 4.
    int I[] = { 0, -12, -12, -12, -12, -12, -10, -10, -10, -10, -8,
                    -8,  -6,  -6,  -6,  -6,  -5,  -4,  -4,  -4, -3,
                    -3,  -3,  -3,  -2,  -2,  -1,   0,   2,   2,  5,
                     6,   8,  10,  14,  14  };
    
    int J[] = { 0,  2, 10, 12, 14, 20,  2, 10, 14, 18,  2,
                    8,  2,  6,  7,  8, 10,  4,  5,  8,  1,
                    3,  5,  6,  0,  1,  0,  3,  0,  1,  0,
                    1,  1,  1,  3,  7  };
    
    double n[] = {   0.0,
                     0.125244360717979e-12,
                    -0.126599322553713e-01,
                     0.506878030140626e+01,
                     0.317847171154202e+02,
                    -0.391041161399932e+06,
                    -0.975733406392044e-10,
                    -0.186312419488279e+02,
                     0.510973543414101e+03,
                     0.373847005822362e+06,
                     0.299804024666572e-07,
                     0.200544393820342e+02,
                    -0.498030487662829e-05,
                    -0.102301806360030e+02,
                     0.552819126990325e+02,
                    -0.206211367510878e+03,
                    -0.794012232324823e+04,
                     0.782248472028153e+01,
                    -0.586544326902468e+02,
                     0.355073647696481e+04,
                    -0.115303107290162e-03,
                    -0.175092403171802e+01,
                     0.257981687748160e+03,
                    -0.727048374179467e+03,
                     0.121644822609198e-03,
                     0.393137871762692e-01,
                     0.704181005909296e-02,
                    -0.829108200698110e+02,
                    -0.265178818131250e+00,
                     0.137531682453991e+02,
                    -0.522394090753046e+02,
                     0.240556298941048e+04,
                    -0.227361631268929e+05,
                     0.890746343932567e+05,
                    -0.239234565822486e+08,
                     0.568795808129714e+10 };
    
    // Reference Values
    double Pstar = 16.6;  // Units of MPa
    double hstar = 2800.0;  // Units of kJ/kg
    double sstar = 5.3; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<36; i++) {
        hold += n[i] * pow(Eta-0.681,I[i]) * pow(Sigma-0.792,J[i]);
    };
    
    return Pstar / hold;
};

// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R3_P_hs(int iRegion, double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 3.
     
     Equations are based on Equation (1) and (2) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=3a, 1=3b
        h       = Specific enthalpy, kJ/kg
        s       = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Pressure, MPa
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 3a
            return IFC97_R3_P_hs_3a(h, s);
            
        case 1: // Utilize Sub-region 3b
            return IFC97_R3_P_hs_3b(h, s);
            
        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};
