//
//  IFC97_R2_Basic.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 23/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R2_Basic.h"
#include "IAPWS_DynamicViscosity_Td.h"
#include "IAPWS_ThermalConductivity_TP.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

double IFC97_R2_Basic(int Property, double P, double T) {
    /* Provides the Basic equation for the Region 2 Gibbs free energy function.
     
     Equations are based on Equation (15) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Property = Indicates property to be returned
        P        = Saturation Pressure, MPa
        T        = Saturation Temperature, K
     
     Return:
        For Property =  0, Pressure, MPa
                        1, Temperature, K
                        2, Specific volume, m^3/kg
                        3, Specific internal energy, kJ/kg
                        4, Specific entropy, kJ/(kg K)
                        5, Specific enthalpy, kJ/kg
                        6, Specific isobaric heat capacity, kJ/(kg K)
                        7, Specific isochoric heat capacity, kJ/(kg K)
                        8, Density, kg/m^3
                        9, Speed of sound, m/s
                        10, Gibbs free energy, kJ/kg
                        11, Hemholtz free energy, kJ/kg
                        12, Quality, dimensionless
                        13, Ratio of specific heats (Cp/Cv), dimensionless
                        14, Surface Tension, N/m
                        15. Dynamic Viscosity, Pa*s
                        16. Thermal Conductivity, W/(K*m)
                        17. Thermal diffusivity, m^2/s
                        18. Kinematic Viscosity, m^2/s
                        19. Prandtl Number, dimensionless
                        20. Drho_DP_T, kg/(m^3 MPa), for thermal conductivity calculation
     
     Errors:
        -10000.0 = Incorrect iType input
        -90000.0 = Unknown error
        -80000.0 = Invalid Property Type
     */
    
    // Numerical coefficients from Table 10 for Real Gas Portion.
    int J_o[] = { 0,  0, 1, -5, -4, -3, -2, -1, 2, 3 };
    
    double n_o[] = {     0.0,
                        -0.96927686500217e+01,
                         0.10086655968018e+02,
                        -0.56087911283020e-02,
                         0.71452738081455e-01,
                        -0.40710498223928e+00,
                         0.14240819171444e+01,
                        -0.43839511319450e+01,
                        -0.28408632460772e+00,
                         0.21268463753307e-01 };
        
    // Numerical coefficients from Table 11 for Residual Portion.
    int I[] = { 0,  1,  1,  1,  1,  1,  2,  2,  2,  2,  2,
                    3,  3,  3,  3,  3,  4,  4,  4,  5,  6,
                    6,  6,  7,  7,  7,  8,  8,  9, 10, 10,
                   10, 16, 16, 18, 20, 20, 20, 21, 22, 23,
                   24, 24, 24 };
    
    int J[] = { 0,  0,  1,  2,  3,  6,  1,  2,  4,  7, 36,
                    0,  1,  3,  6, 35,  1,  2,  3,  7,  3,
                   16, 35,  0, 11, 25,  8, 36, 13,  4, 10,
                   14, 29, 50, 57, 20, 35, 48, 21, 53, 39,
                   26, 40, 58 };
    
    double n[] = {   0.0,
                    -0.17731742473213e-02,
                    -0.17834862292358e-01,
                    -0.45996013696365e-01,
                    -0.57581259083432e-01,
                    -0.50325278727930e-01,
                    -0.33032641670203e-04,
                    -0.18948987516315e-03,
                    -0.39392777243355e-02,
                    -0.43797295650573e-01,
                    -0.26674547914087e-04,
                     0.20481737692309e-07,
                     0.43870667284435e-06,
                    -0.32277677238570e-04,
                    -0.15033924542148e-02,
                    -0.40668253562649e-01,
                    -0.78847309559367e-09,
                     0.12790717852285e-07,
                     0.48225372718507e-06,
                     0.22922076337661e-05,
                    -0.16714766451061e-10,
                    -0.21171472321355e-02,
                    -0.23895741934104e+02,
                    -0.59059564324270e-17,
                    -0.12621808899101e-05,
                    -0.38946842435739e-01,
                     0.11256211360459e-10,
                    -0.82311340897998e+01,
                     0.19809712802088e-07,
                     0.10406965210174e-18,
                    -0.10234747095929e-12,
                    -0.10018179379511e-08,
                    -0.80882908646985e-10,
                     0.10693031879409e+00,
                    -0.33662250574171e+00,
                     0.89185845355421e-24,
                     0.30629316876232e-12,
                    -0.42002467698208e-05,
                    -0.59056029685639e-25,
                     0.37826947613457e-05,
                    -0.12768608934681e-14,
                     0.73087610595061e-28,
                     0.55414715350778e-16,
                    -0.94369707241210e-06 };
        
    // Reference Values
    double Pstar = 1.00;  // Units of MPa
    double Tstar = 540.0;  // Units of K
    double R     = 0.461526; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Tau = Tstar/T;
    
    // Temporary Variables
    double hold = 0.0;
    double temp = 0.0;
    double denom = 0.0;
    
    // Return requested property
    switch (Property) {
        case 0: // return Pressure,
            return P;
            
        case 1: // return Temperature
            return T;
            
        case 2: // return specific volume
            return Pi*(IFC97_R2_Get_Go_p(Pi)+IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J)) * R*T/(1000.0*P);
            
        case 3: // return specific internal energy
            return (  (Tau*(IFC97_R2_Get_Go_t(Tau, n_o, J_o)+IFC97_R2_Get_Gr_t(Pi, Tau, n, I, J))) -
                      (Pi*(IFC97_R2_Get_Go_p(Pi)+IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J)))                 ) * R * T;
            
        case 4: // return specific entropy
            return (  (Tau*(IFC97_R2_Get_Go_t(Tau, n_o, J_o)+IFC97_R2_Get_Gr_t(Pi, Tau, n, I, J))) -
                      (IFC97_R2_Get_Go(Pi, Tau, n_o, J_o)+IFC97_R2_Get_Gr(Pi, Tau, n, I, J))               ) * R;
            
        case 5: // return specific enthalpy
            return Tau * (IFC97_R2_Get_Go_t(Tau, n_o, J_o)+IFC97_R2_Get_Gr_t(Pi, Tau, n, I, J)) * R * T;
            
        case 6: // return specific isobaric heat capacity
            return -pow(Tau,2.0) * (IFC97_R2_Get_Go_tt(Tau, n_o, J_o)+IFC97_R2_Get_Gr_tt(Pi, Tau, n, I, J)) * R;
            
        case 7: // return specific isochoric heat capacity
            hold  = pow(1.0+Pi*IFC97_R2_Get_Gr_p(Pi,Tau,n,I,J)-Tau*Pi*IFC97_R2_Get_Gr_pt(Pi,Tau,n,I,J), 2.0);
            denom = 1.0 - pow(Pi,2.0)*IFC97_R2_Get_Gr_pp(Pi, Tau, n, I, J);
            return (-pow(Tau,2.0) * (IFC97_R2_Get_Go_tt(Tau,n_o,J_o)+IFC97_R2_Get_Gr_tt(Pi,Tau,n,I,J)) - hold/denom) * R;
            
        case 8: // return density
            return (1000.0*P) / ( Pi*(IFC97_R2_Get_Go_p(Pi)+IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J)) * R*T );
        
        case 9: // return speed of sound
            hold  = pow(1.0+Pi*IFC97_R2_Get_Gr_p(Pi,Tau,n,I,J)-Tau*Pi*IFC97_R2_Get_Gr_pt(Pi,Tau,n,I,J), 2.0);
            temp  = pow(Tau,2.0)*(IFC97_R2_Get_Go_tt(Tau, n_o, J_o)+IFC97_R2_Get_Gr_tt(Pi, Tau, n, I, J));
            denom = (1.0-pow(Pi,2.0)*IFC97_R2_Get_Gr_pp(Pi, Tau, n, I, J)) + (hold/temp);
            hold  = 1.0 + 2.0*Pi*IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J) + pow(Pi,2.0)*pow(IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J),2.0);
            return sqrt( hold * 1000.0 * R * T / denom);
            
        case 10: // return Gibbs free energy
            return IFC97_R2_Gibbs_ND(Pi, Tau, n, I, J, n_o, J_o)*R*T;
            
        case 11: // return Hemholtz free energy
            return IFC97_R2_Gibbs_ND(Pi, Tau, n, I, J, n_o, J_o)*R*T - P*IFC97_R2_Basic(2,P,T)*1000.0;
            
        case 12: // return Quality (Region 2 is vapor which is 1)
            return 1.0;
            
        case 13: // return ratio of specific heats (Cp/Cv)
            temp = -pow(Tau,2.0) * (IFC97_R2_Get_Go_tt(Tau, n_o, J_o)+IFC97_R2_Get_Gr_tt(Pi, Tau, n, I, J)) * R; // Cp
            hold  = pow(1.0+Pi*IFC97_R2_Get_Gr_p(Pi,Tau,n,I,J)-Tau*Pi*IFC97_R2_Get_Gr_pt(Pi,Tau,n,I,J), 2.0); // Part of Cv
            denom = 1.0 - pow(Pi,2.0)*IFC97_R2_Get_Gr_pp(Pi, Tau, n, I, J); // Part of Cv
            return temp / ( (-pow(Tau,2.0) * (IFC97_R2_Get_Go_tt(Tau,n_o,J_o)+IFC97_R2_Get_Gr_tt(Pi,Tau,n,I,J)) - hold/denom) * R );
            
        case 14: // return Surface Tension, N/m
            return -80000.0;
            
        case 15: // return Dynamic Viscosity, Pa*s
            hold = (1000.0*P) / ( Pi*(IFC97_R2_Get_Go_p(Pi)+IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J)) * R*T );
            return IAPWS_DynamicViscosity_Td(T, hold);
            
        case 16: // return Thermal Conductivity, W/(K*m)
            return IAPWS_ThermalConductivity_TP(T, P);
  
        case 17:  // Thermal diffusivity, m^2/sec
            hold = (1000.0*P) / ( Pi*(IFC97_R2_Get_Go_p(Pi)+IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J)) * R*T );        // Density
            temp = -pow(Tau,2.0) * (IFC97_R2_Get_Go_tt(Tau, n_o, J_o)+IFC97_R2_Get_Gr_tt(Pi, Tau, n, I, J)) * R; // Cp
            return IAPWS_ThermalConductivity_TP(T, P) / (hold * temp * 1000.0 );
            
        case 18: // Kinematic viscosity, m^2/sec
            hold = (1000.0*P) / ( Pi*(IFC97_R2_Get_Go_p(Pi)+IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J)) * R*T );        // Density
            return IAPWS_DynamicViscosity_Td(T, hold) / hold;
            
        case 19: // Prandtl Number, dimensionless
            hold = (1000.0*P) / ( Pi*(IFC97_R2_Get_Go_p(Pi)+IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J)) * R*T );        // Density
            temp = -pow(Tau,2.0) * (IFC97_R2_Get_Go_tt(Tau, n_o, J_o)+IFC97_R2_Get_Gr_tt(Pi, Tau, n, I, J)) * R; // Cp
            return IAPWS_DynamicViscosity_Td(T, hold)*temp*1000.0/IAPWS_ThermalConductivity_TP(T, P);
            
        case 20: // Drho_DP_T, kg/(m^3 MPa) for thermal conductivity calculation
            hold = IFC97_R2_Get_Go_pp(Pi)+IFC97_R2_Get_Gr_pp(Pi, Tau, n, I, J);
            temp = IFC97_R2_Get_Go_p(Pi) + IFC97_R2_Get_Gr_p(Pi, Tau, n, I, J);
            return (-1000.0*hold) / ( R * T * pow(temp, 2.0) );
            
        default:
            return -10000.0;
    };
    
    return -90000.0;
};

// Non-Dimensional Gibbs free energy. Equation (15)
double IFC97_R2_Gibbs_ND(double Pi, double Tau, double *n, int *I, int *J, double *n_o, int *J_o) {
    /* Provides the non-dimensional Gibbs free energy for Region 2.
     
     Equations are based on Equation (15) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n   = Table 11 n-coefficient array
        I   = Table 11 I-coefficient array
        J   = Table 11 J-coefficient array
        n_o = Table 10 n_o-coefficient array
        J_o = Table 10 J_o-coefficient array
     
     Return:
        Non-dimensional Gibbs free energy
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Get_Go(Pi, Tau, n_o, J_o) + IFC97_R2_Get_Gr(Pi, Tau, n, I, J);
};

// Derivatives for the Ideal Gas Portion of the Gibbs free energy per Table 13.
double IFC97_R2_Get_Go(double Pi, double Tau, double *n_o, int *J_o) {
    /* Provides the non-dimensional Ideal Gas Portion of the Gibbs free energy.
     
     Equations are based on Table 13 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n_o = Table 10 n_o-coefficient array
        J_o = Table 10 J_o-coefficient array
     
     Return:
        Non-dimensional ideal gas portion of the Gibbs free energy
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */

    double hold = log(Pi); // summer variable
    
    for (int i=1; i<10; i++) {
        hold += n_o[i] * pow(Tau,J_o[i]);  // Equation (7)
    };
    
    return hold;
};


double IFC97_R2_Get_Go_p(double Pi) {
    /* Provides the non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt P .
     
     Equations are based on Table 13 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
          
     Return:
        Non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt P .
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return 1.0 /Pi;
};


double IFC97_R2_Get_Go_pp(double Pi) {
    /* Provides the non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt P, P .
     
     Equations are based on Table 13 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
     
     Return:
        Non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt P, P .
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return -1.0 / pow(Pi,2.0);
};

double IFC97_R2_Get_Go_t(double Tau, double *n_o, int *J_o) {
    /* Provides the non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt T .
     
     Equations are based on Table 13 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Tau = Non-dimensional temperature for Region 2
        n_o = Table 10 n_o-coefficient array
        J_o = Table 10 J_o-coefficient array
     
     Return:
        Non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt T .
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    double hold = 0.0; // summer variable
    
    for (int i=1; i<10; i++) {
        hold += n_o[i] * J_o[i] * pow(Tau,J_o[i]-1);  // Equation (7)
    };
    
    return hold;
};
double IFC97_R2_Get_Go_tt(double Tau, double *n_o, int *J_o) {
    /* Provides the non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt T,T .
     
     Equations are based on Table 13 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Tau = Non-dimensional temperature for Region 2
        n_o = Table 10 n_o-coefficient array
        J_o = Table 10 J_o-coefficient array
     
     Return:
        Non-dimensional derivative of the Ideal Gas Portion of the Gibbs free energy wrt T, T .
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    double hold = 0.0; // summer variable
    
    for (int i=1; i<10; i++) {
        hold += n_o[i] * J_o[i] * (J_o[i]-1) * pow(Tau,J_o[i]-2);  // Equation (7)
    };
    
    return hold;
};


// Derivatives for the Residual Portion of the Gibbs free energy per Table 14.
double IFC97_R2_Get_Gr(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional residual portion of the Gibbs free energy.
     
     Equations are based on Table 14 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n   = Table 11 n-coefficient array
        I   = Table 11 I-coefficient array
        J   = Table 11 J-coefficient array
     
     Return:
        Non-dimensional residual portion of the Gibbs free energy.
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<44; i++) {
        hold += n[i] * pow(Pi,I[i]) * pow(Tau-0.5,J[i]);
    };
    
    return hold;
};


double IFC97_R2_Get_Gr_p(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the residual portion of the Gibbs free energy wrt P.
     
     Equations are based on Table 14 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n   = Table 11 n-coefficient array
        I   = Table 11 I-coefficient array
        J   = Table 11 J-coefficient array
     
     Return:
        Non-dimensional derivative of the residual portion of the Gibbs free energy wrt P.
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<44; i++) {
        hold += n[i] * I[i] * pow(Pi,I[i]-1) * pow(Tau-0.5,J[i]);
    };
    
    return hold;
};



double IFC97_R2_Get_Gr_pp(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the residual portion of the Gibbs free energy wrt P, P.
     
     Equations are based on Table 14 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n   = Table 11 n-coefficient array
        I   = Table 11 I-coefficient array
        J   = Table 11 J-coefficient array
     
     Return:
        Non-dimensional derivative of the residual portion of the Gibbs free energy wrt P, P.
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<44; i++) {
        hold += n[i] * I[i] * (I[i]-1) * pow(Pi,I[i]-2) * pow(Tau-0.5,J[i]);
    };
    
    return hold;
};


double IFC97_R2_Get_Gr_t(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the residual portion of the Gibbs free energy wrt T.
     
     Equations are based on Table 14 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n   = Table 11 n-coefficient array
        I   = Table 11 I-coefficient array
        J   = Table 11 J-coefficient array
     
     Return:
        Non-dimensional derivative of the residual portion of the Gibbs free energy wrt T.
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<44; i++) {
        hold += n[i] * pow(Pi,I[i]) * J[i] * pow(Tau-0.5,J[i]-1);
    };
    
    return hold;
    
};


double IFC97_R2_Get_Gr_tt(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the residual portion of the Gibbs free energy wrt T, T.
     
     Equations are based on Table 14 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n   = Table 11 n-coefficient array
        I   = Table 11 I-coefficient array
        J   = Table 11 J-coefficient array
     
     Return:
        Non-dimensional derivative of the residual portion of the Gibbs free energy wrt T, T.
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<44; i++) {
        hold += n[i] * pow(Pi,I[i]) * J[i] * (J[i]-1) * pow(Tau-0.5,J[i]-2);
    };
    
    return hold;
};


double IFC97_R2_Get_Gr_pt(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the residual portion of the Gibbs free energy wrt P, T.
     
     Equations are based on Table 14 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 2
        Tau = Non-dimensional temperature for Region 2
        n   = Table 11 n-coefficient array
        I   = Table 11 I-coefficient array
        J   = Table 11 J-coefficient array
     
     Return:
        Non-dimensional derivative of the residual portion of the Gibbs free energy wrt P, T.
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<44; i++) {
        hold += n[i] * I[i] * pow(Pi,I[i]-1) * J[i] * pow(Tau-0.5,J[i]-1);
    };
    
    return hold;
};

// Interface functions to obtain specific properties
double IFC97_R2_SV_pt(double P, double T) {
    /* Provides the Region 2 specific volume given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific volume, m^3/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(2, P, T);
};


double IFC97_R2_U_pt(double P, double T) {
    /* Provides the Region 2 specific internal energy given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific internal energy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(3, P, T);
};


double IFC97_R2_S_pt(double P, double T) {
    /* Provides the Region 2 specific entropy given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific entropy, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(4, P, T);
};


double IFC97_R2_H_pt(double P, double T) {
    /* Provides the Region 2 specific enthalpy given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific enthalpy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(5, P, T);
};


double IFC97_R2_Cp_pt(double P, double T) {
    /* Provides the Region 2 specific isobaric heat capacity given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific isobaric heat capacity, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(6, P, T);
};


double IFC97_R2_Cv_pt(double P, double T) {
    /* Provides the Region 2 specific isochoric heat capacity given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific isochoric heat capacity, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(7, P, T);
};


double IFC97_R2_SOS_pt(double P, double T) {
    /* Provides the Region 2 speed of sound given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Speed of sound, m/s
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(8, P, T);
};


double IFC97_R2_Gibbs_pt(double P, double T) {
    /* Provides the Region 2 Gibbs free energy pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Gibbs free energy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(9, P, T);
};


double IFC97_R2_Hemholtz_pt(double P, double T) {
    /* Provides the Region 2 Hemholtz free energy given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Hemholtz free energy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Basic(10, P, T);
    
};

double IFC97_R2_k_Ratio_pt(double P, double T){
    /* Provides the Region 2 ration of specific heats given pressure and temperature
     
     Functions calls the basic Region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Ratio of specific heats, k, dimensionless
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R2_Cp_pt(P, T) / IFC97_R2_Cv_pt(P, T);
};

double IFC97_R2_RHO_pt(double P, double T) {
    /* Provides the Region 2 density given pressure and temperature
     
     Functions calls the basic region 2 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Density, kg/m^3
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return 1.0 / IFC97_R2_SV_pt(P, T);
};