//
//  IFC97_R2_Backward_Tps.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 26/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R2_Backward_Tps.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

// Sub-region 2a specific Equation (25) from above referenced IAPWS document
double IFC97_R2_T_ps_2a(double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 2a.
     
     Equations are based on Equation (25) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 25.
    double I[] = { 0, -1.5,  -1.5,  -1.5,  -1.5,  -1.5,
                      -1.5, -1.25, -1.25, -1.25,  -1.0,
                      -1.0,  -1.0,  -1.0,  -1.0,  -1.0,
                     -0.75, -0.75,  -0.5,  -0.5,  -0.5,
                      -0.5, -0.25, -0.25, -0.25, -0.25,
                      0.25,  0.25,  0.25,  0.25,   0.5,
                       0.5,   0.5,   0.5,   0.5,   0.5,
                       0.5,  0.75,  0.75,  0.75,  0.75,
                       1.0,   1.0,  1.25,  1.25,   1.5, 1.5  };
    
    int J[] = { 0, -24, -23, -19, -13, -11, -10, -19, -15, -6, -26,
                   -21, -17, -16,  -9,  -8, -15, -14, -26, -13, -9,
                    -7, -27, -25, -11,  -6,   1,   4,   8,  11,  0,
                     1,   5,   6,  10,  14,  16,   0,   4,   9, 17,
                     7,  18,   3,  15,   5,  18 };
    
    double n[] = {   0.0,
                    -0.39235983861984e+06,
                     0.51526573827270e+06,
                     0.40482443161048e+05,
                    -0.32193790923902e+03,
                     0.96961424218694e+02,
                    -0.22867846371773e+02,
                    -0.44942914124357e+06,
                    -0.50118336020166e+04,
                     0.35684463560015e+00,
                     0.44235335848190e+05,
                    -0.13673388811708e+05,
                     0.42163260207864e+06,
                     0.22516925837475e+05,
                     0.47442144865646e+03,
                    -0.14931130797647e+03,
                    -0.19781126320452e+06,
                    -0.23554399470760e+05,
                    -0.19070616302076e+05,
                     0.55375669883164e+05,
                     0.38293691437363e+04,
                    -0.60391860580567e+03,
                     0.19363102620331e+04,
                     0.42660643698610e+04,
                    -0.59780638872718e+04,
                    -0.70401463926862e+03,
                     0.33836784107553e+03,
                     0.20862786635187e+02,
                     0.33834172656196e-01,
                    -0.43124428414893e-04,
                     0.16653791356412e+03,
                    -0.13986292055898e+03,
                    -0.78849547999872e+00,
                     0.72132411753872e-01,
                    -0.59754839398283e-02,
                    -0.12141358953904e-04,
                     0.23227096733871e-06,
                    -0.10538463566194e+02,
                     0.20718925496502e+01,
                    -0.72193155260427e-01,
                     0.20749887081120e-06,
                    -0.18340657911379e-01,
                     0.29036272348696e-06,
                     0.21037527893619e+00,
                     0.25681239729999e-03,
                    -0.12799002933781e-01,
                    -0.82198102652018e-05 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double sstar = 2.0; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<47; i++) {
        hold += n[i] * pow(Pi,I[i]) * pow(Sigma-2.0,J[i]);
    };
    
    return hold * Tstar;
};


// Sub-region 2b specific Equation (26) from above referenced IAPWS document
double IFC97_R2_T_ps_2b(double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 2b.
     
     Equations are based on Equation (26) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 26.
    int I[] = { 0, -6, -6, -5, -5, -4, -4, -4, -3, -3, -3,
                   -3, -2, -2, -2, -2, -1, -1, -1, -1, -1,
                    0,  0,  0,  0,  0,  0,  0,  1,  1,  1,
                    1,  1,  1,  2,  2,  2,  3,  3,  3,  4,
                    4,  5,  5,  5 };
    
    int J[] = { 0,   0, 11, 0, 11,  0, 1, 11, 0, 1, 11,
                    12,  0, 1,  6, 10, 0,  1, 5, 8,  9,
                     0,  1, 2,  4,  5, 6,  9, 0, 1,  2,
                     3,  7, 8,  0,  1, 5,  0, 1, 3,  0,
                     1,  0, 1,  2    };
    
    double n[] = {   0.0,
                     0.31687665083497e+06,
                     0.20864175881858e+02,
                    -0.39859399803599e+06,
                    -0.21816058518877e+02,
                     0.22369785194242e+06,
                    -0.27841703445817e+04,
                     0.99207436071480e+01,
                    -0.75197512299157e+05,
                     0.29708605951158e+04,
                    -0.34406878548526e+01,
                     0.38815564249115e+00,
                     0.17511295085750e+05,
                    -0.14237112854449e+04,
                     0.10943803364167e+01,
                     0.89971619308495e+00,
                    -0.33759740098958e+04,
                     0.47162885818355e+03,
                    -0.19188241993679e+01,
                     0.41078580492196e+00,
                    -0.33465378172097e+00,
                     0.13870034777505e+04,
                    -0.40663326195838e+03,
                     0.41727347159610e+02,
                     0.21932549434532e+01,
                    -0.10320050009077e+01,
                     0.35882943516703e+00,
                     0.52511453726066e-02,
                     0.12838916450705e+02,
                    -0.28642437219381e+01,
                     0.56912683664855e+00,
                    -0.99962954584931e-01,
                    -0.32632037778459e-02,
                     0.23320922576723e-03,
                    -0.15334809857450e+00,
                     0.29072288239902e-01,
                     0.37534702741167e-03,
                     0.17296691702411e-02,
                    -0.38556050844504e-03,
                    -0.35017712292608e-04,
                    -0.14566393631492e-04,
                     0.56420857267269e-05,
                     0.41286150074605e-07,
                    -0.20684671118824e-07,
                     0.16409393674725e-08    };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double sstar = 0.7853; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<45; i++) {
        hold += n[i] * pow(Pi,I[i]) * pow(10.0-Sigma,J[i]);
    };
    
    return hold * Tstar;
};


// Sub-region 2c specific Equation (27) from above referenced IAPWS document
double IFC97_R2_T_ps_2c(double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 2c.
     
     Equations are based on Equation (27) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 27.
    int I[] = { 0, -2, -2, -1, 0, 0, 0, 0, 1, 1, 1,
                    1,  2,  2, 2, 3, 3, 3, 4, 4, 4,
                    5,  5,  5, 6, 6, 7, 7, 7, 7, 7 };
    
    int J[] = { 0, 0, 1, 0, 0, 1, 2, 3, 0, 1, 3,
                   4, 0, 1, 2, 0, 1, 5, 0, 1, 4,
                   0, 1, 2, 0, 1, 0, 1, 3, 4, 5  };
    
    double n[] = {   0.0,
                     0.90968501005365e+03,
                     0.24045667088420e+04,
                    -0.59162326387130e+03,
                     0.54145404128074e+03,
                    -0.27098308411192e+03,
                     0.97976525097926e+03,
                    -0.46966772959435e+03,
                     0.14399274604723e+02,
                    -0.19104204230429e+02,
                     0.53299167111971e+01,
                    -0.21252975375934e+02,
                    -0.31147334413760e+00,
                     0.60334840894623e+00,
                    -0.42764839702509e-01,
                     0.58185597255259e-02,
                    -0.14597008284753e-01,
                     0.56631175631027e-02,
                    -0.76155864584577e-04,
                     0.22440342919332e-03,
                    -0.12561095013413e-04,
                     0.63323132660934e-06,
                    -0.20541989675375e-05,
                     0.36405370390082e-07,
                    -0.29759897789215e-08,
                     0.10136618529763e-07,
                     0.59925719692351e-11,
                    -0.20677870105164e-10,
                    -0.20874278181886e-10,
                     0.10162166825089e-09,
                    -0.16429828281347e-09  };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double sstar = 2.9251; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<31; i++) {
        hold += n[i] * pow(Pi,I[i]) * pow(2.0-Sigma,J[i]);
    };
    
    return hold * Tstar;
};

// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R2_T_ps(int iRegion, double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 2.
     
     Equations are based on Equation (25), (26), and (27) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=2a, 1=2b, 2=2c
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 2a
            return IFC97_R2_T_ps_2a(P, s);
            
        case 1: // Utilize Sub-region 2b
            return IFC97_R2_T_ps_2b(P, s);
            
        case 2: // Utilize Sub-region 2c
            return IFC97_R2_T_ps_2c(P, s);
            
        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};