//
//  IFC97_R2_Backward_Tph.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 26/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R2_Backward_Tph.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

// Sub-region 2a specific Equation (22) from above referenced IAPWS document
double IFC97_R2_T_ph_2a(double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 2a.
     
     Equations are based on Equation (22) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 20.
    int I[] = { 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                   1, 1, 1, 1, 1, 2, 2, 2, 2, 2,
                   2, 2, 2, 3, 3, 4, 4, 4, 5, 5,
                   5, 6, 6, 7 };
     
    int J[] = { 0, 0,  1,  2,  3,  7, 20,  0,  1,  2,  3,
                   7,  9, 11, 18, 44,  0,  2,  7, 36, 38,
                  40, 42, 44, 24, 44, 12, 32, 44, 32, 36,
                  42, 34, 44, 28 };
    
    double n[] = {   0.0,
                     0.10898952318288e+04,
                     0.84951654495535e+03,
                    -0.10781748091826e+03,
                     0.33153654801263e+02,
                    -0.74232016790248e+01,
                     0.11765048724356e+02,
                     0.18445749355790e+01,
                    -0.41792700549624e+01,
                     0.62478196935812e+01,
                    -0.17344563108114e+02,
                    -0.20058176862096e+03,
                     0.27196065473796e+03,
                    -0.45511318285818e+03,
                     0.30919688604755e+04,
                     0.25226640357872e+06,
                    -0.61707422868339e-02,
                    -0.31078046629583e+00,
                     0.11670873077107e+02,
                     0.12812798404046e+09,
                    -0.98554909623276e+09,
                     0.28224546973002e+10,
                    -0.35948971410703e+10,
                     0.17227349913197e+10,
                    -0.13551334240775e+05,
                     0.12848734664650e+08,
                     0.13865724283226e+01,
                     0.23598832556514e+06,
                    -0.13105236545054e+08,
                     0.73999835474766e+04,
                    -0.55196697030060e+06,
                     0.37154085996233e+07,
                     0.19127729239660e+05,
                    -0.41535164835634e+06,
                    -0.62459855192507e+02 };
        
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double hstar = 2000.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<35; i++) {
        hold += n[i] * pow(Pi,I[i]) * pow(Eta-2.1,J[i]);
    };
    
    return hold * Tstar;
};


// Sub-region 2b specific Equation (23) from above referenced IAPWS document
double IFC97_R2_T_ph_2b(double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 2b.
     
     Equations are based on Equation (23) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 21.
    int I[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1,
                   1, 1, 1, 1, 1, 1, 2, 2, 2, 2,
                   3, 3, 3, 3, 4, 4, 4, 4, 4, 4,
                   5, 5, 5, 6, 7, 7, 9, 9 };
    
    int J[] = { 0, 0,  1,  2, 12, 18, 24, 28, 40,  0,  2,
                   6, 12, 18, 24, 28, 40,  2,  8, 18, 40,
                   1,  2, 12, 24,  2, 12, 18, 24, 28, 40,
                  18, 24, 40, 28,  2, 28,  1, 40 };
    
    double n[] = {   0.0,
                     0.14895041079516e+04,
                     0.74307798314034e+03,
                    -0.97708318797837e+02,
                     0.24742464705674e+01,
                    -0.63281320016026e+00,
                     0.11385952129658e+01,
                    -0.47811863648625e+00,
                     0.85208123431544e-02,
                     0.93747147377932e+00,
                     0.33593118604916e+01,
                     0.33809355601454e+01,
                     0.16844539671904e+00,
                     0.73875745236695e+00,
                    -0.47128737436186e+00,
                     0.15020273139707e+00,
                    -0.21764114219750e-02,
                    -0.21810755324761e-01,
                    -0.10829784403677e+00,
                    -0.46333324635812e-01,
                     0.71280351959551e-04,
                     0.11032831789999e-03,
                     0.18955248387902e-03,
                     0.30891541160537e-02,
                     0.13555504554949e-02,
                     0.28640237477456e-06,
                    -0.10779857357512e-04,
                    -0.76462712454814e-04,
                     0.14052392818316e-04,
                    -0.31083814331434e-04,
                    -0.10302738212103e-05,
                     0.28217281635040e-06,
                     0.12704902271945e-05,
                     0.73803353468292e-07,
                    -0.11030139238909e-07,
                    -0.81456365207833e-13,
                    -0.25180545682962e-10,
                    -0.17565233969407e-17,
                     0.86934156344163e-14  };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double hstar = 2000.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<39; i++) {
        hold += n[i] * pow(Pi-2.0,I[i]) * pow(Eta-2.6,J[i]);
    };
    
    return hold * Tstar;
};


// Sub-region 2c specific Equation (24) from above referenced IAPWS document
double IFC97_R2_T_ph_2c(double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 2c.
     
     Equations are based on Equation (24) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 22.
    int I[] = { 0, -7, -7, -6, -6, -5, -5, -2, -2, -1, -1,
                    0,  0,  1,  1,  2,  6,  6,  6,  6,  6,
                    6,  6,  6 };
    
    int J[] = { 0, 0,  4,  0,  2,  0,  2,  0,  1,  0,  2,
                   0,  1,  4,  8,  4,  0,  1,  4, 10, 12,
                  16, 20, 22  };
    
    double n[] = {   0.0,
                    -0.32368398555242e+13,
                     0.73263350902181e+13,
                     0.35825089945447e+12,
                    -0.58340131851590e+12,
                    -0.10783068217470e+11,
                     0.20825544563171e+11,
                     0.61074783564516e+06,
                     0.85977722535580e+06,
                    -0.25745723604170e+05,
                     0.31081088422714e+05,
                     0.12082315865936e+04,
                     0.48219755109255e+03,
                     0.37966001272486e+01,
                    -0.10842984880077e+02,
                    -0.45364172676660e-01,
                     0.14559115658698e-12,
                     0.11261597407230e-11,
                    -0.17804982240686e-10,
                     0.12324579690832e-06,
                    -0.11606921130984e-05,
                     0.27846367088554e-04,
                    -0.59270038474176e-03,
                     0.12918582991878e-02  };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double hstar = 2000.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<24; i++) {
        hold += n[i] * pow(Pi+25.0,I[i]) * pow(Eta-1.8,J[i]);
    };
    
    return hold * Tstar;
};

// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R2_T_ph(int iRegion, double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 2.
     
     Equations are based on Equation (22), (23), and (24) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=2a, 1=2b, 2=2c
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 2 temperature, K
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 2a
            return IFC97_R2_T_ph_2a(P, h);

        case 1: // Utilize Sub-region 2b
            return IFC97_R2_T_ph_2b(P, h);

        case 2: // Utilize Sub-region 2c
            return IFC97_R2_T_ph_2c(P, h);

        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};