//
//  IFC97_R2_Backward_Phs.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 29/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R2_Backward_Phs.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations for Pressure as a Function of Enthalpy and Entropy P(h,s) to the IAWPS Industrial Formualation 1997 for the Thermodynamic Properties of Water and Steam. Gaithersburh, Maryland, USA. September 2001.
 */

// Sub-region 2a specific Equation (25) from above referenced IAPWS document
double IFC97_R2_P_hs_2a(double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 2a.
     
     Equations are based on Equation (3) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Pressure, MPa
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 6.
    double I[] = { 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                      1, 1, 1, 1, 1, 1, 2, 2, 2, 3,
                      3, 3, 3, 3, 4, 5, 5, 6, 7     };
    
    int J[] = { 0, 1, 3,  6, 16, 20, 22,  0,  1,  2, 3,
                   5, 6, 10, 16, 20, 22,  3, 16, 20, 0,
                   2, 3,  6, 16, 16,  3, 16,  3,  1  };
    
    double n[] = {   0.0,
                    -0.182575361923032e-01,
                    -0.125229548799536e+00,
                     0.592290437320145e+00,
                     0.604769706185122e+01,
                     0.238624965444474e+03,
                    -0.298639090222922e+03,
                     0.512250813040750e-01,
                    -0.437266515606486e+00,
                     0.413336902999504e+00,
                    -0.516468254574773e+01,
                    -0.557014838445711e+01,
                     0.128555037824478e+02,
                     0.114144108953290e+02,
                    -0.119504225652714e+03,
                    -0.284777985961560e+04,
                     0.431757846408006e+04,
                     0.112894040802650e+01,
                     0.197409186206319e+04,
                     0.151612444706087e+04,
                     0.141324451421235e-01,
                     0.585501282219601e+00,
                    -0.297258075863012e+01,
                     0.594567314847319e+01,
                    -0.623656565798905e+04,
                     0.965986235133332e+04,
                     0.681500934948134e+01,
                    -0.633207286824489e+04,
                    -0.558919224465760e+01,
                     0.400645798472063e-01   };
    
    // Reference Values
    double hstar = 4200.0;  // Units of kJ/ kg
    double sstar = 12.0; // Units of kJ/(kg K)
    double Pstar = 4.0;  // Units of MPa
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<30; i++) {
        hold += n[i] * pow(Eta-0.5,I[i]) * pow(Sigma-1.2,J[i]);
    };
    
    return pow(hold,4.0) * Pstar;
};


// Sub-region 2b specific Equation (26) from above referenced IAPWS document
double IFC97_R2_P_hs_2b(double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 2b.
     
     Equations are based on Equation (26) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Pressure, MPa

     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 7.
    int I[] = { 0, 0,  0,  0, 0, 0, 1, 1, 1, 1, 1,
                   1,  2,  2, 2, 3, 3, 3, 3, 4, 4,
                   5,  5,  6, 6, 6, 7, 7, 8, 8, 8,
                   8, 12, 14 };
    
    int J[] = { 0,  0,  1,  2,  4,  8, 0, 1,  2,  3,  5,
                   12,  1,  6, 18,  0, 1, 7, 12,  1, 16,
                    1, 12,  1,  8, 18, 1, 16, 1,  3, 14,
                   18, 10, 16    };
    
    double n[] = {   0.0,
                     0.801496989929495e-01,
                    -0.543862807146111e+00,
                     0.337455597421283e+00,
                     0.890555451157450e+01,
                     0.313840736431485e+03,
                     0.797367065977789e+00,
                    -0.121616973556240e+01,
                     0.872803386937477e+01,
                    -0.169769781757602e+02,
                    -0.186552827328416e+03,
                     0.951159274344237e+05,
                    -0.189168510120494e+02,
                    -0.433407037194840e+04,
                     0.543212633012715e+09,
                     0.144793408386013e+00,
                     0.128024559637516e+03,
                    -0.672309534071268e+05,
                     0.336972380095287e+08,
                    -0.586634196762720e+03,
                    -0.221403224769889e+11,
                     0.171606668708389e+04,
                    -0.570817595806302e+09,
                    -0.312109693178482e+04,
                    -0.207841384633010e+07,
                     0.305605946157786e+13,
                     0.322157004314333e+04,
                     0.326810259797295e+12,
                    -0.144104158934487e+04,
                     0.410694867802691e+03,
                     0.109077066873024e+12,
                    -0.247964654258893e+14,
                     0.188801906865134e+10,
                    -0.123651009018773e+15 };
    
    // Reference Values
    double hstar = 4100.0;  // Units of kJ/ kg
    double sstar = 7.9; // Units of kJ/(kg K)
    double Pstar = 100.0;  // Units of MPa
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<34; i++) {
        hold += n[i] * pow(Eta-0.6,I[i]) * pow(Sigma-1.01,J[i]);
    };
    
    return pow(hold,4.0) * Pstar;
};

// Sub-region 2c specific Equation (27) from above referenced IAPWS document
double IFC97_R2_P_hs_2c(double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 2c.
     
     Equations are based on Equation (27) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Pressure, MPa

     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 8.
    int I[] = { 0,  0, 0, 0, 0, 0, 0, 1, 1,  1,  1,
                    1, 2, 2, 2, 2, 2, 3, 3,  3,  3,
                    3, 4, 5, 5, 5, 5, 6, 6, 10, 12,
                   16 };
    
    int J[] = { 0,   0,  1, 2, 3,  4,  8, 0,  2, 5,  8,
                    14,  2, 3, 7, 10, 18, 0,  5, 8, 16,
                    18, 18, 1, 4,  6, 14, 8, 18, 7,  7,
                    10   };
    
    double n[] = {   0.0,
                     0.112225607199012e+00,
                    -0.339005953606712e+01,
                    -0.320503911730094e+02,
                    -0.197597305104900e+03,
                    -0.407693861553446e+03,
                     0.132943775222331e+05,
                     0.170846839774007e+01,
                     0.373694198142245e+02,
                     0.358144365815434e+04,
                     0.423014446424664e+06,
                    -0.751071025760063e+09,
                     0.523446127607898e+02,
                    -0.228351290812417e+03,
                    -0.960652417056937e+06,
                    -0.807059292526074e+08,
                     0.162698017225669e+13,
                     0.772465073604171e+00,
                     0.463929973837746e+05,
                    -0.137317885134128e+08,
                     0.170470392630512e+13,
                    -0.251104628187308e+14,
                     0.317748830835520e+14,
                     0.538685623675312e+02,
                    -0.553089094625169e+05,
                    -0.102861522421405e+07,
                     0.204249418756234e+13,
                     0.273918446626977e+09,
                    -0.263963146312685e+16,
                    -0.107890854108088e+10,
                    -0.296492620980124e+11,
                    -0.111754907323424e+16  };
    
    // Reference Values
    double hstar = 3500.0;  // Units of kJ/ kg
    double sstar = 5.9; // Units of kJ/(kg K)
    double Pstar = 100.0;  // Units of MPa
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<32; i++) {
        hold += n[i] * pow(Eta-0.7,I[i]) * pow(Sigma-1.1,J[i]);
    };
    
    return pow(hold,4.0) * Pstar;

};

// User interface function to call appropriate sub-region function based on input sub-region.
double IFC97_R2_P_hs(int iRegion, double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 2.
     
     Equations are based on Equation (25), (26), and (27) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iRegion = Sub-region indicator. 0=2a, 1=2b, 2=2c
        h       = Specific enthalpy, kJ/kg
        s       = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Pressure, MPa
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    switch ( iRegion) {
        case 0: // Utilize Sub-region 2a
            return IFC97_R2_P_hs_2a(h, s);
            
        case 1: // Utilize Sub-region 2b
            return IFC97_R2_P_hs_2b(h, s);
            
        case 2: // Utilize Sub-region 2c
            return IFC97_R2_P_hs_2c(h, s);
            
        default:
            return -1000.0; // Incorrect sub-region input
    };
    
    return -2000.0;  // Unknown error
};