//
//  IFC97_R1_Basic.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 23/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R1_Basic.h"
#include "IAPWS_Surface_Tension.h"
#include "IAPWS_DynamicViscosity_Td.h"
#include "IAPWS_ThermalConductivity_TP.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

double IFC97_R1_Basic(int Property, double P, double T) {
    /* Provides the Basic equation for the Region 1 Gibbs free energy function.
     
     Equations are based on Equation (7) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Property = Indicates property to be returned
        P        = Saturation Pressure, MPa
        T        = Saturation Temperature, K
     
     Return:
        For Property = 0, Pressure, MPa
                       1, Temperature, K
                       2, Specific volume, m^3/kg
                       3, Specific internal energy, kJ/kg
                       4, Specific entropy, kJ/(kg K)
                       5, Specific enthalpy, kJ/kg
                       6, Specific isobaric heat capacity, kJ/(kg K)
                       7, Specific isochoric heat capacity, kJ/(kg K)
                       8, Density, kg/m^3
                       9, Speed of sound, m/s
                      10, Gibbs free energy, kJ/kg
                      11, Hemholtz free energy, kJ/kg
                      12, Quality, dimensionless
                      13, Ratio of specific heats (Cp/Cv), dimensionless
                      14, Surface Tension, N/m
                      15. Dynamic Viscosity, Pa*s
                      16. Thermal Conductivity, W/(K*m)
                      17. Thermal diffusivity, m^2/s
                      18. Kinematic Viscosity, m^2/s
                      19. Prandtl Number, dimensionless
                      20. Drho_DP_T, kg/(m^3 MPa), for thermal conductivity calculation
     
     Errors:
        -10000.0 = Incorrect Property input
        -90000.0 = Unknown error
        -80000.0 = Invalid Property Type
     */
    
    // Numerical coefficients from Table 2.
    int I[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 8, 8, 21, 23, 29, 30, 31, 32 };
    
    int J[] = { 0, -2, -1,  0,  1,  2,  3,  4,  5, -9,  -7, -1,   0,   1,   3, -3,   0,
                1,  3, 17, -4,  0,  6, -5, -2, 10, -8, -11, -6, -29, -31, -38,-39, -40, -41 };
    
    double n[] = {   0.0,
                     0.14632971213167e+00,
                    -0.84548187169114e+00,
                    -0.37563603672040e+01,
                     0.33855169168385e+01,
                    -0.95791963387872e+00,
                     0.15772038513228e+00,
                    -0.16616417199501e-01,
                     0.81214629983568e-03,
                     0.28319080123804e-03,
                    -0.60706301565874e-03,
                    -0.18990068218419e-01,
                    -0.32529748770505e-01,
                    -0.21841717175414e-01,
                    -0.52838357969930e-04,
                    -0.47184321073267e-03,
                    -0.30001780793026e-03,
                     0.47661393906987e-04,
                    -0.44141845330846e-05,
                    -0.72694996297594e-15,
                    -0.31679644845054e-04,
                    -0.28270797985312e-05,
                    -0.85205128120103e-09,
                    -0.22425281908000e-05,
                    -0.65171222895601e-06,
                    -0.14341729937924e-12,
                    -0.40516996860117e-06,
                    -0.12734301741641e-08,
                    -0.17424871230634e-09,
                    -0.68762131295531e-18,
                     0.14478307828521e-19,
                     0.26335781662795e-22,
                    -0.11947622640071e-22,
                     0.18228094581404e-23,
                    -0.93537087292458e-25 };
    
    // Reference Values
    double Pstar = 16.5300;  // Units of MPa
    double Tstar = 1386.00;  // Units of K
    double R     = 0.461526; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Tau = Tstar/T;
    
    // Temporary Variables
    double hold  = 0.0;
    double hold1 = 0.0;
    
    // Return requested property
    switch (Property) {
        case 0: // return Pressure,
            return P;
        
        case 1: // return Temperature
            return T;
        
        case 2: // return specific volume
            return R * T * Pi * IFC97_R1_Get_Gp(Pi, Tau, n, I, J) / (1000.0*P);
            
        case 3: // return specific internal energy
            return (Tau*IFC97_R1_Get_Gt(Pi, Tau, n, I, J) - Pi*IFC97_R1_Get_Gp(Pi, Tau, n, I, J)) * R * T;
            
        case 4: // return specific entropy
            return (Tau*IFC97_R1_Get_Gt(Pi, Tau, n, I, J) - IFC97_R1_Gibbs_ND(Pi, Tau, n, I, J)) * R;
            
        case 5: // return specific enthalpy
            return Tau * IFC97_R1_Get_Gt(Pi, Tau, n, I, J) * R * T;
            
        case 6: // return specific isobaric heat capacity
            return -pow(Tau,2.0) * IFC97_R1_Get_Gtt(Pi, Tau, n, I, J) * R;
            
        case 7: // return specific isochoric heat capacity
            hold = pow(IFC97_R1_Get_Gp(Pi, Tau, n, I, J) - Tau*IFC97_R1_Get_Gpt(Pi, Tau, n, I, J), 2.0) / IFC97_R1_Get_Gpp(Pi, Tau, n, I, J);
            return (-pow(Tau,2.0) * IFC97_R1_Get_Gtt(Pi, Tau, n, I, J) + hold) * R;
        
        case 8: // return density
            return (1000.0*P) / ( R * T * Pi * IFC97_R1_Get_Gp(Pi, Tau, n, I, J) );
        
        case 9: // return speed of sound
            hold = pow(IFC97_R1_Get_Gp(Pi, Tau, n, I, J)-Tau*IFC97_R1_Get_Gpt(Pi, Tau, n, I, J),2.0) / (pow(Tau,2.0)*IFC97_R1_Get_Gtt(Pi, Tau, n, I, J));
            return sqrt ( ( pow(IFC97_R1_Get_Gp(Pi, Tau, n, I, J),2.0) / (hold - IFC97_R1_Get_Gpp(Pi, Tau, n, I, J)) ) * R * T * 1000.0 );

        case 10: // return Gibbs free energy
            return IFC97_R1_Gibbs_ND(Pi, Tau, n, I, J) * R * T;
            
        case 11: // return Hemholtz free energy
            return IFC97_R1_Gibbs_ND(Pi, Tau, n, I, J)*R*T - P*IFC97_R1_Basic(2,P,T)*1000.0;
            
        case 12: // return Quality (Region 1 is liquid which is 0)
            return 0.0;
            
        case 13: // return ratio of specific heats (Cp/Cv)
            hold = pow(IFC97_R1_Get_Gp(Pi, Tau, n, I, J) - Tau*IFC97_R1_Get_Gpt(Pi, Tau, n, I, J), 2.0) / IFC97_R1_Get_Gpp(Pi, Tau, n, I, J);
            return (-pow(Tau,2.0) * IFC97_R1_Get_Gtt(Pi, Tau, n, I, J) * R)/( (-pow(Tau,2.0) * IFC97_R1_Get_Gtt(Pi, Tau, n, I, J) + hold) * R );
            
        case 14: // return Surface Tension, N/m
            return IAPWS_SurfaceTension(T);
            
        case 15: // return Dynamic Viscosity, Pa*s
            hold = (1000.0*P) / ( R * T * Pi * IFC97_R1_Get_Gp(Pi, Tau, n, I, J) );  // Density
            return IAPWS_DynamicViscosity_Td(T, hold);
            
        case 16: // return Thermal Conductivity, W/(K*m)
            return IAPWS_ThermalConductivity_TP(T, P);
            
        case 17:  // Thermal diffusivity, m^2/sec
            hold  = (1000.0*P) / ( R * T * Pi * IFC97_R1_Get_Gp(Pi, Tau, n, I, J) ); // Density
            hold1 = -pow(Tau,2.0) * IFC97_R1_Get_Gtt(Pi, Tau, n, I, J) * R;          // Cp
            return IAPWS_ThermalConductivity_TP(T, P) / (hold * hold1 * 1000.0 );
            
        case 18: // Kinematic viscosity, m^2/sec
            hold = (1000.0*P) / ( R * T * Pi * IFC97_R1_Get_Gp(Pi, Tau, n, I, J) ); // Density
            return IAPWS_DynamicViscosity_Td(T, hold) / hold;
            
        case 19: // Prandtl Number, dimensionless
            hold  = (1000.0*P) / ( R * T * Pi * IFC97_R1_Get_Gp(Pi, Tau, n, I, J) ); // Density
            hold1 = -pow(Tau,2.0) * IFC97_R1_Get_Gtt(Pi, Tau, n, I, J) * R;          // Cp
            return IAPWS_DynamicViscosity_Td(T, hold)*hold1*1000.0/IAPWS_ThermalConductivity_TP(T, P);
            
        case 20: // Drho_DP_T, kg/(m^3 MPa) for thermal conductivity calculation
            return (-1000.0*IFC97_R1_Get_Gpp(Pi, Tau, n, I, J)) / ( R * T * pow(IFC97_R1_Get_Gp(Pi, Tau, n, I, J),2.0) );
            
        default:
            return -10000.0;
    };
    
    return -90000.0;
};

// Non-Dimensional Gibbs free energy. Equation (7)
double IFC97_R1_Gibbs_ND(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional Gibbs free energy for Region 1.
     
     Equations are based on Equation (7) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 1
        Tau = Non-dimensional temperature for Region 1
        n   = Table 2 n-coefficient array
        I   = Table 2 I-coefficient array
        J   = Table 2 J-coefficient array
     
     Return:
        Non-dimensional Gibbs free energy
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    double hold = 0.0; // summer variable
    
    for (int i=1; i<35; i++) {
        hold += n[i] * pow(7.1-Pi,I[i]) * pow(Tau-1.222,J[i]);  // Equation (7)
    };
    
    return hold;
};

// Derivatives for the Gibbs free energy per Table 4
double IFC97_R1_Get_Gp(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the Gibbs free energy wrt P w/T constant.
     
     Equations are based on Table 4 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 1
        Tau = Non-dimensional temperature for Region 1
        n   = Table 2 n-coefficient array
        I   = Table 2 I-coefficient array
        J   = Table 2 J-coefficient array
     
     Return:
        Non-dimensional derivative of G wrt P
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<35; i++) {
        hold += -n[i] * I[i] * pow(7.1-Pi,I[i]-1) * pow(Tau-1.222,J[i]);
    };
    
    return hold;
};



double IFC97_R1_Get_Gpp(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional second derivative of the Gibbs free energy wrt P,P
     
     Equations are based on Table 4 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 1
        Tau = Non-dimensional temperature for Region 1
        n   = Table 2 n-coefficient array
        I   = Table 2 I-coefficient array
        J   = Table 2 J-coefficient array
     
     Return:
        Non-dimensional derivative of G wrt P,P
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<35; i++) {
        hold += n[i] * I[i] * (I[i]-1) * pow(7.1-Pi,I[i]-2) * pow(Tau-1.222,J[i]);
    };
    
    return hold;
};


double IFC97_R1_Get_Gt(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional derivative of the Gibbs free energy wrt T w/P constant.
     
     Equations are based on Table 4 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 1
        Tau = Non-dimensional temperature for Region 1
        n   = Table 2 n-coefficient array
        I   = Table 2 I-coefficient array
        J   = Table 2 J-coefficient array
     
     Return:
        Non-dimensional derivative of G wrt T
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<35; i++) {
        hold += n[i] * pow(7.1-Pi,I[i]) * J[i] * pow(Tau-1.222,J[i]-1);  // Equation (7)
    };
    
    return hold;

};


double IFC97_R1_Get_Gtt(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional second derivative of the Gibbs free energy wrt T, T
     
     Equations are based on Table 4 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 1
        Tau = Non-dimensional temperature for Region 1
        n   = Table 2 n-coefficient array
        I   = Table 2 I-coefficient array
        J   = Table 2 J-coefficient array
     
     Return:
        Non-dimensional derivative of G wrt T, T
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<35; i++) {
        hold += n[i] * pow(7.1-Pi,I[i]) * J[i] * (J[i]-1) * pow(Tau-1.222,J[i]-2);  // Equation (7)
    };
    
    return hold;
};


double IFC97_R1_Get_Gpt(double Pi, double Tau, double *n, int *I, int *J) {
    /* Provides the non-dimensional second derivative of the Gibbs free energy wrt P, T
     
     Equations are based on Table 4 of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        Pi  = Non-dimensional pressure for Region 1
        Tau = Non-dimensional temperature for Region 1
        n   = Table 2 n-coefficient array
        I   = Table 2 I-coefficient array
        J   = Table 2 J-coefficient array
     
     Return:
        Non-dimensional derivative of G wrt T, T
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    double hold = 0.0; // summer variable
    
    for (int i=1; i<35; i++) {
        hold += -n[i] * I[i] * pow(7.1-Pi,I[i]-1) * J[i] * pow(Tau-1.222,J[i]-1);  // Equation (7)
    };
    
    return hold;
};

// Interface functions to obtain specific properties
double IFC97_R1_SV_pt(double P, double T) {
    /* Provides the Region 1 specific volume given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific volume, m^3/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(2, P, T);
};


double IFC97_R1_U_pt(double P, double T) {
    /* Provides the Region 1 specific internal energy given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific internal energy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(3, P, T);
};


double IFC97_R1_S_pt(double P, double T) {
    /* Provides the Region 1 specific entropy given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific entropy, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(4, P, T);
};


double IFC97_R1_H_pt(double P, double T) {
    /* Provides the Region 1 specific enthalpy given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific enthalpy, kJ/kg
     
     Errors:
     None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(5, P, T);
};


double IFC97_R1_Cp_pt(double P, double T) {
    /* Provides the Region 1 specific isobaric heat capacity given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific isobaric heat capacity, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(6, P, T);
};


double IFC97_R1_Cv_pt(double P, double T) {
    /* Provides the Region 1 specific isochoric heat capacity given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Specific isochoric heat capacity, kJ/(kg K)
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(7, P, T);
};


double IFC97_R1_SOS_pt(double P, double T) {
    /* Provides the Region 1 speed of sound given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Speed of sound, m/s
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(8, P, T);
};


double IFC97_R1_Gibbs_pt(double P, double T) {
    /* Provides the Region 1 Gibbs free energy pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Gibbs free energy, kJ/kg
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(9, P, T);
};


double IFC97_R1_Hemholtz_pt(double P, double T) {
    /* Provides the Region 1 Hemholtz free energy given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Hemholtz free energy, kJ/kg
     
     Errors:
     None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Basic(10, P, T);

};

double IFC97_R1_k_Ratio_pt(double P, double T){
    /* Provides the Region 1 ratio of specific heats given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Ratio of specific heats, k, dimensionless
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return IFC97_R1_Cp_pt(P, T) / IFC97_R1_Cv_pt(P, T);
};

double IFC97_R1_RHO_pt(double P, double T) {
    /* Provides the Region 1 density given pressure and temperature
     
     Functions calls the basic region 1 equation above
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure, MPa
        T = Temperature, K
     
     Return:
        Density, kg/m^3
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    return 1.0 / IFC97_R1_SV_pt(P, T);
};