//
//  IFC97_R1_Backward_Tps.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 26/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R1_Backward_Tps.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

double IFC97_R1_T_ps(double P, double s) {
    /* Provides the backward equation for determining T based on P,s in Region 1.
     
     Equations are based on Equation (13) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        T   = Region 1 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 8.
    int I[] = { 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                   1, 1, 2, 2, 2, 2, 2, 3, 3, 4  };
    
    int J[] = { 0,  0,  1,  2,  3, 11, 31,  0,  1,  2,  3,
                   12, 31,  0,  1,  2,  9, 31, 10, 32, 32  };
    
    double n[] = {   0.0,
                     0.17478268058307e+03,
                     0.34806930892873e+02,
                     0.65292584978455e+01,
                     0.33039981775489e+00,
                    -0.19281382923196e-06,
                    -0.24909197244573e-22,
                    -0.26107636489332e+00,
                     0.22592965981586e+00,
                    -0.64256463395226e-01,
                     0.78876289270526e-02,
                     0.35672110607366e-09,
                     0.17332496994895e-23,
                     0.56608900654837e-03,
                    -0.32635483139717e-03,
                     0.44778286690632e-04,
                    -0.51322156908507e-09,
                    -0.42522657042207e-25,
                     0.26400441360689e-12,
                     0.78124600459723e-28,
                    -0.30732199903668e-30 };
    
    // Reference Values
    double Pstar = 1.0;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double sstar = 1.0; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Pi    = P/Pstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<21; i++) {
        hold += n[i] * pow(Pi,I[i]) * pow(Sigma+2.0,J[i]);
    };
    
    return hold * Tstar;
    
};