//
//  IFC97_R1_Backward_Tph.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 26/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R1_Backward_Tph.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

double IFC97_R1_T_ph(double P, double h) {
    /* Provides the backward equation for determining T based on P,h in Region 1.
     
     Equations are based on Equation (11) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P   = Pressure, MPa
        h   = Specific enthalpy, kJ/kg
     
     Return:
        T   = Region 1 temperature, K
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 6.
    int I[] = { 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
                   1, 1, 1, 2, 2, 3, 3, 4, 5, 6  };
    
    int J[] = { 0, 0,  1,  2,  6, 22, 32,  0,  1,  2,  3,
                   4, 10, 32, 10, 32, 10, 32, 32, 32, 32  };

    double n[] = {   0.0,
                    -0.23872489924521e+03,
                     0.40421188637945e+03,
                     0.11349746881718e+03,
                    -0.58457616048039e+01,
                    -0.15285482413140e-03,
                    -0.10866707695377e-05,
                    -0.13391744872602e+02,
                     0.43211039183559e+02,
                    -0.54010067170506e+02,
                     0.30535892203916e+02,
                    -0.65964749423638e+01,
                     0.93965400878363e-02,
                     0.11573647505340e-06,
                    -0.25858641282073e-04,
                    -0.40644363084799e-08,
                     0.66456186191635e-07,
                     0.80670734103027e-10,
                    -0.93477771213947e-12,
                     0.58265442020601e-14,
                    -0.15020185953503e-16 };
    
    // Reference Values
    double Pstar = 1.00;  // Units of MPa
    double Tstar = 1.0;  // Units of K
    double hstar = 2500.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<21; i++) {
        hold += n[i] * pow(Pi,I[i]) * pow(Eta+1.0,J[i]);
    };
    
    return hold * Tstar;

};