//
//  IFC97_R1_Backward_Phs.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 29/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_R1_Backward_Phs.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Supplemetary Release on Backward Equations for Pressure as a Function of Enthalpy and Entropy P(h,s) to the IAWPS Industrial Formualation 1997 for the Thermodynamic Properties of Water and Steam. Gaithersburh, Maryland, USA. September 2001.
 */

double IFC97_R1_P_hs(double h, double s) {
    /* Provides the backward equation for determining P based on h,s in Region 1.
     
     Equations are based on Equation (1) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Region 1 pressure, MPa
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 2.
    int I[] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1,
                   1, 1, 2, 2, 2, 3, 4, 4, 5  };
    
    int J[] = { 0,  0, 1,  2,  4,  5, 6, 8, 14, 0, 1,
                    4, 6,  0,  1, 10, 4, 1,  4, 0 };
    
    double n[] = {   0.0,
                    -0.691997014660582e+00,
                    -0.183612548787560e+02,
                    -0.928332409297335e+01,
                     0.659639569909906e+02,
                    -0.162060388912024e+02,
                     0.450620017338667e+03,
                     0.854680678224170e+03,
                     0.607523214001162e+04,
                     0.326487682621856e+02,
                    -0.269408844582931e+02,
                    -0.319947848334300e+03,
                    -0.928354307043320e+03,
                     0.303634537455249e+02,
                    -0.650540422444146e+02,
                    -0.430991316516130e+04,
                    -0.747512324096068e+03,
                     0.730000345529245e+03,
                     0.114284032569021e+04,
                    -0.436407041874559e+03    };
    
    // Reference Values
    double Pstar = 100.0;  // Units of MPa
    double hstar = 3400.0;  // Units of kJ/kg
    double sstar = 7.6; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Eta   = h/hstar;
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<20; i++) {
        hold += n[i] * pow(Eta+0.05,I[i]) * pow(Sigma+0.05,J[i]);
    };
    
    return hold * Pstar;
    
};