//
//  IFC97_P3sat_s.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 07/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_P3sat_s.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Supplementary Release on Backward Equations for the Functions T(P,h), v(P,h) and T(P,s), v(P,s) for Region 3 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properites of Water and Steam. Kyoto, Japan, September 2004
 */

// Region 3, Determination of saturation pressure based on entropy input. Equation (11).
double IFC97_P3sat_s(double s) {  // Equation 10
    /* Provides the backward equation for determining Region 3 saturation Pressure given entropy.
     
     Equations are based on Equation (11) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        s   = Specific entropy, kJ/(kg K)
     
     Return:
        P   = Pressure, MPa
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 3.
    int I[] = { 0,  0,  1,  1,  4, 12, 12, 16, 24, 28, 32 };
    
    int J[] = { 0,  0,  1, 32,  7,  4, 14, 36, 10,  0, 18 };
    
    double n[] = {   0.0,
                     0.639767553612785e+00,
                    -0.129727445396014e+02,
                    -0.224595125848403e+16,
                     0.177466741801846e+07,
                     0.717079349571538e+10,
                    -0.378829107169011e+18,
                    -0.955586736431328e+35,
                     0.187269814676188e+24,
                     0.119254746466473e+12,
                     0.110649277244882e+37 };
    
    // Reference Values
    double Pstar = 22.0;  // Units of MPa
    double sstar = 5.2; // Units of kJ/(kg K)
    
    // Non-dimensional values
    double Sigma = s/sstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<11; i++) {
        hold += n[i] * pow(Sigma-1.03,I[i]) * pow(Sigma-0.699,J[i]);
    };
    
    return hold * Pstar;
};
