//
//  IFC97_P3sat_h.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 07/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_P3sat_h.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Supplementary Release on Backward Equations for the Functions T(P,h), v(P,h) and T(P,s), v(P,s) for Region 3 of the IAPWS Industrial Formulation 1997 for the Thermodynamic Properites of Water and Steam. Kyoto, Japan, September 2004
 */

// Region 3, Determination of saturation pressure based on enthalpy input. Equation (10).
double IFC97_P3sat_h(double h) {  // Equation 10
    /* Provides the backward equation for determining Region 3 saturation Pressure given enthalpy.
     
     Equations are based on Equation (10) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h   = Specific enthalpy, kJ/kg
     
     Return:
        P   = Pressure, MPa
     
     Errors:
        None.  Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    // Numerical coefficients from Table 17.
    int I[] = { 0,  0,  1,  1,  1, 1, 5, 7, 8, 14, 20,
                   22, 24, 28, 36 };
    
    int J[] = { 0,  0,  1,  3,  4, 36,  3,  0, 24, 16, 16,
                    3, 18,  8, 24  };
    
    double n[] = {   0.0,
                     0.600073641753024e+00,
                    -0.936203654849857e+01,
                     0.246590798594147e+02,
                    -0.107014222858224e+03,
                    -0.915821315805768e+14,
                    -0.862332011700662e+04,
                    -0.235837344740032e+02,
                     0.252304969384128e+18,
                    -0.389718771997719e+19,
                    -0.333775713645296e+23,
                     0.356499469636328e+11,
                    -0.148547544720641e+27,
                     0.330611514838798e+19,
                     0.813641294467829e+38  };
    
    // Reference Values
    double Pstar = 22.0;  // Units of MPa
    double hstar = 2600.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Eta = h/hstar;
    
    // Solution summer variable
    double hold = 0.0;
    
    for (int i=1; i<15; i++) {
        hold += n[i] * pow(Eta-1.02,I[i]) * pow(Eta-0.608,J[i]);
    };
    
    return hold * Pstar;
};
