//
//  IFC97_GetRegion_hs.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 22/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetRegion_hs.h"

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_GetRegion_Ph.h"
#include "IFC97_GetProperty_Ph.h"
#include "IFC97_GetProperty_Ph_Region4.h"
#include "IFC97_GetRegion_Ps.h"
#include "IFC97_GetProperty_Ps.h"
#include "IFC97_GetProperty_Ps_Region4.h"
#include "IFC97_Bhmax_s.h"
#include "IFC97_Bhmin_s.h"


// For inputs of Enthalpy and Entropy, provides the IFC97 Region where the state resides, and SubRegion if applicable.

int IFC97_GetRegion_hs(double h, double s, int *Region, int *SubRegion) {
    /*
     For cases were Entropy and Enthalpy are input, function determines the appropriate IFC97 region to which the state belongs.
     
     Inputs:
         h         = Specific enthalpy, kJ/kg
         s         = Specific entropy, kJ/(kg K)
         Region    = Location where main IFC97 region is returned
         SubRegion = Location where the IFC97 SubRegion, if applicalble, is returned
     
     Outputs:
         Function  = 0 if no errors.  Otherwise, see below.
         Region    = 1, 2, 3, or 4 (saturated) for main IFC97 region
         SubRegion = For Region 2, this will be 0, 1, or 2 depending on location 2a, 2b, or 2c
                     For Region 3, this will be 0 or 1 depending on location 3a or 3b
                     For Region 1 and 4, this will be 0 (no SubRegion)
     
     Errors:
     -31000  = Enthalpy too low
     -32000  = Enthalpy too high
     -41000  = Entropy too low
     -42000  = Entropy too high
     -90000  = Unknown Error
     */
    // Establish the boundaries for the inputs. Pressures in MPa and Temperatures in K
    double Tmin = 273.15;
    double Tmax = 1073.15;
    double Pmax = 100.0;
    double Pmin = IFC97_Psat(Tmin);
    
    // Maximum range of allowable entropy values
    double smin = IFC97_R1_S_pt(Pmax, Tmin);
    double smax = IFC97_R2_S_pt(Pmin, Tmax);
    
    // Maximum range of allowable enthalpy values
    double hmin = IFC97_Bhmin_s(s);
    double hmax = IFC97_Bhmax_s(s);
    
    // Perform check to verify inputs are within allowable ranges
    if (s < smin) return -41000;
    if (s > smax) return -42000;
    if (h < hmin) return -31000;
    if (h > hmax) return -32000;
    
    // If within allowable boundaries, continue to determine Region
    // Determine entropy check points
    double T_B13          = 623.15; // Isothermal line dividing Region 1 and Region 3
    double s_T_B13        = IFC97_R1_S_pt(IFC97_Psat(T_B13), T_B13); // Saturated liquid entropy at isothermal intersection with saturation region
    double s_critical     = 4.41202148223476;  // Critical point entropy
    double s_intermediate = 5.85;
    double s_Region2      = IFC97_R2_S_pt(IFC97_Psat(Tmin), Tmin); // Saturated vaport entropy at Tmin
    
    // Actual Region determination 
    if ( s <= s_T_B13 ) { // In Region 1, 3, or 4
        if ( h < IFC97_h_liq_1_s(s) )    *Region = 4;
        else { // In Region 1 or 3
            if ( h > IFC97_hB13_s(s) )   *Region = 3;
            else                         *Region = 1;
        };
    }
    else if ( s <= s_critical ) { // In Region 3 or 4
        if ( h < IFC97_h_liq_3a_s(s) )   *Region = 4;
        else                             *Region = 3;
    }
    else if ( s >= s_Region2)            *Region = 2;
    else if ( s >= s_intermediate ) { // In Region 2 or 4
        if ( h < IFC97_h_vap_2ab_s(s) )  *Region = 4;
        else                             *Region = 2;
    }
    else { // In Region 2, 3 or 4
        if ( h < IFC97_h_vap_2c3b_s(s) ) *Region = 4;
        else { // In Region 2 or 3
            // Set boundary limits on bounding box
            double s_min_B23 = 5.048096828; // Units are kJ/(kg K)
            double s_max_B23 = 5.260578707; // Units are kJ/(kg K)
            double h_min_B23 = IFC97_R2_H_pt(IFC97_Psat(T_B13), T_B13);
            double h_max_B23 = IFC97_R2_H_pt(100.0, 863.15);
            
            if      ( s<=s_min_B23 || h<=h_min_B23 ) *Region = 3;
            else if ( s>=s_max_B23 || h>=h_max_B23 ) *Region = 2;
            else {
                double T_B23    = IFC97_T_B23_hs(h, s);
                double P_B23    = IFC97_R2_P_hs_2c(h, s);
                double P_B23_97 = IFC97_B23_GetPFromT(T_B23);
                
                if ( P_B23 > P_B23_97)               *Region = 3;
                else                                 *Region = 2;
            };
        };
    };
    
    // Determine SubRegion as needed
    switch ( *Region ) {
        case 1:
            SubRegion = 0; // No SubRegion
            break;
        
        case 2:
            if      ( h <= IFC97_h2ab(s)  ) *SubRegion = 0; // SubRegion 2a
            else if ( s >= s_intermediate ) *SubRegion = 1; // SubRegion 2b
            else                            *SubRegion = 2; // SubRegion 2c
            break;
            
        case 3:
            if ( s > s_critical ) *SubRegion = 1; // SubRegion 3b
            else                  *SubRegion = 0; // SubRegion 3a
            break;
            
        default:
            *SubRegion = 0; // Case 4, No SubRegion
            break;
    };
    
    return -90000;  // Unknown error
};