//
//  IFC97_GetRegion_Ts.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 24/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetRegion_Ts.h"

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetProperty_PT.h"


// For inputs of Temperature and Enthalpy, provides the IFC97 Region where the state resides, and SubRegion if applicable.
// This is a complete internal (i.e., no IAPWS) evalution.  This is why there are no SubRegions

int IFC97_GetRegion_Ts(double T, double s, int *Region, int *SubRegion, double *P_guess) {
    /*
     For cases were Temperature and Enthalpy are input, function determines the appropriate IFC97 region to which the state belongs.
     
     Inputs:
         T         = Temperature, K
         s         = Specific entropy, kJ/(kg K)
         Region    = Location where main IFC97 region is returned
         SubRegion = Location where the IFC97 SubRegion, if applicalble, is returned
         P_guess   = Variable in which to return the initial guess for pressure at the given T,s. No input is used.
     
     Outputs:
         Function  = 0 if no errors.  Otherwise, see below.
         Region    = 1, 2, 3, or 4 (saturated) for main IFC97 region
         SubRegion = 0 for all cases (i.e., no SubRegions) as this is an internal function (i.e., non-IAPWS) only
         P_guess   = Initial guess for pressure at the given T,s. Return Psat in Region 4 (Saturation)
     
     Errors:
         -11000  = Pressure too low
         -12000  = Pressure too high
         -41000  = Entropy too low
         -42000  = Entropy too high
         -90000  = Unknown Error
     */
    // Establish the boundaries for the inputs. Pressures in MPa and Temperatures in K
    double Tmin      = 273.15;
    double Tmax      = 1073.15;
    double Pmax      = 100.0;
    double Pmin      = IFC97_Psat(Tmin);
    double Tcritical = 647.096;
    double T_B13     = 623.15; // Constant temperature boundary between Region 1 and Region 3
    double T_B23_max = 863.15; // Constant maximum temperature limit between Region 2 and Region 3
    
    // Review if inputs are within allowable boundaries
    if (T < Tmin) return -11000;
    if (T > Tmax) return -12000;
    if (s < IFC97_GetProperty_PT(4, Pmax, T, -1)) return -41000;
    if (s > IFC97_R2_S_pt(Pmin, T))               return -42000;
    
    // If within allowable boundaries, continue to determine Region
    // Take horizontal slices at the three main Temperature break points, 623.15K, Tcritical, and 863.15K
    
    // Declare some placeholders
    double Psat      = 0.0;
    double PB23      = 0.0;  // Boundary pressure PB23 bewteen Region 2 and Region 3
    double SatLiquid = 0.0;
    double SatVapor  = 0.0;
    double s_PB23    = 0.0;  // Entropy at the PB23 boundary between Region 2 and 3
    double s_bound   = 0.0;  // End point boundary based on Region (either s_min for liquid, or s_max for vapor)
    
    if ( T <= T_B13 ) { // Will be in Region 1, Region 2, or Region 4
        // Get saturation conditions
        Psat = IFC97_Psat(T);
        SatLiquid = IFC97_R1_S_pt(Psat, T);
        SatVapor  = IFC97_R2_S_pt(Psat, T);
     
        // Determine Region and Get Initial Guess for Pressure
        if ( s <= SatLiquid ) {     // If less than or equal to saturated liquid, Region 1
            *Region = 1;
            
            // Determine initial guess for pressure. Linear interpolation between P100 value and saturation Pressure.
            s_bound = IFC97_R1_S_pt(Pmax, T);
            *P_guess = Pmax + ( (s-s_bound)/(SatLiquid-s_bound) )*(Psat-Pmax);
        }
        else if ( s < SatVapor  ) { // If less than saturated vapor, Region 4
            *Region = 4;
            
            // Return saturation pressure
            *P_guess = Psat; // Saturated. Saturation Pressure at T
        }
        else {                      // Else in Region 2
            *Region = 2;
            
            // Determine initial guess for pressure. Linear interpolation between saturation Pressure and Pmin.
            s_bound = IFC97_R2_S_pt(Pmin, T);
            *P_guess = Psat + ( (s-SatVapor)/(s_bound-SatVapor) )*(Pmin-Psat);
        };
    }
    else if ( T <= Tcritical ) { // Will be in Region 2, Region 3, or Region 4
        // Get saturation conditions and PB23 boundary condition
        Psat      = IFC97_Psat(T);
        PB23      = IFC97_B23_GetPFromT(T);
        SatLiquid = IFC97_GetProperty_PT(4, 0.0, T, 0);
        SatVapor  = IFC97_GetProperty_PT(4, 0.0, T, 1);
        s_PB23    = IFC97_R2_S_pt(PB23, T);
        
        // Determine Region and Get Initial Guess for Pressure
        if ( s <= SatLiquid )    {  // If <= to saturated liquid s, Region 3
            *Region = 3;
            
            // Determine initial guess for pressure. Linear interpolation between saturation Pressure and Pmin.
            s_bound = IFC97_GetProperty_PT(4, Pmax, T, -1);
            *P_guess = Pmax + ( (s-s_bound)/(SatLiquid-s_bound) )*(Psat-Pmax);
        }
        else if ( s < SatVapor ) {  // If less than saturated vapor s, Region 4
            *Region = 4;
            
            // Return saturation pressure
            *P_guess = Psat; // Saturated. Saturation Pressure at T
        }
        else if ( s < s_PB23 )   {  // If < s at the P23 Boundary, Region 3
            *Region = 3;
            
            // Determine initial guess for pressure. Linear interpolation between saturation Pressure and PB23 boundary.
            *P_guess = Psat + ((s-SatVapor)/(s_PB23-SatVapor))*(PB23-Psat);
        }
        else {                      // Else in Region 2
            *Region = 2;

            // Determine initial guess for pressure. Linear interpolation between PB23 boundary and Pmin.
            s_bound = IFC97_R2_S_pt(Pmin, T);
            *P_guess = PB23 + ( (s-s_PB23)/(s_bound-s_PB23) )*(Pmin-PB23);
        };
    }
    else if ( T < T_B23_max ){ // Will be in Region 3 or Region 2
        // Get boundary conditions
        PB23   = IFC97_B23_GetPFromT(T);
        s_PB23 = IFC97_R2_S_pt(PB23, T);
                
        // Determine Region and Get Initial Guess for Pressure
        if ( s <  s_PB23 ) { // If <= to the s at the P23 boundary temp in Region 3
            *Region = 3;
            
            // Determine initial guess for pressure. Linear interpolation between P100 value and PB23.
            s_bound = IFC97_GetProperty_PT(4, Pmax, T, -1);
            *P_guess = Pmax + ( (s-s_bound)/(s_PB23-s_bound) )*(PB23-Pmax);
        }
        else {               // Else in Region 2
            *Region = 2;

            // Determine initial guess for pressure. Linear interpolation between PB23 boundary and Pmin.
            s_bound = IFC97_R2_S_pt(Pmin, T);
            *P_guess = PB23 + ( (s-s_PB23)/(s_bound-s_PB23) )*(Pmin-PB23);
        };
    }
    else { // Will be in Region 2
        *Region = 2;
        
        // Determine initial guess for pressure. Linear interpolation between P100 value and Pmin.
        double smin = IFC97_R1_S_pt(Pmax, T);
        s_bound     = IFC97_R2_S_pt(Pmin, T);
        *P_guess     = Pmax + ( (s-smin)/(s_bound-smin) )*(Pmin-Pmax);
    };
    
    // Determine the SubRegion, not applicable
    *SubRegion = 0;
    
    return -90000;
};