//
//  IFC97_GetRegion_Ps.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 22/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetRegion_Ps.h"

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetProperty_PT.h"


// For inputs of Pressure and Enthalpy, provides the IFC97 Region where the state resides, and SubRegion if applicable.

int IFC97_GetRegion_Ps(double P, double s, int *Region, int *SubRegion) {
    /*
     For cases were Pressure and Enthalpy are input, function determines the appropriate IFC97 region to which the state belongs.
     
     Inputs:
         P         = Pressure, MPa
         s         = Specific entropy, kJ/(kg K)
         Region    = Location where main IFC97 region is returned
         SubRegion = Location where the IFC97 SubRegion, if applicalble, is returned
     
     Outputs:
         Function  = 0 if no errors.  Otherwise, see below.
         Region    = 1, 2, 3, or 4 (saturated) for main IFC97 region
         SubRegion = For Region 2, this will be 0, 1, or 2 depending on location 2a, 2b, or 2c
                     For Region 3, this will be 0 or 1 depending on location 3a or 3b
                     For Region 1 and 4, this will be 0 (no SubRegion)
     
     Errors:
         -21000  = Pressure too low
         -22000  = Pressure too high
         -41000  = Entropy too low
         -42000  = Entropy too high
         -90000  = Unknown Error
     */
    // Establish the boundaries for the inputs. Pressures in MPa and Temperatures in K
    double Tmin = 273.15;
    double Tmax = 1073.15;
    double Pmax = 100.0;
    double Pmin = IFC97_Psat(Tmin);
    double Pcritical = 22.064;
    double T_B13 = 623.15; // Constant temperature boundary between Region 1 and Region 3
    
    // Review if inputs are within allowable boundaries
    if (P < Pmin) return -21000;
    if (P > Pmax) return -22000;
    if (s < IFC97_R1_S_pt(P, Tmin)) return -41000;
    if (s > IFC97_R2_S_pt(P, Tmax)) return -42000;
    
    // If within allowable boundaries, continue to determine Region
    // Take horizontal slices at the two main Pressure break points, Psat(623.15K) and Pcritical,
    
    if ( P <= IFC97_Psat(T_B13) ) { // Will be in Region 1, Region 2, or Region 4
        if      ( s <= IFC97_R1_S_pt(P, IFC97_Tsat(P)) ) *Region = 1; // If less than or equal to saturated liquid, Region 1
        else if ( s <  IFC97_R2_S_pt(P, IFC97_Tsat(P)) ) *Region = 4; // If less than saturated vapor, Region 4
        else                                             *Region = 2; // Else in Region 2
    }
    else if ( P <= Pcritical ) { // Can be in any region
        if      ( s <= IFC97_R1_S_pt(P, T_B13) )                  *Region = 1; // If <= to the h at the 1-3 boundary temp in Region 1
        else if ( s <= IFC97_GetProperty_PT(4, P, 0.0, 0) )       *Region = 3; // If <= to saturated liquid h, Region 3
        else if ( s <  IFC97_GetProperty_PT(4, P, 0.0, 1) )       *Region = 4; // If less than saturated vapor h, Region 4
        else if ( s <  IFC97_R2_S_pt(P, IFC97_B23_GetTFromP(P)) ) *Region = 3; // If < h at the P23 Boundary, Region 3
        else                                                      *Region = 2; // Else in Region 2
    }
    else { // Will be in Region 1, Region 3, or Region 2
        if      ( s <= IFC97_R1_S_pt(P, T_B13) )                  *Region = 1; // If <= to the h at the 1-3 boundary temp in Region 1
        else if ( s <  IFC97_R2_S_pt(P, IFC97_B23_GetTFromP(P)) ) *Region = 3; // If < h at the P23 Boundary, Region 3
        else                                                      *Region = 2; // Else in Region 2
    };
    
    // Determine the SubRegion, if applicable
    switch (*Region) {
        case 1:
            SubRegion = 0; // No SubRegion
            break;
            
        case 2: // From Reference 1, Article 6.3
            if      ( P < 4.0 )   *SubRegion = 0; // SubRegion 2a
            else if ( s >= 5.85 ) *SubRegion = 1; // SubRegion 2b
            else                  *SubRegion = 2; // SubRegion 2c
            break;
            
        case 3: // From Reference 4, Article 3.2
            if ( s <= 4.41202148223476 ) *SubRegion = 0; // SubRegion 3b
            else                         *SubRegion = 1; // SubRegion 3a
            break;
            
        case 4:
            SubRegion = 0; // No SubRegion
            break;
            
        default:
            return -90000; // Unknown error
    };
    
    return 0;
};