//
//  IFC97_GetRegion_PT.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 20/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetRegion_PT.h"

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
//#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"

// For inputs of Pressure and Temperature, provides the IFC97 Region where the state resides.

int IFC97_GetRegion_PT(double P, double T, int *Region, int *SubRegion) {
  /*
    For cases were Pressure and Temperature are input, function determines the appropriate IFC97 region to which the state belongs.
   
   Inputs:
    P         = Pressure, MPa
    T         = Temperatre, K
    Region    = Location where main IFC97 region is returned
    SubRegion = Location where the IFC97 SubRegion, if applicalble, is returned
   
   Outputs:
    Function  = 0 if no errors.  Otherwise, see below.
    Region    = 1, 2, or 3 for main IFC97 region
    SubRegion = For Region 3, this will be 0-25 depending on location for v(P,T) backward/auxiliary function.
                For Region 1 and 2, this will be 0 (no SubRegion)
   
   Errors:
    -11000  = Temperature too low
    -12000  = Temperature too high
    -21000  = Pressure too low
    -22000  = Pressure too high
    -90000  = Unknown Error
   */
    // Establish the boundaries for the inputs
    double Tmin = 273.15;
    double Tmax = 1073.15;
    double Pmax = 100.0;
    double Pmin = IFC97_Psat(Tmin);
    
    // Review if inputs are within allowable boundaries
    if (T < Tmin) return -11000;
    if (T > Tmax) return -12000;
    if (P < Pmin) return -21000;
    if (P > Pmax) return -22000;
    
    // If within allowable boundaries, continue to determine Region and SubRegion
    
    if (T <= 623.15) { // Either Region 1 or Region 2
        if ( P > IFC97_Psat(T) ) { // Check to see if above saturation pressure to indicate Region 1
            *Region    = 1;
            *SubRegion = 0;
            return 0;
        } // If not, the state is in Region 2. 
        else {
            *Region    = 2;
            *SubRegion = 0;
            return 0;
        };
    }
    else { // Either Region 2 or Region 3
        if ( T <= 863.15 ) {
            if ( P > IFC97_B23_GetPFromT(T)) { // In Region 3
                *Region    = 3;
                // Determine Sub-Region below
            }
            else { // In Region 2
                *Region    = 2;
                *SubRegion = 0;
                return 0;
            };
        }
        else { // In Region 2
            *Region    = 2;
            *SubRegion = 0;
            return 0;
        };
    };
    
    // If reached this location, then determine the SubRegion within Region 3
    
    // Set some hard coded Pressure check points
    double P3cd = 1.900881189173929e+01;
    double Psat_00264 = 2.193161551e+01; // Auxiliary Region check
    double Psat_00385 = 2.190096265e+01; // Auxiliary Region check
    
    // Set up pressure boundary vector
    double P_bounds[] = { 40.0, 25.0, 23.5, 23.0, 22.5, 22.11, 22.064, Psat_00264, Psat_00385, IFC97_Psat(643.15), 20.5, P3cd };
    
    *SubRegion = 0;
    
    while (*SubRegion < 12) {
        if (P > P_bounds[*SubRegion]) break;
        (*SubRegion)++;
    };
    
    switch (*SubRegion) {
        case 0:
            if      ( T > IFC97_R3_T3ab_P(P) ) *SubRegion = 1; // SubRegion 3b
            else                               *SubRegion = 0; // SubRegion 3a
            break;
            
        case 1:
            if      ( T > IFC97_R3_T3ef_P(P) ) *SubRegion = 5; // SubRegion 3f
            else if ( T > IFC97_R3_T3ab_P(P) ) *SubRegion = 4; // SubRegion 3e
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 3; // SubRegion 3d
            else                               *SubRegion = 2; // SubRegion 3c
            break;

        case 2:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3ij_P(P) ) *SubRegion = 9; // SubRegion 3j
            else if ( T > IFC97_R3_T3ef_P(P) ) *SubRegion = 8; // SubRegion 3i
            else if ( T > IFC97_R3_T3gh_P(P) ) *SubRegion = 7; // SubRegion 3h
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 6; // SubRegion 3g
            else                               *SubRegion = 2; // SubRegion 3c
            break;

        case 3:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3ij_P(P) ) *SubRegion = 9; // SubRegion 3j
            else if ( T > IFC97_R3_T3ef_P(P) ) *SubRegion = 8; // SubRegion 3i
            else if ( T > IFC97_R3_T3gh_P(P) ) *SubRegion = 7; // SubRegion 3h
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 11;// SubRegion 3l
            else                               *SubRegion = 2; // SubRegion 3c
            break;
            
        case 4:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3ij_P(P) ) *SubRegion = 9; // SubRegion 3j
            else if ( T > IFC97_R3_T3op_P(P) ) *SubRegion = 15;// SubRegion 3p
            else if ( T > IFC97_R3_T3ef_P(P) ) *SubRegion = 14;// SubRegion 3o
            else if ( T > IFC97_R3_T3mn_P(P) ) *SubRegion = 13;// SubRegion 3n
            else if ( T > IFC97_R3_T3gh_P(P) ) *SubRegion = 12;// SubRegion 3m
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 11;// SubRegion 3l
            else                               *SubRegion = 2; // SubRegion 3c
            break;

        case 5:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3rx_P(P) ) *SubRegion = 17;// SubRegion 3r
            else if ( T > IFC97_R3_T3wx_P(P) ) *SubRegion = 23;// SubRegion 3x
            else if ( T > IFC97_R3_T3ef_P(P) ) *SubRegion = 22;// SubRegion 3w
            else if ( T > IFC97_R3_T3uv_P(P) ) *SubRegion = 21;// SubRegion 3v
            else if ( T > IFC97_R3_T3qu_P(P) ) *SubRegion = 20;// SubRegion 3u
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 16;// SubRegion 3q
            else                               *SubRegion = 2; // SubRegion 3c
            break;
            
        case 6:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3rx_P(P) ) *SubRegion = 17;// SubRegion 3r
            else if ( T > IFC97_R3_T3wx_P(P) ) *SubRegion = 23;// SubRegion 3x
            else if ( T > IFC97_R3_T3ef_P(P) ) *SubRegion = 25;// SubRegion 3z
            else if ( T > IFC97_R3_T3uv_P(P) ) *SubRegion = 24;// SubRegion 3y
            else if ( T > IFC97_R3_T3qu_P(P) ) *SubRegion = 20;// SubRegion 3u
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 16;// SubRegion 3q
            else                               *SubRegion = 2; // SubRegion 3c
            break;
            
        case 7:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3rx_P(P) ) *SubRegion = 17;// SubRegion 3r
            else if ( T > IFC97_R3_T3wx_P(P) ) *SubRegion = 23;// SubRegion 3x
            else if ( T > IFC97_Tsat(P)      ) *SubRegion = 25;// SubRegion 3z
            else if ( T > IFC97_R3_T3uv_P(P) ) *SubRegion = 24;// SubRegion 3y
            else if ( T > IFC97_R3_T3qu_P(P) ) *SubRegion = 20;// SubRegion 3u
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 16;// SubRegion 3q
            else                               *SubRegion = 2; // SubRegion 3c
            break;
            
        case 8:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3rx_P(P) ) *SubRegion = 17;// SubRegion 3r
            else if ( T > IFC97_R3_T3wx_P(P) ) *SubRegion = 23;// SubRegion 3x
            else if ( T > IFC97_Tsat(P)      ) *SubRegion = 25;// SubRegion 3z
            else if ( T > IFC97_R3_T3qu_P(P) ) *SubRegion = 20;// SubRegion 3u
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 16;// SubRegion 3q
            else                               *SubRegion = 2; // SubRegion 3c
            break;

        case 9:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_R3_T3rx_P(P) ) *SubRegion = 17;// SubRegion 3r
            else if ( T > IFC97_Tsat(P)      ) *SubRegion = 23;// SubRegion 3x
            else if ( T > IFC97_R3_T3uv_P(P) ) *SubRegion = 24;// SubRegion 3y
            else if ( T > IFC97_R3_T3qu_P(P) ) *SubRegion = 20;// SubRegion 3u
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 16;// SubRegion 3q
            else                               *SubRegion = 2; // SubRegion 3c
            break;

        case 10:
            if      ( T > IFC97_R3_T3jk_P(P) ) *SubRegion = 10;// SubRegion 3k
            else if ( T > IFC97_Tsat(P)      ) *SubRegion = 17;// SubRegion 3r
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 18;// SubRegion 3s
            else                               *SubRegion = 2; // SubRegion 3c
            break;

        case 11:
            if      ( T > IFC97_Tsat(P)      ) *SubRegion = 19;// SubRegion 3t
            else if ( T > IFC97_R3_T3cd_P(P) ) *SubRegion = 18;// SubRegion 3s
            else                               *SubRegion = 2; // SubRegion 3c
            break;
            
        case 12:
            if      ( T > IFC97_Tsat(P)      ) *SubRegion = 19;// SubRegion 3t
            else                               *SubRegion = 2; // SubRegion 3c
            break;

        default:
            return -90000; // Unknown error
    };
    
    return -90000;
    
};
