//
//  IFC97_GetProperty_hs_R4.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 24/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetProperty_hs_R4.h"
#include <math.h>

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_GetRegion_Ph.h"
#include "IFC97_GetProperty_Ph_Region4.h"
#include "IFC97_GetRegion_Ps.h"
#include "IFC97_GetProperty_Ps_Region4.h"
#include "IFC97_GetRegion_hs.h"
#include "IFC97_Bhmin_s.h"
#include "IFC97_R4_h_Tx.h"
#include "IFC97_R4_s_Tx.h"

/*
 Internal function to get the quality and saturation conditions of a state within the saturation region based on Pressure and Entropy.
 */

double IFC97_GetProperty_hs_R4(int Property, double h, double s) {
    /* Returns the requested property in the saturation region based on given h, s input.
     
     Inputs:
         Property   = Indicates property to be returned
         h          = Specific enthalpy, kJ/kg
         s          = Specific entropy, kJ/(kg K)
     
     Return:
         For Property =  0, Pressure, MPa
                         1, Temperature, K
                         2, Specific volume, m^3/kg
                         3, Specific internal energy, kJ/kg
                         4, Specific entropy, kJ/(kg K)
                         5, Specific enthalpy, kJ/kg
                         6, Specific isobaric heat capacity, kJ/(kg K)
                         7, Specific isochoric heat capacity, kJ/(kg K)
                         8, Density, kg/m^3
                         9, NOT APPLICABLE (Speed of sound, m/s)
                         10, Gibbs free energy, kJ/kg
                         11, Hemholtz free energy, kJ/kg
                         12, Quality, dimensionless
                         13, Ratio of specific heats (Cp/Cv), dimensionless
                         14, NOT APPLICABLE Surface Tension, N/m
                         15. NOT APPLICABLE Dynamic Viscosity, Pa*s
                         16. NOT APPLICABLE Thermal Conductivity, W/(K*m)
                         17. NOT APPLICABLE Thermal diffusivity, m^2/s
                         18. NOT APPLICABLE Kinematic Viscosity, m^2/s
                         19. NOT APPLICABLE Prandtl Number, dimensionless
     
     Errors:
         -31000  = Enthalpy too low
         -32000  = Enthalpy too high
         -41000  = Entropy too low
         -42000  = Entropy too high
         -80000  = Invalid Property
         -99990  = Iterations exceeded
         -90000  = Unknown Error
     */
    // First determine the saturation temperature and quality
    
    // Declare placeholders for the saturation temperature and pressure and quality
    double Tsat    = 0.0;
    double Quality = 0.0;
    
    // Establish entropy break point for IAPWS initial guess and internally generated initial guess
    double s_IAPWS = IFC97_R2_Basic(4, IFC97_Psat(623.15), 623.15); // If equal to or greater than this, use IAPWS equations
    
    // Determine initial guess for saturation temperature and quality
    if ( s >= s_IAPWS ) { // Use IAPWS equations from Reference 3, Equation 9
        Tsat    = IFC97_Tsat_hs(h, s);
        Quality = (h-IFC97_GetProperty_PT(5, 0.0, Tsat, 0)) / (IFC97_GetProperty_PT(5, 0.0, Tsat, 1)-IFC97_GetProperty_PT(5, 0.0, Tsat, 0));
    }
    else { // Use internal (i.e., non-IAPWS) correlation for initial guess.
        // Establish coefficients for internal (i.e., non-IAPWS) correlation between saturation enthalpy and temperature
        double Coeff[2][5] = { {-9.561784800E-12, 1.233425742E-08, -1.496868029E-05, 2.447153153E-01,  2.726769809E+02},   // Saturated Liquid
                               {-1.641116059E-10, 1.337225506E-06, -4.141383296E-03, 5.782551640E+00, -2.425078545E+03} }; // Saturated Vapor to 623.15K
        
        // Get the upper and lower bounds for saturation enthalpy based on input entropy. Use IAPWS enthalpy vs. entropy relations
        
        // Establish entropy break points
        double s_critical = 4.41202148223476;               // Equal to or above, use h".2c3b(s) from Reference 3, Eq. (6)
        double s_623_15   = IFC97_R1_Basic(4, 0.0, 623.15); // Equal to or above but below s_critical, use h'.3a(s) from Reference 3, Eq. (4)
        
        // Get upper bounds for enthalpy and temperature
        double h_upper = 0.0;
        double T_upper = 0.0;
        
        if ( s > s_critical ) { // Obtain from saturated vapor conditions
            h_upper = IFC97_h_vap_2c3b_s(s);
            T_upper = pow(h_upper, 4.)*Coeff[1][0]+pow(h_upper, 3.)*Coeff[1][1]+pow(h_upper, 2.)*Coeff[1][2]+h_upper*Coeff[1][3]+Coeff[1][4];
        }
        else if (s > s_623_15 ) { // Obtain from saturated liquid conditions in Region 3
            h_upper = IFC97_h_liq_3a_s(s);
            T_upper = pow(h_upper, 4.)*Coeff[0][0]+pow(h_upper, 3.)*Coeff[0][1]+pow(h_upper, 2.)*Coeff[0][2]+h_upper*Coeff[0][3]+Coeff[0][4];
        }
        else { // Obtain from saturated liquid conditions in Region 1
            h_upper = IFC97_h_liq_1_s(s);
            T_upper = pow(h_upper, 4.)*Coeff[0][0]+pow(h_upper, 3.)*Coeff[0][1]+pow(h_upper, 2.)*Coeff[0][2]+h_upper*Coeff[0][3]+Coeff[0][4];
        };
        
        // Get lower bounds for ethalpy and temperature
        double h_lower = IFC97_Bhmin_s(s);
        double T_lower = 273.15; // Minimum temperature
        
        // Get initial guess for saturation temperature and quality
        Tsat    = T_lower + ((h-h_lower)/(h_upper-h_lower))*(T_upper-T_lower);
        Quality = (h-IFC97_GetProperty_PT(5, 0.0, Tsat, 0)) / (IFC97_GetProperty_PT(5, 0.0, Tsat, 1)-IFC97_GetProperty_PT(5, 0.0, Tsat, 0));
    };
    
    // Obtain actual saturation temperature and quality based on double iteration on known enthalpy and entropy in Region 4
    
    // Establish allowable tolerances on solutions to iterations
    double Tol_h = 0.000005;  // Tolerance on enthalpy iterations, kJ/kg
    double Tol_s = 0.000001;  // Tolerance on entropy iterations, kJ/(kg K)
    
    // Set the maximum allowed number of iterations
    int MaxIterations  = 500;
    
    // Get actual saturation temperature and quality
    if ( DoubleIteration(IFC97_R4_h_Tx, IFC97_R4_s_Tx, &Tsat, &Quality, Tol_h, Tol_s, h, s, MaxIterations) < 0 ) return -99990.0;
    
    // Determine Property and Return Value
    switch ( Property ) {
        case 0:
            return IFC97_Psat(Tsat);
            
        case 1:
            return Tsat;
            
        case 2:
        case 3:
            return (1.0-Quality)*IFC97_GetProperty_PT(Property, 0.0, Tsat, 0) + Quality*IFC97_GetProperty_PT(Property, 0.0, Tsat, 1);
        case 4:
            return s;
        
        case 5:
            return h;
            
        case 6:
        case 7:
            return (1.0-Quality)*IFC97_GetProperty_PT(Property, 0.0, Tsat, 0) + Quality*IFC97_GetProperty_PT(Property, 0.0, Tsat, 1);
            
        case 8:
            return 1.0/(1.0-Quality)*IFC97_GetProperty_PT(2, 0.0, Tsat, 0) + Quality*IFC97_GetProperty_PT(2, 0.0, Tsat, 1);
            
        case 9:
            return -80000; // Invalid saturation property
            
        case 10:
        case 11:
            return (1.0-Quality)*IFC97_GetProperty_PT(Property, 0.0, Tsat, 0) + Quality*IFC97_GetProperty_PT(Property, 0.0, Tsat, 1);
            
        case 12:
            return Quality;
            
        case 13:
            return (1.0-Quality)*IFC97_GetProperty_PT(6, 0.0, Tsat, 0) + Quality*IFC97_GetProperty_PT(6, 0.0, Tsat, 1) /
                   (1.0-Quality)*IFC97_GetProperty_PT(7, 0.0, Tsat, 0) + Quality*IFC97_GetProperty_PT(7, 0.0, Tsat, 1);

        case 14:
        case 15:
        case 16:
        case 17:
        case 18:
        case 19:
            return -80000; // Invalid saturation property
            
        default:
            break;
    };
    
    return -90000; // Unknown error
};
