//
//  IFC97_GetProperty_Ts_R4.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 24/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetProperty_Ts_R4.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_Saturation_Curve.h"


/*
 Internal function to get the quality and saturation conditions of a state within the saturation region based on Temperature and Entropy.
 */

double IFC97_GetProperty_Ts_R4(int Property, double T, double s) {
    /* Returns the requested property in the saturation region based on given T, s input.
     
     Inputs:
         Property   = Indicates property to be returned
         T          = Temperature, K
         s          = Specific entropy, kJ/(kg K)
     
     Return:
         For Property =  0, Pressure, MPa
                         1, Temperature, K
                         2, Specific volume, m^3/kg
                         3, Specific internal energy, kJ/kg
                         4, Specific entropy, kJ/(kg K)
                         5, Specific enthalpy, kJ/kg
                         6, Specific isobaric heat capacity, kJ/(kg K)
                         7, Specific isochoric heat capacity, kJ/(kg K)
                         8, Density, kg/m^3
                         9, NOT APPLICABLE (Speed of sound, m/s)
                         10, Gibbs free energy, kJ/kg
                         11, Hemholtz free energy, kJ/kg
                         12, Quality, dimensionless
                         13, Ratio of specific heats (Cp/Cv), dimensionless
                         14, NOT APPLICABLE Surface Tension, N/m
                         15. NOT APPLICABLE Dynamic Viscosity, Pa*s
                         16. NOT APPLICABLE Thermal Conductivity, W/(K*m)
                         17. NOT APPLICABLE Thermal diffusivity, m^2/s
                         18. NOT APPLICABLE Kinematic Viscosity, m^2/s
                         19. NOT APPLICABLE Prandtl Number, dimensionless
     
     Errors:
         -11000  = Temperature too low
         -12000  = Temperature too high
         -41000  = Entropy too low
         -42000  = Entropy too high
         -80000  = Invalid Property
         -99990  = Iterations exceeded
         -90000  = Unknown Error
     */
    // Get the saturated liquid and vapor enthalpies
    double s_liq = IFC97_GetProperty_PT(4, 0.0, T, 0);
    double s_vap = IFC97_GetProperty_PT(4, 0.0, T, 1);
    
    // Determine the quality
    double Qual = (s-s_liq) / (s_vap-s_liq);
    
    // Determine Property and Return Value
    switch ( Property ) {
        case 0:
            return IFC97_Psat(T);
            
        case 1:
            return T;
            
        case 2:
        case 3:
            return (1.0-Qual)*IFC97_GetProperty_PT(Property, 0.0, T, 0) + Qual*IFC97_GetProperty_PT(Property, 0.0, T, 1);
        case 4:
            return s;
            
        case 5:
        case 6:
        case 7:
            return (1.0-Qual)*IFC97_GetProperty_PT(Property, 0.0, T, 0) + Qual*IFC97_GetProperty_PT(Property, 0.0, T, 1);
            
        case 8:
            return 1.0/(1.0-Qual)*IFC97_GetProperty_PT(2, 0.0, T, 0) + Qual*IFC97_GetProperty_PT(2, 0.0, T, 1);
            
        case 9:
            return -80000; // Invalid saturation property
            
        case 10:
        case 11:
            return (1.0-Qual)*IFC97_GetProperty_PT(Property, 0.0, T, 0) + Qual*IFC97_GetProperty_PT(Property, 0.0, T, 1);
            
        case 12:
            return Qual;
            
        case 13:
            s_liq = (1.0-Qual)*IFC97_GetProperty_PT(6, 0.0, T, 0) + Qual*IFC97_GetProperty_PT(6, 0.0, T, 1); // Placeholder for Cp
            s_vap = (1.0-Qual)*IFC97_GetProperty_PT(7, 0.0, T, 0) + Qual*IFC97_GetProperty_PT(7, 0.0, T, 1); // Placeholder for Cv
            return s_liq / s_vap;

        case 14:
        case 15:
        case 16:
        case 17:
        case 18:
        case 19:
            return -80000; // Invalid saturation property
            
        default:
            break;
    };
    
    return -90000; // Unknown error
};