//
//  IFC97_GetProperty_Ts.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 24/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetProperty_Ts.h"

#include "IFC97_GetRegion_Ts.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_GetProperty_Ts_R4.h"

/*
 User interface function to provide the requested property based on Pressure and Enthalpy inputs.
 */

double IFC97_GetProperty_Ts(int Property, double T, double s) {
    /* Returns the requested property based on given T, s input.
     
     Inputs:
         Property   = Indicates property to be returned
         T          = Temperature, K
         s          = Specific entropy, kJ/(kg K)
     
     Return:
         For Property =  0, Pressure, MPa
                         1, Temperature, K
                         2, Specific volume, m^3/kg
                         3, Specific internal energy, kJ/kg
                         4, Specific entropy, kJ/(kg K)
                         5, Specific enthalpy, kJ/kg
                         6, Specific isobaric heat capacity, kJ/(kg K)
                         7, Specific isochoric heat capacity, kJ/(kg K)
                         8, Density, kg/m^3
                         9, Speed of sound, m/s
                         10, Gibbs free energy, kJ/kg
                         11, Hemholtz free energy, kJ/kg
                         12, Quality, dimensionless
                         13, Ratio of specific heats (Cp/Cv), dimensionless
                         14, Surface Tension, N/m
                         15. Dynamic Viscosity, Pa*s
                         16. Thermal Conductivity, W/(K*m)
                         17. Thermal diffusivity, m^2/s
                         18. Kinematic Viscosity, m^2/s
                         19. Prandtl Number, dimensionless
                         100, Region/Subregion (XYY: X = Region, YY=SubRegion)
     
     Errors:
         -11000  = Temperature too low
         -12000  = Temperature too high
         -41000  = Entropy too low
         -42000  = Entropy too high
         -80000  = Invalid Property
         -99990  = Iterations exceeded
         -90000  = Unknown Error
     */
    // Get Region and SubRegion information
    int Region     = -1; // Overall Region
    int SubRegion  = -1; // SubRegion, if in Region 3. Else will be zero
    double P_guess = 0.0; // Placeholder for initial pressure guess.
    
    int Check = IFC97_GetRegion_Ts(T, s, &Region, &SubRegion, &P_guess); // Get Region and initial pressure and see if there are errors
    
    if (  Check < 0 ) return (double)Check; // Exit on Error
    
    if ( Property == 100 ) return Region*100.0 + SubRegion; // Return Region/SubRegion information if requested
    
    // Get Requested Property Information
    double RHO_BW = 0.0; // Holder for backward density function (1/SV) function results
    
    // Establish allowable tolerances on solutions to iterations
    double Tol_s = 0.000001;  // Tolerance on entropy iterations, kJ/(kg K)
    
    // Set the maximum allowed number of iterations
    int MaxIterations  = 5000;
    
    switch ( Region ) {
        case 1:
            if ( SingleIteration(IFC97_R1_S_pt, &P_guess, &T, 0, Tol_s, s, MaxIterations) < 0 ) return -99990; // Exit. Iterations exceeded
            return IFC97_R1_Basic(Property, P_guess, T);
            
        case 2:
            if ( SingleIteration(IFC97_R2_S_pt, &P_guess, &T, 0, Tol_s, s, MaxIterations) < 0 ) return -99990; // Exit. Iterations exceeded
            return IFC97_R2_Basic(Property, P_guess, T);
            
        case 3:
            // Obtain the estimate of the SubRegion for the specific volume initial guess
            IFC97_GetRegion_PT(P_guess, T, &Region, &SubRegion);
            
            // Get the initial guess for density based on SubRegion determination above
            RHO_BW = 1.0/IFC97_R3_V_pt(SubRegion, P_guess, T);
            
            // Get actual density.
            if ( SingleIteration(IFC97_R3_S_dt, &RHO_BW, &T, 0, Tol_s, s, MaxIterations) < 0 ) return -99990; // Exit. Iterations exceeded
            return IFC97_R3_Basic(Property, RHO_BW, T);
            
        case 4:
            return IFC97_GetProperty_Ts_R4(Property, T, s);
            
        default:
            break;
    };
    
    return -90000.0;
};
