//
//  IFC97_GetProperty_Ph_Region4.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 22/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetProperty_Ph_Region4.h"

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_GetRegion_Ph.h"

/*
 Internal function to get the quality and saturation conditions of a state within the saturation region based on Pressure and Enthalpy.
 */

double IFC97_GetProperty_Ph_Region4(int Property, double P, double h) {
    /* Returns the requested property in the saturation region based on given P, h input.
     
     Inputs:
         Property   = Indicates property to be returned
         P          = Pressure, MPa
         h          = Specific enthalpy, kJ/kg
     
     Return:
         For Property =  0, Pressure, MPa
                         1, Temperature, K
                         2, Specific volume, m^3/kg
                         3, Specific internal energy, kJ/kg
                         4, Specific entropy, kJ/(kg K)
                         5, Specific enthalpy, kJ/kg
                         6, Specific isobaric heat capacity, kJ/(kg K)
                         7, Specific isochoric heat capacity, kJ/(kg K)
                         8, Density, kg/m^3
                         9, NOT APPLICABLE (Speed of sound, m/s)
                         10, Gibbs free energy, kJ/kg
                         11, Hemholtz free energy, kJ/kg
                         12, Quality, dimensionless
                         13, Ratio of specific heats (Cp/Cv), dimensionless
                         14, NOT APPLICABLE Surface Tension, N/m
                         15. NOT APPLICABLE Dynamic Viscosity, Pa*s
                         16. NOT APPLICABLE Thermal Conductivity, W/(K*m)
                         17. NOT APPLICABLE Thermal diffusivity, m^2/s
                         18. NOT APPLICABLE Kinematic Viscosity, m^2/s
                         19. NOT APPLICABLE Prandtl Number, dimensionless

     Errors:
     -21000  = Pressure too low
     -22000  = Pressure too high
     -31000  = Enthalpy too low
     -32000  = Enthalpy too high
     -80000  = Invalid Property
     -99990  = Iterations exceeded
     -90000  = Unknown Error
     */
    // Get the saturated liquid and vapor enthalpies
    double h_liq = IFC97_GetProperty_PT(5, P, 0.0, 0);
    double h_vap = IFC97_GetProperty_PT(5, P, 0.0, 1);
    
    // Determine the quality
    double Qual = (h-h_liq) / (h_vap-h_liq);
    
    // Determine Property and Return Value
    switch ( Property ) {
        case 0:
            return P;
            
        case 1:
            return IFC97_Tsat(P);
        
        case 2:
        case 3:
        case 4:
            return (1.0-Qual)*IFC97_GetProperty_PT(Property, P, 0.0, 0) + Qual*IFC97_GetProperty_PT(Property, P, 0.0, 1);
            
        case 5:
            return h;
            
        case 6:
        case 7:
            return (1.0-Qual)*IFC97_GetProperty_PT(Property, P, 0.0, 0) + Qual*IFC97_GetProperty_PT(Property, P, 0.0, 1);
        
        case 8:
            return 1.0/(1.0-Qual)*IFC97_GetProperty_PT(2, P, 0.0, 0) + Qual*IFC97_GetProperty_PT(2, P, 0.0, 1);
        
        case 9:
            return -80000; // Invalid saturation property
        
        case 10:
        case 11:
            return (1.0-Qual)*IFC97_GetProperty_PT(Property, P, 0.0, 0) + Qual*IFC97_GetProperty_PT(Property, P, 0.0, 1);
            
        case 12:
            return Qual;
            
        case 13:
            h_liq = (1.0-Qual)*IFC97_GetProperty_PT(6, P, 0.0, 0) + Qual*IFC97_GetProperty_PT(6, P, 0.0, 1); // Placeholder for Cp
            h_vap = (1.0-Qual)*IFC97_GetProperty_PT(7, P, 0.0, 0) + Qual*IFC97_GetProperty_PT(7, P, 0.0, 1); // Placeholder for Cv
            return h_liq / h_vap;

        case 14:
        case 15:
        case 16:
        case 17:
        case 18:
        case 19:
            return -80000; // Invalid saturation property

        default:
            break;
    };
    
    return -90000; // Unknown error
};