//
//  IFC97_GetProperty_Ph.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 22/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetProperty_Ph.h"

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_GetRegion_Ph.h"
#include "IFC97_GetProperty_Ph_Region4.h"


/*
 User interface function to provide the requested property based on Pressure and Enthalpy inputs.
 */

double IFC97_GetProperty_Ph(int Property, double P, double h) {
    /* Returns the requested property based on given P, h input.
     
     Inputs:
         Property   = Indicates property to be returned
         P          = Pressure, MPa
         h          = Specific enthalpy, kJ/kg
                  
     Return:
         For Property =  0, Pressure, MPa
                         1, Temperature, K
                         2, Specific volume, m^3/kg
                         3, Specific internal energy, kJ/kg
                         4, Specific entropy, kJ/(kg K)
                         5, Specific enthalpy, kJ/kg
                         6, Specific isobaric heat capacity, kJ/(kg K)
                         7, Specific isochoric heat capacity, kJ/(kg K)
                         8, Density, kg/m^3
                         9, Speed of sound, m/s
                         10, Gibbs free energy, kJ/kg
                         11, Hemholtz free energy, kJ/kg
                         12, Quality, dimensionless
                         13, Ratio of specific heats (Cp/Cv), dimensionless
                         14, Surface Tension, N/m
                         15. Dynamic Viscosity, Pa*s
                         16. Thermal Conductivity, W/(K*m)
                         17. Thermal diffusivity, m^2/s
                         18. Kinematic Viscosity, m^2/s
                         19. Prandtl Number, dimensionless
                         100, Region/Subregion (XYY: X = Region, YY=SubRegion)
     Errors:
         -21000  = Pressure too low
         -22000  = Pressure too high
         -31000  = Enthalpy too low
         -32000  = Enthalpy too high
         -80000  = Invalid Property
         -99990  = Iterations exceeded
         -90000  = Unknown Error
     */
    // Get Region and SubRegion information
    
    int Region    = -1; // Overall Region
    int SubRegion = -1; // SubRegion, if in Region 3. Else will be zero
    
    int Check = IFC97_GetRegion_Ph(P, h, &Region, &SubRegion); // Get Region and see if there are errors
    
    if (  Check < 0 ) return (double)Check; // Exit on Error
    
    if ( Property == 100 ) return Region*100.0 + SubRegion; // Return Region/SubRegion information if requested
    
    // Get Requested Property Information
    double T_BW   = 0.0; // Holder for backward temperature function result
    double RHO_BW = 0.0; // Holder for backward density function (1/SV) function results
    
    // Establish allowable tolerances on solutions to iterations
    double Tol_P = 0.00000000005; // Tolerance on pressure iterations, MPa
    double Tol_h = 0.000005;  // Tolerance on enthalpy iterations, kJ/kg
    
    // Set the maximum allowed number of iterations
    int MaxIterations  = 500;
    
    switch ( Region)  {
        case 1:
            T_BW = IFC97_R1_T_ph(P, h); // Get initial guess for temperature
            if ( SingleIteration(IFC97_R1_H_pt, &P, &T_BW, 1, Tol_h, h, MaxIterations) < 0 ) return -99990; // Exit. Iterations exceeded
            return IFC97_R1_Basic(Property, P, T_BW);
            
        case 2:
            T_BW = IFC97_R2_T_ph(SubRegion, P, h); // Get initial guess for temperature
            if ( SingleIteration(IFC97_R2_H_pt, &P, &T_BW, 1, Tol_h, h, MaxIterations) < 0 ) return -99990; // Exit. Iterations exceeded
            return IFC97_R2_Basic(Property, P, T_BW);
            
        case 3:
            T_BW   = IFC97_R3_T_ph(SubRegion, P, h);     // Get initial guess for temperature
            RHO_BW = 1.0/IFC97_R3_V_ph(SubRegion, P, h); // Get initial guess for density
            if ( DoubleIteration(IFC97_R3_P_dt, IFC97_R3_H_dt, &RHO_BW, &T_BW, Tol_P, Tol_h, P, h, MaxIterations) < 0 ) return -99990; // Exit. Iterations Exceeded
            return IFC97_R3_Basic(Property, RHO_BW, T_BW);
        
        case 4:
            return IFC97_GetProperty_Ph_Region4(Property, P, h); // Need to add function.
        
        default:
            break; // Unknown error
    };
    
    return -90000; // Unknown error
};
