//
//  IFC97_GetProperty_PT.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 21/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_GetProperty_PT.h"

#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_R3_Saturation_V.h"


double IFC97_GetProperty_PT(int Property, double P, double T, int Saturation) {
    /* Returns the requested property based on given P, T input.
     
     Inputs:
         Property   = Indicates property to be returned
         P          = Pressure, MPa
         T          = Temperature, K
         Saturation = Indication if Saturation Condition is requested
                      0 = Saturated Liquid Conditions (either P or T must be zero)
                      1 = Saturated Vapor Conditions  (either P or T must be zero)
                     -1 = Non-saturated conditions to be provided 
     
     Return:
         For Property =  0, Pressure, MPa
                         1, Temperature, K
                         2, Specific volume, m^3/kg
                         3, Specific internal energy, kJ/kg
                         4, Specific entropy, kJ/(kg K)
                         5, Specific enthalpy, kJ/kg
                         6, Specific isobaric heat capacity, kJ/(kg K)
                         7, Specific isochoric heat capacity, kJ/(kg K)
                         8, Density, kg/m^3
                         9, Speed of sound, m/s
                         10, Gibbs free energy, kJ/kg
                         11, Hemholtz free energy, kJ/kg
                         12, Quality, dimensionless
                         13, Ratio of specific heats (Cp/Cv), dimensionless
                         14, Surface Tension, N/m
                         15. Dynamic Viscosity, Pa*s
                         16. Thermal Conductivity, W/(K*m)
                         17. Thermal diffusivity, m^2/s
                         18. Kinematic Viscosity, m^2/s
                         19. Prandtl Number, dimensionless
                         20. Drho_DP_T, kg/(m^3 MPa), for thermal conductivity calculation
                        100, Region/Subregion (XYY: X = Region, YY=SubRegion)
     Errors:
         -11000  = Temperature too low
         -12000  = Temperature too high
         -21000  = Pressure too low
         -22000  = Pressure too high
         -99990  = Iteration Error
         -90000  = Unknown Error
     */

    // Place holder for density if Region 3
    double RHO;

    // Establish if saturated or non-saturated conditions are being requested
    
    if ( Saturation > -1) { // Requested saturation conditions
        if ( Property == 100 ) {
            if ( T == 0.0 ) return 400.0+Saturation; // Saturation conditions at given Pressure wanted
            else            return 410.0+Saturation; // Saturation conditions at given Temperature wanted
        };
      
        // For all othe properties, get values
        double Tmin = 273.15;           // Minimum allowed Temperature, K
        double Pmin = IFC97_Psat(Tmin); // Minimum allowed Pressure, MPa
        double T_critial = 647.096;     // Critical Temperature, K
        double P_critial = 22.064;      // Critical Pressure, MPa
        double T_B13 = 623.15;          // Isothermal boundary condition between Region 1 and 3, K
        
        if ( T == 0.0 ) { // Property based on Pressure
            if ( P < Pmin        ) return -21000.0;                                    // Error. Pressure below limit
            if ( P > P_critial   ) return -22000.0;                                    // Error. Pressure above critical pressure
            if ( Saturation == 0 ) { // Get Liquid conditions
                if ( P <= IFC97_Psat(T_B13) ) return IFC97_R1_Basic(Property, P, IFC97_Tsat(P));  // Get Region 1 saturated liquid property
                else { // Need Region 3 Saturation Condition
                    RHO = 1.0/IFC97_R3_Saturation_V(P, 0.0, 0);          // Get saturated liquid density in Region 3
                    return IFC97_R3_Basic(Property, RHO, IFC97_Tsat(P)); // Return Region 3 saturated liquid property
                };
            }
            else { // Get Vapor Conditions
                if ( P <= IFC97_Psat(T_B13) ) return IFC97_R2_Basic(Property, P, IFC97_Tsat(P)); // Get Region 2 saturated vapor property
                else { // Need Region 3 Saturation Condition
                    RHO = 1.0/IFC97_R3_Saturation_V(P, 0.0, 1);          // Get saturated vapor density in Region 3
                    return IFC97_R3_Basic(Property, RHO, IFC97_Tsat(P)); // Return Region 3 saturated vapor property
                };
            };
        }
        else {                   // Property based on Temperature
            if ( T < Tmin        ) return -11000.0;                                    // Error. Temperature below limit
            if ( T > T_critial   ) return -12000.0;                                    // Error. Temperature above critical pressure
            if ( Saturation == 0 ) { // Get Liquid conditions
                if ( T <= T_B13 )  return IFC97_R1_Basic(Property, IFC97_Psat(T), T); // Get Region 1 saturated liquid property
                else { // Need Region 3 Saturation Condition
                    RHO = 1.0/IFC97_R3_Saturation_V(0.0, T, 0);          // Get saturated liquid density in Region 3
                    return IFC97_R3_Basic(Property, RHO, T);             // Return Region 3 saturated liquid property
                };
            }
            else { // Get Vapor Conditions
                if ( T <= T_B13 )  return IFC97_R2_Basic(Property, IFC97_Psat(T), T); // Get Region 2 saturated vapor property
                else {
                    RHO = 1.0/IFC97_R3_Saturation_V(0.0, T, 1);          // Get saturated vapor density in Region 3
                    return IFC97_R3_Basic(Property, RHO, T);             // Return Region 3 saturated vapor property                    
                };
            };
        };
    }
    else {                 // Non-Saturation conditions
        int Region    = -1; // Overall Region
        int SubRegion = -1; // SubRegion, if in Region 3. Else will be zero
        
        int Check = IFC97_GetRegion_PT(P, T, &Region, &SubRegion); // Get Region and see if there are errors
        
        if (  Check < 0 ) return (double)Check; // Exit on Error
        
        if ( Property == 100 ) return Region*100.0 + SubRegion; // Return Region/SubRegion information if requested
        
        // Establish allowable tolerances on solutions to iterations
        double Tol_P = 0.00000000005; // Tolerance on pressure iterations, MPa
        
        // Set the maximum allowed number of iterations
        int MaxIterations  = 500;
        
        // If no error, get Property
        switch (Region) {
            case 1:
                return IFC97_R1_Basic(Property, P, T);
                
            case 2:
                return IFC97_R2_Basic(Property, P, T);
                
            case 3:
                RHO = 1.0/IFC97_R3_V_pt(SubRegion, P, T); // Get initial guess for density
                if ( SingleIteration(IFC97_R3_P_dt, &RHO, &T, 0, Tol_P, P, MaxIterations) < 0 ) return -99990.0; // Get actual density or iteration error
                return IFC97_R3_Basic(Property, RHO, T);
                
            default:
                return -90000.0; // Unknown error
        };
    };
    
    return -90000;
};