//
//  IFC97_Double_Iteration.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 20/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_Double_Iteration.h"
#include <math.h>

/*
 Function peforms two dimensional Newton's Methond on the two given two variable input functions meeting the indicated tolerance for a know value of Solutions
 */

int DoubleIteration(FUNC F0, FUNC F1, double *InVar0, double *InVar1, double TolSol0, double TolSol1, double Sol0, double Sol1, int MaxIterations) {
    /*
     Function performs two dimensional Newton's Method on the two provided two variable functions to find variable to make the Functions equal to the Solutions. The required variables are returned in the input variables.
     
     Inputs:
        F0:             Two-variable (each a double) function that returns a double
        F1:             Two-variable (each a double) function that returns a double
        Var0:           Initial guess for first variable in each function (F0 and F1 must have the same first variable)
        Var1:           Initial guess for second variable in each function (F0 and F1 must have the same second variable)
        TolSol0:        Absolute tolerance on solution for F0. Units must be the same as F0.
        TolSol1:        Absolute tolerance on solution for F1. Units must be the same as F1.
        Sol0:           Required solution for F0
        Sol1:           Required solution for F1
        MaxIterations:  Maximum allowed iterations for solution
        
     Outputs:
        return:         Number of iterations required for solution
     
     Errors:
        -1.  Iterations exceeded
     */
    // Transfer Inputs to internal variables
    double Var0 = *InVar0;
    double Var1 = *InVar1;
    
    // Establish ChangeVariable iteration step for determination of derivative. Use 0.5% change as basis.
    double DeltaPercent = 0.5; // Percent of variable to utilize as the step value for the derivative
    double DeltaVar0 = DeltaPercent * Var0 / 100.0;
    double DeltaVar1 = DeltaPercent * Var1 / 100.0;
    
    // Establish the initial errors for the evaluation
    double E0 = F0(Var0,Var1) - Sol0;
    double E1 = F1(Var0,Var1) - Sol1;
    
    // Declare the approximate partial derivatives for the evaluation
    double dE0_dVar0 = 0.0;
    double dE0_dVar1 = 0.0;
    double dE1_dVar0 = 0.0;
    double dE1_dVar1 = 0.0;
    
    // Declare the incremental step change variables
    double DVar0 = 100.0;
    double DVar1 = 100.0;
    
    for (int Count=0; Count<MaxIterations; Count++) {
        
        // Check to see if the tolerances have been met
        if (fabs(E0)<TolSol0  &&  fabs(E1)<TolSol1) { // If met, exist and indicate number of iterations.
            // Transfer Solution to input variables
            *InVar0 = Var0;
            *InVar1 = Var1;
            
            // Return number of iterations
            return Count+1;  
        };
        
        // If not met, peform Newton's method to determine next incremental value
        
        // Determine approximate slopes for the partial derivatives of the first Function error
        dE0_dVar0 = (E0 - F0(Var0-DeltaVar0,Var1) + Sol0) / DeltaVar0;
        dE0_dVar1 = (E0 - F0(Var0,Var1-DeltaVar1) + Sol0) / DeltaVar1;
        
        // Determine approximate slopes for the partial derivatives of the second Function error
        dE1_dVar0 = (E1 - F1(Var0-DeltaVar0,Var1) + Sol1) / DeltaVar0;
        dE1_dVar1 = (E1 - F1(Var0,Var1-DeltaVar1) + Sol1) / DeltaVar1;
        
        // Determine Var1 step change to remove the error
        DVar1 = ( E1 - (dE1_dVar0/dE0_dVar0)*E0 ) / ( dE1_dVar1 - (dE1_dVar0*dE0_dVar1/dE0_dVar0) ) ;
        
        // Determine Var0 step change to remove the error
        DVar0 = ( E0 - (dE0_dVar1)*DVar1 ) / dE0_dVar0;
        
        // The new estimates of the required variables are
        Var0 -= DVar0;
        Var1 -= DVar1;
        
        // Determine new errors are
        E0 = F0(Var0,Var1) - Sol0;
        E1 = F1(Var0,Var1) - Sol1;
    };

    return -1; // Return error based on too many iterations.
};




