//
//  IFC97_Bhmin_p.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 23/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_Bhmin_s.h"
#include <math.h>

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_GetRegion_Ph.h"
#include "IFC97_GetProperty_Ph.h"
#include "IFC97_GetProperty_Ph_Region4.h"
#include "IFC97_GetRegion_Ps.h"
#include "IFC97_GetProperty_Ps.h"
#include "IFC97_GetProperty_Ps_Region4.h"
#include "IFC97_GetRegion_hs.h"

/*
 Provides an approximation of the maximum enthalpy boundary based on input entropy.  Approximation line not provided in IAPWS documents. This is an internal polynomial fit which will allow a maximum enthalpy of 13 kJ/kg over actual maximum boundary value (in Region 2) to be allowed to be calculated.
 */

double  IFC97_Bhmin_s(double s) {
    /*
     Determine minimum allowable input enthalpy based on a given entropy. This divides the maximum line into three regions.
     Region 0 is the constant enthalpy extension from saturated liquid entropy at saturation at T=273.15K down to lowest possible entropy
     Region 1 is from saturation line at constant saturation pressure at T=273.15K.
     Region 2 is from the s_sat_vap at 273.15K to the enthalpy at Pmin and T=1073.25. Pmin is the saturation pressure at T=273.15K
     
     Inputs:
        s         = Specific entropy, kJ/(kg K)
     
     Outputs:
        h         = Specific enthalpy, kJ/kg
     
     Errors:
        -9000 = Unknown error
     */
    // Get Region break points
    double Tmin = 273.15; // Units of K
    
    // Note, s_liq is the same as the boundary between Region 0-1 and s_vap is the same as the boundary between Region 1-2
    double s_liq = IFC97_GetProperty_PT(4, 0.0, Tmin, 0); // Value is -0.0001545496 kJ/(kg K). Saturated liquid entropy
    double s_vap = IFC97_GetProperty_PT(4, 0.0, Tmin, 1); // Value is  9.1557593952 kJ/(kg K). Saturated vapor entropy
    
    // Get saturated enthalpy conditions for Region 1 minimum enthalpy determination
    double h_liq = IFC97_GetProperty_PT(5, 0.0, Tmin, 0); // Saturated liquid enthalpy
    double h_vap = IFC97_GetProperty_PT(5, 0.0, Tmin, 1); // Saturated vapor enthalpy
    
    // Coefficients for polynomical curve fit for Region 2 lower boundary
    double Coeff[] = { -1.215075521E-01, 7.283823287E+00, -1.525784894E+02, 1.533666505E+03, -7.532172933E+03, 1.663816371E+04 };
    
    /*
     Offsets for curve fits to ensure capturing all possible values.
     For Region 0. Extension to capture to minimum entropy at Pmax, Tmin. Not truely valid, but will allow to simply boundary
     For Region 1. No offset. Straight saturated enthalpy value based on quality from IAWPS calculations
     For Region 2. Value allows for a minimum value that is up to 0.020 kJ/kg lower than IAWPS curves allow. This is a maximum 0.0008% deviation
     */
    double Offset_R2 = -0.0198; // Units are kJ/kg
    
    // Determine maximum allowed value of enthalpy for region determination
    if ( s <= s_liq ) { // In Region 0. Use constant enthalpy value at saturated liquid enthalpy at Tmin
        return h_liq;
    }
    else if (s <= s_vap ){ // Region 1. Get actual quality and determine allowable minimum enthalpy from IAPWS equations
        double Quality = (s-s_liq)/(s_vap-s_liq);
        return (1.0-Quality)*h_liq + Quality*h_vap;
    }
    else { // Region 2. Hold maximum possible value at Pmin and Tmax
        return pow(s,5.)*Coeff[0]+pow(s,4.)*Coeff[1]+pow(s,3.)*Coeff[2]+pow(s,2.)*Coeff[3]+s*Coeff[4]+Coeff[5]+Offset_R2;
    };
    
    return -9000.0; // Unknown error
};