//
//  IFC97_Bhmax_s.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 23/Dec/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_Bhmax_s.h"
#include <math.h>

#include "IFC97_B23_Equations.h"
#include "IFC97_Saturation_Curve.h"
#include "IFC97_R1_Basic.h"
#include "IFC97_R2_Basic.h"
#include "IFC97_R3_Basic.h"
#include "IFC97_R1_Backward_Tph.h"
#include "IFC97_R1_Backward_Tps.h"
#include "IFC97_B2bc_Equation.h"
#include "IFC97_R2_Backward_Tph.h"
#include "IFC97_R2_Backward_Tps.h"
#include "IFC97_R1_Backward_Phs.h"
#include "IFC97_h2ab_Equation.h"
#include "IFC97_R2_Backward_Phs.h"
#include "IFC97_h3ab_Equation.h"
#include "IFC97_R3_Backward_Tph.h"
#include "IFC97_R3_Backward_Vph.h"
#include "IFC97_R3_Backward_Tps.h"
#include "IFC97_R3_Backward_Vps.h"
#include "IFC97_P3sat_s.h"
#include "IFC97_P3sat_h.h"
#include "IFC97_R3_Backward_Phs.h"
#include "IFC97_h_liq_1_s.h"
#include "IFC97_h_liq_3a_s.h"
#include "IFC97_h_vap_2ab_s.h"
#include "IFC97_h_vap_2c3b_s.h"
#include "IFC97_hB13_s.h"
#include "IFC97_T_B23_hs.h"
#include "IFC97_Tsat_hs.h"
#include "IFC97_R3_T_Boundaries.h"
#include "IFC97_R3_Backward_Vpt.h"
#include "IFC97_Single_Iteration.h"
#include "IFC97_Double_Iteration.h"
#include "IFC97_GetRegion_PT.h"
#include "IFC97_GetProperty_PT.h"
#include "IFC97_GetRegion_Ph.h"
#include "IFC97_GetProperty_Ph.h"
#include "IFC97_GetProperty_Ph_Region4.h"
#include "IFC97_GetRegion_Ps.h"
#include "IFC97_GetProperty_Ps.h"
#include "IFC97_GetProperty_Ps_Region4.h"
#include "IFC97_GetRegion_hs.h"

/*
 Provides an approximation of the maximum enthalpy boundary based on input entropy.  Approximation line not provided in IAPWS documents. This is an internal polynomial fit which will allow a maximum enthalpy of 13 kJ/kg over actual maximum boundary value (in Region 2) to be allowed to be calculated.
 */

double  IFC97_Bhmax_s(double s) {
    /*
     Determine maximum allowable input enthalpy based on a given entropy. This divides the maximum line into three regions.  
     Region 0 is the P=100 MPa line and extends from h_sat_liq at 273.15 K to the P=100 MPa and T=1073.15K location.
     Region 1 is from the P=100 MPa and T=1073.15K point to the s_sat_vap at 273.15K at a constant temperature of T=1073.15K.
     Region 2 is from the s_sat_vap at 273.15K to the enthalpy at Pmin and T=1073.25. Pmin is the saturation pressure at T=273.15K
     
     Inputs:
        s         = Specific entropy, kJ/(kg K)
     
     Outputs:
        h         = Specific enthalpy, kJ/kg
     
     Errors:
        None
     */
    // Get Region break points
    double Tmax = 1073.15; // Units of K
    double Pmax = 100.0; // Units of MPa
    double Tmin = 273.15; // Units of K
    double Pmin = IFC97_Psat(Tmin);
    
    double s01 = IFC97_GetProperty_PT(4, Pmax, Tmax, -1); // Value is 6.0404836717 kJ/(kg K)
    double s12 = IFC97_R2_S_pt(Pmin, Tmin);               // Value is 9.1557593952 kJ/(kg K)
    
    // Coefficients for polynomical curve fit
    double Coeff[2][6] = { { 1.406066172e-01, -2.133756155e+00,  1.435723181e+01, 1.103028511e+01,  2.915069447e+02, 9.475200700e+01}, // P=100 Boundary
                           {-7.429987647e+00,  2.768797634e+02, -4.075147019e+03, 2.950544154e+04, -1.045148387e+05, 1.477580696e+05} }; // T=1073.15 Boundary

    /* 
     Offsets for curve fits to ensure capturing all possible values.
        For Region 0. Value allows for a maximum value that is up to 5.95 kJ/kg higher than IAWPS curves allow. This is a maximum 2.78% deviation
        For Region 1. Value allows for a maximum value that is up to 13.0 kJ/kg higher than IAWPS curves allow. This is a maximum 0.313% deviation
        For Region 2. Value allows for a maximum value that is up to 1.13 kJ/kg higher than IAWPS curves allow. This is a maximum 0.027% deviation
     */
    double Offset_R0 = 4.54; // Units are kJ/kg
    double Offset_R1 = 9.91; // Units are kJ/kg
        
    // Determine maximum allowed value of enthalpy for region determination
    if ( s <= s01 ) { // In Region 0. Use constant P=100 MPa boundary line
        return pow(s,5.)*Coeff[0][0]+pow(s,4.)*Coeff[0][1]+pow(s,3.)*Coeff[0][2]+pow(s,2.)*Coeff[0][3]+s*Coeff[0][4]+Coeff[0][5]+Offset_R0;
    }
    else if (s <= s12 ){ // Region 1. Use constant T=1073.15K boundary line
        return pow(s,5.)*Coeff[1][0]+pow(s,4.)*Coeff[1][1]+pow(s,3.)*Coeff[1][2]+pow(s,2.)*Coeff[1][3]+s*Coeff[1][4]+Coeff[1][5]+Offset_R1;
    }
    else { // Region 2. Hold maximum possible value at Pmin and Tmax
        return IFC97_R2_H_pt(Pmin, Tmax);
    };
    
    return -9000.0; // Unknown error
};