//
//  IFC97_B2bc_Equation.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 26/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_B2bc_Equation.h"
#include <math.h>

/*
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

// Overall base equation providing the coefficients for both P(h) and h(P). Table 19 coefficients and Equation (20) and (21).
double IFC97_B2bc(int iType, double P, double h) {
    /* Provides the Basic equation for the boundary between b-area and c-area in Region 2.  This is the B2bc Equation.
     
     Equations are based on Equation (20) and (12) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iType = Indicates property to be returned
        P     = Pressure, MPa
        h     = Specific enthalpy, kJ/kg
     
     Return:
        For iType = 0, Pressure, MPa
                    5, Specific enthalpy, kJ/kg
     
     Errors:
        -1000.0 = Incorrect iType input
        -2000.0 = Unknown error
     */
    
    // Numerical coefficients from Table 19.
    double n[] = {   0.0,
                     0.90584278514723e+03,
                    -0.67955786399241e+00,
                     0.12809002730136e-03,
                     0.26526571908428e+04,
                     0.45257578905948e+01 };
    
    // Reference Values
    double Pstar = 1.0; // Units of MPa
    double hstar = 1.0; // Units of kJ/kg
    
    // Non-dimensional values
    double Pi  = P/Pstar;
    double Eta = h/hstar;
    
    // Return either the Pressure or enthalpy boundary
    switch (iType) {
        case 0: // Return the Pressure boundary, MPa Equation (20)
            return (n[1] + n[2]*Eta + n[3]*pow(Eta,2.0)) * Pstar;
      
        case 5: // Return the enthalpy boundary, kJ/kg  Equation (21)
            return (n[4] + sqrt( (Pi-n[5])/n[3] )) * hstar;
            
        default:
            return -1000.0;  // Return error on incorrect input
    };
    
    return -2000.0; // Return unknown error.
};


// User interface function to obtain the B2bc Pressure boundary given h. Calls IFC97_B2bc
double IFC97_B2bc_P_h(double h) {
    /* Provides the Basic equation for the Pressure boundary between b-area and c-area in Region 2.  This calls the B2bc Equation.
          
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        h     = Specific enthalpy, kJ/kg
     
     Return:
        P     = Pressure, MPa
     
     Errors:
        None.
     */
    
    return IFC97_B2bc(0, 0.0, h);
};


// User interface function to obtain the B2bc enthalpy boundary given P. Calls IFC97_B2bc
double IFC97_B2bc_h_p(double P) {
    /* Provides the Basic equation for the enthalpy boundary between b-area and c-area in Region 2.  This calls the B2bc Equation.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P     = Pressure, MPa
     
     Return:
        h     = Specific enthalpy, kJ/kg
     
     Errors:
        None.
     */
    
    return IFC97_B2bc(5, P, 0.0);
};

