//
//  IFC97_B23_Equations.cpp
//  IFC97
//
//  Created by Rodrigo Carvajal on 18/Nov/12.
//  Copyright (c) 2012 Rodrigo Carvajal. All rights reserved.
//

#include "IFC97_B23_Equations.h"
#include <math.h>


/* 
 Equations given below are based on the information provided in the IAPWS, Revised Release on the IAPWS Industrial 
 Formulation 1997 for the Thermodynamic Properties of Water and Steam,. Lucerne, Switzerland, August 2007.
 */

double IFC97_B23_Basic(int iType, double P, double T){
    /* Provides the Basic equations for the Boundary between Regions 2 and 3.
     
     Equations are based on Equation (5) and (6) of IAPWS reference discussed above.
     
     Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        iType = Indicates if Pressure or Temperature is to be determined based on inputs
        P     = Pressure at B23 boundary point, MPa
        T     = Temperature at B23 boundary point, K
     
     Return:
        For iType 0, Boundary Pressure, MPa
        For iType 1, Boundary Temperature, K
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function.
     */
    
    double Tstar = 1.0;
    double Pstar = 1.0;
    
    // Numerical coefficients from Table 1.
    
    double n[] = {      0.0,
                        0.34805185628969e+03,
                       -0.11671859879975e+01,
                        0.10192970039326e-02,
                        0.57254459862746e+03,
                        0.13918839778870e+02 };
    
    if ( iType == 0 ){  // Temperature is input
        double theta = T/Tstar;
        return (n[1] + n[2]*theta + n[3]*theta*theta)*Pstar;// Equation (5)
    }
    else {              // Pressure is input
        double pi   = P/Pstar;
        return (n[4] + sqrt((pi-n[5])/n[3]))*Tstar;// Equation (6)
    };
};


double IFC97_B23_GetPFromT(double T) {
    /* Used as the interface function to get the B23 boundary pressure given a temperature. Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        T = Temperature at B23 boundary, Kelvin
     
     Outputs:
        P = Pressure at B23 boundary, MPa
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function
     
     */
    
    return IFC97_B23_Basic(0, 0.0, T);
};



double IFC97_B23_GetTFromP(double P) {
    /* Used as the interface function to get the B23 boundary temperature given a pressure. Note, no out of range checks performed in this function. These should be performed prior to calling this function.
     
     Inputs:
        P = Pressure at B23 boundary, MPa
     
     Outputs:
        T = Temperature at B23 boundary, Kelvin
     
     Errors:
        None. No out of range checks performed in this function. These should be performed prior to calling this function
     
     */
    
    return IFC97_B23_Basic(1, P, 0.0);
};
