(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     21882,        627]*)
(*NotebookOutlinePosition[     22664,        653]*)
(*  CellTagsIndexPosition[     22620,        649]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Math 384", "Subtitle"],

Cell[CellGroupData[{

Cell["\<\
A simple constrained minimization example using the penalty method\
\>", "Subsubsection",
  CellDingbat->None],

Cell[TextData[{
  "Say we want to minimize ",
  Cell[BoxData[
      \(f[x, y] = x\^2 + 2\ y\^2\)]],
  " subject to the constraint ",
  Cell[BoxData[
      \(g[x, y] = 1 - x\^2 - y\^2 \[LessEqual] 0\)]],
  ". It is fairly easy to see that the minimizers for this constrained \
problem are ",
  Cell[BoxData[
      \({\(\[PlusMinus]1\), 0}\)]],
  ". You may know how to tackle this sort of problem with a Lagrange \
multiplier (and if not we cover that in chapter 7). Let us see how to \
approach this with penalty functions. We begin by defining ",
  Cell[BoxData[
      \(\(f[k_]\)[x_, y_]\  := \ f[x, y] + k\ g[x, y]\^2\)]],
  ". We find the minimizer for ",
  Cell[BoxData[
      \(k = 1\)]],
  ", given some starting point; it does not really matter where we start \
provided ",
  Cell[BoxData[
      \(FindMinimum\)]],
  " converges. We use the minimizer to plug in for the starting point for \
minimizing ",
  Cell[BoxData[
      \(f[2]\)]],
  ". We proceed in this manner. Below I do two iterations explicitly."
}], "Text"],

Cell[BoxData[{
    \(f[x_, y_] := x\^2 + 2\ y\^2\), "\[IndentingNewLine]", 
    \(g[x_, y_] := \ 1 - x\^2 - y\^2\), "\[IndentingNewLine]", 
    \(gplus[x_, y_]\  := \ Which[g[x, y] > 0, \ g[x, y]\^2, \ True, \ 0]\), 
  "\[IndentingNewLine]", 
    \(\(f[k_]\)[x_, y_]\  := \ f[x, y]\  + \ k*gplus[x, y]\)}], "Input"],

Cell[CellGroupData[{

Cell[BoxData[{
    \({minvalue[1], vals[1]} = 
      FindMinimum[\(f[1]\)[x, y], \ {x, 50, 55}, \ {y, 50, 55}]\), 
  "\[IndentingNewLine]", 
    \({newx[1], newy[1]}\  = \ {x, y}\  /. \ vals[1]\)}], "Input"],

Cell[BoxData[
    \({0.7500000000003143`, 
      {x \[Rule] 0.7071070817015978`, y \[Rule] \(-3.656485310869881`*^-7\)}}
      \)], "Output"],

Cell[BoxData[
    \({0.7071070817015978`, \(-3.656485310869881`*^-7\)}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
    \({minvalue[1], vals[1]} = FindMinimum[\(f[1]\)[x, y], \ ]\), 
  "\[IndentingNewLine]", 
    \({newx[1], newy[1]}\  = \ {x, y}\  /. \ vals[1]\)}], "Input"],

Cell[BoxData[
    \({0.7500000000000232`, 
      {x \[Rule] 0.7071068716081402`, y \[Rule] 8.30308191770773`*^-8}}\)], 
  "Output"],

Cell[BoxData[
    \({0.7071068716081402`, 8.30308191770773`*^-8}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[{
    \({minvalue[2], vals[2]} = 
      FindMinimum[\(f[2]\)[x, y], \ {x, newx[1], newx[1] + 1}, \ 
        {y, newy[1], newy[1] + 1}]\), "\[IndentingNewLine]", 
    \({newx[2], newy[2]}\  = \ {x, y}\  /. \ vals[2]\)}], "Input"],

Cell[BoxData[
    \({0.875`, 
      {x \[Rule] 0.8660254043633556`, y \[Rule] 6.601822947505456`*^-11}}\)], 
  "Output"],

Cell[BoxData[
    \({0.8660254043633556`, 6.601822947505456`*^-11}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Now I show an algorithm for this. We may have multiple constraints so \
these will be given in a list. Just to be clever I will double the penalty \
multiplier each time so at step ",
  Cell[BoxData[
      \(j\)]],
  " it is ",
  Cell[BoxData[
      \(2\^j\)]],
  ". This is a common means by which to speed convergence; it works quite \
well for the particular example above. For the first iteration we pick random \
values in the range ",
  Cell[BoxData[
      \({\(-50\), 50}\)]],
  " for each variable. This can of course be modified for different problems \
in cases where it is clearly unsuitable. The code is a bit awkward due to \
vagaries of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " ",
  Cell[BoxData[
      \(FindMinimum\)]],
  " needs, but you get the idea. We iteratively find a minimum for a modified \
function-with-penalty, then double the penalty and use the previous minimizer \
as starting point for the next ",
  Cell[BoxData[
      \(FindMinimum\)]],
  " call. I add optional ",
  Cell[BoxData[
      \(Print\)]],
  " statements so you can see the successive minimizers as they approach the \
constrained minimizer for ",
  Cell[BoxData[
      \(f[x, y]\)]],
  "."
}], "Text"],

Cell[BoxData[{
    \(Clear[minvalue, vals, newx, newy, penaltyMethod]\), 
  "\[IndentingNewLine]", 
    \(penaltyMethod[func_, constraints_List, vars_, iters_, prnt_:  False]\  
      := \ Module[\[IndentingNewLine]{posconstraints, sumconstraints, next, 
          vals, j, k}, \[IndentingNewLine]posconstraints\  = 
          Map[Which[# > 0, #^2, True, 0] &, 
            constraints]; \[IndentingNewLine]sumconstraints = 
          Apply[Plus, posconstraints]; \[IndentingNewLine]inits\  = 
          Table[100*Random[] - 50, 
            {Length[vars]}]; \[IndentingNewLine]inits\  = 
          Table[\ {vars[\([j]\)], inits[\([j]\)], inits[\([j]\)] + 1}, 
            {j, Length[vars]}]; \[IndentingNewLine]next\  = \ 
          FindMinimum[Evaluate[func + sumconstraints], \ 
            Evaluate[Apply[Sequence, inits]]]; \[IndentingNewLine]If[prnt, 
          Print[next]]; \[IndentingNewLine]Do\ [\[IndentingNewLine]vals\  = \ 
            Table[Apply[List, \ next[\([2, k]\)]], \ 
              {k, Length[vars]}]; \[IndentingNewLine]next\  = \ 
            FindMinimum[Evaluate[func + 2^j*sumconstraints], \ 
              Evaluate[Apply[Sequence, vals]]]; \[IndentingNewLine]If[prnt, 
            Print[next]], \[IndentingNewLine]{j, 
            iters - 1}]; \[IndentingNewLine]{func /. next[\([2]\)], 
          next[\([2]\)]}\[IndentingNewLine]]\)}], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethod[f[x, y], {g[x, y]}, {x, y}, 20]\)], "Input"],

Cell[BoxData[
    \({0.9999995231628418`, 
      {x \[Rule] \(-0.9999995231628657`\), y \[Rule] 4.012895365775546`*^-13}}
      \)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethod[f[x, y], {g[x, y]}, {x, y}, 20, \ True]\)], "Input"],

Cell[BoxData[
    \({0.7500000000868052`, 
      {x \[Rule] 0.707112849269099`, y \[Rule] 3.6278254373664515`*^-6}}\)], 
  "Print"],

Cell[BoxData[
    \({0.875`, 
      {x \[Rule] 0.8660254037844384`, y \[Rule] 9.018081995481052`*^-11}}\)], 
  "Print"],

Cell[BoxData[
    \({0.9375`, 
      {x \[Rule] 0.9354143466838999`, y \[Rule] 2.026566216499583`*^-11}}\)], 
  "Print"],

Cell[BoxData[
    \({0.96875`, 
      {x \[Rule] 0.9682458365580481`, y \[Rule] 1.616075598511441`*^-11}}\)], 
  "Print"],

Cell[BoxData[
    \({0.984375`, 
      {x \[Rule] 0.9842509842530353`, y \[Rule] 6.2427882950743605`*^-12}}
      \)], "Print"],

Cell[BoxData[
    \({0.9921875`, 
      {x \[Rule] 0.9921567416492214`, y \[Rule] 6.193242344420114`*^-12}}\)], 
  "Print"],

Cell[BoxData[
    \({0.99609375`, 
      {x \[Rule] 0.9960860906498794`, y \[Rule] 6.1312971135596994`*^-12}}
      \)], "Print"],

Cell[BoxData[
    \({0.998046875`, 
      {x \[Rule] 0.9980449639315003`, 
        y \[Rule] \(-2.2872183416975305`*^-11\)}}\)], "Print"],

Cell[BoxData[
    \({0.9990234375`, 
      {x \[Rule] 0.9990229601966375`, y \[Rule] 1.318258096686212`*^-16}}\)], 
  "Print"],

Cell[BoxData[
    \({0.99951171875`, 
      {x \[Rule] 0.9995115994825141`, 
        y \[Rule] \(-1.9345321107009794`*^-15\)}}\)], "Print"],

Cell[BoxData[
    \({0.999755859375`, 
      {x \[Rule] 0.9997558295653994`, 
        y \[Rule] \(-1.9340595822094233`*^-15\)}}\)], "Print"],

Cell[BoxData[
    \({0.9998779296875`, 
      {x \[Rule] 0.9998779222551446`, y \[Rule] \(-1.932665680620097`*^-15\)}}
      \)], "Print"],

Cell[BoxData[
    \({0.99993896484375`, 
      {x \[Rule] 0.9999389629621971`, y \[Rule] 1.417571901571675`*^-10}}\)], 
  "Print"],

Cell[BoxData[
    \({0.999969482421875`, 
      {x \[Rule] 0.9999694819791763`, y \[Rule] \(-5.539383503781165`*^-17\)}}
      \)], "Print"],

Cell[BoxData[
    \({0.9999847412109375`, 
      {x \[Rule] 0.9999847410717193`, y \[Rule] 3.637579447975584`*^-11}}\)], 
  "Print"],

Cell[BoxData[
    \({0.9999923706054688`, 
      {x \[Rule] 0.9999923705878826`, y \[Rule] 6.494268841515651`*^-20}}\)], 
  "Print"],

Cell[BoxData[
    \({0.9999961853027344`, 
      {x \[Rule] 0.9999961852839904`, y \[Rule] \(-1.146796002870645`*^-11\)}}
      \)], "Print"],

Cell[BoxData[
    \({0.9999980926513672`, 
      {x \[Rule] 0.9999980926533453`, 
        y \[Rule] \(-1.9794696147625958`*^-11\)}}\)], "Print"],

Cell[BoxData[
    \({0.9999990463256836`, 
      {x \[Rule] 0.9999990463262617`, y \[Rule] 4.211925248436159`*^-12}}\)], 
  "Print"],

Cell[BoxData[
    \({0.9999995231628418`, 
      {x \[Rule] 0.999999523164784`, y \[Rule] \(-1.6437154878102506`*^-11\)}}
      \)], "Print"],

Cell[BoxData[
    \({0.9999995231628418`, 
      {x \[Rule] 0.999999523164784`, y \[Rule] \(-1.6437154878102506`*^-11\)}}
      \)], "Output"]
}, Open  ]],

Cell["\<\
Not bad. Note that in successive attempts one may approach one or the other \
global minimizer.\
\>", "Text"],

Cell["\<\
Now we work with our friend The Fiendish Problem From Chapter 3. This time it \
has a pair of constraint functions. \
\>", "Text"],

Cell[BoxData[{
    \(fhard[x1_, x2_, x3_, x4_] := 
      3*x1^2*x2\  - \ x1*x2*x3\  + \ x2*x4^2\  + \ 2*x1\  - \ 7*x3\  + 
        Exp[x1^2 + x2^4 + x3^2 + x4^2]\), "\[IndentingNewLine]", 
    \(\(vars\  = \ {x1, x2, x3, x4};\)\), "\[IndentingNewLine]", 
    \(g1[x1_, x2_, x3_, x4_] := \ x1^2\  - \ 3*x2\  + \ x3*x4\  + \ 5\), 
  "\[IndentingNewLine]", 
    \(g2[x1_, x2_, x3_, x4_] := \ x1*x2\  - \ x4^2\  + \ x3\  + \ 6\)}], 
  "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethod[fhard[x1, x2, x3, x4], 
      {g1[x1, x2, x3, x4], g2[x1, x2, x3, x4]}, {x1, x2, x3, x4}, 20]\)], 
  "Input"],

Cell[BoxData[
    \({3738.2891681640563`, 
      {x1 \[Rule] \(-0.1923975091869087`\), x2 \[Rule] 0.9233538540828495`, 
        x3 \[Rule] 0.8748106916033538`, x4 \[Rule] \(-2.587315857212749`\)}}
      \)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethod[fhard[x1, x2, x3, x4], 
      {g1[x1, x2, x3, x4], g2[x1, x2, x3, x4]}, {x1, x2, x3, x4}, 20]\)], 
  "Input"],

Cell[BoxData[
    \({720.466613599871`, 
      {x1 \[Rule] \(-0.19567764147899172`\), x2 \[Rule] 0.833189999560709`, 
        x3 \[Rule] \(-1.175726100655262`\), x4 \[Rule] 2.1588639031776657`}}
      \)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethod[fhard[x1, x2, x3, x4], 
      {g1[x1, x2, x3, x4], g2[x1, x2, x3, x4]}, {x1, x2, x3, x4}, 20]\)], 
  "Input"],

Cell[BoxData[
    \(General::"ovfl" \( : \ \) "Overflow occurred in computation."\)], 
  "Message"]
}, Open  ]],

Cell["\<\
Note that different runs give at least two different results after 20 \
iterations (not counting sporadic runs that overflow and thus fail entirely). \
It seems that one converges to a local minimum value near 3738 which is in \
fact not a global minimizer for our constrained domain. This points to a \
certain problem: when one converts to an unconstrained minimization problem, \
one is still stuck with the fact that unconstrained minimizers are imperfect \
and frequently give local rather than global minimizers. As a remark it is \
quite common to have simple constraint functions defined, for example, by \
polynomials. So the constraints in this example are not atypical.\
\>", "Text"],

Cell[TextData[{
  "If you do problems using the penalty method by hand, you run into an \
obstacle that does not arise when using software. How does one find an \
unconstrained minimizer for a piece-wise defined function? This is a bit \
tricky. It is also not terribly important because nontrivial problems cannot \
be done by hand anyway. So let us ask why it poses no real difficulty for the \
software. This is because, at any given point in ",
  Cell[BoxData[
      \(\[DoubleStruckCapitalR]\^n\)]],
  ", the function is known. Hence a minimizer can use methods of chapter 3, \
for example Newton or BFGS, assuming the function is differentiable at that \
point. If the objective function ",
  Cell[BoxData[
      \(f\)]],
  " and all the constraint functions ",
  Cell[BoxData[
      \(g\_j\)]],
  " are differentiable, then the unconstrained function-plus-penalty will be \
differentiable almost everywhere, and numeric approximations to derivatives \
will work just fine even where the function is not differentiable. When one \
works by hand, in contrast, one uses methods of calculus to find roots of \
critical points, and this requires a global definition that is not \
piecewise."
}], "Text"],

Cell[TextData[{
  "There is an interesting class of problems for which one may use penalty \
functions and avoid the piecewise nature of the definitions. This is when the \
constraints are equations rather than inequalities. In this setting we want \
to impose a penalty wherever ",
  Cell[BoxData[
      \(g\_j \[NotEqual] 0\)]],
  ", not just when ",
  Cell[BoxData[
      \(g\_j > 0\)]],
  ". I will show an example, again using the test problem from chapter 3. I \
call the function ",
  Cell[BoxData[
      \(penaltyMethodEC\)]],
  " where the last two letters stands for \"equation constraints.\" It is of \
course trivial to combine it with ",
  Cell[BoxData[
      \(penaltyMethod\)]],
  " into one function; I omit the details."
}], "Text"],

Cell[BoxData[
    \(penaltyMethodEC[func_, constraints_List, vars_, iters_, 
        prnt_:  False]\  := \ 
      Module[\[IndentingNewLine]{sumconstraints, next, vals, j, 
          k}, \[IndentingNewLine]sumconstraints = 
          constraints . constraints; \[IndentingNewLine]inits\  = 
          Table[100*Random[] - 50, 
            {Length[vars]}]; \[IndentingNewLine]inits\  = 
          Table[\ {vars[\([j]\)], inits[\([j]\)]}, 
            {j, Length[vars]}]; \[IndentingNewLine]next\  = \ 
          FindMinimum[Evaluate[func + sumconstraints], \ 
            Evaluate[Apply[Sequence, inits]]]; \[IndentingNewLine]If[prnt, 
          Print[next]]; \[IndentingNewLine]Do\ [\[IndentingNewLine]vals\  = \ 
            Table[Apply[List, \ next[\([2, k]\)]], \ 
              {k, Length[vars]}]; \[IndentingNewLine]next\  = \ 
            FindMinimum[Evaluate[func + 2^j*sumconstraints], \ 
              Evaluate[Apply[Sequence, vals]]]; \[IndentingNewLine]If[prnt, 
            Print[next]], \[IndentingNewLine]{j, 
            iters - 1}]; \[IndentingNewLine]{func /. next[\([2]\)], 
          next[\([2]\)]}\[IndentingNewLine]]\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethodEC[fhard[x1, x2, x3, x4], 
      {g1[x1, x2, x3, x4], g2[x1, x2, x3, x4]}, {x1, x2, x3, x4}, 20]\)], 
  "Input"],

Cell[BoxData[
    \({720.4666135998594`, 
      {x1 \[Rule] \(-0.19567764003252475`\), x2 \[Rule] 0.8331899990952756`, 
        x3 \[Rule] \(-1.1757261009366904`\), x4 \[Rule] 2.1588639034281103`}}
      \)], "Output"]
}, Open  ]],

Cell["\<\
No surprise here, we already knew the minimizer for the \
inequality-constrained problem was a point for which both constraints are \
zero. Here is a question: Assuming a minimizer exists and is not in the \
interior of the constrained region, when and why is it likely that it will be \
at a \"vertex\" of the boundary (i.e. a point where all the constraint \
functions vanish) , rather than at a point where just one of the constraints \
vanishes and the others are negative? If you are familiar with linear \
programming you will have some insight into this matter.\
\>", "Text"],

Cell[TextData[{
  "Actually when working with inequality constraints there is a way to avoid \
piecewise functions. Instead of using the absolute value or Courant-Beltrami \
penalty functions, work with an exponential. In particular for each \
constraint ",
  Cell[BoxData[
      \(g\_j\)]],
  " I will (for now) use ",
  Cell[BoxData[
      \(\[ExponentialE]\^\(k\ g\_j[x, y]\)\/k\)]],
  ". Below I illustrate with our simple example."
}], "Text"],

Cell[BoxData[{
    \(f[x_, y_] := x\^2 + 2\ y\^2\), "\[IndentingNewLine]", 
    \(g[x_, y_] := \ 1 - x\^2 - y\^2\), "\[IndentingNewLine]", 
    \(gpen[x_, y_, k_]\  := Exp[k*g[x, y]]/k\), "\[IndentingNewLine]", 
    \(\(f[k_]\)[x_, y_]\  := \ f[x, y]\  + \ gpen[x, y, k]\)}], "Input"],

Cell[CellGroupData[{

Cell[BoxData[{
    \({minvalue[1], vals[1]} = 
      FindMinimum[\(f[1]\)[x, y], \ {x, 50, 55}, \ {y, 50, 55}]\), 
  "\[IndentingNewLine]", 
    \({newx[1], newy[1]}\  = \ {x, y}\  /. \ vals[1]\)}], "Input"],

Cell[BoxData[
    \({2.0000000000000417`, 
      {x \[Rule] \(-1.000000133951174`\), 
        y \[Rule] \(-7.810890295260019`*^-8\)}}\)], "Output"],

Cell[BoxData[
    \(General::"spell1" \( : \ \) 
      "Possible spelling error: new symbol name \"\!\(newy\)\" is similar to \
existing symbol \"\!\(newx\)\"."\)], "Message"],

Cell[BoxData[
    \({\(-1.000000133951174`\), \(-7.810890295260019`*^-8\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Why did it work so well? Indeed, why does it work at all? The idea is \
quite simple. For ",
  Cell[BoxData[
      \(g\_j > 0\)]],
  " the penalty will get large as ",
  Cell[BoxData[
      \(k \[Rule] \[Infinity]\)]],
  " because the exponential will dominate the factor of ",
  Cell[BoxData[
      \(1\/k\)]],
  ". For ",
  Cell[BoxData[
      \(g\_j < 0\)]],
  " the exponential is less than ",
  Cell[BoxData[
      \(1\)]],
  ", hence the penalty term will be less than ",
  Cell[BoxData[
      \(1\/k\)]],
  " everywhere in our domain."
}], "Text"],

Cell["\<\
I modified the penalty a bit below. It appears to work tolerably in test runs \
with our harder problem.\
\>", "Text"],

Cell[BoxData[
    \(penaltyMethodExpCon[func_, constraints_List, vars_, iters_, 
        prnt_:  False]\  := \ 
      Module[\[IndentingNewLine]{posconstraints, sumconstraints, next, vals, 
          j, k}, \[IndentingNewLine]inits\  = 
          Table[100*Random[] - 50, 
            {Length[vars]}]; \[IndentingNewLine]inits\  = 
          Table[\ {vars[\([j]\)], inits[\([j]\)]}, 
            {j, Length[vars]}]; \[IndentingNewLine]posconstraints\  = 
          Map[Exp[#] &, constraints]; \[IndentingNewLine]sumconstraints = 
          Apply[Plus, posconstraints]; \[IndentingNewLine]next\  = \ 
          FindMinimum[Evaluate[func + sumconstraints], \ 
            Evaluate[Apply[Sequence, inits]]]; \[IndentingNewLine]If[prnt, 
          Print[next]]; \[IndentingNewLine]Do\ [\[IndentingNewLine]vals\  = \ 
            Table[Apply[List, \ next[\([2, k]\)]], \ 
              {k, Length[vars]}]; \[IndentingNewLine]posconstraints\  = 
            Map[\((1.5^\((\(-j\))\)*Exp[j^2*#])\) &, 
              constraints]; \[IndentingNewLine]sumconstraints = 
            Apply[Plus, posconstraints]; \[IndentingNewLine]next\  = \ 
            FindMinimum[Evaluate[func + sumconstraints], \ 
              Evaluate[Apply[Sequence, vals]]], \[IndentingNewLine]{j, 
            iters - 1}]; \[IndentingNewLine]{func /. next[\([2]\)], 
          next[\([2]\)]}\[IndentingNewLine]]\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethodExpCon[fhard[x1, x2, x3, x4], 
      {g1[x1, x2, x3, x4], g2[x1, x2, x3, x4]}, {x1, x2, x3, x4}, 200]\)], 
  "Input"],

Cell[BoxData[
    \({3750.863471947298`, 
      {x1 \[Rule] \(-0.19222318405943575`\), x2 \[Rule] 0.9234515409921666`, 
        x3 \[Rule] 0.8758333307746985`, x4 \[Rule] \(-2.588150378635404`\)}}
      \)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(penaltyMethodExpCon[fhard[x1, x2, x3, x4], 
      {g1[x1, x2, x3, x4], g2[x1, x2, x3, x4]}, {x1, x2, x3, x4}, 200]\)], 
  "Input"],

Cell[BoxData[
    \({721.0082514996312`, 
      {x1 \[Rule] \(-0.19565173869067426`\), x2 \[Rule] 0.8332375832791118`, 
        x3 \[Rule] \(-1.1759172946933094`\), x4 \[Rule] 2.159004176565584`}}
      \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "We still have a situation wherein we might converge to different (local) \
constrained minimizers. This may be unavoidable. Also we do not really get to \
the boundary in a hurry, and moreovere we lose in efficiency because ",
  Cell[BoxData[
      \(FindMinimum\)]],
  " is now called with a more complicated function. I think the moral is that \
it pays to keep the penalty function simple whenever possible."
}], "Text"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 640}, {0, 424}},
WindowSize->{496, 340},
WindowMargins->{{Automatic, 29}, {Automatic, -21}},
PrintingCopies->1,
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"PrintingMargins"->{{36, 36}, {36, 43.1875}}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 28, 0, 64, "Subtitle"],

Cell[CellGroupData[{
Cell[1792, 55, 120, 3, 43, "Subsubsection"],
Cell[1915, 60, 1029, 28, 185, "Text"],
Cell[2947, 90, 315, 5, 91, "Input"],

Cell[CellGroupData[{
Cell[3287, 99, 207, 4, 70, "Input"],
Cell[3497, 105, 141, 3, 29, "Output"],
Cell[3641, 110, 84, 1, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[3762, 116, 174, 3, 50, "Input"],
Cell[3939, 121, 131, 3, 29, "Output"],
Cell[4073, 126, 78, 1, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[4188, 132, 241, 4, 90, "Input"],
Cell[4432, 138, 120, 3, 29, "Output"],
Cell[4555, 143, 80, 1, 29, "Output"]
}, Open  ]],
Cell[4650, 147, 1233, 35, 204, "Text"],
Cell[5886, 184, 1374, 22, 490, "Input"],

Cell[CellGroupData[{
Cell[7285, 210, 78, 1, 30, "Input"],
Cell[7366, 213, 142, 3, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[7545, 221, 86, 1, 30, "Input"],
Cell[7634, 224, 131, 3, 25, "Print"],
Cell[7768, 229, 119, 3, 25, "Print"],
Cell[7890, 234, 120, 3, 25, "Print"],
Cell[8013, 239, 121, 3, 25, "Print"],
Cell[8137, 244, 127, 3, 25, "Print"],
Cell[8267, 249, 123, 3, 25, "Print"],
Cell[8393, 254, 129, 3, 25, "Print"],
Cell[8525, 259, 137, 3, 25, "Print"],
Cell[8665, 264, 126, 3, 25, "Print"],
Cell[8794, 269, 139, 3, 25, "Print"],
Cell[8936, 274, 140, 3, 25, "Print"],
Cell[9079, 279, 138, 3, 25, "Print"],
Cell[9220, 284, 130, 3, 25, "Print"],
Cell[9353, 289, 140, 3, 25, "Print"],
Cell[9496, 294, 132, 3, 25, "Print"],
Cell[9631, 299, 132, 3, 25, "Print"],
Cell[9766, 304, 141, 3, 25, "Print"],
Cell[9910, 309, 144, 3, 25, "Print"],
Cell[10057, 314, 132, 3, 25, "Print"],
Cell[10192, 319, 141, 3, 25, "Print"],
Cell[10336, 324, 142, 3, 29, "Output"]
}, Open  ]],
Cell[10493, 330, 119, 3, 52, "Text"],
Cell[10615, 335, 140, 3, 52, "Text"],
Cell[10758, 340, 439, 8, 130, "Input"],

Cell[CellGroupData[{
Cell[11222, 352, 143, 3, 70, "Input"],
Cell[11368, 357, 217, 4, 48, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[11622, 366, 143, 3, 70, "Input"],
Cell[11768, 371, 215, 4, 48, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[12020, 380, 143, 3, 70, "Input"],
Cell[12166, 385, 99, 2, 25, "Message"]
}, Open  ]],
Cell[12280, 390, 704, 10, 166, "Text"],
Cell[12987, 402, 1205, 23, 242, "Text"],
Cell[14195, 427, 749, 19, 147, "Text"],
Cell[14947, 448, 1150, 19, 430, "Input"],

Cell[CellGroupData[{
Cell[16122, 471, 145, 3, 70, "Input"],
Cell[16270, 476, 218, 4, 48, "Output"]
}, Open  ]],
Cell[16503, 483, 592, 9, 128, "Text"],
Cell[17098, 494, 448, 11, 96, "Text"],
Cell[17549, 507, 284, 4, 90, "Input"],

Cell[CellGroupData[{
Cell[17858, 515, 207, 4, 70, "Input"],
Cell[18068, 521, 147, 3, 29, "Output"],
Cell[18218, 526, 175, 3, 63, "Message"],
Cell[18396, 531, 88, 1, 29, "Output"]
}, Open  ]],
Cell[18499, 535, 573, 21, 93, "Text"],
Cell[19075, 558, 128, 3, 52, "Text"],
Cell[19206, 563, 1389, 22, 470, "Input"],

Cell[CellGroupData[{
Cell[20620, 589, 150, 3, 70, "Input"],
Cell[20773, 594, 217, 4, 48, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[21027, 603, 150, 3, 70, "Input"],
Cell[21180, 608, 217, 4, 48, "Output"]
}, Open  ]],
Cell[21412, 615, 442, 8, 109, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

