(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     66345,       1597]*)
(*NotebookOutlinePosition[     67109,       1623]*)
(*  CellTagsIndexPosition[     67065,       1619]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Math 384", "Subtitle"],

Cell[CellGroupData[{

Cell["Inequality-constrained minimization using multipliers", "Subsubsection",
  CellDingbat->None],

Cell["\<\
Below is a simple example of a function we wish to minimize subject to two \
constraints. We will use the method of chapter 5, section 2. Specifically we \
set up the equations that must be solved for the saddle point condition to \
hold, then solve them for the minimizer and Karush-Kuhn-Tucker (KKT) \
multipliers.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(f[x_, y_] := x\^3 - 3\ x\ y\^2 + y\^2 + x\), "\n", 
    \(g1[x_, y_] := x + y - 3\), "\n", 
    \(g2[x_, y_] := y\^2 - x - 5\), "\n", 
    \(\(eqns = {lam1\ \((x + y - 3)\), lam2\ \((y\^2 - x - 5)\), 
          3\ x\^2 - 3\ y\^2 + 1 - lam1 - lam2, \(-6\)\ x\ y + 2\ y + lam1 - 
            2\ lam2\ y};\)\), "\n", 
    \(\(solns = Solve[eqns == 0, {x, y, lam1, lam2}];\)\), "\n", 
    \(N[solns]\)}], "Input"],

Cell[BoxData[
    \({{lam1 \[Rule] \(-56.`\), lam2 \[Rule] 0.`, 
        x \[Rule] \(-1.6666666666666667`\), y \[Rule] 4.666666666666667`}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] 0.`, x \[Rule] 0.3333333333333333`, 
        y \[Rule] \(-0.6666666666666666`\)}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] 0.`, x \[Rule] 0.3333333333333333`, 
        y \[Rule] 0.6666666666666666`}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] 0.`, 
        x \[Rule] 
          \(0.`\[InvisibleSpace]\) - 0.5773502691896258`\ \[ImaginaryI], 
        y \[Rule] 0.`}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] 0.`, 
        x \[Rule] 
          \(0.`\[InvisibleSpace]\) + 0.5773502691896258`\ \[ImaginaryI], 
        y \[Rule] 0.`}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] 76.`, x \[Rule] \(-5.`\), 
        y \[Rule] 0.`}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] \(-5.708203932499369`\), 
        x \[Rule] 2.23606797749979`, y \[Rule] \(-2.6899940478558295`\)}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] \(-5.708203932499369`\), 
        x \[Rule] 2.23606797749979`, y \[Rule] 2.6899940478558295`}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] 7.708203932499369`, 
        x \[Rule] \(-2.23606797749979`\), 
        y \[Rule] \(-1.6625077511098136`\)}, 
      {lam1 \[Rule] 0.`, lam2 \[Rule] 7.708203932499369`, 
        x \[Rule] \(-2.23606797749979`\), y \[Rule] 1.6625077511098136`}, 
      {lam1 \[Rule] 10.`, lam2 \[Rule] 0.`, x \[Rule] 2.`, y \[Rule] 1.`}, 
      {lam1 \[Rule] \(-11.412518790589255`\), 
        lam2 \[Rule] \(-3.2885450282530075`\), x \[Rule] 0.6277186767309857`, 
        y \[Rule] 2.3722813232690143`}, 
      {lam1 \[Rule] 125.41251879058926`, 
        lam2 \[Rule] \(-36.711454971746996`\), x \[Rule] 6.372281323269014`, 
        y \[Rule] \(-3.3722813232690143`\)}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Now some have complex values, and others violate the nonnegativity of the \
KKT multipliers (",
  Cell[BoxData[
      \(lam1\)]],
  " and ",
  Cell[BoxData[
      \(lam2\)]],
  " above). Some also violate the constraints. Once we remove all these \
violators, we are left with several candidate minimizers (and corresponding \
KKT multipliers)."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(s2\  = \ 
      Select[solns, \ \((FreeQ[#, \ 
                Complex]\  && \ \((Simplify[lam1 /. #])\) \[GreaterEqual] 
                0\  && \ \((Simplify[lam2 /. #])\) \[GreaterEqual] 
                0\  && \ \((Simplify[g1[x, y] /. #])\) \[LessEqual] 
                0\  && \ \((Simplify[g2[x, y] /. #])\) \[LessEqual] 
                0)\)\  &]\)], "Input"],

Cell[BoxData[
    \({{lam1 \[Rule] 0, lam2 \[Rule] 0, x \[Rule] 1\/3, 
        y \[Rule] \(-\(2\/3\)\)}, 
      {lam1 \[Rule] 0, lam2 \[Rule] 0, x \[Rule] 1\/3, y \[Rule] 2\/3}, 
      {lam1 \[Rule] 0, lam2 \[Rule] 76, x \[Rule] \(-5\), y \[Rule] 0}, 
      {lam1 \[Rule] 0, lam2 \[Rule] 1 + 3\ \@5, x \[Rule] \(-\@5\), 
        y \[Rule] \(-\@\(5 - \@5\)\)}, 
      {lam1 \[Rule] 0, lam2 \[Rule] 1 + 3\ \@5, x \[Rule] \(-\@5\), 
        y \[Rule] \@\(5 - \@5\)}, 
      {lam1 \[Rule] 10, lam2 \[Rule] 0, x \[Rule] 2, y \[Rule] 1}}\)], 
  "Output"]
}, Open  ]],

Cell["\<\
There is another issue. The theorem about this states there should be only \
one candidate minimizer. What went wrong, and how do we fix it? Actually \
nothing really went wrong. We are simply in a situation where the theorem of \
that section does not quite apply. Specifically, our objective function is \
not convex (though the constraint domain is a convex set). We will learn in \
chapter 7 that this means we may get critical points that are not minimizers, \
but, if a minimizer exists, it will be at a critical point (under mild \
assumptions; I'll not elaborate here on that topic). So all we need do is \
test the candidates and select the one that gives the smallest value for our \
objective function.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[f[x, y]\  /. \ s2]\)], "Input"],

Cell[BoxData[
    \({0.37037037037037035`, 0.37037037037037035`, \(-130.`\), 
      7.888543819998315`, 7.888543819998315`, 5.`}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(s2[\([3]\)]\)], "Input"],

Cell[BoxData[
    \({lam1 \[Rule] 0, lam2 \[Rule] 76, x \[Rule] \(-5\), y \[Rule] 0}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "Our third candidate, with ",
  Cell[BoxData[
      \(x = \(-5\)\)]],
  " nd ",
  Cell[BoxData[
      \(y = 0\)]],
  ", gives the minimum for this constrained problem. A graph shows that this \
is plausible."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Plot3D[f[x, y], \ {x, \(-7\), 3}, \ {y, \(-3\), 3}, \ 
        Shading \[Rule] False];\)\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0 1.04977 -0.0679587 1.04977 [
[.08608 .22315 -12 -8.87306 ]
[.08608 .22315 0 .12694 ]
[.20731 .17545 -11.6738 -9 ]
[.20731 .17545 .32622 0 ]
[.3345 .12542 -11.1734 -9 ]
[.3345 .12542 .8266 0 ]
[.46808 .07289 -5.33651 -9 ]
[.46808 .07289 .66349 0 ]
[.60856 .01769 -5.08632 -9 ]
[.60856 .01769 .91368 0 ]
[.75679 .08071 0 -6.17777 ]
[.75679 .08071 12 2.82223 ]
[.8565 .23339 0 -6.03127 ]
[.8565 .23339 6 2.96873 ]
[.9429 .3657 0 -5.9083 ]
[.9429 .3657 6 3.0917 ]
[.02003 .29069 -24 -2.76392 ]
[.02003 .29069 0 6.23608 ]
[.01166 .34314 -24 -2.80835 ]
[.01166 .34314 0 6.19165 ]
[.00301 .3974 -24 -2.85445 ]
[.00301 .3974 0 6.14555 ]
[-0.00595 .45355 -6 -2.90234 ]
[-0.00595 .45355 0 6.09766 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.09549 .2323 m
.1002 .23687 L
s
[(-6)] .08608 .22315 1 .97179 Mshowa
.21633 .18498 m
.22084 .18975 L
s
[(-4)] .20731 .17545 .94563 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(-2)] .3345 .12542 .86223 1 Mshowa
.47614 .08324 m
.48017 .08842 L
s
[(0)] .46808 .07289 .77884 1 Mshowa
.61605 .02846 m
.6198 .03385 L
s
[(2)] .60856 .01769 .69544 1 Mshowa
.125 Mabswid
.12517 .22067 m
.12797 .22345 L
s
.1552 .20892 m
.15797 .21172 L
s
.18558 .19702 m
.18832 .19985 L
s
.24744 .1728 m
.25012 .17569 L
s
.27893 .16047 m
.28157 .16339 L
s
.3108 .14799 m
.31341 .15094 L
s
.37572 .12257 m
.37826 .12558 L
s
.40878 .10962 m
.41128 .11266 L
s
.44225 .09652 m
.44471 .09959 L
s
.51046 .06981 m
.51284 .07294 L
s
.54521 .0562 m
.54755 .05937 L
s
.58041 .04242 m
.5827 .04562 L
s
.06616 .24378 m
.06901 .2465 L
s
.65216 .01432 m
.65436 .01759 L
s
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.7445 .0853 m
.73835 .08759 L
s
[(-2)] .75679 .08071 -1 .37284 Mshowa
.84407 .23761 m
.83786 .23973 L
s
[(0)] .8565 .23339 -1 .34028 Mshowa
.93037 .36962 m
.92411 .37158 L
s
[(2)] .9429 .3657 -1 .31296 Mshowa
.125 Mabswid
.7708 .12553 m
.7671 .12688 L
s
.79613 .16428 m
.79242 .1656 L
s
.82054 .20161 m
.81682 .20291 L
s
.86679 .27235 m
.86305 .2736 L
s
.88871 .30589 m
.88497 .30711 L
s
.9099 .3383 m
.90615 .3395 L
s
.71716 .04348 m
.71349 .04489 L
s
.95018 .39992 m
.94642 .40107 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.03228 .28596 m
.0384 .2836 L
s
[(-300)] .02003 .29069 1 -0.38579 Mshowa
.02395 .33852 m
.03009 .33621 L
s
[(-200)] .01166 .34314 1 -0.37592 Mshowa
.01533 .39289 m
.02149 .39064 L
s
[(-100)] .00301 .3974 1 -0.36568 Mshowa
.00641 .44916 m
.0126 .44697 L
s
[(0)] -0.00595 .45355 1 -0.35504 Mshowa
.125 Mabswid
.03063 .29634 m
.03431 .29492 L
s
.02898 .30678 m
.03265 .30537 L
s
.02731 .31729 m
.03099 .31589 L
s
.02563 .32787 m
.02932 .32648 L
s
.02225 .34925 m
.02593 .34787 L
s
.02053 .36005 m
.02422 .35868 L
s
.01881 .37092 m
.0225 .36956 L
s
.01708 .38187 m
.02077 .38051 L
s
.01357 .40399 m
.01727 .40265 L
s
.0118 .41517 m
.0155 .41383 L
s
.01002 .42642 m
.01372 .42509 L
s
.00822 .43775 m
.01193 .43643 L
s
.03391 .27566 m
.03758 .27424 L
s
.03553 .26543 m
.0392 .264 L
s
.03714 .25526 m
.04081 .25382 L
s
.00459 .46065 m
.0083 .45934 L
s
.00276 .47223 m
.00647 .47092 L
s
.00091 .48388 m
.00463 .48258 L
s
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
1 g
.38023 .65285 .40181 .69851 .43865 .72583 .41681 .68206 Metetra
1 g
.41681 .68206 .43865 .72583 .47697 .7442 .45499 .7025 Metetra
1 g
.45499 .7025 .47697 .7442 .51652 .7542 .49451 .71468 Metetra
1 g
.49451 .71468 .51652 .7542 .55704 .75657 .53513 .71934 Metetra
1 g
.53513 .71934 .55704 .75657 .59833 .75222 .57663 .71733 Metetra
1 g
.57663 .71733 .59833 .75222 .64022 .74214 .61883 .70963 Metetra
1 g
.61883 .70963 .64022 .74214 .68256 .72744 .6616 .6973 Metetra
1 g
.6616 .6973 .68256 .72744 .72527 .70924 .70483 .68148 Metetra
1 g
.70483 .68148 .72527 .70924 .76833 .68872 .74849 .66333 Metetra
1 g
.74849 .66333 .76833 .68872 .81174 .66704 .7926 .644 Metetra
1 g
.7926 .644 .81174 .66704 .85561 .6454 .83725 .6247 Metetra
1 g
.83725 .6247 .85561 .6454 .90006 .62494 .88256 .60659 Metetra
1 g
.88256 .60659 .90006 .62494 .94531 .60683 .92875 .59085 Metetra
1 g
.92875 .59085 .94531 .60683 .99163 .59227 .97611 .57871 Metetra
1 g
.35822 .61059 .38023 .65285 .41681 .68206 .39455 .64137 Metetra
1 g
.39455 .64137 .41681 .68206 .45499 .7025 .4326 .66351 Metetra
1 g
.4326 .66351 .45499 .7025 .49451 .71468 .4721 .6775 Metetra
1 g
.4721 .6775 .49451 .71468 .53513 .71934 .51282 .68404 Metetra
1 g
.51282 .68404 .53513 .71934 .57663 .71733 .55452 .68396 Metetra
1 g
.55452 .68396 .57663 .71733 .61883 .70963 .59703 .67821 Metetra
1 g
.59703 .67821 .61883 .70963 .6616 .6973 .64019 .66785 Metetra
1 g
.64019 .66785 .6616 .6973 .70483 .68148 .68391 .654 Metetra
1 g
.68391 .654 .70483 .68148 .74849 .66333 .72815 .63779 Metetra
1 g
.72815 .63779 .74849 .66333 .7926 .644 .77292 .62042 Metetra
1 g
.77292 .62042 .7926 .644 .83725 .6247 .8183 .60305 Metetra
1 g
.8183 .60305 .83725 .6247 .88256 .60659 .86442 .58689 Metetra
1 g
.86442 .58689 .88256 .60659 .92875 .59085 .9115 .57314 Metetra
1 g
.9115 .57314 .92875 .59085 .97611 .57871 .95982 .56303 Metetra
1 g
.33569 .57167 .35822 .61059 .39455 .64137 .3718 .6037 Metetra
1 g
.3718 .6037 .39455 .64137 .4326 .66351 .40973 .62719 Metetra
1 g
.40973 .62719 .4326 .66351 .4721 .6775 .44922 .64262 Metetra
1 g
.44922 .64262 .4721 .6775 .51282 .68404 .49003 .65065 Metetra
1 g
.49003 .65065 .51282 .68404 .55452 .68396 .53194 .65209 Metetra
1 g
.53194 .65209 .55452 .68396 .59703 .67821 .57474 .64789 Metetra
1 g
.57474 .64789 .59703 .67821 .64019 .66785 .6183 .63908 Metetra
1 g
.6183 .63908 .64019 .66785 .68391 .654 .66249 .62677 Metetra
1 g
.66249 .62677 .68391 .654 .72815 .63779 .70728 .6121 Metetra
1 g
.70728 .6121 .72815 .63779 .77292 .62042 .75268 .59626 Metetra
1 g
.75268 .59626 .77292 .62042 .8183 .60305 .79874 .58042 Metetra
1 g
.79874 .58042 .8183 .60305 .86442 .58689 .84562 .56581 Metetra
1 g
.84562 .56581 .86442 .58689 .9115 .57314 .89352 .55363 Metetra
1 g
.89352 .55363 .9115 .57314 .95982 .56303 .94274 .54515 Metetra
1 g
.31255 .53603 .33569 .57167 .3718 .6037 .34846 .569 Metetra
1 g
.34846 .569 .3718 .6037 .40973 .62719 .38629 .59351 Metetra
1 g
.38629 .59351 .40973 .62719 .44922 .64262 .42579 .61002 Metetra
1 g
.42579 .61002 .44922 .64262 .49003 .65065 .46672 .61916 Metetra
1 g
.46672 .61916 .49003 .65065 .53194 .65209 .50883 .62173 Metetra
1 g
.50883 .62173 .53194 .65209 .57474 .64789 .55193 .61866 Metetra
1 g
.55193 .61866 .57474 .64789 .6183 .63908 .59587 .61098 Metetra
1 g
.59587 .61098 .6183 .63908 .66249 .62677 .64053 .59979 Metetra
1 g
.64053 .59979 .66249 .62677 .70728 .6121 .68585 .58623 Metetra
1 g
.68585 .58623 .70728 .6121 .75268 .59626 .73183 .57149 Metetra
1 g
.73183 .57149 .75268 .59626 .79874 .58042 .77855 .55677 Metetra
1 g
.77855 .55677 .79874 .58042 .84562 .56581 .82614 .54327 Metetra
1 g
.82614 .54327 .84562 .56581 .89352 .55363 .87481 .53224 Metetra
1 g
.87481 .53224 .89352 .55363 .94274 .54515 .92485 .52496 Metetra
1 g
.28871 .50363 .31255 .53603 .34846 .569 .32443 .53725 Metetra
1 g
.32443 .53725 .34846 .569 .38629 .59351 .3622 .56246 Metetra
1 g
.3622 .56246 .38629 .59351 .42579 .61002 .40173 .57968 Metetra
1 g
.40173 .57968 .42579 .61002 .46672 .61916 .44279 .58955 Metetra
1 g
.44279 .58955 .46672 .61916 .50883 .62173 .48513 .59285 Metetra
1 g
.48513 .59285 .50883 .62173 .55193 .61866 .52854 .5905 Metetra
1 g
.52854 .5905 .55193 .61866 .59587 .61098 .57286 .58353 Metetra
1 g
.57286 .58353 .59587 .61098 .64053 .59979 .61798 .57303 Metetra
1 g
.61798 .57303 .64053 .59979 .68585 .58623 .66381 .56016 Metetra
1 g
.66381 .56016 .68585 .58623 .73183 .57149 .71037 .5461 Metetra
1 g
.71037 .5461 .73183 .57149 .77855 .55677 .7577 .53205 Metetra
1 g
.7577 .53205 .77855 .55677 .82614 .54327 .80596 .51923 Metetra
1 g
.80596 .51923 .82614 .54327 .87481 .53224 .85534 .50892 Metetra
1 g
.85534 .50892 .87481 .53224 .92485 .52496 .90614 .50239 Metetra
1 g
.26405 .47446 .28871 .50363 .32443 .53725 .29963 .50845 Metetra
1 g
.29963 .50845 .32443 .53725 .3622 .56246 .33736 .53402 Metetra
1 g
.33736 .53402 .3622 .56246 .40173 .57968 .37696 .55161 Metetra
1 g
.37696 .55161 .40173 .57968 .44279 .58955 .41818 .56183 Metetra
1 g
.41818 .56183 .44279 .58955 .48513 .59285 .46076 .56546 Metetra
1 g
.46076 .56546 .48513 .59285 .52854 .5905 .5045 .56343 Metetra
1 g
.5045 .56343 .52854 .5905 .57286 .58353 .54922 .55674 Metetra
1 g
.54922 .55674 .57286 .58353 .61798 .57303 .59479 .5465 Metetra
1 g
.59479 .5465 .61798 .57303 .66381 .56016 .64113 .53387 Metetra
1 g
.64113 .53387 .66381 .56016 .71037 .5461 .68824 .52004 Metetra
1 g
.68824 .52004 .71037 .5461 .7577 .53205 .73617 .50621 Metetra
1 g
.73617 .50621 .7577 .53205 .80596 .51923 .78505 .49364 Metetra
1 g
.78505 .49364 .80596 .51923 .85534 .50892 .83509 .48357 Metetra
1 g
.83509 .48357 .85534 .50892 .90614 .50239 .88659 .47733 Metetra
1 g
.23846 .44853 .26405 .47446 .29963 .50845 .27394 .4826 Metetra
1 g
.27394 .4826 .29963 .50845 .33736 .53402 .31167 .50823 Metetra
1 g
.31167 .50823 .33736 .53402 .37696 .55161 .35138 .52583 Metetra
1 g
.35138 .52583 .37696 .55161 .41818 .56183 .39279 .53602 Metetra
1 g
.39279 .53602 .41818 .56183 .46076 .56546 .43566 .53958 Metetra
1 g
.43566 .53958 .46076 .56546 .5045 .56343 .47975 .53743 Metetra
1 g
.47975 .53743 .5045 .56343 .54922 .55674 .52489 .53059 Metetra
1 g
.52489 .53059 .54922 .55674 .59479 .5465 .57092 .52017 Metetra
1 g
.57092 .52017 .59479 .5465 .64113 .53387 .61778 .50734 Metetra
1 g
.61778 .50734 .64113 .53387 .68824 .52004 .66543 .49328 Metetra
1 g
.66543 .49328 .68824 .52004 .73617 .50621 .71394 .47923 Metetra
1 g
.71394 .47923 .73617 .50621 .78505 .49364 .76341 .46642 Metetra
1 g
.76341 .46642 .78505 .49364 .83509 .48357 .81406 .45614 Metetra
1 g
.81406 .45614 .83509 .48357 .88659 .47733 .8662 .44971 Metetra
1 g
.21181 .42588 .23846 .44853 .27394 .4826 .24724 .45974 Metetra
1 g
.24724 .45974 .27394 .4826 .31167 .50823 .28502 .4851 Metetra
1 g
.28502 .4851 .31167 .50823 .35138 .52583 .32488 .50235 Metetra
1 g
.32488 .50235 .35138 .52583 .39279 .53602 .36654 .51213 Metetra
1 g
.36654 .51213 .39279 .53602 .43566 .53958 .40973 .51521 Metetra
1 g
.40973 .51521 .43566 .53958 .47975 .53743 .45421 .51251 Metetra
1 g
.45421 .51251 .47975 .53743 .52489 .53059 .49979 .50508 Metetra
1 g
.49979 .50508 .52489 .53059 .57092 .52017 .54632 .49403 Metetra
1 g
.54632 .49403 .57092 .52017 .61778 .50734 .5937 .48053 Metetra
1 g
.5937 .48053 .61778 .50734 .66543 .49328 .6419 .46579 Metetra
1 g
.6419 .46579 .66543 .49328 .71394 .47923 .69096 .45104 Metetra
1 g
.69096 .45104 .71394 .47923 .76341 .46642 .74101 .43753 Metetra
1 g
.74101 .43753 .76341 .46642 .81406 .45614 .79223 .42655 Metetra
1 g
.79223 .42655 .81406 .45614 .8662 .44971 .84494 .41943 Metetra
1 g
.18394 .40656 .21181 .42588 .24724 .45974 .21938 .43993 Metetra
1 g
.21938 .43993 .24724 .45974 .28502 .4851 .25727 .46469 Metetra
1 g
.25727 .46469 .28502 .4851 .32488 .50235 .29735 .48123 Metetra
1 g
.29735 .48123 .32488 .50235 .36654 .51213 .33931 .49019 Metetra
1 g
.33931 .49019 .36654 .51213 .40973 .51521 .38288 .49237 Metetra
1 g
.38288 .49237 .40973 .51521 .45421 .51251 .42781 .48869 Metetra
1 g
.42781 .48869 .45421 .51251 .49979 .50508 .47388 .48021 Metetra
1 g
.47388 .48021 .49979 .50508 .54632 .49403 .52093 .46806 Metetra
1 g
.52093 .46806 .54632 .49403 .5937 .48053 .56885 .45343 Metetra
1 g
.56885 .45343 .5937 .48053 .6419 .46579 .6176 .43752 Metetra
1 g
.6176 .43752 .6419 .46579 .69096 .45104 .66722 .42159 Metetra
1 g
.66722 .42159 .69096 .45104 .74101 .43753 .71782 .4069 Metetra
1 g
.71782 .4069 .74101 .43753 .79223 .42655 .76958 .39473 Metetra
1 g
.76958 .39473 .79223 .42655 .84494 .41943 .82282 .38641 Metetra
1 g
.15471 .39066 .18394 .40656 .21938 .43993 .19021 .42325 Metetra
1 g
.19021 .42325 .21938 .43993 .25727 .46469 .22829 .44706 Metetra
1 g
.22829 .44706 .25727 .46469 .29735 .48123 .26866 .46252 Metetra
1 g
.26866 .46252 .29735 .48123 .33931 .49019 .31099 .47026 Metetra
1 g
.31099 .47026 .33931 .49019 .38288 .49237 .35501 .47109 Metetra
1 g
.35501 .47109 .38288 .49237 .42781 .48869 .40044 .46598 Metetra
1 g
.40044 .46598 .42781 .48869 .47388 .48021 .44706 .45598 Metetra
1 g
.44706 .45598 .47388 .48021 .52093 .46806 .49467 .44226 Metetra
1 g
.49467 .44226 .52093 .46806 .56885 .45343 .54317 .426 Metetra
1 g
.54317 .426 .56885 .45343 .6176 .43752 .59251 .40844 Metetra
1 g
.59251 .40844 .6176 .43752 .66722 .42159 .64269 .39084 Metetra
1 g
.64269 .39084 .66722 .42159 .71782 .4069 .69383 .37446 Metetra
1 g
.69383 .37446 .71782 .4069 .76958 .39473 .74611 .3606 Metetra
1 g
.74611 .3606 .76958 .39473 .82282 .38641 .79982 .35058 Metetra
1 g
.12391 .37832 .15471 .39066 .19021 .42325 .15956 .40981 Metetra
1 g
.15956 .40981 .19021 .42325 .22829 .44706 .19791 .43232 Metetra
1 g
.19791 .43232 .22829 .44706 .26866 .46252 .23865 .44629 Metetra
1 g
.23865 .44629 .26866 .46252 .31099 .47026 .28146 .45238 Metetra
1 g
.28146 .45238 .31099 .47026 .35501 .47109 .32601 .45142 Metetra
1 g
.32601 .45142 .35501 .47109 .40044 .46598 .37202 .44439 Metetra
1 g
.37202 .44439 .40044 .46598 .44706 .45598 .41925 .43239 Metetra
1 g
.41925 .43239 .44706 .45598 .49467 .44226 .46749 .41659 Metetra
1 g
.46749 .41659 .49467 .44226 .54317 .426 .51662 .39821 Metetra
1 g
.51662 .39821 .54317 .426 .59251 .40844 .56656 .3785 Metetra
1 g
.56656 .3785 .59251 .40844 .64269 .39084 .61733 .35873 Metetra
1 g
.61733 .35873 .64269 .39084 .69383 .37446 .66901 .34016 Metetra
1 g
.66901 .34016 .69383 .37446 .74611 .3606 .72179 .32409 Metetra
1 g
.72179 .32409 .74611 .3606 .79982 .35058 .77594 .31185 Metetra
1 g
.09133 .36968 .12391 .37832 .15956 .40981 .12723 .39975 Metetra
1 g
.12723 .39975 .15956 .40981 .19791 .43232 .16596 .42058 Metetra
1 g
.16596 .42058 .19791 .43232 .23865 .44629 .20718 .43265 Metetra
1 g
.20718 .43265 .23865 .44629 .28146 .45238 .25055 .43663 Metetra
1 g
.25055 .43663 .28146 .45238 .32601 .45142 .29574 .43339 Metetra
1 g
.29574 .43339 .32601 .45142 .37202 .44439 .34243 .42395 Metetra
1 g
.34243 .42395 .37202 .44439 .41925 .43239 .39036 .40944 Metetra
1 g
.39036 .40944 .41925 .43239 .46749 .41659 .43931 .39105 Metetra
1 g
.43931 .39105 .46749 .41659 .51662 .39821 .48912 .37003 Metetra
1 g
.48912 .37003 .51662 .39821 .56656 .3785 .53972 .34766 Metetra
1 g
.53972 .34766 .56656 .3785 .61733 .35873 .5911 .32519 Metetra
1 g
.5911 .32519 .61733 .35873 .66901 .34016 .64334 .30393 Metetra
1 g
.64334 .30393 .66901 .34016 .72179 .32409 .69661 .28514 Metetra
1 g
.69661 .28514 .72179 .32409 .77594 .31185 .75119 .27015 Metetra
1 g
.05674 .36495 .09133 .36968 .12723 .39975 .09299 .39324 Metetra
1 g
.09299 .39324 .12723 .39975 .16596 .42058 .13221 .41199 Metetra
1 g
.13221 .41199 .16596 .42058 .20718 .43265 .17404 .4217 Metetra
1 g
.17404 .4217 .20718 .43265 .25055 .43663 .21811 .42309 Metetra
1 g
.21811 .42309 .25055 .43663 .29574 .43339 .26406 .41707 Metetra
1 g
.26406 .41707 .29574 .43339 .34243 .42395 .31155 .40469 Metetra
1 g
.31155 .40469 .34243 .42395 .39036 .40944 .36029 .38713 Metetra
1 g
.36029 .38713 .39036 .40944 .43931 .39105 .41004 .36561 Metetra
1 g
.41004 .36561 .43931 .39105 .48912 .37003 .46062 .34143 Metetra
1 g
.46062 .34143 .48912 .37003 .53972 .34766 .51195 .31585 Metetra
1 g
.51195 .31585 .53972 .34766 .5911 .32519 .56399 .29018 Metetra
1 g
.56399 .29018 .5911 .32519 .64334 .30393 .61681 .26569 Metetra
1 g
.61681 .26569 .64334 .30393 .69661 .28514 .67058 .24367 Metetra
1 g
.67058 .24367 .69661 .28514 .75119 .27015 .72555 .22543 Metetra
1 g
.01985 .36439 .05674 .36495 .09299 .39324 .05659 .39051 Metetra
1 g
.05659 .39051 .09299 .39324 .13221 .41199 .09645 .40674 Metetra
1 g
.09645 .40674 .13221 .41199 .17404 .4217 .13904 .4136 Metetra
1 g
.13904 .4136 .17404 .4217 .21811 .42309 .18395 .41188 Metetra
1 g
.18395 .41188 .21811 .42309 .26406 .41707 .23081 .40252 Metetra
1 g
.23081 .40252 .26406 .41707 .31155 .40469 .27924 .38665 Metetra
1 g
.27924 .38665 .31155 .40469 .36029 .38713 .32892 .36547 Metetra
1 g
.32892 .36547 .36029 .38713 .41004 .36561 .37959 .34027 Metetra
1 g
.37959 .34027 .41004 .36561 .46062 .34143 .43104 .31235 Metetra
1 g
.43104 .31235 .46062 .34143 .51195 .31585 .48317 .28303 Metetra
1 g
.48317 .28303 .51195 .31585 .56399 .29018 .53593 .25362 Metetra
1 g
.53593 .25362 .56399 .29018 .61681 .26569 .58939 .22539 Metetra
1 g
.58939 .22539 .61681 .26569 .67058 .24367 .64367 .19963 Metetra
1 g
.64367 .19963 .67058 .24367 .72555 .22543 .69905 .17761 Metetra
0 g
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.09549 .2323 m
.1002 .23687 L
s
[(-6)] .08608 .22315 1 .97179 Mshowa
.21633 .18498 m
.22084 .18975 L
s
[(-4)] .20731 .17545 .94563 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(-2)] .3345 .12542 .86223 1 Mshowa
.47614 .08324 m
.48017 .08842 L
s
[(0)] .46808 .07289 .77884 1 Mshowa
.61605 .02846 m
.6198 .03385 L
s
[(2)] .60856 .01769 .69544 1 Mshowa
.125 Mabswid
.12517 .22067 m
.12797 .22345 L
s
.1552 .20892 m
.15797 .21172 L
s
.18558 .19702 m
.18832 .19985 L
s
.24744 .1728 m
.25012 .17569 L
s
.27893 .16047 m
.28157 .16339 L
s
.3108 .14799 m
.31341 .15094 L
s
.37572 .12257 m
.37826 .12558 L
s
.40878 .10962 m
.41128 .11266 L
s
.44225 .09652 m
.44471 .09959 L
s
.51046 .06981 m
.51284 .07294 L
s
.54521 .0562 m
.54755 .05937 L
s
.58041 .04242 m
.5827 .04562 L
s
.06616 .24378 m
.06901 .2465 L
s
.65216 .01432 m
.65436 .01759 L
s
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 233.563},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jA000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00;=oo`@006Uo
o`00/goo00<007ooOol0JWoo002dOol00`00Oomoo`1YOol00;Eoo`03001oogoo01=oo`8005=oo`00
/goo00@007ooOol0019oo`800003Ool00000059oo`00]7oo0P0047oo0`000goo0P00DWoo0034Ool2
0006Ool00`00Ool0001AOol00<5oo`<000Qoo`04001oogoo001@Ool00;moo`80009oo`03001oogoo
00Ioo`04001oogoo001@Ool00;aoo`<000eoo`05001oogooOol0001?Ool00;Yoo`80011oo`03001o
ogoo009oo`03001oogoo04aoo`00]goo0`004Woo00<007ooOol00Woo00<007ooOol0C7oo002eOol2
0003Ool00`00Oomoo`0@Ool00`00Oomoo`02Ool00`00Oomoo`1;Ool00;9oo`<001Qoo`03001oogoo
009oo`03001oogoo04]oo`00/7oo0P006goo00<007ooOol00goo00<007ooOol0BWoo002?Ool2000L
Ool4000LOol00`00Oomoo`03Ool2001;Ool008ioo`04001oogoo000IOol2000POol00`00Oomoo`04
Ool00`00Oomoo`19Ool008ioo`04001oogoo000FOol3000ROol00`00Oomoo`05Ool00`00Oomoo`18
Ool008ioo`04001oogoo000DOol2000UOol00`00Oomoo`06Ool00`00Oomoo`17Ool008ioo`04001o
ogoo000AOol300000goo001oo`0TOol00`00Oomoo`06Ool00`00Oomoo`17Ool008moo`8000moo`<0
02Yoo`03001oogoo00Moo`03001oogoo04Ioo`00WWoo0P00;Goo00<007ooOol027oo00<007ooOol0
2Woo1000=goo002KOol3000_Ool00`00Oomoo`08Ool00`00Oomoo`0:Ool00`00Oomoo`0hOol009Uo
o`8000=oo`03001oogoo02aoo`03001oogoo00Uoo`03001oogoo00Yoo`03001oogoo03Moo`00UWoo
0`00=7oo00<007ooOol02Woo00<007ooOol00Woo100017oo00<007ooOol0=Woo002DOol2000gOol0
0`00Oomoo`0:Ool00`00Oomoo`08Ool01000Oomoo`00=goo002AOol300000goo001oo`0fOol00`00
Oomoo`0;Ool00`00Oomoo`08Ool2000hOol008moo`8000Aoo`03001oogoo03Eoo`03001oogoo00Uo
o`800003Ool007oo049oo`00Jgoo10007Goo0`00?goo00<007ooOol02goo00<007ooOol0@Goo001[
Ool00`00Oomoo`0LOol20012Ool00`00Oomoo`0<Ool00`00Oomoo`10Ool006aoo`03001oogoo01Qo
o`<0009oo`03001oogoo03moo`03001oogoo00eoo`03001oogoo03moo`00IGoo100017oo00<007oo
Ool05Goo0P00Agoo00<007ooOol03Goo00<007ooOol0?goo001[Ool01000Oomoo`004goo0`00BGoo
00<007ooOol03Woo00<007ooOol0?Woo001/Ool2000BOol200000goo001oo`19Ool00`00Oomoo`0>
Ool00`00Oomoo`0nOol007eoo`<004ioo`03001oogoo00moo`03001oogoo03eoo`00Ngoo0P00DGoo
00<007ooOol047oo00<007ooOol0?7oo001hOol3001COol00`00Oomoo`0@Ool00`00Oomoo`0lOol0
07Ioo`80009oo`03001oogoo055oo`03001oogoo011oo`8003eoo`00Lgoo0`00F7oo00<007ooOol0
4Woo00<007ooOol0>Woo001aOol2001KOol00`00Oomoo`0BOol00`00Oomoo`0jOol006ioo`<00003
Ool007oo05Yoo`03001oogoo01=oo`03001oogoo03Uoo`00K7oo0P00H7oo00<007ooOol057oo00<0
07ooOol0>7oo001;Ool3000KOol3001ROol00`00Oomoo`0DOol00`00Oomoo`0hOol004aoo`03001o
ogoo01Qoo`<006Aoo`03001oogoo01Eoo`03001oogoo03Moo`00BGoo1@005Woo0`00J7oo00<007oo
Ool05Goo00<007ooOol0=Woo0013Ool40002Ool01000Oomoo`005Goo0P00Jgoo00<007ooOol05Goo
00<007ooOol0=Woo001:Ool00`00Ool0000BOol3001]Ool00`00Oomoo`0FOol00`00Oomoo`0eOol0
04]oo`80011oo`80071oo`03001oogoo01Ioo`8003Ioo`00FWoo0`00LWoo00<007ooOol05goo00<0
07ooOol0=7oo001HOol2001eOol00`00Oomoo`0HOol00`00Oomoo`0cOol005Eoo`<00003Ool007oo
07=oo`8001]oo`03001oogoo039oo`00Dgoo0P00Mgoo0P0000=oo`0000006Woo00<007ooOol0<Woo
001@Ool3001gOol20003Ool2000KOol00`00Oomoo`0aOol004ioo`800003Ool007oo07Aoo`<000Eo
o`03001oo`0001Yoo`03001oogoo035oo`00:7oo0`0087oo0`00Mgoo0P0027oo00<007oo00006goo
00<007ooOol0<7oo000XOol01000Oomoo`0077oo0`00N7oo0P002Woo00@007ooOol001]oo`03001o
ogoo02moo`00:7oo00@007ooOol001Yoo`<007Moo`<000aoo`04001oogoo000KOol00`00Oomoo`0_
Ool0029oo`@0009oo`<001Qoo`<007Qoo`80009oo`03001oogoo00Yoo`05001oogooOol0000JOol2
000`Ool002Uoo`03001oogoo01Eoo`8007Uoo`8000Aoo`03001oogoo00Yoo`05001oogooOol0000L
Ool00`00Oomoo`0]Ool002Uoo`<0019oo`<007Uoo`8000Moo`03001oogoo00Uoo`03001oogoo009o
o`03001oogoo01Uoo`03001oogoo02eoo`00?7oo0P00NWoo0P002Goo00<007ooOol02Woo00D007oo
Oomoo`0001aoo`03001oogoo02aoo`00>Goo0`00NWoo0P0037oo00<007ooOol02Goo00<007ooOol0
0Woo00<007ooOol06Woo00<007ooOol01Woo0P008goo000gOol2001kOol2000>Ool00`00Oomoo`09
Ool00`00Oomoo`02Ool00`00Oomoo`0JOol00`00Oomoo`05Ool01000Oomoo`008Woo000dOol3001j
Ool3000AOol00`00Oomoo`08Ool00`00Oomoo`02Ool2000LOol00`00Oomoo`04Ool01000Oomoo`00
8Woo000bOol2001kOol20002Ool00`00Oomoo`0@Ool00`00Oomoo`07Ool00`00Ool000020002Ool0
0`00Oomoo`0JOol00`00Oomoo`03Ool01000Oomoo`008Woo000_Ool3001kOol20004Ool00`00Oomo
o`0@Ool00`00Oomoo`07Ool20005Ool00`00Oomoo`0JOol00`00Oomoo`03Ool01000Oomoo`008Woo
000]Ool20002Ool00`00Oomoo`1gOol20007Ool00`00Oomoo`0@Ool00`00Oomoo`03Ool40007Ool0
0`00Oomoo`0JOol00`00Oomoo`03Ool2000SOol002Yoo`<007aoo`8000Yoo`03001oogoo00moo`03
001oogoo00<000=oo`03001oogoo00Eoo`03001oogoo01Uoo`03001oo`0002Uoo`00:7oo0P00OGoo
0P0037oo00<007ooOol03goo0`001Goo0P0027oo00<007ooOol06Woo00<007ooOol09goo000UOol3
001mOol2000?Ool00`00Oomoo`0;Ool30002Ool00`00Oomoo`030002Ool00`00Oomoo`06Ool00`00
Oomoo`0JOol00`00Oomoo`0VOol0025oo`04001oo`00001oOol00`00Oomoo`0@Ool00`00Oomoo`07
Ool30005Ool30005Ool00`00Oomoo`06Ool00`00Oomoo`0JOol00`00Oomoo`0VOol0025oo`8007mo
o`800003Ool007oo015oo`03001oogoo00=oo`<000Ioo`800003Ool007oo00Ioo`03001oogoo00Io
o`03001oogoo01Yoo`03001oogoo02Eoo`008Goo0P00OGoo0P000goo00<007ooOol047oo00<007oo
Ool00`001Woo0`000goo00<007ooOol01Woo00<007ooOol01Goo00<007ooOol06goo00<007ooOol0
97oo000POol200000goo001oo`1iOol20006Ool00`00Oomoo`0@Ool20006Ool30007Ool00`00Oomo
o`06Ool00`00Oomoo`05Ool00`00Oomoo`0JOol00`00Oomoo`0TOol0025oo`04001oogoo001hOol0
0`00Oomoo`07Ool00`00Oomoo`0=Ool200000goo001oo`02Ool3000:Ool00`00Oomoo`06Ool00`00
Oomoo`05Ool00`00Oomoo`0KOol00`00Oomoo`0SOol0021oo`03001oogoo009oo`03001oogoo07=o
o`8000Yoo`03001oogoo00]oo`8000=oo`@000ioo`03001oogoo00Ioo`03001oogoo00Aoo`8001eo
o`03001oogoo029oo`0087oo0P0017oo00<007ooOol0L7oo0P003Goo00<007ooOol027oo0P000goo
100047oo00<007ooOol01goo00<007oo00000`000Woo00<007ooOol06Goo0P0097oo000POol00`00
Oomoo`04Ool2001^Ool2000@Ool00`00Oomoo`05Ool20002Ool30004Ool00`00Oomoo`0>Ool00`00
Oomoo`04Ool40006Ool00`00Oomoo`0KOol00`00Oomoo`0QOol0021oo`8000Moo`03001oogoo06Yo
o`03001oogoo015oo`03001oogoo009oo`80009oo`8000Moo`03001oogoo00ioo`03001oogoo00@0
00=oo`03001oogoo00Eoo`03001oogoo01]oo`03001oogoo021oo`007goo0P002Goo00<007ooOol0
Igoo0P0000=oo`00Ool04Goo00<007ooOol00P0000Aoo`00000000Yoo`03001oogoo00eoo`<000Qo
o`03001oogoo00Aoo`03001oogoo01]oo`03001oogoo021oo`0037oo0P0017oo0P0017oo0P001Woo
00<007ooOol027oo00<007ooOol0I7oo0P0017oo00<007ooOol047oo1@003Goo00<007ooOol02Woo
0`000Woo00<007ooOol01goo00<007ooOol017oo00<007ooOol06goo00<007ooOol07goo000;Ool0
1000Oomoo`000Woo00@007ooOol0009oo`04001oogoo0004Ool2000;Ool00`00Oomoo`1QOol20007
Ool00`00Oomoo`0=Ool4000AOol00`00Oomoo`07Ool20005Ool00`00Oomoo`08Ool00`00Oomoo`03
Ool00`00Oomoo`0LOol00`00Oomoo`0NOol000eoo`05001oogooOol00002Ool01000Oomoo`000Woo
00<007ooOol00Woo00<007ooOol02goo00<007ooOol0Ggoo00<007ooOol027oo00<007ooOol02Woo
0`000goo00<007ooOol03goo00<007ooOol00goo0`0027oo00<007ooOol01goo00<007ooOol017oo
00<007ooOol06goo00<007ooOol07Woo0005Ool40005Ool01000Oomoo`000Woo00@007ooOol0009o
o`03001oogoo009oo`03001oogoo00aoo`03001oogoo05aoo`8000]oo`03001oogoo00Moo`<000Mo
o`03001oogoo00ioo`03001oogoo00<000aoo`03001oogoo00Moo`03001oogoo00=oo`03001oogoo
01aoo`03001oogoo01eoo`002goo00@007ooOol0009oo`04001oogoo0002Ool01000Oomoo`000goo
0P003goo00<007ooOol0FGoo0P003Woo00<007ooOol017oo0`002Woo00<007ooOol03Goo0`003goo
00<007ooOol027oo00<007ooOol00goo00<007ooOol06goo0P007Woo000<Ool20004Ool20004Ool2
0005Ool00`00Oomoo`0>Ool00`00Oomoo`1FOol2000AOol00`00Oomoo`04000<Ool00`00Oomoo`0;
Ool20002Ool00`00Oomoo`0>Ool00`00Oomoo`07Ool00`00Oomoo`03Ool00`00Oomoo`0LOol00`00
Oomoo`0LOol001moo`03001oogoo00moo`03001oogoo05Aoo`03001oogoo015oo`D000moo`03001o
ogoo00Moo`<000Eoo`03001oogoo00ioo`03001oogoo00Moo`04001oogooOol3000NOol00`00Oomo
o`0KOol001ioo`80019oo`03001oogoo055oo`800003Ool007oo00ioo`<00003Ool00000019oo`03
001oogoo00=oo`<000Qoo`03001oogoo00moo`03001oogoo00Ioo`@000=oo`03001oogoo01aoo`03
001oogoo01Yoo`007Woo00<007ooOol04Woo00<007ooOol0CWoo0P0017oo00<007ooOol02Goo0`00
0Woo0P000Woo00<007ooOol047oo00<007ooOol00P0037oo00<007ooOol03Woo00@007ooOomoo`D0
009oo`03001oogoo009oo`03001oogoo01aoo`03001oogoo01Yoo`007Woo00<007ooOol04goo00<0
07ooOol0Bgoo0P001goo0P001Woo0`000goo0P001Goo00<007ooOol03goo0`003goo00<007ooOol0
3Goo10001goo00<007ooOol00goo00<007ooOol077oo00<007ooOol06Goo000MOol2000FOol20019
Ool2000:Ool20002Ool30003Ool30008Ool00`00Oomoo`0<Ool20002Ool00`00Oomoo`0>Ool00`00
Oomoo`08Ool40002Ool00`00Oomoo`07Ool00`00Oomoo`03Ool00`00Oomoo`0KOol00`00Oomoo`0I
Ool001ioo`03001oogoo01Ioo`03001oogoo04Aoo`8000ioo`8000Aoo`8000]oo`03001oogoo00Yo
o`8000Eoo`03001oogoo00eoo`03001oogoo00Eoo`<000Ioo`03001oogoo00Qoo`03001oogoo00=o
o`03001oogoo01]oo`03001oogoo01Qoo`007Woo0P0067oo00<007ooOol0@Goo0P003Goo0`0000Ao
o`00Oomoo`8000ioo`03001oogoo00Ioo`<000Moo`03001oogoo00ioo`03001oogoo00@000Yoo`03
001oogoo00Qoo`03001oogoo009oo`03001oogoo01aoo`03001oogoo01Moo`007Goo00<007ooOol0
6Goo00<007ooOol0?Woo0P0037oo0`001Goo0P004Goo00<007ooOol00goo0P002goo00<007ooOol0
3Goo0`003goo00<007ooOol01goo00<007ooOol00goo00<007ooOol06goo0P0067oo0009Ool40003
Ool20004Ool20005Ool00`00Oomoo`0JOol00`00Oomoo`0kOol20002Ool00`00Oomoo`06Ool30006
Ool200000goo001oo`0AOol00`00Oomoo`02000>Ool00`00Oomoo`0:Ool20002Ool00`00Oomoo`0=
Ool00`00Oomoo`08Ool00`00Oomoo`03Ool00`00Oomoo`0KOol00`00Oomoo`0FOol000Uoo`03001o
ogoo00=oo`04001oogoo0002Ool01000Oomoo`0017oo0P0077oo00<007ooOol0>7oo0P001Goo00<0
07ooOol00Woo0`001goo0P0017oo00<007ooOol03goo0`004Goo00<007ooOol01Woo0`001Goo00<0
07ooOol03Goo00<007ooOol027oo00<007ooOol00Woo00<007ooOol077oo00<007ooOol05Goo000:
Ool00`00Oomoo`02Ool01000Oomoo`000Woo00@007ooOol000Aoo`03001oogoo01aoo`03001oogoo
03Eoo`8000Qoo`@000Qoo`8000Moo`03001oogoo00aoo`80009oo`03001oogoo00moo`03001oogoo
00=oo`<000Qoo`03001oogoo00ioo`03001oogoo00Moo`04001oogooOol3000MOol00`00Oomoo`05
Ool4000<Ool000=oo`@000Aoo`05001oogooOol00002Ool01000Oomoo`000Woo00<007ooOol00Woo
00<007ooOol07Goo00<007ooOol0<Woo0P002Goo10001goo0P002Woo00<007ooOol02Goo0P001Goo
00<007ooOol03goo00<007ooOol00P0037oo00<007ooOol03Woo00<007ooOol01Woo10000goo00<0
07ooOol06goo00<007ooOol017oo00<007ooOol03Goo0009Ool01000Oomoo`000Woo00@007ooOol0
009oo`04001oogoo0003Ool2000POol00`00Oomoo`0_Ool20008Ool30004Ool00`00Oomoo`02Ool2
000=Ool00`00Oomoo`06Ool20008Ool00`00Oomoo`0>Ool3000?Ool00`00Oomoo`0=Ool01000Oomo
ogoo1@000Woo00<007ooOol00Woo00<007ooOol077oo00<007ooOol017oo00<007ooOol037oo000:
Ool20004Ool20004Ool20004Ool00`00Oomoo`0POol00`00Oomoo`0/Ool20007Ool30008Ool01000
Ool0000047oo00<007ooOol00goo0P002Woo00<007ooOol02goo0`000Woo00<007ooOol03Woo00<0
07ooOol02goo1@0027oo00<007ooOol00Woo00<007ooOol06goo00<007ooOol01Goo00<007ooOol0
2goo000LOol?000EOol00`00Oomoo`0YOol20006Ool3000;Ool2000COol00`00Oomoo`02000=Ool0
0`00Oomoo`08Ool20005Ool00`00Oomoo`0>Ool00`00Oomoo`07Ool40003Ool00`00Oomoo`07Ool0
0`00Oomoo`03Ool00`00Oomoo`0KOol00`00Oomoo`02Ool01000Oomoo`0037oo000LOol30008Ool0
1000Oomoogoo1`003goo00<007ooOol09Woo0P000Woo0P0000Aoo`00000000aoo`80009oo`03001o
ogoo011oo`<0011oo`03001oogoo00Aoo`<000Qoo`03001oogoo00ioo`04001oogooOol50007Ool0
0`00Oomoo`08Ool00`00Oomoo`02Ool00`00Oomoo`0JOol30004Ool2000=Ool001aoo`03001oogoo
008000Moo`8000Ioo`03001oo`0000<000aoo`03001oogoo029oo`<000Aoo`<000eoo`8000Eoo`03
001oogoo00eoo`80009oo`03001oogoo00moo`03001oogoo00<000aoo`03001oogoo00eoo`@000eo
o`03001oogoo00Qoo`03001oogoo009oo`03001oogoo01]oo`03001oogoo015oo`0077oo00<007oo
Ool00Woo00<007ooOol01Woo00<007ooOol017oo00<007ooOol00Woo0`002Woo0P007goo0`0017oo
0`000goo0P002Goo0P0027oo0P002goo0P001Goo00<007ooOol03Woo0`0047oo00<007ooOol027oo
10000Woo00<007ooOol03Goo00<007ooOol027oo00<007ooOol00Woo00<007ooOol06goo00<007oo
Ool047oo000KOol20005Ool20007Ool00`00Oomoo`04Ool00`00Oomoo`03Ool40009Ool00`00Oomo
o`0JOol20005Ool20008Ool00`00Oomoo`04Ool2000;Ool20008Ool20008Ool00`00Oomoo`0;Ool2
0002Ool00`00Oomoo`0>Ool00`00Oomoo`05Ool30006Ool00`00Oomoo`0=Ool00`00Oomoo`08Ool0
0`00Oomoo`02Ool00`00Oomoo`0LOol00`00Oomoo`0?Ool001aoo`03001oogoo00Eoo`8000Ioo`80
00Eoo`8000Aoo`03001oogoo00<000Moo`03001oogoo01Ioo`<000Aoo`<000]oo`80009oo`8000mo
o`03001oogoo00=oo`8000]oo`03001oogoo00Moo`<000Eoo`03001oogoo00ioo`03001oogoo00@0
00Yoo`03001oogoo00eoo`03001oogoo00Qoo`03001oogoo00<001eoo`03001oogoo00moo`006goo
0P002Goo1@000goo00<007ooOol017oo00<007ooOol00Woo00<007ooOol00Woo0P001Woo00<007oo
Ool04goo0P000Woo1@003goo0`004Woo00<007ooOol00P003Woo00<007ooOol00goo0`002Goo00<0
07ooOol03Goo0`003goo00<007ooOol03Goo00<007ooOol01Woo10000goo00<007ooOol06goo00<0
07ooOol03Woo000KOol00`00Oomoo`09Ool20002Ool70004Ool00`00Oomoo`02Ool20004Ool20005
Ool00`00Oomoo`0?Ool30003Ool5000=Ool30002Ool2000AOol3000AOol00`00Oomoo`02000<Ool0
0`00Oomoo`0;Ool20002Ool00`00Oomoo`0=Ool00`00Oomoo`0=Ool01000Oomoogoo1@000goo00D0
07ooOomoo`0001ioo`03001oogoo00eoo`001goo1@000Woo0P0017oo0P001Goo00<007ooOol02goo
0P001Woo1@0000=oo`00Ool017oo00<007ooOol00Woo0P001Goo00<007ooOol037oo0P000goo0`00
1Goo0P001goo10001goo0P0037oo0`000Woo00<007ooOol03goo0`003goo00<007ooOol01goo0`00
1Goo00<007ooOol03Goo00<007ooOol02goo1@0027oo00<007ooOol00Woo00<007ooOol06goo00<0
07ooOol03Goo0009Ool01@00Oomoogoo00000Woo00@007ooOol0009oo`03001oogoo009oo`03001o
ogoo00eoo`8000Ioo`800003Ool00000008000Aoo`03001oogoo009oo`8000Eoo`03001oogoo00Uo
o`80009oo`<000Yoo`80009oo`<000aoo`8000Moo`@000Ioo`03001oogoo00aoo`80009oo`03001o
ogoo00ioo`03001oogoo00=oo`<000Qoo`03001oogoo00ioo`03001oogoo00Ioo`@000=oo`03001o
ogoo00Qoo`03001oogoo009oo`03001oogoo01Yoo`8000ioo`002Goo00D007ooOomoo`00009oo`04
001oogoo0002Ool01000Oomoogoo0P004Goo0P001Woo0P000goo0`000Woo00<007ooOol00Woo0`00
17oo00<007ooOol00goo1@000Woo0P003Goo10004Goo0P000Woo0`002goo0P002Goo0`001Goo00<0
07ooOol03Woo00<007ooOol00P0037oo00<007ooOol03Goo00@007ooOomoo`D000Moo`03001oogoo
00Uoo`05001oogooOol0000MOol00`00Oomoo`0<Ool00003Ool00000008000Aoo`05001oogooOol0
0002Ool01000Oomoo`000Woo00<007ooOol00Woo00<007ooOol04Goo1`0000=oo`0000000goo1@00
17oo00<007oo00000goo1P0017oo10002Woo10000goo0P004Goo0`0047oo00<007ooOol00goo0`00
2Goo00<007ooOol03Goo0`003goo00<007ooOol037oo10003Goo00<007ooOol027oo00<007ooOol0
0Woo00<007ooOol06goo00<007ooOol02goo0009Ool01@00Oomoogoo00000Woo00@007ooOol0009o
o`05001oogooOol0000HOol:00000goo0000000;0007Ool30003Ool30004Ool40009Ool2000<Ool3
0002Ool00`00Oomoo`0?Ool00`00Oomoo`02000<Ool00`00Oomoo`0;Ool20002Ool00`00Oomoo`0>
Ool00`00Oomoo`08Ool30002Ool00`00Oomoo`0=Ool00`00Oomoo`08Ool00`00Oomoo`02Ool00`00
Oomoo`0KOol00`00Oomoo`0:Ool000Qoo`8000Aoo`8000Aoo`8000Aoo`8001eoo`P000Qoo`<000=o
o`<000Qoo`D000ioo`8000Moo`@000Ioo`03001oogoo00ioo`<000moo`03001oogoo00Moo`<000Eo
o`03001oogoo00eoo`03001oogoo00Aoo`@000Eoo`03001oogoo00ioo`03001oogoo00Qoo`05001o
ogooOol0000MOol00`00Oomoo`0:Ool001Yoo`03001oogoo025oo`D000Uoo`<000Eoo`H000=oo`80
011oo`80009oo`<000]oo`8000]oo`<0009oo`03001oogoo00ioo`03001oogoo00=oo`<000Uoo`03
001oogoo00eoo`03001oogoo00<000Yoo`03001oogoo00eoo`03001oogoo00Uoo`D001eoo`03001o
ogoo00Uoo`006Woo00<007ooOol09Woo4@002goo0P003goo0`0047oo00<007ooOol017oo10001Woo
00<007ooOol03Woo00<007ooOol00P0037oo00<007ooOol03Goo0`003Woo00<007ooOol03Goo00<0
07ooOol00Woo1`0017oo00<007ooOol06goo00<007ooOol027oo000IOol2000[Ool40008Ool2000=
Ool20009Ool40002Ool2000@Ool00`00Oomoo`03000;Ool00`00Oomoo`0=Ool3000?Ool00`00Oomo
o`09Ool30002Ool00`00Oomoo`0=Ool00`00Oomoo`0;Ool60007Ool01@00Oomoogoo00007Goo00<0
07ooOol027oo000JOol00`00Oomoo`0WOol20003Ool30008Ool2000=Ool20003Ool40007Ool2000?
Ool3000?Ool00`00Oomoo`09Ool30002Ool00`00Oomoo`0>Ool00`00Oomoo`04Ool40006Ool00`00
Oomoo`0<Ool00`00Oomoo`07Ool40003Ool00`00Oomoo`08Ool01@00Oomoogoo000077oo0P002Goo
000IOol00`00Oomoo`0UOol30008Ool80002Ool20008Ool8000=Ool00`00Oomoo`09Ool30002Ool0
0`00Oomoo`0>Ool00`00Oomoo`04Ool40006Ool00`00Oomoo`0>Ool00`00Oomoo`03000;Ool00`00
Oomoo`0<Ool01000Oomoogoo1@001goo00<007ooOol02Goo00D007ooOomoo`0001Moo`@00003Ool0
07oo00Qoo`006Goo0P008goo0`003Goo100000=oo`0000002`001Woo0`003Goo0P001Woo0`001Woo
00<007ooOol03Woo00<007ooOol00`002goo00<007ooOol03Goo0`003Woo00<007ooOol02goo1@00
3Goo00<007ooOol027oo00D007ooOomoo`0001Aoo`<000Eoo`03001oogoo00Moo`006Goo00<007oo
Ool07goo0`004Woo1@001Goo0`002goo0`0037oo0P000Woo0`002Woo0P003Woo0`003goo00<007oo
Ool02Goo0`000Woo00<007ooOol03Goo00<007ooOol01Woo10000goo00<007ooOol037oo00<007oo
Ool02Goo00D007ooOomoo`00011oo`<000Qoo`03001oogoo00Moo`006Goo0P007Goo0`005goo1`00
0goo0`002goo0`001goo1`003Woo0P002Woo0`000Woo00<007ooOol03Woo00<007ooOol017oo1000
1Woo00<007ooOol03Goo00@007ooOomoo`@000Qoo`03001oogoo00aoo`03001oogoo00Uoo`04001o
ogoo000<Ool4000<Ool00`00Oomoo`06Ool001Qoo`8001]oo`<001aoo`d000Moo``000Ioo`8000mo
o`03001oogoo00Aoo`<000Ioo`03001oogoo00ioo`03001oogoo00<000]oo`03001oogoo00aoo`@0
00eoo`03001oogoo00aoo`03001oogoo00Qoo`D000Qoo`<0011oo`03001oogoo00Ioo`004Goo0P00
1Woo00<007ooOol05goo0P008Goo1P000goo2@001Woo00<007ooOol02goo00<007ooOol03Goo00<0
07ooOol00`002Woo00<007ooOol03Goo0`003Woo00<007ooOol02Goo0`000Woo00<007ooOol037oo
00<007ooOol037oo00<007ooOol00Woo20000goo00<007ooOol00Woo0`004goo00<007ooOol01Woo
000@Ool01000Oomoo`0017oo00<007ooOol05Goo0`0097oo0P0000=oo`0000000P0017oo0`002Woo
00<007ooOol02goo00<007ooOol02Woo1@003Woo00<007ooOol02Goo0`000Woo00<007ooOol03Goo
00<007ooOol017oo10001Goo00<007ooOol03Goo00<007ooOol02Goo1`0027oo00@007ooOomoo`@0
01Ioo`03001oogoo00Ioo`0047oo00@007ooOol000Aoo`8001=oo`<002Qoo`<0009oo`@000=oo`<0
00Uoo`8000aoo`8000Aoo`H000Aoo`03001oogoo00eoo`03001oogoo00Aoo`@000Ioo`03001oogoo
00eoo`03001oogoo00<000Yoo`03001oogoo00eoo`03001oogoo009oo`H000Aoo`03001oogoo00Uo
o`05001oogooOol0000HOol00`00Oomoo`06Ool0011oo`04001oogoo0004Ool00`00Oomoo`0?Ool3
000/Ool40002Ool:0009Ool00`00Oomoo`09Ool6000;Ool00`00Oomoo`0=Ool00`00Oomoo`03000;
Ool00`00Oomoo`0<Ool3000>Ool00`00Oomoo`0<Ool5000;Ool00`00Oomoo`08Ool01@00Oomoogoo
000067oo00<007ooOol01Woo000@Ool01000Oomoo`0017oo00<007ooOol037oo0`00<7oo1@000Woo
00<007ooOol00goo200017oo00<007ooOol00Woo1P000goo0P003Goo0P003Goo0`003Woo00<007oo
Ool02Goo0`000Woo00<007ooOol03Goo00<007ooOol027oo0`000Woo00<007ooOol037oo00<007oo
Ool027oo00D007ooOomoo`0001Qoo`03001oogoo00Eoo`004Goo0P0017oo0P002goo0`00=7oo2000
1goo00<007ooOol00Woo2@002Woo0P003Woo00<007ooOol01Woo10000Woo00<007ooOol03Goo00<0
07ooOol017oo10001Goo00<007ooOol03Goo00<007ooOol017oo10001Goo00<007ooOol037oo00<0
07ooOol02Goo00D007ooOomoo`0001Moo`03001oogoo00Eoo`0067oo00<007ooOol01Woo0`00>7oo
10000Woo0`001Woo00<007ooOol01goo0P003Goo00<007ooOol037oo00@007ooOomoo`@000Moo`03
001oogoo00eoo`03001oogoo00<000Yoo`03001oogoo00eoo`03001oogoo00<000Yoo`03001oogoo
00aoo`03001oogoo00Uoo`D001Ioo`03001oogoo00Eoo`005goo00<007ooOol01Goo0P00?7oo0P00
00=oo`0000000Woo0`001Goo0P002Goo0P0037oo0P002goo1@0037oo00<007ooOol037oo0`003Woo
00<007ooOol037oo0`003Woo00<007ooOol037oo00<007ooOol00Woo1`0017oo00<007ooOol04goo
00<007ooOol01Goo000GOol20003Ool3000oOol20002Ool20002Ool50003Ool00`00Oomoo`07Ool2
000<Ool20004Ool60003Ool00`00Oomoo`0<Ool00`00Oomoo`08Ool30002Ool00`00Oomoo`0=Ool0
0`00Oomoo`08Ool30002Ool00`00Oomoo`0=Ool00`00Oomoo`0:Ool60007Ool01@00Oomoogoo0000
5Goo00<007ooOol01Goo000GOol00`00Ool000020013Ool300001Woo00000000Ool000=oo`D000Yo
o`8000Moo`T000Yoo`03001oogoo00aoo`03001oogoo00Aoo`<000Ioo`03001oogoo00aoo`03001o
ogoo00Aoo`@000Ioo`03001oogoo00aoo`03001oogoo00Ioo`@000=oo`03001oogoo00Qoo`05001o
ogooOol0000DOol00`00Oomoo`05Ool001Moo`8004Moo`X000Eooa@000Ioo`03001oogoo00]oo`03
001oogoo00aoo`03001oogoo00<000Yoo`03001oogoo00aoo`03001oogoo00<000]oo`03001oogoo
00aoo`04001oogooOol40007Ool00`00Oomoo`09Ool01@00Oomoogoo00004goo00<007ooOol01Goo
000HOol00`00Oomoo`16Ool200000goo000000050006Ool2000:Ool00`00Oomoo`0:Ool00`00Oomo
o`0;Ool00`00Oomoo`0;Ool3000>Ool00`00Oomoo`0;Ool3000>Ool00`00Oomoo`0;Ool5000<Ool0
0`00Oomoo`08Ool00`00Oomoo`02Ool00`00Oomoo`0AOol00`00Oomoo`04Ool001Uoo`8004Moo`80
0005Ool00000001oo`030007Ool00`00Oomoo`08Ool00`00Oomoo`0:Ool00`00Oomoo`0;Ool00`00
Oomoo`06Ool40002Ool00`00Oomoo`0=Ool00`00Oomoo`06Ool40002Ool00`00Oomoo`0=Ool00`00
Oomoo`06Ool40003Ool00`00Oomoo`0<Ool00`00Oomoo`08Ool01@00Oomoogoo00004goo00<007oo
Ool017oo000KOol00`00Oomoo`15Ool20002Ool20002Ool40005Ool00`00Oomoo`08Ool2000;Ool0
0`00Oomoo`0;Ool01000Oomoogoo10001goo00<007ooOol03Goo00@007ooOomoo`@000Moo`03001o
ogoo00eoo`04001oogooOol40008Ool00`00Oomoo`0<Ool00`00Oomoo`08Ool5000BOol00`00Oomo
o`04Ool001aoo`03001oogoo04Eoo`8000=oo`80009oo`03001oo`0000L000Yoo`03001oogoo00Uo
o`03001oogoo00Uoo`D000aoo`03001oogoo00aoo`@000aoo`03001oogoo00aoo`@000aoo`03001o
ogoo00aoo`03001oogoo009oo`L000Aoo`03001oogoo00moo`03001oogoo00Aoo`007Goo00<007oo
Ool0A7oo0`000goo0`0000=oo`00Ool01Goo2@0017oo00<007ooOol02Goo00<007ooOol00Woo1P00
0goo00<007ooOol02goo00<007ooOol02Goo0`000Woo00<007ooOol02goo00<007ooOol02Goo0`00
0Woo00<007ooOol037oo00<007ooOol02Goo1`001goo00<007ooOol00Woo00<007ooOol03Woo00<0
07ooOol017oo000NOol20015Ool40002Ool60007Ool00`00Oomoo`02OolF000:Ool00`00Oomoo`0;
Ool00`00Oomoo`04Ool40006Ool00`00Oomoo`0;Ool00`00Oomoo`04Ool40006Ool00`00Oomoo`0<
Ool00`00Oomoo`02Ool60004Ool00`00Oomoo`08Ool00`00Oomoo`02Ool00`00Oomoo`0=Ool00`00
Oomoo`04Ool0021oo`03001oogoo04=oo`D00003Ool007oo00@000Moo`03001oogoo00Qoo`03001o
ogoo00Uoo`03001oogoo00]oo`03001oogoo00]oo`03001oogoo00<000]oo`03001oogoo00]oo`03
001oogoo00<000]oo`03001oogoo00]oo`D000]oo`03001oogoo00Qoo`05001oogooOol0000@Ool0
0`00Oomoo`03Ool0025oo`03001oogoo049oo`08001oo`00001oo`00Ool0009oo`<000Moo`03001o
ogoo00Qoo`03001oogoo00Uoo`03001oogoo00]oo`03001oogoo00Uoo`@000ioo`03001oogoo00]o
o`<000ioo`03001oogoo00Qoo`<0009oo`03001oogoo00]oo`03001oogoo00Qoo`03001oogoo009o
o`03001oogoo00eoo`03001oogoo00=oo`008Woo00<007ooOol0@Woo00<007oo00000Woo0`000goo
0`001Woo00<007ooOol027oo00<007ooOol02Goo00<007ooOol02goo00<007ooOol00Woo1P000goo
00<007ooOol03Goo00<007ooOol01Woo10000Woo00<007ooOol03Goo00<007ooOol017oo0`001Woo
00<007ooOol02goo00<007ooOol027oo00<007oo00000`003Woo00<007ooOol00goo000SOol20013
Ool00`00Ool00003Ool20004Ool50003Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`09
Ool6000:Ool00`00Oomoo`0=Ool01000Oomoogoo10001goo00<007ooOol03Goo00<007ooOol00`00
2Woo00<007ooOol02goo00<007ooOol00Woo1`0017oo00<007ooOol02goo00<007ooOol00goo000U
Ool00`00Oomoo`10Ool01000Oomoo`0017oo0P000goo00@007ooOomoo`D000Yoo`03001oogoo00Uo
o`03001oogoo009oo`H000=oo`03001oogoo00]oo`03001oogoo00aoo`@000aoo`03001oogoo00]o
o`@000eoo`03001oogoo00]oo`D000Moo`03001oogoo009oo`03001oogoo00Yoo`03001oogoo00=o
o`009Woo00<007ooOol0@7oo00@007ooOol000Aoo`<0009oo`03001oogoo00Aoo`P000Aoo`03001o
ogoo00Aoo`T000Yoo`03001oogoo00]oo`03001oogoo00Qoo`<0009oo`03001oogoo00aoo`03001o
ogoo00Ioo`@000=oo`03001oogoo00aoo`03001oogoo00Ioo`@0009oo`03001oogoo00Moo`03001o
ogoo009oo`03001oogoo00Yoo`03001oogoo00=oo`009goo00<007ooOol0@7oo00<007ooOol00P00
0Woo00<007oo00000P0000=oo`00Ool01Woo00<007ooOol00Woo30001Woo00<007ooOol02goo00<0
07ooOol02goo00<007ooOol017oo0`001Woo00<007ooOol037oo00@007ooOomoo`@000Qoo`03001o
ogoo00aoo`04001oogooOol40007Ool00`00Oomoo`07Ool00`00Oomoo`02Ool00`00Oomoo`09Ool0
0`00Oomoo`03Ool002Qoo`80041oo`<00003Ool00000009oo`03001oogoo00@000Moo`03001oogoo
00Qoo`03001oogoo00Uoo`03001oogoo00]oo`03001oogoo00]oo`03001oogoo00<000Yoo`03001o
ogoo00]oo`@000aoo`03001oogoo00]oo`D000aoo`03001oogoo00Moo`03001oo`0000<000]oo`03
001oogoo009oo`00:Woo00<007ooOol0?Woo00@007oo001oo`<00003Ool007oo009oo`<000Moo`03
001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00]oo`03001oogoo00Uoo`@000eoo`03001oogoo
00Qoo`<0009oo`03001oogoo00aoo`03001oogoo00Ioo`@000=oo`03001oogoo00aoo`03001oogoo
009oo`H000Aoo`03001oogoo00Qoo`03001oogoo009oo`00:goo00<007ooOol0?Woo00D007oo0000
07oo00800003Ool007oo00=oo`<000Ioo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00]o
o`03001oogoo009oo`H000=oo`03001oogoo00aoo`03001oogoo00Aoo`<000Ioo`03001oogoo00ao
o`04001oogooOol40008Ool00`00Oomoo`0;Ool50005Ool00`00Oomoo`03Ool00`00Oomoo`07Ool0
0`00Oomoo`02Ool002aoo`03001oogoo03eoo`03001oogoo0080009oo`8000Ioo`D000=oo`03001o
ogoo00Moo`03001oogoo00Yoo`03001oogoo00Uoo`H000Yoo`03001oogoo00aoo`03001oogoo00<0
00Yoo`03001oogoo00]oo`@000aoo`03001oogoo00Qoo`<0009oo`03001oogoo00Ioo`03001oogoo
00=oo`03001oogoo00Ioo`03001oogoo009oo`00;Goo0P00?Woo00<007ooOol00P000Woo0P001Woo
00@007ooOomoo`D000Uoo`03001oogoo00Yoo`03001oogoo009oo`H000=oo`03001oogoo00]oo`03
001oogoo00]oo`<000eoo`03001oogoo00Qoo`<0009oo`03001oogoo00aoo`03001oogoo00Aoo`<0
00Ioo`03001oogoo00Ioo`03001oogoo00=oo`03001oogoo00Eoo`03001oogoo009oo`00;goo00<0
07ooOol0?7oo00D007ooOomoo`0000=oo`8000Eoo`03001oogoo00Aoo`L000Aoo`03001oogoo00Eo
o`T000Yoo`03001oogoo00Yoo`03001oogoo00Qoo`<0009oo`03001oogoo00aoo`03001oogoo00Ao
o`<000Eoo`03001oogoo00eoo`03001oogoo00<000Uoo`03001oogoo00Ioo`03001oogoo009oo`<0
00Ioo`03001oogoo009oo`00<7oo00<007ooOol0>goo00<007ooOol00Woo00@007ooOomoo`@000=o
o`03001oogoo00Eoo`03001oogoo009oo``000Ioo`03001oogoo00]oo`03001oogoo00Yoo`03001o
ogoo00Aoo`<000Ioo`03001oogoo00aoo`03001oogoo00<000Uoo`03001oogoo00aoo`<000eoo`03
001oogoo00Ioo`@000=oo`03001oogoo00=oo`03001oogoo009oo`00<Goo00<007ooOol0>goo00<0
07ooOol00Woo0P000Woo00<007ooOol00P000Woo00<007ooOol01Goo00<007ooOol01goo00<007oo
Ool02Woo00<007ooOol02Woo00<007ooOol02goo00<007ooOol00`002Woo00<007ooOol02goo0`00
3Goo00<007ooOol027oo0`000Woo00<007ooOol037oo00@007ooOomoo`@00003Ool007oo00Eoo`03
001oogoo00=oo`03001oogoo005oo`00<Woo0P00?7oo00L007ooOomoo`00Ool00002Ool01000Oomo
ogoo10001goo00<007ooOol01goo00<007ooOol02Woo00<007ooOol02Woo00<007ooOol02Goo1000
3Goo00<007ooOol027oo0`000Woo00<007ooOol037oo00<007ooOol00goo10001Woo00<007ooOol0
2goo10001Woo00<007ooOol017oo00<007ooOol00Woo00<007ooOol00Goo000dOol00`00Oomoo`0i
Ool20003Ool00`00Ool000020006Ool30006Ool00`00Oomoo`07Ool00`00Oomoo`09Ool00`00Oomo
o`0;Ool00`00Oomoo`02Ool60003Ool00`00Oomoo`0<Ool00`00Oomoo`04Ool30005Ool00`00Oomo
o`0<Ool00`00Oomoo`03000;Ool00`00Oomoo`06Ool40002Ool00`00Oomoo`05Ool00`00Oomoo`04
Ool20003Ool00`00Oomoo`01Ool003Eoo`03001oogoo03Uoo`80009oo`03001oogoo00<000Moo`D0
00=oo`03001oogoo00Moo`03001oogoo00Uoo`03001oogoo00Uoo`H000Yoo`03001oogoo00aoo`03
001oogoo00<000Uoo`03001oogoo00]oo`<000moo`04001oogooOol40007Ool20006Ool00`00Ool0
00030002Ool01000Oomoo`000goo000fOol00`00Oomoo`0hOol01P00Ool00000Ool000=oo`8000Io
o`03001oogoo009oo`D000Uoo`03001oogoo00Uoo`03001oogoo009oo`H000=oo`03001oogoo00]o
o`03001oogoo00]oo`<000eoo`03001oogoo00Moo`<0009oo`03001oogoo00eoo`@000eoo`04001o
ogooOol40007Ool00`00Ool00003Ool003Moo`8003Uoo`06001oo`00001oo`000goo0P001Woo00<0
07ooOol01Goo1`0017oo00<007ooOol017oo2@002Goo00<007ooOol02goo00<007ooOol027oo0`00
0Woo00<007ooOol037oo00<007ooOol00goo0`001Woo0P002Woo0`000Woo00<007ooOol037oo1000
17oo00<007ooOol01Goo0P000goo000iOol00`00Oomoo`0gOol00`00Oomoo`020005Ool20005Ool0
0`00Oomoo`06Ool00`00Oomoo`02Ool;0006Ool00`00Oomoo`0:Ool00`00Oomoo`0;Ool00`00Oomo
o`04Ool30005Ool00`00Oomoo`0<Ool00`00Oomoo`03000:Ool20006Ool30006Ool00`00Oomoo`08
Ool30002Ool00`00Oomoo`03Ool00`00Oomoo`05Ool30002Ool003Yoo`03001oogoo03Ioo`04001o
ogooOol20005Ool30004Ool00`00Oomoo`05Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomo
o`0;Ool00`00Oomoo`0;Ool00`00Oomoo`030009Ool00`00Oomoo`0;Ool3000?Ool00`00Oomoo`03
000:Ool00`00Oomoo`04Ool30006Ool00`00Oomoo`03Ool00`00Oomoo`040002Ool10001Ool1Ool0
03]oo`03001oogoo03Ioo`05001oogooOol00005Ool00`00Oomoo`0300000goo001oo`07Ool00`00
Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`0;Ool00`00Oomoo`0:Ool3000=Ool00`00Oomoo`07
Ool30002Ool00`00Oomoo`0=Ool3000>Ool00`00Oomoo`03000:Ool01000Oomoogoo10001Woo0@00
0Goo0Goo000lOol2000fOol00`00Oomoo`02Ool00`00Oomoo`03Ool00`00Oomoo`02Ool30008Ool0
0`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`0:Ool00`00Oomoo`06Ool40002Ool00`00Oomo
o`0<Ool00`00Oomoo`03Ool30006Ool00`00Oomoo`0:Ool20002Ool00`00Oomoo`0<Ool3000=Ool4
0002Ool00`00Oomoo`05Ool10001Ool1Ool003ioo`03001oogoo03Aoo`03001oogoo009oo`8000Ao
o`03001oogoo00=oo`<000Moo`03001oogoo00Moo`03001oogoo00Uoo`03001oogoo00Yoo`04001o
ogooOol40006Ool00`00Oomoo`0<Ool00`00Oomoo`03000:Ool00`00Oomoo`07Ool20005Ool00`00
Oomoo`09Ool20002Ool00`00Oomoo`09Ool20007Ool00`00Oomoo`04Ool10001Ool1Ool003moo`03
001oogoo03=oo`8000=oo`04001oo`000002Ool00`00Oomoo`05Ool50003Ool00`00Oomoo`08Ool0
0`00Oomoo`08Ool00`00Oomoo`09Ool5000;Ool00`00Oomoo`0;Ool3000>Ool00`00Oomoo`03Ool3
0007Ool00`00Oomoo`07Ool20005Ool00`00Oomoo`06Ool2000:Ool00`00Oomoo`03Ool10001Ool1
Ool0041oo`03001oogoo03=oo`8000=oo`04001oogoo0002Ool00`00Oomoo`04Ool00`00Oomoo`02
Ool5000:Ool00`00Oomoo`08Ool00`00Oomoo`02Ool60003Ool00`00Oomoo`0;Ool00`00Oomoo`07
Ool30002Ool00`00Oomoo`0=Ool00`00Oomoo`02000;Ool00`00Oomoo`03Ool30007Ool00`00Oomo
o`03Ool3000=Ool00`00Oomoo`02Ool10001Ool1Ool0045oo`8003=oo`03001oo`00009oo`04001o
ogooOol200000goo001oo`05Ool00`00Oomoo`05OolK000:Ool00`00Oomoo`0;Ool00`00Oomoo`03
Ool30006Ool00`00Oomoo`0<Ool3000>Ool00`00Oomoo`02000;Ool00`00Oomoo`02000@Ool00`00
Oomoo`03Ool10001Ool004=oo`03001oogoo035oo`06001oo`00001oo`0017oo0P001goo00<007oo
Ool01Woo00<007ooOol027oo00<007ooOol027oo00<007ooOol02goo00<007ooOol02Woo00<007oo
Ool00`002Woo00<007ooOol02Goo0P000Woo00<007ooOol037oo0`003Woo0P004goo00<007ooOol0
0Woo0@000Goo0014Ool00`00Oomoo`0`Ool00`00Oomoo`0200000goo001oo`04Ool20005Ool00`00
Oomoo`06Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool00`00Oomoo`0:Ool3000=
Ool00`00Oomoo`07Ool20005Ool00`00Oomoo`09Ool20002Ool00`00Oomoo`0:Ool2000FOol01@00
Oomoogoo00000Goo0015Ool00`00Oomoo`0`Ool01000Oomoogoo0P001Woo0`0017oo00<007ooOol0
1Woo00<007ooOol027oo00<007ooOol02Goo00<007ooOol02Woo00<007ooOol01Goo10000Woo00<0
07ooOol037oo00<007ooOol00goo0`001goo00<007ooOol01goo0P0017oo00<007ooOol027oo0P00
6Goo00@007ooOol0005oo`00AWoo0P00<7oo00<007ooOol00Woo0P001Goo00<007ooOol00`000Woo
00<007ooOol01Woo00<007ooOol027oo00<007ooOol027oo00<007ooOol02goo00<007ooOol01000
1goo00<007ooOol037oo00<007ooOol00P002goo00<007ooOol00goo0`001goo00<007ooOol01Goo
0P006goo00@007ooOol0005oo`00B7oo00<007ooOol0;Woo00<007ooOol00Woo00<007ooOol017oo
00<007ooOol00Woo0`002Goo00<007ooOol027oo00<007ooOol027oo00<007ooOol02Goo100037oo
00<007ooOol02goo0`003Woo00<007ooOol00P002goo00<007ooOol00Woo0P007Woo00<007oo0000
0Goo0019Ool00`00Oomoo`0]Ool00`00Oomoo`03Ool20005Ool00`00Oomoo`03Ool40006Ool00`00
Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`02Ool60003Ool00`00Oomoo`0:Ool00`00Oomoo`08
Ool30002Ool00`00Oomoo`0<Ool3000=Ool00`00Oomoo`02000QOol10001Ool1000004Yoo`03001o
ogoo02eoo`03001oogoo00=oo`<000=oo`03001oogoo00Eoo`03001oo`0000D00003Ool007oo00Uo
o`03001oogoo00Aoo`T000Uoo`03001oogoo00]oo`03001oogoo00Aoo`<000Ioo`03001oogoo00Uo
o`80009oo`03001oogoo00]oo`<002Aoo`800000Bgoo0P00;Goo00<007ooOol00goo00<007oo0000
0P000Woo00<007ooOol017oo00<007ooOol01Goo4`001Woo00<007ooOol02Woo00<007ooOol02goo
00<007ooOol00`002Goo00<007ooOol01goo0P001Goo00<007ooOol027oo0P009goo0P00001=Ool0
0`00Oomoo`0[Ool00`00Oomoo`03Ool00`00Oomoo`0300000goo001oo`05Ool00`00Oomoo`07Ool0
0`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`0;Ool00`00Oomoo`0:Ool3000=Ool00`00Oomo
o`03Ool30007Ool00`00Oomoo`06Ool2000YOol2000004ioo`03001oogoo02Yoo`<000=oo`03001o
ogoo009oo`<000Moo`03001oogoo00Ioo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00]o
o`03001oogoo00Ioo`<0009oo`03001oogoo00aoo`03001oogoo008000]oo`03001oogoo00=oo`80
02Moo`@0009oo`00Cgoo00<007ooOol0:Woo00<007oo00000goo00<007ooOol00goo0`001Goo00<0
07ooOol01goo00<007ooOol027oo00<007ooOol02Goo00<007ooOol02Woo00<007ooOol017oo0P00
1Woo00<007ooOol02goo0`003Woo00<007ooOol00P009Goo10001Woo001@Ool2000ZOol00`00Oomo
o`020002Ool00`00Oomoo`04Ool30004Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`09
Ool00`00Oomoo`0:Ool00`00Oomoo`030009Ool00`00Oomoo`08Ool20002Ool00`00Oomoo`0=Ool2
000SOol4000:Ool0059oo`03001oogoo02Qoo`03001oogoo00800003Ool007oo00Eoo`03001oogoo
00<0009oo`03001oogoo00Ioo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Uoo`@000ao
o`03001oogoo00Ioo`8000Aoo`03001oogoo00]oo`80025oo`@000ioo`00Dgoo00<007ooOol09goo
00<007ooOol00Woo00<007oo00001Woo00<007ooOol00Woo0`002Goo00<007ooOol027oo00<007oo
Ool02Goo00<007ooOol00Woo1P000goo00<007ooOol02goo00<007ooOol00goo0P001goo00<007oo
Ool027oo0P007goo10004Woo001DOol00`00Oomoo`0WOol00`00Oomoo`02Ool20006Ool00`00Oomo
o`04Ool40007Ool00`00Oomoo`08Ool00`00Oomoo`07Ool60009Ool00`00Oomoo`0<Ool00`00Oomo
o`02000:Ool00`00Oomoo`05Ool2000MOol4000FOol005Eoo`8002Moo`03001oogoo00Aoo`03001o
ogoo00Aoo`03001oogoo00Eoo`03001oo`0000D00003Ool007oo00Uoo`04001oogooOol60003Ool0
0`00Oomoo`0:Ool00`00Oomoo`0;Ool3000=Ool00`00Oomoo`02Ool2000KOol4000JOol005Moo`03
001oogoo02Eoo`03001oogoo00=oo`<000Aoo`03001oogoo00Eoo`03001oogoo00Eooa0000Uoo`03
001oogoo00]oo`03001oogoo00Moo`<0009oo`03001oogoo00]oo`03001oogoo008001Yoo`<001io
o`00F7oo00<007ooOol097oo00<007ooOol017oo00@007oo000000=oo`03001oogoo00Eoo`03001o
ogoo00Moo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00]oo`03001oogoo00=oo`<000Eo
o`03001oogoo00]oo`<001Qoo`@0025oo`00FGoo00<007ooOol097oo00<007ooOol00goo00D007oo
Oomoo`00009oo`03001oogoo00Eoo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Yoo`03
001oogoo00Yoo`03001oogoo00<000Uoo`03001oogoo00Qoo`8001Moo`@002Eoo`00FWoo0P0097oo
00<007ooOol017oo00@007ooOomoo`800003Ool007oo00Ioo`03001oogoo00Moo`03001oogoo00Qo
o`03001oogoo00Uoo`03001oogoo00Yoo`<000eoo`03001oogoo00Eoo`8001Eoo`@002Uoo`00G7oo
00<007ooOol08Woo00<007ooOol00goo00<007ooOol00goo0P0027oo00<007ooOol01goo00<007oo
Ool027oo00<007ooOol02Goo00<007ooOol01Woo0`000Woo00<007ooOol037oo00<007ooOol00Woo
0P004goo1000;Goo001MOol00`00Oomoo`0QOol20004Ool00`00Oomoo`05Ool20006Ool00`00Oomo
o`07Ool00`00Oomoo`08Ool00`00Oomoo`0:Ool00`00Oomoo`03Ool20005Ool00`00Oomoo`0<Ool0
0`00Oomoo`02000AOol4000aOol005ioo`03001oogoo021oo`04001oo`000003Ool00`00Oomoo`05
Ool50003Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`030008Ool0
0`00Oomoo`0;Ool3000?Ool4000eOol005moo`80021oo`05001oogooOol00002Ool00`00Oomoo`05
Ool01000Oomoogoo1@002Woo00<007ooOol027oo00<007ooOol01goo100037oo00<007ooOol027oo
0P003Woo1000>Goo001QOol00`00Oomoo`0MOol00`00Oomoo`02Ool01000Oomoo`001goo00<007oo
Ool017oo1`0017oo00<007ooOol027oo00@007ooOomoo`H000=oo`03001oogoo00]oo`03001oogoo
00Eoo`8000eoo`<003eoo`00HWoo00<007ooOol077oo00<007ooOol00goo0`001goo00<007ooOol0
1Woo00<007ooOol00Woo4`002Goo00<007ooOol02goo00<007ooOol00goo0P002goo1000@7oo001S
Ool00`00Oomoo`0KOol00`00Oomoo`05Ool20007Ool00`00Oomoo`05Ool00`00Oomoo`08Ool00`00
Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`0;Ool00`00Oomoo`020009Ool40014Ool006Aoo`80
01]oo`03001oogoo00Ioo`8000Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Qoo`03
001oogoo00Yoo`03001oogoo00Yoo`<000Moo`@004Qoo`00IWoo00<007ooOol067oo00<007ooOol0
27oo0P001Goo00<007ooOol01Goo00<007ooOol027oo00<007ooOol027oo00<007ooOol02Woo00<0
07ooOol01Woo0`001Woo1000C7oo001WOol00`00Oomoo`0GOol00`00Oomoo`0:Ool20003Ool00`00
Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`09Ool00`00Oomoo`09Ool00`00Oomoo`04Ool20005
Ool4001@Ool006Qoo`03001oogoo01Ioo`03001oogoo00aoo`80009oo`03001oogoo00Ioo`03001o
ogoo00Moo`03001oogoo00Qoo`03001oogoo00Yoo`03001oogoo00<000=oo`@005Aoo`00JGoo0P00
5Woo00<007ooOol03Woo0`0027oo00<007ooOol027oo00<007ooOol027oo00<007ooOol02Goo0`00
0Woo1000F7oo001[Ool00`00Oomoo`0COol00`00Oomoo`0@Ool40006Ool00`00Oomoo`07Ool00`00
Oomoo`09Ool00`00Oomoo`04Ool400000goo00000002001LOol006aoo`03001oogoo019oo`03001o
ogoo01Aoo`D00003Ool007oo00Uoo`03001oogoo00Qoo`03001oogoo00@0009oo`<0061oo`00KGoo
00<007ooOol04Goo00<007ooOol06Goo200017oo00<007ooOol017oo1`000Woo1000Hgoo001^Ool2
000AOol00`00Oomoo`0QOol;0005Ool4001WOol0071oo`03001oogoo00eoo`03001oogoo02ioo`@0
06]oo`00LGoo00<007ooOol037oo00<007ooOol0:Woo1000Kgoo001bOol00`00Oomoo`0;Ool00`00
Oomoo`0VOol4001cOol007=oo`8000]oo`03001oogoo029oo`@007Moo`00MGoo00<007ooOol027oo
00<007ooOol07Woo1000Ngoo001fOol00`00Oomoo`07Ool00`00Oomoo`0KOol3001oOol007Moo`03
001oogoo00Ioo`03001oogoo01Moo`@0089oo`00N7oo0P001Woo00<007ooOol04goo1000QWoo001j
Ool00`00Oomoo`03Ool00`00Oomoo`0?Ool4002:Ool007]oo`03001oogoo009oo`03001oogoo00]o
o`@008ioo`00O7oo00D007ooOomoo`0000Uoo`@0099oo`00OGoo0P0000=oo`00Ool017oo1000UWoo
001oOol200000goo00000002002JOol0081oo`8009ioo`00ogoo8Goo003oOolQOol00?moob5oo`00
ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo0000\
\>"],
  ImageRangeCache->{{{60.5625, 289.938}, {271.875, 86.0625}} -> {-0.357566, 
  0.412311, 0.00361501, 0.00361501}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Inequality-constrained minimization of posynomials", "Subsubsection",
  CellDingbat->None],

Cell["\<\
There is another sort of constrained minimization problem from chapter five \
for which we need not assume convexity of our functions or domain. If our \
objective and constraint functions are all posynomials then we may use the \
method of section 3. For example suppose we have an objective function that \
is now a posynomial, and a domain constraint as shown below. \
\>", "Text"],

Cell[BoxData[{
    \(f[x_, y_] := x*y\), "\n", 
    \(g2[x_, y_] := 1\/y\^2 + 2\/x\)}], "Input"],

Cell[TextData[{
  "We will set up a dual problem and try to solve that instead. We require a \
new variable for each term in the set of functions. Our constraint is ",
  Cell[BoxData[
      \(g1[x, y] \[LessEqual] 1\)]],
  " (this is the standard form). For each term ",
  Cell[BoxData[
      \(c\_j\ x\^\[Alpha]\_j\ y\^\[Beta]\_j\)]],
  " we have (from the arithmetic-geometric mean and its extended cousin) a \
factor in the dual function of the form ",
  Cell[BoxData[
      \(\((c\_j\/\[Delta]\_j)\)\^\[Delta]\_j\)]],
  ". Specifically, we obtain"
}], "Text"],

Cell[TextData[{
  " ",
  Cell[BoxData[
      \(f[x, y] \[GreaterEqual] 
        \((\(x*y\)\/\[Delta]\_1)\)\^\[Delta]\_1\ \ \(\((\[Delta]\_2 + 
                  \[Delta]\_3)\)\^\(\[Delta]\_2 + \[Delta]\_3\)\) 
          \((1\/\(\(\[Delta]\_2\) y\^2\))\)\^\[Delta]\_2\ \
\((2\/\(\(\[Delta]\_3\) x\))\)\^\[Delta]\_3\)]]
}], "Text"],

Cell[TextData[{
  "We must choose our ",
  Cell[BoxData[
      \(\[Delta]\)]],
  "'s so as to eliminate all factors of ",
  Cell[BoxData[
      \(x\)]],
  " and ",
  Cell[BoxData[
      \(y\)]],
  ", as we did back in chapter 2. Collecting factors in each of these \
variables and setting exponents to zero, we get:"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(eqns = 
        {\[Delta]\_1 \[Equal] 1, \[Delta]\_1 - \[Delta]\_3 \[Equal] 0, 
          \[Delta]\_1 - 2*\[Delta]\_2 \[Equal] 0};\)\), 
  "\[IndentingNewLine]", 
    \(deltas = Solve[eqns]\)}], "Input"],

Cell[BoxData[
    \({{\[Delta]\_1 \[Rule] 1, \[Delta]\_2 \[Rule] 1\/2, 
        \[Delta]\_3 \[Rule] 1}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "As the ",
  Cell[BoxData[
      \(\[Delta]\)]],
  "'s are nonnegative and the solution is unique, we have found the values \
that maximize the dual program ",
  Cell[BoxData[
      \(v[\[Delta]\_1, \[Delta]\_2, \[Delta]\_3] = 
        \((1\/\[Delta]\_1)\)\^\[Delta]\_1\ \((\[Delta]\_2 + \
\[Delta]\_3)\)\^\(\[Delta]\_2 + \[Delta]\_3\)\ \((1\/\[Delta]\_2)\)\^\[Delta]\
\_2\ \((2\/\[Delta]\_3)\)\^\[Delta]\_3\)]]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(v[\[Delta]\_1, \[Delta]\_2, \[Delta]\_3] := 
      \((1\/\[Delta]\_1)\)\^\[Delta]\_1\ \((\[Delta]\_2 + \[Delta]\_3)\)\^\(\
\[Delta]\_2 + \[Delta]\_3\)\ \((1\/\[Delta]\_2)\)\^\[Delta]\_2\ \
\((2\/\[Delta]\_3)\)\^\[Delta]\_3\), "\[IndentingNewLine]", 
    \(vmax\  = \ 
      v[\[Delta]\_1, \[Delta]\_2, \[Delta]\_3]\  /. \ First[deltas]\)}], 
  "Input"],

Cell[BoxData[
    \(3\ \@3\)], "Output"]
}, Open  ]],

Cell["\<\
In general we may get no solution for the dual that satisfies the \
constraints, in which case the theory tells us that the original problem has \
no minimizer. Another case that may arise is that we may not have a unique \
solution for the dual program. Handling this situation is beyond the scope of \
the course.\
\>", "Text"],

Cell[TextData[{
  "Our next task is to find the values of ",
  Cell[BoxData[
      \(x\)]],
  " and ",
  Cell[BoxData[
      \(y\)]],
  " that minimize the original program. For this we require the equality \
conditions of the ordinary and extended AGM inequalities. We need to solve:"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(neweqns = 
      {\(x\ y\)\/\[Delta]\_1 == vmax, 
          1\/\(\[Delta]\_2\ y\^2\) == 2\/\(\[Delta]\_3\ x\)} 
        /. \[InvisibleSpace]\ First[deltas]\), "\[IndentingNewLine]", 
    \(Solve[neweqns, \ {x, y}]\)}], "Input"],

Cell[BoxData[
    \({x\ y == 3\ \@3, 2\/y\^2 == 2\/x}\)], "Output"],

Cell[BoxData[
    \({{x \[Rule] 3, y \[Rule] \@3}, 
      {x \[Rule] \(-3\)\ \((\(-1\))\)\^\(1/3\), 
        y \[Rule] \((\(-1\))\)\^\(2/3\)\ \@3}, 
      {x \[Rule] 3\ \((\(-1\))\)\^\(2/3\), 
        y \[Rule] \(-\((\(-1\))\)\^\(1/3\)\)\ \@3}}\)], "Output"]
}, Open  ]],

Cell["\<\
Only one solution is positive. Note that, as in chapter 2, we can avoid all \
the extraneous solutions by switching to log-linear equations.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(logneweqns\  = \ Map[Log, neweqns, {2}];\)\), "\[IndentingNewLine]", 
    \(logneweqnsexpanded\  = \ PowerExpand[logneweqns]\), 
  "\[IndentingNewLine]", 
    \(\(logvars\  = \ Map[Log, {x, y}];\)\), "\[IndentingNewLine]", 
    \(\(logsoln\  = \ First[Solve[logneweqnsexpanded, logvars]];\)\), 
  "\[IndentingNewLine]", 
    \(newsoln\  = \ Map[Exp, logsoln, {2}]\)}], "Input"],

Cell[BoxData[
    \({Log[x] + Log[y] == \(3\ Log[3]\)\/2, 
      Log[2] - 2\ Log[y] == Log[2] - Log[x]}\)], "Output"],

Cell[BoxData[
    \({x \[Rule] 3, y \[Rule] \@3}\)], "Output"]
}, Open  ]],

Cell["\<\
We see that this agrees with the more basic method above. We should also \
check that the value at our minimizer is equal to the maximim value we just \
found for the dual program.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((f[x, y]\  /. \ newsoln)\)\  \[Equal] \ vmax\)], "Input"],

Cell[BoxData[
    \(True\)], "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 640}, {0, 407}},
WindowSize->{496, 309},
WindowMargins->{{Automatic, 28}, {Automatic, 7}},
PrintingCopies->1,
PrintingPageRange->{3, 3},
PrintingOptions->{"PrintingMargins"->{{36, 36}, {36, 43.1875}}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 28, 0, 64, "Subtitle"],

Cell[CellGroupData[{
Cell[1792, 55, 99, 1, 43, "Subsubsection"],
Cell[1894, 58, 340, 6, 90, "Text"],

Cell[CellGroupData[{
Cell[2259, 68, 430, 8, 152, "Input"],
Cell[2692, 78, 1768, 32, 276, "Output"]
}, Open  ]],
Cell[4475, 113, 376, 11, 90, "Text"],

Cell[CellGroupData[{
Cell[4876, 128, 386, 7, 110, "Input"],
Cell[5265, 137, 548, 10, 185, "Output"]
}, Open  ]],
Cell[5828, 150, 737, 11, 166, "Text"],

Cell[CellGroupData[{
Cell[6590, 165, 53, 1, 30, "Input"],
Cell[6646, 168, 142, 2, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[6825, 175, 44, 1, 30, "Input"],
Cell[6872, 178, 101, 2, 29, "Output"]
}, Open  ]],
Cell[6988, 183, 238, 9, 52, "Text"],

Cell[CellGroupData[{
Cell[7251, 196, 123, 2, 30, "Input"],
Cell[7377, 200, 53379, 1215, 242, 18405, 779, "GraphicsData", "PostScript", \
"Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[60805, 1421, 96, 1, 43, "Subsubsection"],
Cell[60904, 1424, 394, 6, 109, "Text"],
Cell[61301, 1432, 96, 2, 65, "Input"],
Cell[61400, 1436, 563, 13, 101, "Text"],
Cell[61966, 1451, 329, 8, 39, "Text"],
Cell[62298, 1461, 328, 12, 52, "Text"],

Cell[CellGroupData[{
Cell[62651, 1477, 226, 5, 50, "Input"],
Cell[62880, 1484, 117, 2, 42, "Output"]
}, Open  ]],
Cell[63012, 1489, 441, 11, 58, "Text"],

Cell[CellGroupData[{
Cell[63478, 1504, 373, 7, 66, "Input"],
Cell[63854, 1513, 40, 1, 30, "Output"]
}, Open  ]],
Cell[63909, 1517, 339, 6, 90, "Text"],
Cell[64251, 1525, 297, 9, 71, "Text"],

Cell[CellGroupData[{
Cell[64573, 1538, 248, 5, 65, "Input"],
Cell[64824, 1545, 67, 1, 44, "Output"],
Cell[64894, 1548, 258, 5, 54, "Output"]
}, Open  ]],
Cell[65167, 1556, 164, 3, 52, "Text"],

Cell[CellGroupData[{
Cell[65356, 1563, 400, 7, 110, "Input"],
Cell[65759, 1572, 117, 2, 67, "Output"],
Cell[65879, 1576, 62, 1, 32, "Output"]
}, Open  ]],
Cell[65956, 1580, 204, 4, 52, "Text"],

Cell[CellGroupData[{
Cell[66185, 1588, 79, 1, 30, "Input"],
Cell[66267, 1591, 38, 1, 29, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

