(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     25238,        810]*)
(*NotebookOutlinePosition[     26019,        836]*)
(*  CellTagsIndexPosition[     25975,        832]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Math 384", "Subtitle"],

Cell[CellGroupData[{

Cell["A typical problem for the least-squares method", "Subsubsection",
  CellDingbat->None],

Cell["\<\
Recall from a chapter 2 that we once wanted to solve the following \
overdetermined linear system of equations.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(A\  = \ {{0.5`, 1.`, 0, \(-3.`\)}, {\(-1\), \(-2.`\), \(-1\), 
            0}, {\(-1\), 1.`, 2.`, 0}, {0, 0, \(-2.`\), 
            1.`}, {2.`, \(-1\), \(-1\), 0}};\)\), "\n", 
    \(\(b\  = \ {0.20790942220887798`, 
          0.8469893814985481`, \(-0.7884240173388126`\), 0.7187350459748684`, 
          0.1739843117166322`};\)\), "\n", 
    \(LinearSolve[A, b]\)}], "Input"],

Cell[BoxData[
    \(LinearSolve::"nosol" \( : \ \) 
      "Linear equation encountered which has no solution."\)], "Message"],

Cell[BoxData[
    \(LinearSolve[
      {{0.5`, 1.`, 0, \(-3.`\)}, {\(-1\), \(-2.`\), \(-1\), 0}, 
        {\(-1\), 1.`, 2.`, 0}, {0, 0, \(-2.`\), 1.`}, 
        {2.`, \(-1\), \(-1\), 0}}, 
      {0.20790942220887798`, 0.8469893814985481`, \(-0.7884240173388126`\), 
        0.7187350459748684`, 0.1739843117166322`}]\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "We have an overdetermined system of the form ",
  Cell[BoxData[
      \(A  x\&\[RightArrow] = b\&\[RightArrow]\)]],
  " which we wish to solve for ",
  Cell[BoxData[
      \(x\&\[RightArrow]\)]],
  ". As we have seen, this can be viewed as a least-squares problem: minimize \
\[LeftDoubleBracketingBar]",
  Cell[BoxData[
      \(A  x\&\[RightArrow] - b\&\[RightArrow]\)]],
  "\[RightDoubleBracketingBar]. Provided the matrix A has full column rank, \
this in turn was seen to be equivalent to solving ",
  Cell[BoxData[
      \(\(A\^t\) A  x\&\[RightArrow] = \(A\^t\) b\&\[RightArrow]\)]],
  " . We will do just that."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(transA\  = \ Transpose[A];\)\), "\[IndentingNewLine]", 
    \(soln\  = \ LinearSolve[transA . A, transA . b]\)}], "Input"],

Cell[BoxData[
    \({\(-0.1855767439479109`\), \(-0.1162748379381834`\), 
      \(-0.4288629616742703`\), \(-0.13899087737367227`\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Let us check the residual, that is ",
  Cell[BoxData[
      \(\(A . soln\&\[RightArrow] - b\&\[RightArrow]\ \)\)]],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(A . soln - b\)], "Input"],

Cell[BoxData[
    \({0.`, \(-1.1102230246251565`*^-16\), \(-4.440892098500626`*^-16\), 
      \(-1.1102230246251565`*^-16\), \(-3.3306690738754696`*^-16\)}\)], 
  "Output"]
}, Open  ]],

Cell[TextData[{
  "Note that the system need not have been solvable. This same method will \
find the closest vector to an actual solution. The product ",
  Cell[BoxData[
      \(A . soln\&\[RightArrow]\)]],
  " is the projection of ",
  Cell[BoxData[
      \(b\&\[RightArrow]\)]],
  " onto the range subspace of the matrix ",
  Cell[BoxData[
      \(A\)]],
  ". For example, suppose I take ",
  Cell[BoxData[
      \(bnew\&\[RightArrow] = {1, 2, 3, 4, 5}\)]],
  ". Let us see what happens to our least-squares solution."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(bnew\  = \ {1, 2, 3, 4, 5}\), "\[IndentingNewLine]", 
    \(newsoln\  = \ 
      LinearSolve[transA . A, transA . bnew]\), "\[IndentingNewLine]", 
    \(residual\  = \ A . newsoln - bnew\)}], "Input"],

Cell[BoxData[
    \({1, 2, 3, 4, 5}\)], "Output"],

Cell[BoxData[
    \({0.5644627004107591`, \(-0.24102292301576764`\), 
      \(-0.612163773684908`\), \(-0.010070226580098025`\)}\)], "Output"],

Cell[BoxData[
    \({\(-0.9285808930700941`\), \(-1.4702530806943157`\), 
      \(-5.029813170796343`\), \(-2.7857426792102817`\), 
      \(-3.0178879024778063`\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This difference ",
  Cell[BoxData[
      \(residual\&\[RightArrow]\)]],
  " is as close as one can get to ",
  Cell[BoxData[
      \(bnew\&\[RightArrow]\)]],
  " from the range space of the matrix ",
  Cell[BoxData[
      \(A\)]],
  "."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Another typical problem for the least-squares method", "Subsubsection",
  CellDingbat->None],

Cell[TextData[{
  "This time we have a consistent underdetermined system of the form ",
  Cell[BoxData[
      \(A  x\&\[RightArrow] = b\&\[RightArrow]\)]],
  " which we wish to solve for ",
  Cell[BoxData[
      \(x\&\[RightArrow]\)]],
  " and furthermore we want the minimal solution, that is, the solution ",
  Cell[BoxData[
      \(\(x\^*\)\&\[RightArrow]\)]],
  " with smallest Euclidean norm. This, too, may be viewed as a least-squares \
problem. I demonstrated two ways to tackle it in class and here I will repeat \
in brief the method not found in the text. This in turn will lead us to some \
linear algebra techniques that are of general interest in applied math."
}], "Text"],

Cell[TextData[{
  "We first suppose we have a specific solution ",
  Cell[BoxData[
      \(\(x\^0\)\&\[RightArrow]\)]],
  ". Then we know from basic linear algebra that all solutions are of the \
form ",
  Cell[BoxData[
      \(\(x\^0\)\&\[RightArrow] - y\&\[RightArrow]\)]],
  " where ",
  Cell[BoxData[
      \(y\&\[RightArrow]\)]],
  " ranges over solutions to the homogeneous linear system ",
  Cell[BoxData[
      \(A  x\&\[RightArrow] = 0\&\[RightArrow]\)]],
  ". In other words, ",
  Cell[BoxData[
      \(y\&\[RightArrow]\)]],
  " ranges over the null space of ",
  Cell[BoxData[
      \(A\)]],
  ". Just as with finding ",
  Cell[BoxData[
      \(\(x\^0\)\&\[RightArrow]\)]],
  ", one can use basic Gaussian elimination techniques to find a basis for \
the null space of ",
  Cell[BoxData[
      \(A\)]],
  ". If we regard those basis vectors as columns of a new matrix ",
  Cell[BoxData[
      \(B\)]],
  ", then we have written this null space as the range space of ",
  Cell[BoxData[
      \(B\)]],
  " (later I will show how to do the reverse, that is, given a subspace as \
the range space of a matrix, recast it as the null space of another matrix)."
}], "Text"],

Cell[TextData[{
  "To continue, we want to minimize \[LeftDoubleBracketingBar]",
  Cell[BoxData[
      \(\(\(x\^0\)\& \[RightArrow] \) - B\ \(z\& \[RightArrow] \)\)]],
  "\[RightDoubleBracketingBar] where ",
  Cell[BoxData[
      \(\(z\& \[RightArrow] \)\)]],
  " ranges over ",
  Cell[BoxData[
      \(\[DoubleStruckCapitalR]\^k\)]],
  ", ",
  Cell[BoxData[
      \(k\)]],
  " being the dimension of the null space of ",
  Cell[BoxData[
      \(A\)]],
  ". Notice that we have just reduced to the sort of problem tackled in \
section 2 of chapter 4. We know that the minimum for this expression will \
happen at ",
  Cell[BoxData[
      \(\(z\& \[RightArrow] \) = 
        \(B\^\[Dagger]\) \(\(x\^0\)\& \[RightArrow] \)\)]],
  " (question: how do we know ",
  Cell[BoxData[
      \(B\)]],
  " has full column rank?). Hence we obtain ",
  Cell[BoxData[
      \(\(y\& \[RightArrow] \)\  = \ 
        \(B\ \(z\& \[RightArrow] \)\  = \ 
          B\ \(B\^\[Dagger]\) \(A\^\[Dagger]\) \(A\^\[Dagger]\) 
            \(\(x\^0\)\& \[RightArrow] \)\)\)]],
  ", thus ",
  Cell[BoxData[
      \(\(\(x\^*\)\& \[RightArrow] \) = 
        \(\(x\^0\)\& \[RightArrow] \)\  - \ 
          B\ \(B\^\[Dagger]\) \(\(x\^0\ \)\& \[RightArrow] \)\)]],
  "."
}], "Text"],

Cell[TextData[{
  "I will remark that computationally this method has about the same \
complexity of that presented in the book. One advantage is that it does not \
assume that the initial matrix ",
  Cell[BoxData[
      \(A\)]],
  " has full row rank (that is, has linearly independent rows). This is \
important because, while such an assumption will be satisfied essentially \
always, there may be an issue of numeric stability in the sense that rows \
might be \"close\" to linearly dependent. When this happens, solving the \
system ",
  Cell[BoxData[
      \(A\ A\^t\ w\&\[RightArrow]\  = \ b\&\[RightArrow]\)]],
  " for ",
  Cell[BoxData[
      \(w\&\[RightArrow]\)]],
  " can be numerically unstable because ",
  Cell[BoxData[
      \(A\ A\^t\)]],
  " will have a large condition number, whereas there are numerically stable \
ways both to get a reasonable basis for the null space of ",
  Cell[BoxData[
      \(A\)]],
  " and to obtain its generalized inverse."
}], "Text"],

Cell[TextData[{
  "Here is a simple example. We will find the solution of smallest norm to \
the matrix equation\n",
  Cell[BoxData[
      TagBox[
        RowBox[{"(", "\[NoBreak]", GridBox[{
              {"1", "5", "3", \(-8\)},
              {"2", "8", \(-2\), "1"}
              }], "\[NoBreak]", ")"}],
        (MatrixForm[ #]&)]]],
  Cell[BoxData[
      RowBox[{"(", "\[NegativeThinSpace]", GridBox[{
            {\(x\_1\)},
            {\(x\_2\)},
            {\(x\_3\)},
            {\(x\_4\)}
            }], "\[NegativeThinSpace]", ")"}]]],
  " = ",
  Cell[BoxData[
      RowBox[{"(", "\[NegativeThinSpace]", GridBox[{
            {\(-2\)},
            {"1"}
            }], "\[NegativeThinSpace]", ")"}]]]
}], "Text"],

Cell[BoxData[{
    \(\(A\  = \ {{1, 5, 3, \(-8\)}, {2, 8, \(-2\), 
            1}};\)\), "\[IndentingNewLine]", 
    \(\(b\  = \ {\(-2\), 1};\)\)}], "Input"],

Cell["First we obtain a particular solution.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol = \ LinearSolve[A, b]\)], "Input"],

Cell[BoxData[
    \({21\/2, \(-\(5\/2\)\), 0, 0}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Now find the appropriate matrix to represent the null space of the matrix \
",
  Cell[BoxData[
      \(A\)]],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(B\  = \ Transpose[NullSpace[A]]\)], "Input"],

Cell[BoxData[
    \({{\(-69\), 17}, {17, \(-4\)}, {0, 1}, {2, 0}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(minsol\  = \ sol\  - \ B\  . \ PseudoInverse[B] . sol\)], "Input"],

Cell[BoxData[
    \({8\/379, 370\/6443, \(-\(832\/6443\)\), 91\/379}\)], "Output"]
}, Open  ]],

Cell["\<\
We will check that the minimal length solution is indeed smaller than the \
original solution.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol . sol // N\)], "Input"],

Cell[BoxData[
    \(116.5`\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(minsol . minsol // N\)], "Input"],

Cell[BoxData[
    \(0.07806922241191991`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Note that one need not directly find the ",
  StyleBox["PseudoInverse",
    FontWeight->"Bold"],
  StyleBox[" in order to use this technique. We show how to use ",
    FontVariations->{"CompatibilityType"->0}],
  StyleBox["LinearSolve",
    FontWeight->"Bold"],
  StyleBox[" below but for purposes of computational efficiency one should \
use a Cholesky solver or, if the matrix ",
    FontVariations->{"CompatibilityType"->0}],
  StyleBox["Transpose[B].B",
    FontWeight->"Bold"],
  StyleBox[" may be rank deficient, a method based on ",
    FontVariations->{"CompatibilityType"->0}],
  StyleBox["QRDecomposition",
    FontWeight->"Bold",
    FontVariations->{"CompatibilityType"->0}],
  StyleBox[" or ",
    FontVariations->{"CompatibilityType"->0}],
  StyleBox["SingularValues",
    FontWeight->"Bold",
    FontVariations->{"CompatibilityType"->0}],
  StyleBox[".",
    FontVariations->{"CompatibilityType"->0}]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(s2\  = \ 
      LinearSolve[Transpose[B] . B, \ 
        Transpose[B] . sol]\), "\[IndentingNewLine]", 
    \(minsol2\  = \ sol\  - \ B\  . \ s2\)}], "Input"],

Cell[BoxData[
    \({\(-\(91\/758\)\), 832\/6443}\)], "Output"],

Cell[BoxData[
    \({8\/379, 370\/6443, \(-\(832\/6443\)\), 91\/379}\)], "Output"]
}, Open  ]],

Cell["Finally let us do this with the method from the text.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(w\  = \ LinearSolve[A . Transpose[A], b]\)], "Input"],

Cell[BoxData[
    \({\(-\(174\/6443\)\), 155\/6443}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(minsol3\  = \ Transpose[A] . w\)], "Input"],

Cell[BoxData[
    \({8\/379, 370\/6443, \(-\(832\/6443\)\), 91\/379}\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["A pair of related problems from linear algebra", "Subsubsection",
  CellDingbat->None],

Cell[TextData[{
  "(i) We just saw how to recast a null space as a range space. More \
generally, suppose ",
  Cell[BoxData[
      \(P\)]],
  " is the set of all points of the form ",
  Cell[BoxData[
      \({\(x\& \[RightArrow] \) : A \( x\& \[RightArrow] \) = 
          \(b\& \[RightArrow] \)}\)]],
  ". This is called an affine linear set (or, more simply, a linear set). It \
is simply a translation by ",
  Cell[BoxData[
      \(\(b\& \[RightArrow] \)\)]],
  " of all points in the vector subspace ",
  Cell[BoxData[
      \({\(x\& \[RightArrow] \) : A \( x\& \[RightArrow] \) = 
          \(0\& \[RightArrow] \)}\)]],
  ". We have expressed ",
  Cell[BoxData[
      \(P\)]],
  " as the set of all points that satisfy a given linear system of equations. \
This is known as an implicit form for ",
  Cell[BoxData[
      \(P\)]],
  ". Suppose instead we wish to express it in a manner similar to the way we \
present range subspaces. That is, we want to write ",
  Cell[BoxData[
      \(P\)]],
  " as a set of the form ",
  Cell[BoxData[
      \({\(z\& \[RightArrow] \) : \ \(z\& \[RightArrow] \)\  = \ 
          B\ \(y\& \[RightArrow] \)\  + \(v\& \[RightArrow] \)}\)]],
  " where everyone has an appropriate dimension and in particular ",
  Cell[BoxData[
      \(\(y\& \[RightArrow] \)\)]],
  " ranges over ",
  Cell[BoxData[
      \(\[DoubleStruckCapitalR]\^k\)]],
  " with ",
  Cell[BoxData[
      \(k\)]],
  " being the number of columns of ",
  Cell[BoxData[
      \(B\)]],
  " (which we will soon see how to find). Then from the previous section we \
pretty much know how to do this. We first find a particular ",
  Cell[BoxData[
      \(\(\(x\^0\)\& \[RightArrow] \)\)]],
  " that satisfies ",
  Cell[BoxData[
      \(A \( x\& \[RightArrow] \) = \(b\& \[RightArrow] \)\)]],
  ". Then as above we next construct ",
  Cell[BoxData[
      \(B\)]],
  " as the matrix with columns comprising a basis for the null space of ",
  Cell[BoxData[
      \(A\)]],
  ". Then it is simple to check that ",
  Cell[BoxData[
      \(P\)]],
  " consists exactly of all vectors of the form ",
  Cell[BoxData[
      \(B\ \(y\& \[RightArrow] \)\  + \(\(x\^0\)\& \[RightArrow] \)\)]],
  ". Voila. By the way, this manner of describing a subspace is known as the \
parametric form."
}], "Text"],

Cell[TextData[{
  "(ii) We have seen how to rewrite a null space as a range space, and we \
have generalized this to affine linear sets. In other words, we saw how to \
convert from implicit to parametric representation. What about the reverse? \
Suppose you are given a parametric representation of a plane in ",
  Cell[BoxData[
      \(\[DoubleStruckCapitalR]\^3\)]],
  ", and asked to cough up an equation for that plane. Recalling your vector \
calculus, you might do it using a cross product of any pair of linearly \
independent direction vectors for that plane. This method does not generalize \
well to arbitrary linear sets in arbitrary dimensional spaces. Let us see a \
more general approach using basic linear algebra technology. We suppose our \
linear set ",
  Cell[BoxData[
      \(P\)]],
  " is now given as the set of all vectors ",
  Cell[BoxData[
      \({\(z\& \[RightArrow] \) : \ \(z\& \[RightArrow] \)\  = \ 
          B\ \(y\& \[RightArrow] \)\  + \(v\& \[RightArrow] \)}\)]],
  ". Suppose ",
  Cell[BoxData[
      \(\(z\& \[RightArrow] \)\)]],
  " has coordinates ",
  Cell[BoxData[
      \({z\_1,  ... , z\_m}\)]],
  ", and ",
  Cell[BoxData[
      \(B\)]],
  " has dimensions ",
  Cell[BoxData[
      \(m\ x\ n\)]],
  ", ",
  Cell[BoxData[
      \(\(v\& \[RightArrow] \)\)]],
  " is, like ",
  Cell[BoxData[
      \(\(z\& \[RightArrow] \)\)]],
  ", in ",
  Cell[BoxData[
      \(\[DoubleStruckCapitalR]\^m\)]],
  ", and ",
  Cell[BoxData[
      \(\(y\& \[RightArrow] \)\)]],
  " lives in ",
  Cell[BoxData[
      \(\[DoubleStruckCapitalR]\^n\)]],
  " and so has coordinates ",
  Cell[BoxData[
      \({y\_1,  ... , y\_n}\)]],
  ". To get an implicit form first rewrite this vector equation as ",
  Cell[BoxData[
      \(\(\ 
      B\ \(y\& \[RightArrow] \)\ \  - \ \(z\& \[RightArrow] \)\  = \ 
        \(-\(v\& \[RightArrow] \)\)\)\)]],
  ". Now form the augmented matrix that goes with this system of linear \
equations, putting the columns for the ",
  Cell[BoxData[
      \({y\_1,  ... , y\_n}\)]],
  " before the columns for the ",
  Cell[BoxData[
      \({z\_1,  ... , z\_m}\)]],
  ". Next put this (augmented) matrix into row echelon form. Finally, discard \
all rows that have pivots in the first ",
  Cell[BoxData[
      \(n\)]],
  " columns. All remaining rows give the linear relations that must be \
satisfied by the ",
  Cell[BoxData[
      \(\(z\& \[RightArrow] \)\)]],
  " coordinates, in other words, they give an implicit form for the linear \
set."
}], "Text"],

Cell[TextData[{
  "We illustrate with a simple example. Since ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " has no natural representation for augmented matrices we will do something \
that is mathematically equivalent. Specifically, we will add a column at the \
end for the vector ",
  Cell[BoxData[
      \(\(v\& \[RightArrow] \)\)]],
  ", do our row reducing, and then take the negative of that column for the \
right-hand-side (using only those rows we use from the rest of the matrix). \
Say we are given the linear set consisting of all vectors of the form"
}], "Text"],

Cell[BoxData[
    RowBox[{
      RowBox[{"(", "\[NegativeThinSpace]", GridBox[{
            {\(z\_1\)},
            {\(z\_2\)},
            {\(z\_3\)},
            {\(z\_4\)}
            }], "\[NegativeThinSpace]", ")"}], " ", "=", " ", 
      RowBox[{
        RowBox[{\(y\_1\), " ", 
          RowBox[{"(", "\[NegativeThinSpace]", GridBox[{
                {"1"},
                {\(-2\)},
                {"3"},
                {\(-4\)}
                }], "\[NegativeThinSpace]", ")"}]}], "+", " ", 
        RowBox[{\(y\_2\), " ", 
          RowBox[{"(", "\[NegativeThinSpace]", GridBox[{
                {"2"},
                {\(-7\)},
                {"1"},
                {"4"}
                }], "\[NegativeThinSpace]", ")"}]}], "+", " ", 
        RowBox[{"(", "\[NegativeThinSpace]", GridBox[{
              {"3"},
              {"0"},
              {"2"},
              {"1"}
              }], "\[NegativeThinSpace]", ")"}]}]}]], "Text"],

Cell[BoxData[{
    \(\(yvars\  = \ {y\_1, y\_2};\)\), "\[IndentingNewLine]", 
    \(\(zvars\  = \ {z\_1, z\_2, z\_3, z\_4};\)\), "\[IndentingNewLine]", 
    \(\(eqns\  = \ 
        {z\_1, z\_2, z\_3, z\_4}\  \[Equal] \ 
          {{1, 2}, {\(-2\), \(-7\)}, \ {3, 1}, {\(-4\), 4}} . {y\_1, y\_2}\  
            + \ {3, 0, 2, 1};\)\), "\[IndentingNewLine]", 
    \(\(vars\  = \ Flatten[{yvars, zvars}];\)\)}], "Input"],

Cell["We reformulate this as noted above.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(neweqns\  = \ Thread[eqns[\([2]\)] - eqns[\([1]\)] \[Equal] 0]\)], 
  "Input"],

Cell[BoxData[
    \({3 + y\_1 + 2\ y\_2 - z\_1 == 0, \(-2\)\ y\_1 - 7\ y\_2 - z\_2 == 0, 
      2 + 3\ y\_1 + y\_2 - z\_3 == 0, 1 - 4\ y\_1 + 4\ y\_2 - z\_4 == 0}\)], 
  "Output"]
}, Open  ]],

Cell["Now we want to extract a coefficient matrix.", "Text"],

Cell[BoxData[
    \(<< LinearAlgebra`MatrixManipulation`\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \({mat, negvec}\  = LinearEquationsToMatrices[neweqns, \ vars]\)], "Input"],

Cell[BoxData[
    \({{{1, 2, \(-1\), 0, 0, 0}, {\(-2\), \(-7\), 0, \(-1\), 0, 0}, 
        {3, 1, 0, 0, \(-1\), 0}, {\(-4\), 4, 0, 0, 0, \(-1\)}}, 
      {\(-3\), 0, \(-2\), \(-1\)}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(newmat\  = \ Transpose[Append[Transpose[mat], \(-negvec\)]]\)], "Input"],

Cell[BoxData[
    \({{1, 2, \(-1\), 0, 0, 0, 3}, {\(-2\), \(-7\), 0, \(-1\), 0, 0, 0}, 
      {3, 1, 0, 0, \(-1\), 0, 2}, {\(-4\), 4, 0, 0, 0, \(-1\), 1}}\)], 
  "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(redmat\  = \ RowReduce[newmat]\)], "Input"],

Cell[BoxData[
    \({{1, 0, 0, 0, \(-\(1\/4\)\), 1\/16, 7\/16}, 
      {0, 1, 0, 0, \(-\(1\/4\)\), \(-\(3\/16\)\), 11\/16}, 
      {0, 0, 1, 0, \(-\(3\/4\)\), \(-\(5\/16\)\), \(-\(19\/16\)\)}, 
      {0, 0, 0, 1, 9\/4, 19\/16, \(-\(91\/16\)\)}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Now we drop all rows with pivots in the ",
  Cell[BoxData[
      \(\(y\& \[RightArrow] \)\)]],
  " columns. There are several ways to automate this. I will do it as \
follows. We define a function that extracts from a vector the subvector \
consisting of the entries corresponding to the ",
  Cell[BoxData[
      \(\(y\& \[RightArrow] \)\)]],
  " columns. We then define its length squared. For our selection function we \
keep only those rows that have this squared norm equal to zero; this is \
exactly the case for which all entries are zero in the columns that \
correspond to the ",
  Cell[BoxData[
      \(\(y\& \[RightArrow] \)\)]],
  " variables."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(ysubvec[vec_]\  := \ Take[vec, Length[yvars]]\), 
  "\[IndentingNewLine]", 
    \(normsq[vec_]\  := \ vec . vec\), "\[IndentingNewLine]", 
    \(keepers\  = \ Select[redmat, \ normsq[ysubvec[#]] \[Equal] 0 &]\)}], 
  "Input"],

Cell[BoxData[
    \({{{0, 0, 1, 0, \(-\(3\/4\)\), \(-\(5\/16\)\)}, 
        {0, 0, 0, 1, 9\/4, 19\/16}}, {\(-\(19\/16\)\), \(-\(91\/16\)\)}}\)], 
  "Output"]
}, Open  ]],

Cell["To convert back to equation form we may do as follows.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({newredmat, newrhs}\  = \ 
        {Map[Drop[#, \(-1\)] &, keepers], \ Map[Last, keepers]};\)\), 
  "\[IndentingNewLine]", 
    \(Thread[newredmat\  . \ vars\  == \ newrhs]\)}], "Input"],

Cell[BoxData[
    \({z\_1 - \(3\ z\_3\)\/4 - \(5\ z\_4\)\/16 == \(-\(19\/16\)\), 
      z\_2 + \(9\ z\_3\)\/4 + \(19\ z\_4\)\/16 == \(-\(91\/16\)\)}\)], 
  "Output"]
}, Open  ]],

Cell["\<\
There are more general methods for doing parametric-to-implicit conversion \
that do not rely on the equations being linear. Here is one such technique.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Eliminate[neweqns, \ yvars]\)], "Input"],

Cell[BoxData[
    \(19\ z\_1 == 51 - 5\ z\_2 + 3\ z\_3 && 
      16\ z\_2 == 91 - 36\ z\_3 - 19\ z\_4\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "An alternative is to use ",
  Cell[BoxData[
      \(GroebnerBasis\)]],
  ". For this we need to use algebraic expressions rather than equations."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(GroebnerBasis[Map[First, neweqns], \ zvars, yvars]\)], "Input"],

Cell[BoxData[
    \({\(-91\) + 16\ z\_2 + 36\ z\_3 + 19\ z\_4, 
      \(-19\) + 16\ z\_1 - 12\ z\_3 - 5\ z\_4}\)], "Output"]
}, Open  ]],

Cell["\<\
It is left as an exercise to verify that all three implicit forms are \
equivalent.\
\>", "Text"],

Cell["\<\
We have discussed how to go from implicit to parametric representations of \
linear sets, and vice versa. It is useful to have this conversion capacity \
for various applications. One that comes to mind is ray-tracing in computer \
graphics. In typical problems from this domain one wants to express a given \
ray in parametric form, and a planar surface in implicit form, in order to \
solve for the coordinates of the point where the ray strikes the surface.\
\>", "Text"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 640}, {0, 407}},
WindowSize->{496, 313},
WindowMargins->{{41, Automatic}, {Automatic, 14}},
PrintingCopies->1,
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"PrintingMargins"->{{36, 36}, {36, 43.1875}}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 28, 0, 64, "Subtitle"],

Cell[CellGroupData[{
Cell[1792, 55, 92, 1, 43, "Subsubsection"],
Cell[1887, 58, 135, 3, 52, "Text"],

Cell[CellGroupData[{
Cell[2047, 65, 400, 7, 130, "Input"],
Cell[2450, 74, 125, 2, 42, "Message"],
Cell[2578, 78, 330, 6, 67, "Output"]
}, Open  ]],
Cell[2923, 87, 649, 16, 93, "Text"],

Cell[CellGroupData[{
Cell[3597, 107, 145, 2, 50, "Input"],
Cell[3745, 111, 146, 2, 29, "Output"]
}, Open  ]],
Cell[3906, 116, 152, 5, 34, "Text"],

Cell[CellGroupData[{
Cell[4083, 125, 45, 1, 30, "Input"],
Cell[4131, 128, 172, 3, 48, "Output"]
}, Open  ]],
Cell[4318, 134, 533, 15, 92, "Text"],

Cell[CellGroupData[{
Cell[4876, 153, 221, 4, 70, "Input"],
Cell[5100, 159, 49, 1, 29, "Output"],
Cell[5152, 162, 142, 2, 29, "Output"],
Cell[5297, 166, 177, 3, 29, "Output"]
}, Open  ]],
Cell[5489, 172, 267, 11, 53, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[5793, 188, 98, 1, 43, "Subsubsection"],
Cell[5894, 191, 687, 14, 133, "Text"],
Cell[6584, 207, 1177, 35, 179, "Text"],
Cell[7764, 244, 1247, 37, 121, "Text"],
Cell[9014, 283, 982, 24, 167, "Text"],
Cell[9999, 309, 728, 23, 111, "Text"],
Cell[10730, 334, 157, 3, 50, "Input"],
Cell[10890, 339, 54, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[10969, 343, 58, 1, 30, "Input"],
Cell[11030, 346, 62, 1, 42, "Output"]
}, Open  ]],
Cell[11107, 350, 146, 6, 33, "Text"],

Cell[CellGroupData[{
Cell[11278, 360, 64, 1, 30, "Input"],
Cell[11345, 363, 79, 1, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[11461, 369, 86, 1, 30, "Input"],
Cell[11550, 372, 82, 1, 42, "Output"]
}, Open  ]],
Cell[11647, 376, 118, 3, 33, "Text"],

Cell[CellGroupData[{
Cell[11790, 383, 47, 1, 30, "Input"],
Cell[11840, 386, 40, 1, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[11917, 392, 53, 1, 30, "Input"],
Cell[11973, 395, 54, 1, 29, "Output"]
}, Open  ]],
Cell[12042, 399, 946, 25, 90, "Text"],

Cell[CellGroupData[{
Cell[13013, 428, 179, 4, 50, "Input"],
Cell[13195, 434, 63, 1, 42, "Output"],
Cell[13261, 437, 82, 1, 42, "Output"]
}, Open  ]],
Cell[13358, 441, 69, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[13452, 445, 73, 1, 30, "Input"],
Cell[13528, 448, 65, 1, 42, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[13630, 454, 63, 1, 30, "Input"],
Cell[13696, 457, 82, 1, 42, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[13827, 464, 92, 1, 43, "Subsubsection"],
Cell[13922, 467, 2283, 65, 299, "Text"],
Cell[16208, 534, 2505, 71, 337, "Text"],
Cell[18716, 607, 588, 12, 109, "Text"],
Cell[19307, 621, 949, 28, 75, "Text"],
Cell[20259, 651, 416, 7, 130, "Input"],
Cell[20678, 660, 51, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[20754, 664, 98, 2, 30, "Input"],
Cell[20855, 668, 179, 3, 48, "Output"]
}, Open  ]],
Cell[21049, 674, 60, 0, 33, "Text"],
Cell[21112, 676, 69, 1, 30, "Input"],

Cell[CellGroupData[{
Cell[21206, 681, 93, 1, 30, "Input"],
Cell[21302, 684, 196, 3, 67, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[21535, 692, 92, 1, 30, "Input"],
Cell[21630, 695, 171, 3, 48, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[21838, 703, 63, 1, 30, "Input"],
Cell[21904, 706, 258, 4, 110, "Output"]
}, Open  ]],
Cell[22177, 713, 686, 16, 128, "Text"],

Cell[CellGroupData[{
Cell[22888, 733, 246, 5, 70, "Input"],
Cell[23137, 740, 157, 3, 76, "Output"]
}, Open  ]],
Cell[23309, 746, 70, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[23404, 750, 209, 4, 70, "Input"],
Cell[23616, 756, 165, 3, 42, "Output"]
}, Open  ]],
Cell[23796, 762, 176, 3, 52, "Text"],

Cell[CellGroupData[{
Cell[23997, 769, 60, 1, 30, "Input"],
Cell[24060, 772, 115, 2, 29, "Output"]
}, Open  ]],
Cell[24190, 777, 176, 5, 52, "Text"],

Cell[CellGroupData[{
Cell[24391, 786, 83, 1, 30, "Input"],
Cell[24477, 789, 124, 2, 29, "Output"]
}, Open  ]],
Cell[24616, 794, 107, 3, 33, "Text"],
Cell[24726, 799, 484, 7, 128, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

