(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     13010,        345]*)
(*NotebookOutlinePosition[     13790,        371]*)
(*  CellTagsIndexPosition[     13746,        367]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Math 384", "Subtitle"],

Cell[CellGroupData[{

Cell["\<\
Applying LinearSolve or QRDecomposition to a harder least squares problem\
\>", "Subsubsection",
  CellDingbat->None],

Cell["\<\
We pretty well beat to death that warm-up example in the previous handout. \
This next one is a bit more of a challenge. It is a minor modification of a \
question taken from the news group comp.soft-sys.math.mathematica several \
years ago.\
\>", "Text"],

Cell[TextData[{
  "The problem is as follows. Given points that fall approximately on a \
sphere, find the  radius and center of that sphere. One can perform the \
approximation as a best fit in the sense of least squares. This can be set up \
as a calculus problem (set some derivatives to zero and solve) or as a linear \
algebra problem. The first two methods defined below use calculus. Of these, \
",
  Cell[BoxData[
      \(centerRadius1\)]],
  " is conceptually simpler: find ",
  Cell[BoxData[
      \({a, b, c, Rsquare}\)]],
  " that minimizes the sums of squares of the form\n",
  Cell[BoxData[
      \(\((x - a)\)\^2 + \((y - b)\)\^2 + \((z - c)\)\^2 - Rsquare\)]],
  "\nwhere you explicitly substitute in the data for ",
  Cell[BoxData[
      \({x, y, z}\)]],
  ". The second method, ",
  Cell[BoxData[
      \(centerRadius2\)]],
  ", sets up the sphere expression as\n",
  Cell[BoxData[
      \(x\^2 + y\^2 + z\^2 + d\ x + e\ y + f\ z + g\)]],
  "\nand finds ",
  Cell[BoxData[
      \({d, e, f, g}\)]],
  "which it then uses to calculate the center and radius. This is seen to be \
faster than the first method because the derivative equations are much \
simpler. I note that one can use chapter 3 methods a la ",
  Cell[BoxData[
      \(FindMinimum\)]],
  " rather than using a root-finder to get the critical point; the \
modifications needed to do it this way are quite simple."
}], "Text"],

Cell[TextData[{
  "The later methods, ",
  Cell[BoxData[
      \(centerRadius3\)]],
  " and ",
  Cell[BoxData[
      \(centerRadius4\)]],
  ", use linear algebra as we do in chapter 4. Notice that we do not have a \
\"function\" of the variables because a sphere is generally given as an \
implicit equation in all variables (in other words, we do not solve for one \
in terms of the rest). So this will not look exactly like your textbook \
problems. The way to understand these methods is to look again at the \
expression ",
  Cell[BoxData[
      \(x\^2 + y\^2 + z\^2 + d\ x + e\ y + f\ z + g\)]],
  ". We want to solve for a vector of parameters ",
  Cell[BoxData[
      \({e, f, g, h}\)]],
  ". If we augment the data matrix by a column of ",
  Cell[BoxData[
      \(1\)]],
  "'s then we can put the squares on one side, everything else on the other, \
and get a matrix \"equation\" of the form\n",
  Cell[BoxData[
      \({{x\_1, y\_1, z\_1, 1}, {x\_2, y\_2, z\_2, 1},  ... , {x\_n, y\_n, 
              z\_n, 1}} . {e, f, g, h} == {x\_1\%2 + y\_1\%2 + 
            z\_1\%2,  ... , x\_n\%2 + y\_n\%2 + z\_n\%2}\)]],
  "\nwhere we need to \"solve\" for ",
  Cell[BoxData[
      \({e, f, g, h}\)]],
  ". Of course we do not expect this overdetermined system to be solvable, \
but as with curve-fitting we can use a least-squares solution. This is what \
we find in ",
  Cell[BoxData[
      \(centerRadius3\)]],
  " and ",
  Cell[BoxData[
      \(centerRadius4\)]],
  ".\n\nThey are fast compared to ",
  Cell[BoxData[
      \(centerRadius1\)]],
  " and c",
  Cell[BoxData[
      \(enterRadius2\)]],
  ". The function ",
  Cell[BoxData[
      \(centerRadius3\)]],
  " uses the so-called \"normal equations\" i.e. solves ",
  Cell[BoxData[
      \(Transpose[mat] . mat == mat . vec\)]],
  ", while\n",
  Cell[BoxData[
      \(centerRadius4\)]],
  " uses the QR decomposition of a matrix. These linear algebra methods work \
independently of the dimension of the sphere (it is more work to generalize \
the two calculus methods although it can certainly be done). In higher \
dimensions they are still respectably fast."
}], "Text"],

Cell["Now we define the various functions.", "Text"],

Cell[BoxData[
    \(centerRadius1[points_List] := 
      Module[{chisquare, w, x, y, z, a, b, c, R, eqns}, 
        w = \((x - a)\)\^2 + \((y - b)\)\^2 + \((z - c)\)\^2 - R 
            /. \[InvisibleSpace]Thread[{x, y, z} \[Rule] Transpose[points]]; 
        chisquare = Plus@@\(w\^2\); 
        eqns = \((Expand[\[PartialD]\_#1 chisquare] == 0 &)\)/@{a, b, c, R}; 
        {{a, b, c}, \@R} 
          /. \[InvisibleSpace]FindRoot[eqns, {a, 0}, {b, 0}, {c, 0}, {R, 0}, 
            MaxIterations \[Rule] 100]]\)], "Input"],

Cell[BoxData[
    \(centerRadius2[points_List] := 
      Module[{chisquare, w, x, y, z, eqns, d, e, f, g, soln}, 
        w = x\^2 + y\^2 + z\^2 + d\ x + e\ y + f\ z + g 
            /. \[InvisibleSpace]Thread[{x, y, z} \[Rule] Transpose[points]]; 
        chisquare = Plus@@\(w\^2\); 
        eqns = \((Expand[\[PartialD]\_#1 chisquare] == 0 &)\)/@{d, e, f, g}; 
        soln = FindRoot[eqns, {d, 0}, {e, 0}, {f, 0}, {g, 0}, 
            MaxIterations \[Rule] 30]; 
        {d, e, f, g} = {d, e, f, g} /. \[InvisibleSpace]soln; 
        {\(-\(1\/2\)\)\ {d, e, f}, \@\(1\/4\ \((d\^2 + e\^2 + f\^2)\) - g\)}]
        \)], "Input"],

Cell[BoxData[
    \(centerRadius3[points_List] := 
      Module[{mat, sumsquares, dotprod, cen}, \[IndentingNewLine]mat = 
          Map[Append[#, 1] &, points]; \[IndentingNewLine]sumsquares\  = \ 
          Map[# . # &, points]; \[IndentingNewLine]dotprod = 
          LinearSolve[Transpose[mat] . mat, 
            Transpose[mat] . sumsquares]; \[IndentingNewLine]cen = 
          Drop[dotprod, \(-1\)]/2; \[IndentingNewLine]{cen, 
          Sqrt[Last[dotprod] + cen . cen]}]\)], "Input"],

Cell[BoxData[
    \(centerRadius4[points_List] := 
      Module[{mat, sumsquares, Q, R, b, cen}, 
        sumsquares = Map[# . # &, points]; \[IndentingNewLine]mat = 
          Map[Append[#, 1] &, points]; \[IndentingNewLine]{Q, R} = 
          QRDecomposition[mat]; \[IndentingNewLine]b = 
          LinearSolve[R, Q . sumsquares]; \[IndentingNewLine]cen = 
          Drop[b, \(-1\)]/2; \[IndentingNewLine]{cen, 
          Sqrt[Last[b] + cen . cen]}]\)], "Input"],

Cell[TextData[{
  "In the following examples we will work with data sets that contain up to \
two percent noise. Our approximate sphere will be centered near ",
  Cell[BoxData[
      \({1, 2, 3}\)]],
  " with radius near ",
  Cell[BoxData[
      \(1\)]],
  "."
}], "Text"],

Cell[BoxData[{
    \(\(randomNorm[x_] := \((0.98 + 0.04*Random[])\)*x/Sqrt[x . x];\)\), 
  "\[IndentingNewLine]", 
    \(\(points = 
        Map[{1, 2, 3} + randomNorm[#] &\ , 
          Table[Random[] - 0.5, {1000}, {3}]];\)\)}], "Input"],

Cell["\<\
Finally we check that they all give the same result and compare how long each \
takes to calculate. Note that the results are consistent,and the linear \
algebra methods are much faster than the calculus methods.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(Timing[centerRadius1[points]]\), "\n", 
    \(Timing[centerRadius2[points]]\), "\n", 
    \(Timing[centerRadius3[points]]\), "\[IndentingNewLine]", 
    \(Timing[centerRadius4[points]]\)}], "Input"],

Cell[BoxData[
    \({6.150000000001455`\ Second, 
      {{0.9991038567453061`, 2.0002523437659105`, 3.0005646640774857`}, 
        1.0000466881136707`}}\)], "Output"],

Cell[BoxData[
    \({2.4199999999982538`\ Second, 
      {{0.999103856745366`, 2.0002523437658923`, 3.0005646640773795`}, 
        1.0000466881136711`}}\)], "Output"],

Cell[BoxData[
    \({0.11000000000058208`\ Second, 
      {{0.999103856745379`, 2.0002523437659154`, 3.000564664077525`}, 
        1.000046688113672`}}\)], "Output"],

Cell[BoxData[
    \({0.049999999999272404`\ Second, 
      {{0.9991038567453561`, 2.0002523437659074`, 3.0005646640775363`}, 
        1.0000466881136711`}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "I will remark that if we did ",
  Cell[BoxData[
      \(centerRadius3\)]],
  " the \"correct\" way, using a Cholesky decomposition and solver, it would \
most likely be faster than ",
  Cell[BoxData[
      \(centerRadius4\)]],
  "."
}], "Text"],

Cell[TextData[{
  "This function produces an arbitrary number of points in an arbitrary \
dimension. We use it to produce a more challenging example: ",
  Cell[BoxData[
      \(1000\)]],
  " data points in ",
  Cell[BoxData[
      \(25\)]],
  " dimensions."
}], "Text"],

Cell[BoxData[{
    \(\(genpoints[dim_, num_] := 
        \((Range[dim] + #1 &)\)/@
          \(randomNorm/@Table[Random[] - 0.5, {num}, {dim}]\);\)\), 
  "\[IndentingNewLine]", 
    \(\(points2 = genpoints[25, 1000];\)\)}], "Input"],

Cell["The linear algebra methods are still quite fast.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(Timing[centerRadius3[points2]]\), "\[IndentingNewLine]", 
    \(Timing[centerRadius4[points2]]\)}], "Input"],

Cell[BoxData[
    \({0.21999999999752617`\ Second, 
      {{0.9998101019575653`, 2.001144596156933`, 2.999414293383798`, 
          4.00079616182511`, 4.998345694327188`, 5.999884676317952`, 
          6.998658460186641`, 7.999824202196051`, 9.001023529891254`, 
          10.00057616010372`, 11.000460598243578`, 12.001634998000268`, 
          13.00178551782295`, 14.002679519450101`, 14.997700425696651`, 
          16.001436907836847`, 17.000541935836008`, 18.00103751850584`, 
          18.999927556735013`, 20.002627416540342`, 20.998846770830685`, 
          22.001559248206316`, 23.00008268379416`, 24.001831566569635`, 
          25.000040217677988`}, 1.0003794889418207`}}\)], "Output"],

Cell[BoxData[
    \({0.15999999999985448`\ Second, 
      {{0.9998101020954089`, 2.0011445957575242`, 2.999414293594194`, 
          4.000796161793843`, 4.998345693762993`, 5.9998846753560455`, 
          6.998658459951467`, 7.999824202242176`, 9.001023528362314`, 
          10.000576158623092`, 11.000460597876803`, 12.001634998198384`, 
          13.00178551851859`, 14.002679519385977`, 14.997700425333226`, 
          16.00143690563027`, 17.000541934383737`, 18.001037517980535`, 
          18.999927556421657`, 20.00262741849755`, 20.998846768132783`, 
          22.001559245534406`, 23.000082681703546`, 24.001831566747953`, 
          25.00004021434021`}, 1.0003794889481847`}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "As you may well realize we can do these using ",
  Cell[BoxData[
      \(PseudoInverse\)]],
  " instead of explicitly solving linear equations. While this is said to be \
a bad idea in general, it can be quite effective in cases where the columns \
of our matrix are not linearly independent. This might happen if, say, we \
have redundant data e.g. one coordinate is \"yoked\" to another."
}], "Text"],

Cell["\<\
As a final remark, let me point out that we were fortunate to be able to \
coerce the problem to a form suitable for linear least squares techniques. In \
other words, we were able to formulate it as a problem that was linear in the \
parameters for which we solved. This cannot in general be done. But there are \
nonlinear tachniques for least-squares problems as well, for example the \
Levenberg-Marquardt method. References are Numerical Recipes in C by Press et \
al or Practical Optimization by Gill, Murray, and Wright.\
\>", "Text"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 640}, {0, 407}},
WindowSize->{496, 313},
WindowMargins->{{Automatic, 14}, {Automatic, 4}},
PrintingCopies->1,
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"PrintingMargins"->{{36, 36}, {36, 43.1875}}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 28, 0, 64, "Subtitle"],

Cell[CellGroupData[{
Cell[1792, 55, 127, 3, 43, "Subsubsection"],
Cell[1922, 60, 265, 5, 71, "Text"],
Cell[2190, 67, 1407, 34, 272, "Text"],
Cell[3600, 103, 2133, 57, 409, "Text"],
Cell[5736, 162, 52, 0, 33, "Text"],
Cell[5791, 164, 523, 9, 195, "Input"],
Cell[6317, 175, 629, 11, 214, "Input"],
Cell[6949, 188, 491, 8, 170, "Input"],
Cell[7443, 198, 464, 8, 170, "Input"],
Cell[7910, 208, 272, 9, 52, "Text"],
Cell[8185, 219, 239, 5, 70, "Input"],
Cell[8427, 226, 236, 4, 71, "Text"],

Cell[CellGroupData[{
Cell[8688, 234, 219, 4, 90, "Input"],
Cell[8910, 240, 166, 3, 29, "Output"],
Cell[9079, 245, 166, 3, 29, "Output"],
Cell[9248, 250, 165, 3, 29, "Output"],
Cell[9416, 255, 169, 3, 29, "Output"]
}, Open  ]],
Cell[9600, 261, 263, 9, 52, "Text"],
Cell[9866, 272, 269, 9, 52, "Text"],
Cell[10138, 283, 232, 5, 90, "Input"],
Cell[10373, 290, 64, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[10462, 294, 129, 2, 50, "Input"],
Cell[10594, 298, 696, 10, 124, "Output"],
Cell[11293, 310, 699, 10, 124, "Output"]
}, Open  ]],
Cell[12007, 323, 421, 8, 90, "Text"],
Cell[12431, 333, 551, 8, 128, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

