(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[      7966,        273]*)
(*NotebookOutlinePosition[      8747,        299]*)
(*  CellTagsIndexPosition[      8703,        295]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Math 384", "Subtitle"],

Cell[CellGroupData[{

Cell["Various ways to approach linear least-squares problems", "Subsubsection",
  CellDingbat->None],

Cell[TextData[{
  "Example from class: we want to find a quadratic function in x that best \
approximates the data (given as pairs of the form ",
  Cell[BoxData[
      \({x, f[x]}\)]],
  "): ",
  Cell[BoxData[
      \({{0, 1}, {1, 3}, {2, 2}, {3, 7}, {5, 2}}\)]],
  ". Our right-hand-side vector is ",
  Cell[BoxData[
      \({1, 3, 2, 7, 2}\)]],
  " and our matrix is\n",
  Cell[BoxData[
      RowBox[{"A", "=", 
        TagBox[
          RowBox[{"(", "\[NoBreak]", GridBox[{
                {"1", "0", "0"},
                {"1", "1", "1"},
                {"1", "2", "4"},
                {"1", "3", "9"},
                {"1", "5", "25"}
                }], "\[NoBreak]", ")"}],
          (MatrixForm[ #]&)]}]]],
  "\nLet us see how to obtain a least-squares solution to this curve fitting \
problem. I will use a machine number entry just to force machine arithmetic \
throughout."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(data\  = \ {{0, 1}, {1, 3}, {2, 2}, {3, 7}, {5, 2}};\)\), 
  "\[IndentingNewLine]", 
    \(\(A = {{1. , 0, 0}, {1, 1, 1}, {1, 2, 4}, {1, 3, 9}, {1, 5, 25}};\)\), 
  "\[IndentingNewLine]", 
    \(\(y\  = \ {1, 3, 2, 7, 2};\)\), "\n", 
    \(transAdotA\  = \ Transpose[A] . A\), "\[IndentingNewLine]", 
    \(transAdoty\  = \ Transpose[A] . y\)}], "Input"],

Cell[BoxData[
    \({{5.`, 11.`, 39.`}, {11.`, 39.`, 161.`}, {39.`, 161.`, 723.`}}\)], 
  "Output"],

Cell[BoxData[
    \({15.`, 38.`, 124.`}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(LinearSolve[transAdotA, transAdoty]\  . \ {1, x, x^2}\)], "Input"],

Cell[BoxData[
    \(\(0.5625920471281347`\[InvisibleSpace]\) + 2.8858615611192877`\ x - 
      0.5014727540500727`\ x\^2\)], "Output"]
}, Open  ]],

Cell["\<\
This is our \"best-fitting\" curve in the sense of least-squares. Notice that \
I could do this in several other ways. I'll illustrate a few of them here.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\((PseudoInverse[A] . y)\)\  . \ {1, x, x^2}\)], "Input"],

Cell[BoxData[
    \(\(0.5625920471281252`\[InvisibleSpace]\) + 2.885861561119293`\ x - 
      0.501472754050074`\ x\^2\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This use of ",
  Cell[BoxData[
      \(PseudoInverse\)]],
  " is not recommended at least for nontrivial problems in high dimensions \
because there are better ways to handle it. Actually ",
  Cell[BoxData[
      \(LinearSolve\)]],
  " is also not recommended because one can take advantage of the fact that \
the matrix ",
  Cell[BoxData[
      \(A\^t . A\)]],
  " is positive definite and hence can use a Cholesky decomposition and \
solver. Because this capability is well hidden in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " I will defer on the details. It is my hope that we make this more readily \
available in a future release."
}], "Text"],

Cell[TextData[{
  "Here is the way one would use the QR decomposition of ",
  Cell[BoxData[
      \(A\)]],
  " to obtain the best-fit curve. Notice (from the dimensions, or inspired \
guesswork) that the ",
  Cell[BoxData[
      \(Q\)]],
  " matrix is transposed from that of the literature. Hence we do not \
transpose ",
  Cell[BoxData[
      \(Q\)]],
  " in the code below."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({qq, rr}\  = \ QRDecomposition[A];\)\), "\[IndentingNewLine]", 
    \(Dimensions[qq]\), "\[IndentingNewLine]", 
    \(Dimensions[rr]\), "\[IndentingNewLine]", 
    \(Transpose[qq] . rr - A\  // \ Chop\)}], "Input"],

Cell[BoxData[
    \({3, 5}\)], "Output"],

Cell[BoxData[
    \({3, 3}\)], "Output"],

Cell[BoxData[
    \({{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(LinearSolve[R, \ Q . y]\  . \ {1, x, x^2}\)], "Input"],

Cell[BoxData[
    \(\(0.5625920471281269`\[InvisibleSpace]\) + 2.885861561119294`\ x - 
      0.5014727540500737`\ x\^2\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This use of ",
  Cell[BoxData[
      \(LinearSolve\)]],
  " is fair game, by the way, in the sense that it does not impose \
computational inefficiency. The ",
  Cell[BoxData[
      \(R\)]],
  " matrix is already upper triangular, hence the Gaussian elimination phase \
is trivial. So this can safely be used for \"hard\" problems without \
incurring serious penalty."
}], "Text"],

Cell["\<\
Here is a technique that uses multivariable minimization technology, just as \
in chapter 3.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(sumsquares\  = \ 
      Apply[Plus, \ 
        Map[\((#[\([2]\)]\  - \ 
                  \((a + b*#[\([1]\)] + c*#[\([1]\)]^2)\))\)^2\  &, \ data]]
        \)], "Input"],

Cell[BoxData[
    \(\((1 - a)\)\^2 + \((2 - a - 5\ b - 25\ c)\)\^2 + 
      \((7 - a - 3\ b - 9\ c)\)\^2 + \((2 - a - 2\ b - 4\ c)\)\^2 + 
      \((3 - a - b - c)\)\^2\)], "Output"]
}, Open  ]],

Cell["\<\
Now we will obtain both the best-fit quadratic curve and the \
sum-of-squares-of-discrepancy.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\({sumsq, vals}\  = \ FindMinimum[sumsquares, {a, 0}, {b, 0}, {c, 0}];
      \)\), "\[IndentingNewLine]", 
    \(\(({a, b, c}\  /. \ vals)\)\  . \ {1, x, x^2}\), 
  "\[IndentingNewLine]", 
    \(sumsq\)}], "Input"],

Cell[BoxData[
    \(\(0.5625920471281494`\[InvisibleSpace]\) + 2.885861561119339`\ x - 
      0.5014727540499024`\ x\^2\)], "Output"],

Cell[BoxData[
    \(11.081001472754044`\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Here is a method built into ",
  StyleBox["Mathematica.",
    FontSlant->"Italic"]
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Fit[{{0, 1}, {1, 3}, {2, 2}, {3, 7}, {5, 2}}, \ {1, x, x^2}, {x}]\)], 
  "Input"],

Cell[BoxData[
    \(\(0.5625920471281252`\[InvisibleSpace]\) + 2.885861561119293`\ x - 
      0.501472754050074`\ x\^2\)], "Output"]
}, Open  ]],

Cell["\<\
The five methods all agree on the result, a fact which I hope you will find \
encouraging; life is not always this fortunate in the wild world of \
computational mathematics.\
\>", "Text"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 640}, {0, 424}},
WindowSize->{496, 340},
WindowMargins->{{Automatic, 29}, {22, Automatic}},
PrintingCopies->1,
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"PrintingMargins"->{{36, 36}, {36, 43.1875}}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 28, 0, 64, "Subtitle"],

Cell[CellGroupData[{
Cell[1792, 55, 100, 1, 43, "Subsubsection"],
Cell[1895, 58, 898, 26, 203, "Text"],

Cell[CellGroupData[{
Cell[2818, 88, 377, 7, 130, "Input"],
Cell[3198, 97, 99, 2, 29, "Output"],
Cell[3300, 101, 53, 1, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[3390, 107, 86, 1, 30, "Input"],
Cell[3479, 110, 134, 2, 29, "Output"]
}, Open  ]],
Cell[3628, 115, 178, 3, 52, "Text"],

Cell[CellGroupData[{
Cell[3831, 122, 77, 1, 30, "Input"],
Cell[3911, 125, 132, 2, 29, "Output"]
}, Open  ]],
Cell[4058, 130, 683, 18, 128, "Text"],
Cell[4744, 150, 389, 13, 71, "Text"],

Cell[CellGroupData[{
Cell[5158, 167, 237, 4, 90, "Input"],
Cell[5398, 173, 40, 1, 29, "Output"],
Cell[5441, 176, 40, 1, 29, "Output"],
Cell[5484, 179, 89, 1, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[5610, 185, 74, 1, 30, "Input"],
Cell[5687, 188, 133, 2, 29, "Output"]
}, Open  ]],
Cell[5835, 193, 399, 11, 90, "Text"],
Cell[6237, 206, 116, 3, 33, "Text"],

Cell[CellGroupData[{
Cell[6378, 213, 190, 5, 70, "Input"],
Cell[6571, 220, 181, 3, 48, "Output"]
}, Open  ]],
Cell[6767, 226, 117, 3, 33, "Text"],

Cell[CellGroupData[{
Cell[6909, 233, 235, 5, 90, "Input"],
Cell[7147, 240, 133, 2, 29, "Output"],
Cell[7283, 244, 53, 1, 29, "Output"]
}, Open  ]],
Cell[7351, 248, 113, 4, 33, "Text"],

Cell[CellGroupData[{
Cell[7489, 256, 101, 2, 50, "Input"],
Cell[7593, 260, 132, 2, 29, "Output"]
}, Open  ]],
Cell[7740, 265, 198, 4, 52, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

