(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     18529,        548]*)
(*NotebookOutlinePosition[     19455,        580]*)
(*  CellTagsIndexPosition[     19374,        574]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Math 384", "Subtitle"],

Cell[CellGroupData[{

Cell["\<\
Using the LU decomposition of a square matrix: a brief introduction\
\>", "Subsubsection",
  CellDingbat->None],

Cell["\<\
What is the LU decomposition of a matrix? Why do we care about it in Math \
384? Is there life in outer space? Several of these questions will be \
addressed in this notebook. First let me say that, properly speaking, this is \
the domain of linear algebra and is used strictly as a tool in nonlinear \
programming. On the other hand, I'll note that about three pages of the text \
is devoted to this and related matrix algabra \"tools\". The fact is that \
this topic is rarely covered in a one-semester course on linear algebra, and \
yet it is quite important in the real world of applied mathematics. As it can \
be used in the realm of optimization it is entirely plausible that we pause \
for a lecture or so to discuss it.\
\>", "Text"],

Cell["\<\
For our purposes we will work with matrices of machine double-precision \
numbers.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(mat\  = \ 
        N[{{4, \(-3\), \(-2\), 8}, {\(-5\), 0, 9, 3}, {3, \(-6\), 1, \(-7\)}, 
            {0, \(-2\), 9, 1}}];\)\), "\n", 
    \(\(decomp\  = LUDecomposition[mat];\)\), "\[IndentingNewLine]", 
    \({lud, perm, cnum}\  = \ decomp\)}], "Input"],

Cell[BoxData[
    \({{{\(-5.`\), 0.`, 9.`, 3.`}, 
        {\(-0.6000000000000001`\), \(-6.`\), 6.4`, \(-5.199999999999999`\)}, 
        {0.`, 0.3333333333333333`, 6.866666666666667`, 2.733333333333333`}, 
        {\(-0.8`\), 0.5`, 0.2912621359223301`, 12.20388349514563`}}, 
      {2, 3, 4, 1}, 8.914897784527854`}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "I set it up this way because I will want the full decomposition list \
later. Right now we will investigate its components individually. What are \
these three items? The first, ",
  Cell[BoxData[
      \(lud\)]],
  ", is the actual LU decomposition of the matrix. It stores a lower \
triangular matrix UNDER the main diagonal; we use the convention that the \
entries on the main diagonal are all ones. The main diagonal and above store \
an upper triangular matrix. The second item, ",
  Cell[BoxData[
      \(perm\)]],
  ", encodes the permutation of rows of the matrix that was done during \
Gaussian elimination. To recover the original matrix we may use this and ",
  Cell[BoxData[
      \(lud\)]],
  ", as will be demonstrated below. It is not (quite) as simple as in class \
for the reason that rows were swapped for purposes of pivoting. More on this \
later. Finally, cnum is an approximation of what is called the \"condition \
number\" of the matrix. Explanation of this would take us a bit afield, but I \
will say a few words about how it is used in practice, and we will see more \
of this in a subsequent example. If one uses the LU decomposition to solve a \
linear system of equations, then the size of the condition number, in digits, \
can be taken as an approximation of how many digits of precision in the \
result one CANNOT trust. That is, if you use machine numbers which have about \
16 digits of precision, and ",
  Cell[BoxData[
      \(cnum\)]],
  " is around ",
  Cell[BoxData[
      \(10\^3\)]],
  ", then you can guess that your result has only around ",
  Cell[BoxData[
      \(13\)]],
  " or so digits of precision. This is only a heuristic, but in practice it \
works reasonably well; it may be found in the numerical linear algebra \
treatise by Golub and Van Loan.\n\nOkay, let us see how to recover the matrix \
",
  Cell[BoxData[
      \(mat\)]],
  " from ",
  Cell[BoxData[
      \(lud\)]],
  " and ",
  Cell[BoxData[
      \(perm\)]],
  ". By the way, this information can be found in the \"further examples\" \
section of the ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " help browser. First we define a pair of functions to extract the upper \
and lower triangular matrices from lud."
}], "Text"],

Cell[BoxData[{
    \(Upper[LU_?MatrixQ] := 
      LU\ Table[If[i \[LessEqual] j, 1, 0], {i, Length[LU]}, {j, Length[LU]}]
          \), "\n", 
    \(Lower[LU_?MatrixQ] := LU - Upper[LU] + IdentityMatrix[Length[LU]]\)}], 
  "Input",
  CellTags->"LUDecomposition"],

Cell["\<\
Then when we multiply these we do not quite get back the original matrix, but \
rather that same matrix with its rows shuffled according to the permutation \
vector perm. Specifically, in place of row(1) we have row(perm(1)), etc. We \
check this below.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(mat[\([perm]\)] - Lower[lud] . Upper[lud]\  // \ Chop\)], "Input"],

Cell[BoxData[
    \({{0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}, {0, 0, 0, 0}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Of course this LU decomposition of ",
  Cell[BoxData[
      \(mat\)]],
  " is good for something beyond showing that we can rewrite ",
  Cell[BoxData[
      \(mat\)]],
  " as above. Specifically it is quite useful for solving linear equations of \
the form\n",
  Cell[BoxData[
      \(mat . \(x\& \[Rule] \) == \(b\& \[Rule] \)\)]],
  " for ",
  Cell[BoxData[
      \(\(x\& \[Rule] \)\)]],
  " given the right-hand-side vector ",
  Cell[BoxData[
      \(\(b\& \[Rule] \)\)]],
  ". Let us try an example."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(b\  = \ {\(-3\), 4, 1, \(-2\)};\)\), "\[IndentingNewLine]", 
    \(x\  = \ LUBackSubstitution[decomp, \ b]\)}], "Input"],

Cell[BoxData[
    \({\(-1.6157517899761338`\), \(-0.9912490055688147`\), 
      \(-0.4371519490851233`\), \(-0.0481304693715195`\)}\)], "Output"]
}, Open  ]],

Cell["Does this vector really solve the linear equations?", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(mat\  . \ x\)], "Input"],

Cell[BoxData[
    \({\(-3.0000000000000004`\), 4.000000000000001`, 1.0000000000000002`, 
      \(-1.9999999999999998`\)}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Looks good. If you are looking carefully you realize that we obtained a \
solution in terms of machine double precision floating point numbers, whereas \
the input right-hand-side vector ",
  Cell[BoxData[
      \(\(b\& \[Rule] \)\)]],
  " was exact integers. Why? This is because ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " uses a model of numerical \"coercion\" in matrix algebra. The matrix ",
  Cell[BoxData[
      \(mat\)]],
  " contains machine floats, and so most matrix algebra computations such as \
",
  Cell[BoxData[
      \(LinearSolve\)]],
  " will perform the entire operation in floating point arithmetic regardless \
of the precision of the right-hand-side vector (indeed, if only one entry in \
",
  Cell[BoxData[
      \(mat\)]],
  " was a machine float and the rest were exact integers, this would still \
happen). Moreover, if the entries of mat were higher-precision numbers than \
machine floats, the matrix computation would be performed at the lowest \
precision of all the entries.\n\nHad we so desired, we could have done the LU \
decomposition and back-solving in exact arithmetic using rational numbers."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(exactmat\  = \ 
        {{4, \(-3\), \(-2\), 8}, {\(-5\), 0, 9, 3}, {3, \(-6\), 1, \(-7\)}, 
          {0, \(-2\), 9, 1}};\)\), "\n", 
    \(exactdecomp\  = LUDecomposition[exactmat]\), "\[IndentingNewLine]", 
    \(exactx\  = \ LUBackSubstitution[exactdecomp, \ b]\), 
  "\[IndentingNewLine]", 
    \(exactmat\  . \ exactx\)}], "Input"],

Cell[BoxData[
    \({{{3, \(-6\), 1, \(-7\)}, {0, \(-2\), 9, 1}, 
        {4\/3, \(-\(5\/2\)\), 115\/6, 119\/6}, 
        {\(-\(5\/3\)\), 5, \(-\(206\/115\)\), 2514\/115}}, {3, 4, 1, 2}, 1}
      \)], "Output"],

Cell[BoxData[
    \({\(-\(677\/419\)\), \(-\(1246\/1257\)\), \(-\(1099\/2514\)\), 
      \(-\(121\/2514\)\)}\)], "Output"],

Cell[BoxData[
    \({\(-3\), 4, 1, \(-2\)}\)], "Output"]
}, Open  ]],

Cell["\<\
So why do we not do this? Becase it is typically much slower and eats much \
more memory than working over inexact numbers.\
\>", "Text"],

Cell[TextData[{
  "The astute reader may recognize that we just did something a bit foreign \
to the experience of most students of linear algebra. We decoupled \
row-reduction of the matrix ",
  Cell[BoxData[
      \(mat\)]],
  " from back-solving the matrix equation ",
  Cell[BoxData[
      \(mat . \(x\& \[Rule] \) == \(b\& \[Rule] \)\)]],
  ". That is, we did not work with the so-called \"augmented matrix\" (",
  Cell[BoxData[
      \(mat | \(b\& \[Rule] \)\)]],
  ") , using Gaussian elimination to put it into reduced row-echelon form \
(and certainly we did not do the outrageous thing by computing the inverse of \
",
  Cell[BoxData[
      \(mat\)]],
  "). Instead we simply performed row reduction on ",
  Cell[BoxData[
      \(mat\)]],
  " and then fed in a specific vector  ",
  Cell[BoxData[
      \(\(b\& \[Rule] \)\)]],
  ", along with the LU decomposition form of ",
  Cell[BoxData[
      \(mat\)]],
  ", into ",
  Cell[BoxData[
      \(LUBackSubstitution\)]],
  ", in order to obtain out solution vector ",
  Cell[BoxData[
      \(\(x\& \[Rule] \)\)]],
  ". Is this somehow better? Yes, for several reasons.\n\nFirst, we only want \
to solve for ",
  Cell[BoxData[
      \(\(x\& \[Rule] \)\)]],
  ". We do not really care about messing with the augmented matrix (",
  Cell[BoxData[
      \(mat | \(b\& \[Rule] \)\)]],
  "), other than to see how the last column turns out. Doing the ",
  Cell[BoxData[
      \(LUDecomposition\)]],
  " and then invoking ",
  Cell[BoxData[
      \(LUBackSubstitution\)]],
  " in effect allows us to do the minimum of labor (assuming the matrix does \
not have some special structure e.g. mostly zeroes or symmetry; these cases \
can be handled more efficiently by other means). I should mention that \
internally, ",
  StyleBox["Mathematica'",
    FontSlant->"Italic"],
  "s ",
  Cell[BoxData[
      \(LinearSolve\)]],
  ", given an inexact number matrix and vector, does pretty much does as we \
did using the LU decomposition and back-solver. In fact, it also warns one \
about possible ill-conditioning of a problem (this means that a small \
perturbation in the input can give a huge change in the output, so if the \
input is only approximate you really cannot trust the result). This brings me \
to a second reason why one might choose to work with the LU decomposition: \
you get the approximated condition number of the matrix, and it can serve as \
a warning of ill-conditioning if its magnitude is comparable to the precision \
of the input matrix.\n\nHere is a third reason: since the matrix \
decomposition and back-solving are separated, we can give different \
right-hand-side vectors to ",
  Cell[BoxData[
      \(LUBackSubstitution\)]],
  " and obtain different solution vectors, without needing to form separate \
augmented matirces and row-reduce all of them individually.\n\nI'll \
illustrate using the LU decomposition and back-solving with a larger example, \
one that is notoriously ill-conditioned. We use a member of the family of \
Hilbert matrices; these are commonly used as benchmarks for testing numeric \
linear algebra functions."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(hilbmat[n_Integer]\  := \ Table[1/\((i + j - 1)\), \ {i, n}, {j, n}]\), 
  "\[IndentingNewLine]", 
    \(MatrixForm[hilbmat[5]]\)}], "Input"],

Cell[BoxData[
    TagBox[
      RowBox[{"(", "\[NoBreak]", GridBox[{
            {"1", \(1\/2\), \(1\/3\), \(1\/4\), \(1\/5\)},
            {\(1\/2\), \(1\/3\), \(1\/4\), \(1\/5\), \(1\/6\)},
            {\(1\/3\), \(1\/4\), \(1\/5\), \(1\/6\), \(1\/7\)},
            {\(1\/4\), \(1\/5\), \(1\/6\), \(1\/7\), \(1\/8\)},
            {\(1\/5\), \(1\/6\), \(1\/7\), \(1\/8\), \(1\/9\)}
            }], "\[NoBreak]", ")"}],
      (MatrixForm[ #]&)]], "Output"]
}, Open  ]],

Cell["We will work with somewhat larger flavors of this matrix.", "Text"],

Cell[BoxData[{
    \(\(h40\  = \ hilbmat[40];\)\), "\[IndentingNewLine]", 
    \(numerichilbmat[n_Integer, prec_:  $MachinePrecision]\  := \ 
      Table[N[1, prec]/\((i + j - 1)\), \ {i, n}, {j, n}]\), 
  "\[IndentingNewLine]", 
    \(\(nh40\  = \ numerichilbmat[40];\)\)}], "Input"],

Cell["\<\
First let us note that it is computationally slow to work with this exact \
matrix.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Timing[\(Inverse[h40];\)]\)], "Input"],

Cell[BoxData[
    \({8.63000000000011`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell["\<\
On the other hand it may be computationally useless to work with such a \
matrix using machine double-precision arithmetic.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Timing[\(Inverse[nh40];\)]\)], "Input"],

Cell[BoxData[
    \(Inverse::"luc" \( : \ \) 
      "Result for \!\(Inverse\) of badly conditioned matrix \!\({\
\[LeftSkeleton] 1 \[RightSkeleton]}\) may contain significant numerical \
errors."\)], "Message"],

Cell[BoxData[
    \({0.15999999999985448`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "We will steer a middle path, using high-precision arithmetic. \
Specifically, I will set the entries to have ",
  Cell[BoxData[
      \(100\)]],
  " digits of precision."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(bignh40\  = \ numerichilbmat[40, 100];\)\), "\[IndentingNewLine]", 
    \(Timing[\(inv\  = \ Inverse[bignh40];\)]\)}], "Input"],

Cell[BoxData[
    \({2.5800000000017462`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell["\<\
This was alot faster than working with the exact matrix. Furthermore, if we \
want to solve linear equations we can do much better than to use the matrix \
inverse.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(Timing[\(decomp\  = \ LUDecomposition[bignh40];\)]\)], "Input"],

Cell[BoxData[
    \({0.8200000000069849`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(decomp[\([3]\)] // N\)], "Input"],

Cell[BoxData[
    \(1.6063822860313575`*^59\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "This is ",
  Cell[BoxData[
      \(59\)]],
  " digits, and ",
  Cell[BoxData[
      \(59\)]],
  " is substantially less than ",
  Cell[BoxData[
      \(100\)]],
  ", so we figure that if we solve linear equations using this LU \
decomposition then we'll have around ",
  Cell[BoxData[
      \(40\)]],
  " digits correct in the result. I'll demonstrate with a random vector of \
integer components in the range ",
  Cell[BoxData[
      \({\(-10\), 10}\)]],
  "."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(rhs\  = \ Table[Random[Integer, \ {\(-10\), 10}], \ {40}];\)\), 
  "\n", 
    \(Timing[\(bigx1\  = \ LinearSolve[bignh40, rhs];\)]\), "\n", 
    \(Timing[\(bigx2\  = \ LUBackSubstitution[decomp, rhs];\)]\)}], "Input"],

Cell[BoxData[
    \({1.0400000000008731`\ Second, Null}\)], "Output"],

Cell[BoxData[
    \({0.11000000000058208`\ Second, Null}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "So we see it is quite efficient, once we have an LU decomposition, just to \
do back-solving. We should look a bit at the results. First, do they both \
appear to solve the linear system ",
  Cell[BoxData[
      \(bignh40 . \(x\& \[Rule] \) == rhs\&\[LongRightArrow]\)]],
  " for ",
  Cell[BoxData[
      \(\(x\& \[Rule] \)\)]],
  " ?"
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(Max[Abs[bignh40\  . \ bigx1\  - \ rhs]]\), "\[IndentingNewLine]", 
    \(Max[Abs[bignh40\  . \ bigx2\  - \ rhs]]\)}], "Input"],

Cell[BoxData[
    \(1.04404871487976`0*^-53\)], "Output"],

Cell[BoxData[
    \(7.9654595556623`0*^-59\)], "Output"]
}, Open  ]],

Cell["\<\
They seem to. As we know, solutions to linear systems of full rank are \
unique. Moreover since the matrix is invertible it has full rank. So of \
course our two solutions are equal.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(N[Max[Abs[bigx1\  - \ bigx2]]]\)], "Input"],

Cell[BoxData[
    \(3.9243826846124516`*^6\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "Seemingly not even close. What is going on? Well the notion of \"full rank\
\", in inexact arithmetic, is not really discrete, but rather a continuum. As \
the condition number increases, in some sense the matrix is approaching \
singularity. (For exact matrices this makes no sense. The matrix is either \
singular or not, and there is no continuum). But our example involves a \
matrix of inexact numbers, and so a very small differences in the internal \
computations can lead to much larger differences in the results. Also if you \
check carefully you will find that the components that differ by around ",
  Cell[BoxData[
      \(10\^6\)]],
  " are quite large by comparison to that amount (somewhere around ",
  Cell[BoxData[
      \(10\^58\)]],
  "), so in terms of percentage they are very close to one another."
}], "Text"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 640}, {0, 424}},
WindowSize->{496, 415},
WindowMargins->{{Automatic, 25}, {Automatic, 146}},
PrintingCopies->1,
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"PrintingMargins"->{{36, 36}, {43.1875, 50.375}}}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "LUDecomposition"->{
    Cell[5701, 149, 262, 6, 90, "Input",
      CellTags->"LUDecomposition"]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"LUDecomposition", 19271, 567}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 28, 0, 64, "Subtitle"],

Cell[CellGroupData[{
Cell[1792, 55, 121, 3, 43, "Subsubsection"],
Cell[1916, 60, 753, 11, 166, "Text"],
Cell[2672, 73, 106, 3, 33, "Text"],

Cell[CellGroupData[{
Cell[2803, 80, 278, 5, 90, "Input"],
Cell[3084, 87, 328, 5, 67, "Output"]
}, Open  ]],
Cell[3427, 95, 2271, 52, 413, "Text"],
Cell[5701, 149, 262, 6, 90, "Input",
  CellTags->"LUDecomposition"],
Cell[5966, 157, 277, 5, 71, "Text"],

Cell[CellGroupData[{
Cell[6268, 166, 86, 1, 30, "Input"],
Cell[6357, 169, 90, 1, 29, "Output"]
}, Open  ]],
Cell[6462, 173, 535, 18, 72, "Text"],

Cell[CellGroupData[{
Cell[7022, 195, 143, 2, 50, "Input"],
Cell[7168, 199, 145, 2, 29, "Output"]
}, Open  ]],
Cell[7328, 204, 67, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[7420, 208, 44, 1, 30, "Input"],
Cell[7467, 211, 134, 2, 29, "Output"]
}, Open  ]],
Cell[7616, 216, 1178, 26, 243, "Text"],

Cell[CellGroupData[{
Cell[8819, 246, 360, 7, 110, "Input"],
Cell[9182, 255, 210, 4, 76, "Output"],
Cell[9395, 261, 122, 2, 42, "Output"],
Cell[9520, 265, 56, 1, 29, "Output"]
}, Open  ]],
Cell[9591, 269, 147, 3, 52, "Text"],
Cell[9741, 274, 3124, 73, 612, "Text"],

Cell[CellGroupData[{
Cell[12890, 351, 162, 3, 50, "Input"],
Cell[13055, 356, 456, 9, 133, "Output"]
}, Open  ]],
Cell[13526, 368, 73, 0, 33, "Text"],
Cell[13602, 370, 284, 5, 90, "Input"],
Cell[13889, 377, 107, 3, 33, "Text"],

Cell[CellGroupData[{
Cell[14021, 384, 58, 1, 30, "Input"],
Cell[14082, 387, 67, 1, 29, "Output"]
}, Open  ]],
Cell[14164, 391, 147, 3, 52, "Text"],

Cell[CellGroupData[{
Cell[14336, 398, 59, 1, 30, "Input"],
Cell[14398, 401, 210, 4, 63, "Message"],
Cell[14611, 407, 70, 1, 29, "Output"]
}, Open  ]],
Cell[14696, 411, 201, 6, 52, "Text"],

Cell[CellGroupData[{
Cell[14922, 421, 150, 2, 50, "Input"],
Cell[15075, 425, 69, 1, 29, "Output"]
}, Open  ]],
Cell[15159, 429, 188, 4, 52, "Text"],

Cell[CellGroupData[{
Cell[15372, 437, 83, 1, 30, "Input"],
Cell[15458, 440, 69, 1, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[15564, 446, 53, 1, 30, "Input"],
Cell[15620, 449, 57, 1, 29, "Output"]
}, Open  ]],
Cell[15692, 453, 492, 19, 71, "Text"],

Cell[CellGroupData[{
Cell[16209, 476, 240, 4, 70, "Input"],
Cell[16452, 482, 69, 1, 29, "Output"],
Cell[16524, 485, 70, 1, 29, "Output"]
}, Open  ]],
Cell[16609, 489, 366, 10, 73, "Text"],

Cell[CellGroupData[{
Cell[17000, 503, 147, 2, 50, "Input"],
Cell[17150, 507, 57, 1, 29, "Output"],
Cell[17210, 510, 56, 1, 29, "Output"]
}, Open  ]],
Cell[17281, 514, 206, 4, 52, "Text"],

Cell[CellGroupData[{
Cell[17512, 522, 63, 1, 30, "Input"],
Cell[17578, 525, 56, 1, 29, "Output"]
}, Open  ]],
Cell[17649, 529, 852, 15, 166, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

