(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 4.0,
MathReader 4.0, or any compatible application. The data for the notebook 
starts with the line containing stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    305257,      10542]*)
(*NotebookOutlinePosition[    305973,      10567]*)
(*  CellTagsIndexPosition[    305929,      10563]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Math 384", "Subtitle"],

Cell[CellGroupData[{

Cell[TextData[{
  "Using Eigensystem in ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to diagonalize a matrix"
}], "Subsubsection",
  CellDingbat->None],

Cell[TextData[{
  "Matrices with non-repeated eigenvalues can be diagonalized using ",
  Cell[BoxData[
      \(Eigensystem\)]],
  ". This includes all random matrices since eigenvalues are always distinct \
for these.\n\nThe eigenvalues give the diagonalized form of the matrix. The \
columns of the change of basis matrix are the eigenvectors. Here is a quick \
example."
}], "Text"],

Cell["\<\
m = N[{{1, 2, 3, 2},
{5, 3, 1, 8},
{6, 2, 8, 9},
{4, 2, 1, 7}}];
{evalues, evectors} = Eigensystem[m];
newbasis = Transpose[evectors];
diag = DiagonalMatrix[evalues];\
\>", "Input"],

Cell["We check the diagonalization.", "Text"],

Cell[CellGroupData[{

Cell["\<\
m == newbasis . diag . Inverse[newbasis]\
\>", "Input"],

Cell[BoxData[
    \(True\)], "Output"]
}, Open  ]],

Cell["\<\
This was a baby example. In general we do not expect these to be exactly \
equal but only close, due to numerical approximation and round-off error in \
the computations. So I'll show how one sees this in the next example. This \
time I want to use a symmetric matrix. First we take a \"random\" matrix.\
\>", "Text"],

Cell["\<\
m1 = Table[Random[], {20}, {20}];\
\>", "Input"],

Cell["Now make it symmetric.", "Text"],

Cell["m = m1 + Transpose[m1];", "Input"],

Cell["We now diagonalize it as above.", "Text"],

Cell["\<\
{evalues, evectors} = Eigensystem[m];
newbasis = Transpose[evectors];
diag = DiagonalMatrix[evalues];\
\>", "Input"],

Cell["\<\
We check the diagonalization. This time we subtract matrices and find the \
largest difference because these numerical techniques cannot give perfect \
precision.

First we show that the transpose of newbasis is equal to its inverse.\
\>", "Text"],

Cell[CellGroupData[{

Cell["\<\
Max[Abs[Inverse[newbasis] - Transpose[newbasis]]]\
\>", "Input"],

Cell[BoxData[
    \(2.699229728619912`*^-15\)], "Output"]
}, Open  ]],

Cell["\<\
Now show that we indeed diagonalized the matrix.\
\>", "Text"],

Cell[CellGroupData[{

Cell["\<\
Max[Abs[m - newbasis . diag . Transpose[newbasis]]]\
\>", "Input"],

Cell[BoxData[
    \(5.773159728050814`*^-15\)], "Output"]
}, Open  ]],

Cell["\<\
By the way, a random example of this dimension has no hope to be positive or \
negative definite. This example certainly is not. The eigenvalues, once \
sorted, are
{-3.8303,-2.74231,-2.59862,-2.3718,-2.07864,-1.1248,-0.772718,-0.575508,-0.\
416335,
-0.0937088,0.0132402,0.371628,0.512506,0.924972,1.49055,1.78089,1.97815,2.\
12925,
2.65701,20.0095}

Question 1: How do I know that random examples of high dimension will be \
indefinite?\
\>", "Text"],

Cell[TextData[{
  "If I produce repeated random examples, I get similar patterns with the \
eigenvalues. Specifically, I get ",
  Cell[BoxData[
      \(19\)]],
  " of them ranging approximately from ",
  Cell[BoxData[
      \(\(-3\)\)]],
  " to ",
  Cell[BoxData[
      \(3\)]],
  ", and one that is around ",
  Cell[BoxData[
      \(20\)]],
  ". If I use a ",
  Cell[BoxData[
      \(50\ x50\)]],
  " matrix, then ",
  Cell[BoxData[
      \(49\)]],
  " eigenvalues range from around ",
  Cell[BoxData[
      \(\(-5\)\)]],
  " to ",
  Cell[BoxData[
      \(5\)]],
  ", with one around ",
  Cell[BoxData[
      \(50\)]],
  ".\n\nQuestion 2: Why is this?"
}], "Text"],

Cell["\<\
I guess I should say that question 1 is not terribly difficult to answer, but \
offhand I have no explanation for the phenomena behind question 2. Well, not \
quite; I know why the one large eigenvalue is what it is. Note to math \
concentrators: this eigenvalue spread is, mathematically speaking, a VERY \
interesting phenomenon.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Computing a symmetric bilinear form from its corresponding homogeneous \
quadratic polynomial\
\>", "Subsubsection",
  CellDingbat->None],

Cell["\<\
Given a homogeneous quadratic polynomial in n variables, one can form a \
symmetric matrix that is the corresponding bilinear form. I was asked several \
years ago to cough up some code to convert from the polynomial to the \
corresponding matrix. What I wrote then was quite convoluted. If you read the \
last handout carefully you may realize that one can do this fairly simply \
just by repeating the computation for our Hessian matrix. First get the \
derivatives (the gradient vector), then take all derivatives of that. Last, \
divide by two because, well, you ponder that one.\
\>", "Text"],

Cell[BoxData[
    \(symMatrix[poly_] := \ 
      With[\[IndentingNewLine]{vars = Variables[poly]}, \[IndentingNewLine]1/
            2\ *\ Outer[D, Map[D[poly, #] &, \ vars], vars]]\)], "Input"],

Cell["Here is a test to convince you that it works.", "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(vars = Array[x, 5]\), "\[IndentingNewLine]", 
    \(mat = Table[Random[Integer, {\(-10\), 10}], {5}, {5}]\), 
  "\[IndentingNewLine]", 
    \(poly = Expand[vars . mat . vars]\), "\[IndentingNewLine]", 
    \(symMatrix[poly]\)}], "Input"],

Cell[BoxData[
    \({{\(-9\), 1\/2, \(-\(7\/2\)\), \(-1\), 19\/2}, 
      {1\/2, 4, \(-3\), \(-1\), \(-\(3\/2\)\)}, 
      {\(-\(7\/2\)\), \(-3\), 5, \(-\(3\/2\)\), \(-\(17\/2\)\)}, 
      {\(-1\), \(-1\), \(-\(3\/2\)\), 10, \(-6\)}, 
      {19\/2, \(-\(3\/2\)\), \(-\(17\/2\)\), \(-6\), \(-5\)}}\)], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Using matrix diagonalization to understand what happens at a critical point\
\>", "Subsubsection",
  CellDingbat->None],

Cell["\<\
Here we use the diagonalization procedure to illustrate the theory behind the \
(multivariate) second derivative test. The idea is simple. Suppose you know a \
critical point of a function and you have found the Hessian matrix at that \
point. Then you can obtain a very simple polynomial function, of degree two, \
that closely approximates the function of interest near the critical point. I \
will illustrate with a function of two variables.\
\>", "Text"],

Cell[BoxData[
    \(f[x_, y_]\  := \ 
      x*E^y\  - \ x^4\  + \ 5*x*y\  - y*Sin[x]\  + \ 4*y^2\  - \ 
        Cos[3*x^2\  + \ 2*x*y\  - \ y^3 - 2]\)], "Input"],

Cell["\<\
\"Holy Terrors!,\" you cringe. \"Surely we can do nothing useful with this \
function.\"
\"Yes we can,\" sez I, \"and don't call me Shirley.\"\
\>", "Text"],

Cell[BoxData[{
    \(\(vars\  = \ {x, y};\)\), "\[IndentingNewLine]", 
    \(\(derivs\  = \ Map[D[f[x, y], #] &, \ vars];\)\)}], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(critpoint\  = \ 
      FindRoot[Evaluate[Thread[derivs \[Equal] 0]], {x, 0}, \ {y, 0}, \ 
        MaxIterations \[Rule] 100]\)], "Input"],

Cell[BoxData[
    \({x \[Rule] 0.14525741155893446`, y \[Rule] \(-0.057288593163086994`\)}
      \)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(cpcoords\  = \ {x, y}\  /. \ critpoint\)], "Input"],

Cell[BoxData[
    \({0.14525741155893446`, \(-0.057288593163086994`\)}\)], "Output"]
}, Open  ]],

Cell["\<\
Actually if I just give it random values in some range then sometimes \
Newton's method will converge in a few iterations, sometimes not. Moreover it \
sometimes gives me the same root as above and sometimes gives different ones \
e.g. as below.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(critpointtoo\  = \ 
      FindRoot[Evaluate[Thread[derivs \[Equal] 0]], \ {x, 25*Random[]}, \ 
        {y, 25*Random[]}, MaxIterations \[Rule] 100]\)], "Input"],

Cell[BoxData[
    \({x \[Rule] \(-1.5991822036182513`\), y \[Rule] \(-5.481645972877704`\)}
      \)], "Output"]
}, Open  ]],

Cell[TextData[{
  "I'll work with ",
  Cell[BoxData[
      \(critpoint\)]],
  ". First we'll plot ",
  Cell[BoxData[
      \(f[x, y]\)]],
  " in a neighborhood of that point."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Plot3D[f[x, y], \ {x,  .1,  .2}, \ {y, \(- .1\), 0}, \ 
        AxesLabel \[Rule] {"\<x  \>", "\<    y\>", \ "\<z   \>"}, \ 
        Shading \[Rule] False];\)\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
1.11022e-016 1.04977 -0.0679587 1.04977 [
[.02757 .24618 -18 -8.70276 ]
[.02757 .24618 0 .29724 ]
[.14598 .19958 -23.848 -9 ]
[.14598 .19958 .15205 0 ]
[.27013 .15073 -22.8472 -9 ]
[.27013 .15073 1.15281 0 ]
[.40046 .09948 -21.8464 -9 ]
[.40046 .09948 2.15358 0 ]
[.53743 .04564 -20.8457 -9 ]
[.53743 .04564 3.15434 0 ]
[.68156 -0.01098 -14.8837 -9 ]
[.68156 -0.01098 3.11632 0 ]
[.29165 .07573 -20.4846 -12.5 ]
[.29165 .07573 1.51543 0 ]
[.70096 -0.00478 0 -6.26206 ]
[.70096 -0.00478 24 2.73794 ]
[.76745 .09703 0 -6.16187 ]
[.76745 .09703 30 2.83813 ]
[.82812 .18993 0 -6.07246 ]
[.82812 .18993 30 2.92754 ]
[.88369 .27502 0 -5.99218 ]
[.88369 .27502 30 3.00782 ]
[.93478 .35327 0 -5.9197 ]
[.93478 .35327 30 3.0803 ]
[.98191 .42546 0 -5.85393 ]
[.98191 .42546 6 3.14607 ]
[.91861 .21225 0 -8.37676 ]
[.91861 .21225 34 4.12324 ]
[.02241 .27581 -24 -2.75136 ]
[.02241 .27581 0 6.24864 ]
[.00942 .3572 -24 -2.82028 ]
[.00942 .3572 0 6.17972 ]
[-0.00427 .44299 -18 -2.89332 ]
[-0.00427 .44299 0 6.10668 ]
[-0.05457 .39542 -28 -3.93527 ]
[-0.05457 .39542 0 8.56473 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0.1)] .02757 .24618 1 .93395 Mshowa
.1552 .20892 m
.15981 .21359 L
s
[(0.12)] .14598 .19958 .98733 1 Mshowa
.27893 .16047 m
.28333 .16533 L
s
[(0.14)] .27013 .15073 .90393 1 Mshowa
.40878 .10962 m
.41294 .11469 L
s
[(0.16)] .40046 .09948 .82054 1 Mshowa
.54521 .0562 m
.54911 .06148 L
s
[(0.18)] .53743 .04564 .73714 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(0.2)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.18558 .19702 m
.18832 .19985 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.3108 .14799 m
.31341 .15094 L
s
.34306 .13535 m
.34564 .13834 L
s
.37572 .12257 m
.37826 .12558 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.51046 .06981 m
.51284 .07294 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
gsave
.29165 .07573 -81.4846 -16.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
81.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.68874 0 m
.68263 .00239 L
s
[(-0.1)] .70096 -0.00478 -1 .39157 Mshowa
.75514 .10158 m
.74899 .10385 L
s
[(-0.08)] .76745 .09703 -1 .3693 Mshowa
.81573 .19425 m
.80953 .19642 L
s
[(-0.06)] .82812 .18993 -1 .34944 Mshowa
.87123 .27915 m
.865 .28122 L
s
[(-0.04)] .88369 .27502 -1 .3316 Mshowa
.92227 .35722 m
.91601 .35919 L
s
[(-0.02)] .93478 .35327 -1 .31549 Mshowa
.96935 .42924 m
.96306 .43113 L
s
[(0)] .98191 .42546 -1 .30087 Mshowa
.125 Mabswid
.70593 .0263 m
.70226 .02771 L
s
.72272 .05198 m
.71904 .05338 L
s
.73912 .07706 m
.73543 .07845 L
s
.7708 .12553 m
.7671 .12688 L
s
.78611 .14895 m
.78241 .15028 L
s
.80108 .17185 m
.79737 .17317 L
s
.83006 .21617 m
.82633 .21745 L
s
.84407 .23761 m
.84035 .23888 L
s
.8578 .2586 m
.85407 .25986 L
s
.88439 .29928 m
.88065 .3005 L
s
.89727 .31899 m
.89353 .3202 L
s
.9099 .3383 m
.90615 .3395 L
s
.93439 .37576 m
.93063 .37693 L
s
.94627 .39394 m
.94251 .3951 L
s
.95792 .41176 m
.95416 .41291 L
s
gsave
.91861 .21225 -61 -12.3768 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
87.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(y) show
93.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.03464 .27106 m
.04075 .26868 L
s
[(0.48)] .02241 .27581 1 -0.38859 Mshowa
.02171 .35261 m
.02786 .35032 L
s
[(0.49)] .00942 .3572 1 -0.37327 Mshowa
.00809 .43858 m
.01427 .43637 L
s
[(0.5)] -0.00427 .44299 1 -0.35704 Mshowa
.125 Mabswid
.03211 .28703 m
.03578 .28562 L
s
.02955 .30317 m
.03322 .30177 L
s
.02696 .31948 m
.03064 .31808 L
s
.02435 .33596 m
.02804 .33457 L
s
.01905 .36944 m
.02274 .36808 L
s
.01635 .38645 m
.02005 .38509 L
s
.01363 .40364 m
.01733 .4023 L
s
.01087 .42102 m
.01458 .41968 L
s
.03714 .25525 m
.04081 .25382 L
s
.00528 .45634 m
.00899 .45502 L
s
.00243 .47429 m
.00615 .47298 L
s
gsave
-0.05457 .39542 -89 -7.93527 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(z) show
87.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
1 g
.38069 .64164 .40256 .67713 .43954 .68938 .41773 .65229 Metetra
1 g
.41773 .65229 .43954 .68938 .47757 .69964 .45582 .66092 Metetra
1 g
.45582 .66092 .47757 .69964 .51663 .7078 .49492 .66741 Metetra
1 g
.49492 .66741 .51663 .7078 .55666 .71374 .53499 .67165 Metetra
1 g
.53499 .67165 .55666 .71374 .59762 .71733 .57599 .67352 Metetra
1 g
.57599 .67352 .59762 .71733 .63945 .71848 .61786 .67291 Metetra
1 g
.61786 .67291 .63945 .71848 .6821 .71706 .66052 .66972 Metetra
1 g
.66052 .66972 .6821 .71706 .72548 .71297 .70392 .66386 Metetra
1 g
.70392 .66386 .72548 .71297 .76953 .70613 .74797 .65524 Metetra
1 g
.74797 .65524 .76953 .70613 .81416 .69643 .79259 .64377 Metetra
1 g
.79259 .64377 .81416 .69643 .85927 .68381 .83767 .62938 Metetra
1 g
.83767 .62938 .85927 .68381 .90477 .6682 .88313 .61203 Metetra
1 g
.88313 .61203 .90477 .6682 .95054 .64954 .92885 .59165 Metetra
1 g
.92885 .59165 .95054 .64954 .99649 .6278 .97472 .56823 Metetra
1 g
.35831 .60864 .38069 .64164 .41773 .65229 .39544 .61774 Metetra
1 g
.39544 .61774 .41773 .65229 .45582 .66092 .43363 .62477 Metetra
1 g
.43363 .62477 .45582 .66092 .49492 .66741 .47283 .62963 Metetra
1 g
.47283 .62963 .49492 .66741 .53499 .67165 .513 .6322 Metetra
1 g
.513 .6322 .53499 .67165 .57599 .67352 .55408 .63238 Metetra
1 g
.55408 .63238 .57599 .67352 .61786 .67291 .59603 .63005 Metetra
1 g
.59603 .63005 .61786 .67291 .66052 .66972 .63878 .62514 Metetra
1 g
.63878 .62514 .66052 .66972 .70392 .66386 .68225 .61753 Metetra
1 g
.68225 .61753 .70392 .66386 .74797 .65524 .72635 .60716 Metetra
1 g
.72635 .60716 .74797 .65524 .79259 .64377 .77102 .59394 Metetra
1 g
.77102 .59394 .79259 .64377 .83767 .62938 .81614 .57782 Metetra
1 g
.81614 .57782 .83767 .62938 .88313 .61203 .86162 .55874 Metetra
1 g
.86162 .55874 .88313 .61203 .92885 .59165 .90735 .53667 Metetra
1 g
.90735 .53667 .92885 .59165 .97472 .56823 .95321 .51158 Metetra
1 g
.33533 .57809 .35831 .60864 .39544 .61774 .3726 .58566 Metetra
1 g
.3726 .58566 .39544 .61774 .43363 .62477 .41093 .59113 Metetra
1 g
.41093 .59113 .43363 .62477 .47283 .62963 .45028 .59439 Metetra
1 g
.45028 .59439 .47283 .62963 .513 .6322 .49059 .59533 Metetra
1 g
.49059 .59533 .513 .6322 .55408 .63238 .53182 .59385 Metetra
1 g
.53182 .59385 .55408 .63238 .59603 .63005 .57391 .58984 Metetra
1 g
.57391 .58984 .59603 .63005 .63878 .62514 .61679 .58321 Metetra
1 g
.61679 .58321 .63878 .62514 .68225 .61753 .66038 .57389 Metetra
1 g
.66038 .57389 .68225 .61753 .72635 .60716 .70461 .56179 Metetra
1 g
.70461 .56179 .72635 .60716 .77102 .59394 .74938 .54684 Metetra
1 g
.74938 .54684 .77102 .59394 .81614 .57782 .7946 .529 Metetra
1 g
.7946 .529 .81614 .57782 .86162 .55874 .84017 .50821 Metetra
1 g
.84017 .50821 .86162 .55874 .90735 .53667 .88597 .48445 Metetra
1 g
.88597 .48445 .90735 .53667 .95321 .51158 .93189 .45771 Metetra
1 g
.31167 .54996 .33533 .57809 .3726 .58566 .34913 .55604 Metetra
1 g
.34913 .55604 .3726 .58566 .41093 .59113 .38765 .55998 Metetra
1 g
.38765 .55998 .41093 .59113 .45028 .59439 .42719 .56166 Metetra
1 g
.42719 .56166 .45028 .59439 .49059 .59533 .4677 .56099 Metetra
1 g
.4677 .56099 .49059 .59533 .53182 .59385 .50913 .55787 Metetra
1 g
.50913 .55787 .53182 .59385 .57391 .58984 .55141 .55219 Metetra
1 g
.55141 .55219 .57391 .58984 .61679 .58321 .59448 .54388 Metetra
1 g
.59448 .54388 .61679 .58321 .66038 .57389 .63827 .53285 Metetra
1 g
.63827 .53285 .66038 .57389 .70461 .56179 .68267 .51903 Metetra
1 g
.68267 .51903 .70461 .56179 .74938 .54684 .72762 .50237 Metetra
1 g
.72762 .50237 .74938 .54684 .7946 .529 .773 .48281 Metetra
1 g
.773 .48281 .7946 .529 .84017 .50821 .81871 .46033 Metetra
1 g
.81871 .46033 .84017 .50821 .88597 .48445 .86465 .43489 Metetra
1 g
.86465 .43489 .88597 .48445 .93189 .45771 .9107 .40649 Metetra
1 g
.28724 .52425 .31167 .54996 .34913 .55604 .32494 .52886 Metetra
1 g
.32494 .52886 .34913 .55604 .38765 .55998 .3637 .53129 Metetra
1 g
.3637 .53129 .38765 .55998 .42719 .56166 .40348 .53142 Metetra
1 g
.40348 .53142 .42719 .56166 .4677 .56099 .44424 .52915 Metetra
1 g
.44424 .52915 .4677 .56099 .50913 .55787 .48592 .5244 Metetra
1 g
.48592 .5244 .50913 .55787 .55141 .55219 .52846 .51707 Metetra
1 g
.52846 .51707 .55141 .55219 .59448 .54388 .57179 .50708 Metetra
1 g
.57179 .50708 .59448 .54388 .63827 .53285 .61582 .49435 Metetra
1 g
.61582 .49435 .63827 .53285 .68267 .51903 .66047 .47882 Metetra
1 g
.66047 .47882 .68267 .51903 .72762 .50237 .70565 .46045 Metetra
1 g
.70565 .46045 .72762 .50237 .773 .48281 .75126 .43918 Metetra
1 g
.75126 .43918 .773 .48281 .81871 .46033 .79719 .41499 Metetra
1 g
.79719 .41499 .81871 .46033 .86465 .43489 .84333 .38786 Metetra
1 g
.84333 .38786 .86465 .43489 .9107 .40649 .88956 .3578 Metetra
1 g
.26195 .50096 .28724 .52425 .32494 .52886 .29993 .50412 Metetra
1 g
.29993 .50412 .32494 .52886 .3637 .53129 .33898 .50505 Metetra
1 g
.33898 .50505 .3637 .53129 .40348 .53142 .37906 .50363 Metetra
1 g
.37906 .50363 .40348 .53142 .44424 .52915 .42013 .49979 Metetra
1 g
.42013 .49979 .44424 .52915 .48592 .5244 .46213 .49341 Metetra
1 g
.46213 .49341 .48592 .5244 .52846 .51707 .50498 .48443 Metetra
1 g
.50498 .48443 .52846 .51707 .57179 .50708 .54862 .47276 Metetra
1 g
.54862 .47276 .57179 .50708 .61582 .49435 .59296 .45833 Metetra
1 g
.59296 .45833 .61582 .49435 .66047 .47882 .63792 .44109 Metetra
1 g
.63792 .44109 .66047 .47882 .70565 .46045 .6834 .421 Metetra
1 g
.6834 .421 .70565 .46045 .75126 .43918 .72931 .398 Metetra
1 g
.72931 .398 .75126 .43918 .79719 .41499 .77552 .3721 Metetra
1 g
.77552 .3721 .79719 .41499 .84333 .38786 .82193 .34327 Metetra
1 g
.82193 .34327 .84333 .38786 .88956 .3578 .86842 .31154 Metetra
1 g
.23569 .48011 .26195 .50096 .29993 .50412 .274 .48183 Metetra
1 g
.274 .48183 .29993 .50412 .33898 .50505 .3134 .48127 Metetra
1 g
.3134 .48127 .33898 .50505 .37906 .50363 .35384 .47832 Metetra
1 g
.35384 .47832 .37906 .50363 .42013 .49979 .39527 .47289 Metetra
1 g
.39527 .47289 .42013 .49979 .46213 .49341 .43764 .46489 Metetra
1 g
.43764 .46489 .46213 .49341 .50498 .48443 .48087 .45425 Metetra
1 g
.48087 .45425 .50498 .48443 .54862 .47276 .52489 .4409 Metetra
1 g
.52489 .4409 .54862 .47276 .59296 .45833 .56961 .42476 Metetra
1 g
.56961 .42476 .59296 .45833 .63792 .44109 .61495 .4058 Metetra
1 g
.61495 .4058 .63792 .44109 .6834 .421 .6608 .38396 Metetra
1 g
.6608 .38396 .6834 .421 .72931 .398 .70707 .35923 Metetra
1 g
.70707 .35923 .72931 .398 .77552 .3721 .75363 .33159 Metetra
1 g
.75363 .33159 .77552 .3721 .82193 .34327 .80039 .30105 Metetra
1 g
.80039 .30105 .82193 .34327 .86842 .31154 .8472 .26761 Metetra
1 g
.20835 .46174 .23569 .48011 .274 .48183 .24704 .46203 Metetra
1 g
.24704 .46203 .274 .48183 .3134 .48127 .28684 .45997 Metetra
1 g
.28684 .45997 .3134 .48127 .35384 .47832 .3277 .45548 Metetra
1 g
.3277 .45548 .35384 .47832 .39527 .47289 .36956 .44846 Metetra
1 g
.36956 .44846 .39527 .47289 .43764 .46489 .41236 .43884 Metetra
1 g
.41236 .43884 .43764 .46489 .48087 .45425 .45604 .42653 Metetra
1 g
.45604 .42653 .48087 .45425 .52489 .4409 .5005 .41148 Metetra
1 g
.5005 .41148 .52489 .4409 .56961 .42476 .54567 .39361 Metetra
1 g
.54567 .39361 .56961 .42476 .61495 .4058 .59146 .3729 Metetra
1 g
.59146 .3729 .61495 .4058 .6608 .38396 .63775 .34931 Metetra
1 g
.63775 .34931 .6608 .38396 .70707 .35923 .68446 .32281 Metetra
1 g
.68446 .32281 .70707 .35923 .75363 .33159 .73145 .29341 Metetra
1 g
.73145 .29341 .75363 .33159 .80039 .30105 .77862 .2611 Metetra
1 g
.77862 .2611 .80039 .30105 .8472 .26761 .82584 .22593 Metetra
1 g
.17979 .4459 .20835 .46174 .24704 .46203 .21893 .44474 Metetra
1 g
.21893 .44474 .24704 .46203 .28684 .45997 .25919 .44119 Metetra
1 g
.25919 .44119 .28684 .45997 .3277 .45548 .30052 .43515 Metetra
1 g
.30052 .43515 .3277 .45548 .36956 .44846 .34288 .42653 Metetra
1 g
.34288 .42653 .36956 .44846 .41236 .43884 .38618 .41527 Metetra
1 g
.38618 .41527 .41236 .43884 .45604 .42653 .43037 .40127 Metetra
1 g
.43037 .40127 .45604 .42653 .5005 .41148 .47536 .38449 Metetra
1 g
.47536 .38449 .5005 .41148 .54567 .39361 .52105 .36488 Metetra
1 g
.52105 .36488 .54567 .39361 .59146 .3729 .56736 .34239 Metetra
1 g
.56736 .34239 .59146 .3729 .63775 .34931 .61418 .317 Metetra
1 g
.61418 .317 .63775 .34931 .68446 .32281 .66139 .2887 Metetra
1 g
.66139 .2887 .68446 .32281 .73145 .29341 .70889 .25749 Metetra
1 g
.70889 .25749 .73145 .29341 .77862 .2611 .75655 .22339 Metetra
1 g
.75655 .22339 .77862 .2611 .82584 .22593 .80425 .18644 Metetra
1 g
.14987 .43265 .17979 .4459 .21893 .44474 .18951 .43005 Metetra
1 g
.18951 .43005 .21893 .44474 .25919 .44119 .23029 .42499 Metetra
1 g
.23029 .42499 .25919 .44119 .30052 .43515 .27218 .41738 Metetra
1 g
.27218 .41738 .30052 .43515 .34288 .42653 .31509 .40715 Metetra
1 g
.31509 .40715 .34288 .42653 .38618 .41527 .35898 .39421 Metetra
1 g
.35898 .39421 .38618 .41527 .43037 .40127 .40375 .3785 Metetra
1 g
.40375 .3785 .43037 .40127 .47536 .38449 .44934 .35996 Metetra
1 g
.44934 .35996 .47536 .38449 .52105 .36488 .49563 .33855 Metetra
1 g
.49563 .33855 .52105 .36488 .56736 .34239 .54254 .31424 Metetra
1 g
.54254 .31424 .56736 .34239 .61418 .317 .58996 .28702 Metetra
1 g
.58996 .28702 .61418 .317 .66139 .2887 .63778 .25687 Metetra
1 g
.63778 .25687 .66139 .2887 .70889 .25749 .68586 .2238 Metetra
1 g
.68586 .2238 .70889 .25749 .75655 .22339 .7341 .18786 Metetra
1 g
.7341 .18786 .75655 .22339 .80425 .18644 .78236 .14908 Metetra
1 g
.11842 .4221 .14987 .43265 .18951 .43005 .15863 .41803 Metetra
1 g
.15863 .41803 .18951 .43005 .23029 .42499 .20001 .41145 Metetra
1 g
.20001 .41145 .23029 .42499 .27218 .41738 .24251 .40225 Metetra
1 g
.24251 .40225 .27218 .41738 .31509 .40715 .28607 .39036 Metetra
1 g
.28607 .39036 .31509 .40715 .35898 .39421 .33061 .37571 Metetra
1 g
.33061 .37571 .35898 .39421 .40375 .3785 .37605 .35824 Metetra
1 g
.37605 .35824 .40375 .3785 .44934 .35996 .42231 .33791 Metetra
1 g
.42231 .33791 .44934 .35996 .49563 .33855 .4693 .31466 Metetra
1 g
.4693 .31466 .49563 .33855 .54254 .31424 .5169 .28848 Metetra
1 g
.5169 .28848 .54254 .31424 .58996 .28702 .56501 .25936 Metetra
1 g
.56501 .25936 .58996 .28702 .63778 .25687 .61351 .22731 Metetra
1 g
.61351 .22731 .63778 .25687 .68586 .2238 .66227 .19233 Metetra
1 g
.66227 .19233 .68586 .2238 .7341 .18786 .71117 .15448 Metetra
1 g
.71117 .15448 .7341 .18786 .78236 .14908 .76008 .1138 Metetra
1 g
.08527 .41435 .11842 .4221 .15863 .41803 .12611 .4088 Metetra
1 g
.12611 .4088 .15863 .41803 .20001 .41145 .16816 .40065 Metetra
1 g
.16816 .40065 .20001 .41145 .24251 .40225 .21136 .38983 Metetra
1 g
.21136 .38983 .24251 .40225 .28607 .39036 .25563 .37625 Metetra
1 g
.25563 .37625 .28607 .39036 .33061 .37571 .30091 .35985 Metetra
1 g
.30091 .35985 .33061 .37571 .37605 .35824 .34712 .34058 Metetra
1 g
.34712 .34058 .37605 .35824 .42231 .33791 .39415 .31839 Metetra
1 g
.39415 .31839 .42231 .33791 .4693 .31466 .44191 .29325 Metetra
1 g
.44191 .29325 .4693 .31466 .5169 .28848 .4903 .26514 Metetra
1 g
.4903 .26514 .5169 .28848 .56501 .25936 .53919 .23406 Metetra
1 g
.53919 .23406 .56501 .25936 .61351 .22731 .58847 .20002 Metetra
1 g
.58847 .20002 .61351 .22731 .66227 .19233 .63801 .16306 Metetra
1 g
.63801 .16306 .66227 .19233 .71117 .15448 .68767 .12322 Metetra
1 g
.68767 .12322 .71117 .15448 .76008 .1138 .73731 .08056 Metetra
1 g
.05019 .40954 .08527 .41435 .12611 .4088 .09175 .40247 Metetra
1 g
.09175 .40247 .12611 .4088 .16816 .40065 .13455 .39274 Metetra
1 g
.13455 .39274 .16816 .40065 .21136 .38983 .17853 .38025 Metetra
1 g
.17853 .38025 .21136 .38983 .25563 .37625 .22361 .36493 Metetra
1 g
.22361 .36493 .25563 .37625 .30091 .35985 .26972 .34673 Metetra
1 g
.26972 .34673 .30091 .35985 .34712 .34058 .31678 .32559 Metetra
1 g
.31678 .32559 .34712 .34058 .39415 .31839 .36468 .30147 Metetra
1 g
.36468 .30147 .39415 .31839 .44191 .29325 .41332 .27436 Metetra
1 g
.41332 .27436 .44191 .29325 .4903 .26514 .4626 .24424 Metetra
1 g
.4626 .24424 .4903 .26514 .53919 .23406 .51238 .21112 Metetra
1 g
.51238 .21112 .53919 .23406 .58847 .20002 .56254 .17503 Metetra
1 g
.56254 .17503 .58847 .20002 .63801 .16306 .61295 .13599 Metetra
1 g
.61295 .13599 .63801 .16306 .68767 .12322 .66347 .09408 Metetra
1 g
.66347 .09408 .68767 .12322 .73731 .08056 .71396 .04936 Metetra
1 g
.01296 .40785 .05019 .40954 .09175 .40247 .05532 .39923 Metetra
1 g
.05532 .39923 .09175 .40247 .13455 .39274 .09895 .38786 Metetra
1 g
.09895 .38786 .13455 .39274 .17853 .38025 .1438 .37364 Metetra
1 g
.1438 .37364 .17853 .38025 .22361 .36493 .18979 .35653 Metetra
1 g
.18979 .35653 .22361 .36493 .26972 .34673 .23684 .33645 Metetra
1 g
.23684 .33645 .26972 .34673 .31678 .32559 .28485 .31337 Metetra
1 g
.28485 .31337 .31678 .32559 .36468 .30147 .33373 .28726 Metetra
1 g
.33373 .28726 .36468 .30147 .41332 .27436 .38336 .25809 Metetra
1 g
.38336 .25809 .41332 .27436 .4626 .24424 .43363 .22587 Metetra
1 g
.43363 .22587 .4626 .24424 .51238 .21112 .48441 .19062 Metetra
1 g
.48441 .19062 .51238 .21112 .56254 .17503 .53557 .15237 Metetra
1 g
.53557 .15237 .56254 .17503 .61295 .13599 .58697 .11116 Metetra
1 g
.58697 .11116 .61295 .13599 .66347 .09408 .63846 .06707 Metetra
1 g
.63846 .06707 .66347 .09408 .71396 .04936 .68991 .02018 Metetra
0 g
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.03716 .25514 m
.04196 .25962 L
s
[(0.1)] .02757 .24618 1 .93395 Mshowa
.1552 .20892 m
.15981 .21359 L
s
[(0.12)] .14598 .19958 .98733 1 Mshowa
.27893 .16047 m
.28333 .16533 L
s
[(0.14)] .27013 .15073 .90393 1 Mshowa
.40878 .10962 m
.41294 .11469 L
s
[(0.16)] .40046 .09948 .82054 1 Mshowa
.54521 .0562 m
.54911 .06148 L
s
[(0.18)] .53743 .04564 .73714 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(0.2)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.18558 .19702 m
.18832 .19985 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.3108 .14799 m
.31341 .15094 L
s
.34306 .13535 m
.34564 .13834 L
s
.37572 .12257 m
.37826 .12558 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.51046 .06981 m
.51284 .07294 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
gsave
.29165 .07573 -81.4846 -16.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
81.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 233.563},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jA000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00
ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQ
Ool00?moob5oo`00^Goo0P0017oo0P000goo1000F7oo002hOol01000Oomoo`0027oo00<007ooOol0
FGoo002hOol01000Oomoo`002Goo00<007ooOol0F7oo002hOol01000Oomoo`002Woo00<007ooOol0
Egoo002hOol01000Oomoo`0027oo00@007ooOol005Qoo`00^Goo0P002Woo0P002goo0P0017oo0P00
0goo1@00?Woo003AOol01000Oomoo`002Woo00<007ooOol0?Woo003AOol01000Oomoo`002Woo00<0
07ooOol0?Woo003;Ool40002Ool01000Oomoo`002Woo00<007ooOol0?Woo003AOol01000Oomoo`00
2Woo00<007ooOol0?Woo0036Ool2000:Ool2000:Ool20010Ool00<Aoo`D005Moo`00`Goo0`000goo
0P00Egoo002AOol20004Ool20003Ool50002Ool2000JOol20006Ool00`00Ool0001FOol0091oo`04
001oogoo000:Ool01@00Oomoogoo00000Woo00<007ooOol057oo10001goo00@007ooOol005Eoo`00
T7oo00@007ooOol000Yoo`05001oogooOol00002Ool00`00Oomoo`0BOol2000;Ool200000goo001o
o`1DOol0091oo`04001oogoo000:Ool00`00Oomoo`02Ool2000BOol3000=Ool01@00Ool007oo0000
E7oo002@Ool01000Oomoo`002Woo00D007ooOomoo`00009oo`03001oogoo00aoo`@000ioo`800005
Ool00000Ool0001COol0095oo`8000Yoo`8000Aoo`8000eoo`8000=oo`03001oogoo00]oo`08001o
ogoo001oo`00Ool005=oo`00[goo0`004Woo00D007ooOomoo`00009oo`03001oo`00059oo`00FGoo
0P0000=oo`000000Cgoo0P004goo0P0017oo00L007ooOomoo`00Ool0001AOol005Yoo`03001oo`00
04eoo`<0009oo`03001oogoo00moo`03001oogoo00Aoo`07001oogooOol007oo0000DGoo001KOol0
0`00Oomoo`1:Ool2000FOol00`00Oomoo`05Ool00`00Oomoo`02Ool00`00Ool0001@Ool005Uoo`80
0003Ool0000004Moo`<001Moo`03001oogoo00Ioo`09001oogooOol007oo001oo`0004moo`00Xgoo
0P000Woo00<007ooOol057oo00<007ooOol01goo00@007ooOol000=oo`8004moo`00LGoo0P0017oo
0P000goo1@0000Aoo`00000001]oo`<001Yoo`03001oogoo00Qoo`03001oo`0000Eoo`8004ioo`00
L7oo00@007ooOol000Yoo`05001oogooOol00002Ool00`00Oomoo`0FOol2000LOol00`00Oomoo`09
Ool20006Ool00`00Ool0001=Ool0071oo`04001oogoo000:Ool01@00Oomoogoo00000Woo00<007oo
Ool04goo0`0000=oo`00Ool06Woo00<007oo00002Goo0P0027oo0P00CGoo001`Ool01000Oomoo`00
2Woo00@007ooOomoo`<001=oo`<001ioo`8000=oo`03001oogoo00Eoo`03001oo`0000Uoo`8000io
o`8000Aoo`8000Aoo`8000Aoo`8002Yoo`00L7oo00@007ooOol000Yoo`03001oogoo009oo`03001o
ogoo011oo`80021oo`03001oogoo00Aoo`03001oogoo00=oo`04001oogoo000:Ool00`00Oomoo`0;
Ool01000Oomoo`0027oo00@007ooOol0009oo`04001oogoo000YOol0075oo`8000Yoo`8000Aoo`<0
00eoo`<00003Ool007oo01ioo`03001oogoo00Ioo`05001oogooOol00003Ool00`00Oomoo`07Ool0
0`00Ool0000<Ool01000Oomoo`0027oo00@007ooOol0009oo`04001oogoo000YOol0095oo`8002=o
o`03001oogoo00Qoo`03001oo`0000Aoo`03001oogoo00Ioo`03001oogoo008000Eoo`@0009oo`04
001oogoo0008Ool01000Oomoo`000goo0P00:Woo002>Ool3000TOol00`00Oomoo`0:Ool00`00Oomo
o`04Ool00`00Oomoo`03Ool20004Ool00`00Oomoo`09Ool01000Oomoo`0027oo00@007ooOol0009o
o`04001oogoo000YOol008aoo`@002=oo`8000aoo`03001oo`0000Eoo`03001oogoo009oo`03001o
ogoo00Eoo`03001oogoo00Uoo`8000Yoo`8000Aoo`8002Yoo`00RGoo0`009Woo00<007ooOol02goo
00@007ooOol000Eoo`05001oogooOol00008Ool20018Ool008Moo`8002Qoo`03001oogoo00Yoo`80
00Aoo`03001oogoo009oo`04001oogoo000:Ool00`00Oomoo`16Ool008Aoo`@002Qoo`03001oogoo
00Yoo`03001oogoo00Eoo`07001oogooOol007oo00002goo0P00Agoo001BOol20004Ool20003Ool5
0003Ool3000JOol20004Ool00`00Oomoo`0SOol200000goo001oo`09Ool00`00Oomoo`07Ool00`00
Oomoo`02000=Ool20016Ool0055oo`04001oogoo000:Ool00`00Oomoo`04Ool00`00Oomoo`0FOol3
000[Ool00`00Oomoo`02Ool00`00Oomoo`06Ool00`00Oomoo`08Ool01000Ool000003Goo00<007oo
0000AWoo001AOol01000Oomoo`002Woo00@007ooOomoo`D001Aoo`D002]oo`03001oogoo00Aoo`03
001oogoo00Aoo`03001oogoo00Yoo`03001oo`0000]oo`80009oo`8004Eoo`00DGoo00@007ooOol0
00Yoo`05001oogooOol00002Ool00`00Oomoo`0AOol3000]Ool20008Ool01000Oomoogoo0P0037oo
00@007oo000000Yoo`03001oogoo00=oo`8004Aoo`00DGoo00@007ooOol000Yoo`03001oogoo009o
o`03001oo`00011oo`<002moo`03001oogoo00Uoo`03001oo`0000aoo`8000=oo`03001oogoo00Mo
o`03001oogoo00Eoo`03001oogoo049oo`00DWoo0P002Woo0P001Goo0P003Woo1000;goo00<007oo
Ool02goo00<007ooOol02Woo00<007ooOol00goo0P001goo00<007ooOol01Woo0P00@goo001bOol3
000aOol2000=Ool00`00Ool0000:Ool00`00Oomoo`04Ool20006Ool00`00Oomoo`08Ool20012Ool0
071oo`8003=oo`03001oogoo00]oo`8000=oo`03001oogoo00Ioo`03001oogoo00Eoo`03001oo`00
00=oo`8000]oo`80049oo`00KGoo0`0000=oo`00Ool0<Goo00<007ooOol02goo00<007ooOol017oo
00<007ooOol017oo00<007ooOol01Woo00H007ooOol007oo000>Ool20011Ool006]oo`8003Eoo`80
0003Ool007oo00Yoo`03001oogoo00Ioo`04001oogooOol2000:Ool01000Oomoo`003Woo00@007oo
0000041oo`00J7oo0`00=Woo00<007ooOol00Woo00<007ooOol01Woo0P002Woo00<007oo000037oo
00D007oo001oo`0000]oo`80009oo`80041oo`00=7oo0P0017oo0P000goo1@0000=oo`0000000P00
7Goo0P000Woo00<007ooOol0<goo00<007ooOol017oo00<007ooOol017oo00<007ooOol02goo00<0
07ooOol02goo0P000Woo00<007ooOol027oo00<007ooOol00goo0P00?goo000cOol01000Oomoo`00
2Woo00D007ooOomoo`0001eoo`<003Qoo`8000Qoo`03001oogoo009oo`03001oogoo00Yoo`800003
Ool007oo00Yoo`8000Aoo`03001oogoo00Ioo`03001oogoo00Eoo`8003ioo`00<goo00@007ooOol0
00Yoo`03001oogoo009oo`03001oogoo01Qoo`8003Yoo`03001oogoo00Uoo`04001oo`00000<Ool0
1@00Oomoogoo00002Woo00<007oo000017oo00<007ooOol01Goo00<007ooOol01Woo0P003Woo0P00
17oo0P0017oo0P000goo0`0077oo000cOol01000Oomoo`002Woo00<007ooOol00goo00<007ooOol0
57oo0`0000=oo`00Ool0>7oo00<007ooOol02goo00<007ooOol02Woo0P001Goo00<007ooOol01Goo
0P000Woo00<007ooOol00goo00<007ooOol00goo00<007ooOol027oo0P0037oo00@007ooOol000Qo
o`04001oogoo0002Ool01000Oomoo`006goo000cOol01000Oomoo`002Woo00D007ooOomoo`00009o
o`03001oogoo011oo`<003aoo`8000aoo`800003Ool007oo00Uoo`03001oogoo00Ioo`03001oogoo
00=oo`03001oogoo009oo`03001oogoo00Aoo`03001oogoo008000]oo`03001oo`0000]oo`04001o
ogoo0008Ool01000Oomoo`000Woo00@007ooOol001]oo`00=7oo0P002Woo0P0017oo0P004Goo0P00
?Woo00<007ooOol02goo00<007ooOol00Woo0P001Woo0P002Woo00D007ooOomoo`0000Eoo`03001o
ogoo00Aoo`03001oo`0000ioo`8000Eoo`@0009oo`04001oogoo0008Ool01000Oomoo`000Woo0`00
77oo001FOol30002Ool00`00Oomoo`0jOol00`00Oomoo`0;Ool00`00Oomoo`04Ool20004Ool00`00
Oomoo`0:Ool01000Ool000001Woo00<007ooOol01Goo00<007ooOol037oo00@007oo000000Yoo`04
001oogoo0008Ool01000Oomoo`000goo00<007ooOol04goo0P001Woo001DOol20010Ool200000goo
00000009Ool20009Ool01000Ool000003Woo00<007ooOol01Woo00<007ooOol00goo0P0000=oo`00
Ool02Woo0P000Woo00<007oo00002Woo0P002Woo0P0017oo0`0057oo00<007ooOol017oo001AOol4
0010Ool00`00Oomoo`03Ool00`00Oomoo`05Ool00`00Oomoo`0:Ool00`00Oomoo`0<Ool200000goo
001oo`06Ool00`00Oomoo`02Ool01@00Oomoogoo00002Woo00<007ooOol00goo0P00=7oo00<007oo
Ool00goo001?Ool20012Ool20007Ool00`00Oomoo`03Ool00`00Oomoo`09Ool200000goo001oo`0;
Ool00`00Oomoo`02Ool00`00Oomoo`04Ool01@00Oomoogoo00001Goo00<007ooOol01Woo00<007oo
Ool017oo00<007oo0000<Woo00<007oo000017oo001<Ool30013Ool00`00Oomoo`08Ool200000goo
0000000;Ool00`00Oomoo`02Ool00`00Oomoo`08Ool00`00Oomoo`04Ool00`00Oomoo`03Ool01000
Ool000001Woo00<007ooOol01Goo00<007ooOol01Woo00<007oo0000<7oo00@007ooOol000Aoo`00
BWoo0`00@goo0P003Goo00<007ooOol02Woo00<007ooOol017oo00<007ooOol01Goo0P001goo00<0
07ooOol00goo0P002Goo00<007ooOol00goo00<007ooOol01goo00<007oo0000;goo0`0000=oo`00
00000goo000NOol20004Ool20003Ool5000IOol30003Ool00`00Oomoo`0oOol00`00Oomoo`0;Ool2
00000goo001oo`08Ool20007Ool00`00Oomoo`04Ool00`00Oomoo`08Ool00`00Oomoo`02Ool2000:
Ool00`00Oomoo`02000;Ool3000gOol001eoo`04001oogoo000:Ool00`00Oomoo`0GOol20016Ool2
000=Ool00`00Oomoo`02Ool20006Ool00`00Oomoo`08Ool00`00Oomoo`02Ool00`00Oomoo`0:Ool0
1P00Ool00000Ool000Yoo`03001oo`0000eoo`03001oo`0003Moo`007Goo00@007ooOol000Yoo`03
001oogoo01Aoo`<00003Ool007oo04Aoo`03001oogoo00]oo`8000Ioo`8000Aoo`03001oogoo00Yo
o`04001oo`00000>Ool01@00Oomoogoo00002goo00<007ooOol03Goo00<007oo0000=Woo000MOol0
1000Oomoo`002Woo00<007ooOol04Goo0`00B7oo0P0000=oo`0000002Woo00<007ooOol027oo00@0
07oo000000ioo`03001oogoo00aoo`800005Ool007ooOol00009Ool200000goo001oo`0<Ool01@00
Ool007oo0000=Goo000MOol01000Oomoo`002Woo00<007ooOol03goo0P00BWoo00<007ooOol00goo
00<007ooOol01Goo0P0037oo00<007ooOol037oo0P0000=oo`00Ool02goo00D007ooOomoo`00009o
o`03001oogoo00Ioo`05001oogooOol0000;Ool20002Ool00`00Ool0000eOol001ioo`8000Yoo`80
00ioo`<0009oo`03001oogoo04Ioo`03001oogoo00Eoo`03001oogoo00=oo`03001oogoo00Yoo`80
0003Ool007oo00]oo`03001oogoo009oo`03001oogoo00Qoo`03001oogoo00=oo`03001oo`0000Mo
o`03001oogoo00=oo`03001oogoo00Moo`03001oogoo00=oo`<003Aoo`00>7oo0P00C7oo0P002Goo
0P0000=oo`00000037oo00<007ooOol00Woo00<007ooOol01goo0P001Woo00<007ooOol01Goo0P00
1Woo00<007oo00001Goo0P001Woo00<007ooOol01Woo00<007ooOol017oo00@007ooOol003=oo`00
=Goo1000C7oo00<007ooOol02goo00<007ooOol02Woo0P001Woo00<007ooOol017oo0P0027oo00<0
07ooOol017oo00<007ooOol01goo0P0017oo00<007ooOol01goo00<007ooOol017oo00<007ooOol0
1Woo00<007oo0000<goo000cOol2001?Ool00`00Oomoo`0:Ool200000goo001oo`09Ool00`00Oomo
o`07Ool00`00Oomoo`02Ool00`00Oomoo`09Ool00`00Oomoo`02Ool00`00Oomoo`05Ool300000goo
00000002Ool00`00Oomoo`08Ool00`00Oomoo`03Ool00`00Oomoo`07Ool01000Oomoo`00<Woo000a
Ool2001?Ool2000<Ool00`00Oomoo`02Ool20006Ool2000;Ool01000Ool000003Goo00@007oo0000
00Eoo`<000Aoo`@000aoo`03001oogoo008000]oo`04001oogoo000aOol0035oo`8004ioo`03001o
ogoo00Yoo`8000Ioo`8000Aoo`03001oogoo00aoo`03001oogoo00ioo`03001oogoo009oo`<000Qo
o`8000eoo`03001oo`0000eoo`04001oogoo000=Ool20004Ool20004Ool20005Ool3000>Ool0035o
o`03001oo`0004]oo`800003Ool0000000Qoo`8000Yoo`04001oo`00000=Ool200000goo001oo`0<
Ool200000goo000000020009Ool200000goo0000000=Ool00`00Oomoo`0=Ool01000Oomoo`002goo
00@007ooOol000Qoo`04001oogoo0005Ool00`00Oomoo`0=Ool0031oo`<00003Ool007oo04Moo`80
00Eoo`8000Eoo`03001oogoo00]oo`03001oogoo00aoo`03001oogoo009oo`03001oogoo00Uoo`03
001oo`00008000]oo`07001oogooOol007oo00002Woo0P0000=oo`00Ool03Goo00D007ooOomoo`00
00Yoo`04001oogoo0008Ool01000Oomoo`000Woo1@003Woo000`Ool00`00Oomoo`02Ool00`00Oomo
o`13Ool20009Ool200000goo0000000<Ool200000goo0000000:Ool20006Ool00`00Oomoo`06Ool4
0003Ool00`00Oomoo`06Ool20005Ool20009Ool01@00Oomoogoo000037oo0P0000Eoo`00Oomoo`00
00Aoo`@0009oo`04001oogoo0008Ool01000Oomoo`000Woo00@007ooOol000moo`005goo0P0017oo
0P001Goo0`000Woo0P001Goo00<007ooOol00goo00<007ooOol0@Goo00<007ooOol02goo00<007oo
Ool02Woo0P001Goo00<007ooOol01Goo0P0027oo00<007ooOol017oo10001Woo00<007ooOol00goo
0P001goo00<007oo00001Woo0P001Goo00<007ooOol027oo00D007ooOomoo`0000=oo`03001oogoo
00Moo`04001oogoo0008Ool01000Oomoo`000goo00<007oo00003goo000FOol01000Oomoo`002goo
00@007ooOol0009oo`03001oogoo009oo`03001oogoo00Aoo`03001oogoo03ioo`8000aoo`800003
Ool0000000Qoo`8000Qoo`03001oogoo00=oo`03001oogoo00Uoo`04001oogooOol3000:Ool01@00
Oomoogoo00002Woo00<007oo00000goo0P001goo00<007ooOol01goo00<007ooOol00goo00<007oo
00000P002Goo0P002Woo0P001Goo0P003goo000FOol01000Oomoo`0027oo1@0000Eoo`00Oomoo`00
00Aoo`03001oogoo00Eoo`03001oogoo03]oo`8000aoo`8000Eoo`03001oogoo00=oo`8000]oo`80
0003Ool0000000eoo`@000aoo`04001oo`00000<Ool01@00Ool007oo00002Woo00<007ooOol017oo
0P001Woo00D007ooOomoo`0002eoo`005Woo00@007ooOol000Qoo`04001oogoo0003Ool20005Ool0
0`00Oomoo`06Ool2000jOol2000;Ool20008Ool200000goo0000000?Ool00`00Oomoo`0:Ool30000
0goo001oo`0>Ool00`00Oomoo`0<Ool01000Ool000002goo00<007ooOol00goo00<007ooOol01goo
00D007ooOomoo`0002aoo`005Woo00@007ooOol000Uoo`03001oo`00009oo`04001oogoo0003Ool0
0`00Oomoo`09Ool00`00Oomoo`0eOol20002Ool20007Ool2000<Ool00`00Oomoo`0=Ool200000goo
001oo`07Ool30002Ool200000goo001oo`0;Ool200000goo001oo`0=Ool00`00Ool0000<Ool01@00
Oomoogoo00002Woo00<007ooOol00Woo00<007ooOol0:Goo000GOol2000;Ool20003Ool20004Ool0
0`00Oomoo`0:Ool00`00Oomoo`0bOol20006Ool20003Ool2000<Ool200000goo0000000;Ool20004
Ool00`00Oomoo`02Ool30003Ool20003Ool00`00Oomoo`09Ool01@00Oomoogoo000037oo0P0000Ao
o`00Ool000]oo`04001oo`00000<Ool01000Oomoogoo0P00:Woo000_Ool2000<Ool00`00Oomoo`0_
Ool2000:Ool3000<Ool20005Ool00`00Oomoo`06Ool20007Ool40004Ool20006Ool00`00Oomoo`06
Ool20005Ool00`00Oomoo`08Ool01`00Oomoogoo001oo`0000aoo`03001oogoo00aoo`04001oogoo
Ool2000ZOol002moo`03001oogoo00aoo`03001oogoo02aoo`8000Yoo`80009oo`8000Uoo`8000Qo
o`03001oogoo00=oo`8000Moo`@000Aoo`8000Uoo`03001oogoo00=oo`8000Moo`03001oogoo00Io
o`8000Eoo`03001oo`0000Uoo`<000moo`03001oogoo009oo`03001oogoo02Moo`00;goo00<007oo
Ool03Goo00<007ooOol0:Goo0P002Woo0P001Woo0P001Goo0P002goo0P0000=oo`0000001Woo0`00
17oo00@007oo000000aoo`05001oogooOol0000:Ool00`00Oomoo`03Ool20007Ool01000Oomoo`00
1goo00D007ooOomoo`0000ioo`03001oogoo009oo`03001oogoo02Moo`00;goo00<007ooOol03Woo
00<007ooOol09Woo0`002Goo0P002Woo0P0000=oo`0000003Woo0P001Goo0`001goo0P003Woo00@0
07oo000000aoo`05001oogooOol0000:Ool01000Oomoo`0017oo0P0017oo00<007ooOol02Woo0`00
1Goo00<007ooOol09Woo000^Ool2000AOol00`00Oomoo`0SOol20003Ool40003Ool2000=Ool2000>
Ool20002Ool00`00Oomoo`020008Ool20002Ool00`00Oomoo`0;Ool2000>Ool01000Ool000002goo
00@007ooOol0009oo`8000Moo`03001oogoo00Qoo`05001oogooOol00004Ool2000WOol002ioo`03
001oogoo015oo`03001oogoo021oo`8000Uoo`<000eoo`80009oo`8000Yoo`8000Aoo`<000Qoo`80
00Eoo`03001oogoo00Qoo`80009oo`03001oogoo00aoo`03001oogoo00aoo`03001oogoo008000Uo
o`03001oogoo00Moo`03001oogoo009oo`03001oogoo00=oo`03001oogoo02Eoo`00;Woo00<007oo
Ool04Woo00<007ooOol07Goo0P002Goo0P000Woo0P002Woo0P001Woo0P001Woo0P000goo0`000Woo
0P001Goo0P001goo00<007ooOol01Woo0P001Goo00<007ooOol02Goo0P0000=oo`00Ool037oo00<0
07oo00000P002Woo00<007ooOol017oo0P001Woo00<007ooOol00goo00<007ooOol097oo000^Ool0
0`00Oomoo`0COol00`00Oomoo`0JOol20009Ool20006Ool20006Ool2000:Ool20002Ool20002Ool3
0007Ool00`00Oomoo`02000:Ool00`00Oomoo`03Ool20008Ool00`00Oomoo`06Ool20003Ool00`00
Oomoo`0<Ool01000Oomoo`002Woo00<007ooOol00goo00<007ooOol01Woo00<007ooOol017oo00<0
07ooOol08goo000^Ool2000EOol00`00Oomoo`0FOol30009Ool2000:Ool20002Ool2000=Ool30000
17oo000000002Woo0`003Goo00<007ooOol00P002Woo00<007ooOol01Goo00<007ooOol017oo00<0
07ooOol02Goo0P0000Eoo`00Oomoo`0000Yoo`05001oogooOol0000:Ool00`00Oomoo`03Ool00`00
Oomoo`0;Ool20004Ool20004Ool20003Ool40003Ool002ioo`03001oogoo01Eoo`03001oogoo01=o
o`8000=oo`@000=oo`8000eoo`<000eoo`800004Ool00000000;Ool20002Ool00`00Oomoo`0<Ool2
000=Ool00`00Oomoo`02Ool20008Ool00`00Oomoo`06Ool20003Ool01@00Oomoogoo00002Goo00@0
07oo000000]oo`03001oogoo00=oo`8000]oo`04001oogoo0008Ool01000Oomoo`000Woo00<007oo
Ool017oo000]Ool00`00Oomoo`0GOol00`00Oomoo`0@Ool20009Ool3000=Ool20002Ool2000:Ool5
0002Ool20008Ool20005Ool00`00Oomoo`09Ool200000goo001oo`0=Ool01000Ool000002goo00<0
07ooOol017oo00<007ooOol017oo00D007ooOomoo`0000Uoo`03001oogoo00aoo`03001oogoo00Ao
o`03001oogoo00Qoo`04001oogoo0008Ool01000Oomoo`000goo00<007ooOol00goo000]Ool3000H
Ool00`00Oomoo`0=Ool2000:Ool20002Ool30007Ool30006Ool20006Ool40006Ool20005Ool20008
Ool00`00Oomoo`06Ool20004Ool00`00Oomoo`0;Ool2000=Ool00`00Oomoo`02Ool20007Ool01@00
Oomoogoo00001goo0`003Woo00<007ooOol017oo00<007ooOol00Woo10000Woo00@007ooOol000Qo
o`04001oogoo0004Ool00`00Oomoo`02Ool002eoo`03001oogoo01Uoo`03001oogoo00Yoo`8000Yo
o`8000Ioo`<000=oo`8000]oo`80009oo`<000]oo`03001oogoo008000Yoo`03001oogoo00Eoo`03
001oogoo00Eoo`03001oogoo00Qoo`80009oo`03001oogoo00]oo`04001oo`00000:Ool01@00Oomo
ogoo000017oo0P000goo00<007ooOol037oo00<007ooOol017oo00<007ooOol01goo00@007ooOol0
00Qoo`04001oogoo0002Ool01000Oomoo`000goo000DOol20004Ool20005Ool3000017oo00000000
1Goo00<007ooOol06Woo0P001goo10002Goo0P002goo0`003Woo0`003Woo0`003Goo00<007ooOol0
0Woo0P0027oo00<007ooOol01Woo0P0017oo00<007ooOol037oo00<007ooOol02Woo00<007ooOol0
0Woo00@007oo000000Eoo`03001oogoo00aoo`03001oogoo00=oo`8000Yoo`8000Yoo`8000Aoo`80
00Aoo`004goo00@007ooOol000]oo`03001oogoo009oo`03001oogoo00=oo`03001oogoo01aoo`04
001oogooOol30003Ool50003Ool2000:Ool30002Ool2000:Ool30002Ool00`00Oomoo`0:Ool20002
Ool00`00Oomoo`0<Ool01000Ool000002goo00<007ooOol00goo0P001goo00<007ooOol02Goo0P00
00=oo`00Ool02goo00@007ooOomoo`8000Qoo`03001oogoo00Uoo`<000Moo`03001oogoo01moo`00
4goo00@007ooOol000Qoo`D0009oo`<000Aoo`03001oogoo01eoo`<000Yoo`@000Yoo`8000Moo`80
00Ioo`<000Eoo`8000Qoo`8000Eoo`03001oogoo00]oo`8000ioo`03001oogoo008000Uoo`03001o
ogoo00Moo`8000=oo`03001oogoo00Yoo`06001oo`00001oo`0027oo00<007ooOol027oo00D007oo
Oomoo`0000Moo`03001oogoo01ioo`004goo00@007ooOol000Qoo`04001oogoo0002Ool01000Oomo
o`000goo0P007Goo0`002Goo10000goo10000goo0`002goo0P0000Aoo`00000000Uoo`8000Eoo`80
00Qoo`03001oogoo00Moo`<0009oo`03001oogoo00]oo`<000aoo`03001oogoo00Aoo`8000Ioo`03
001oogoo00Yoo`03001oogoo009oo`03001oogoo00Ioo`03001oogoo00Eoo`8000Aoo`03001oogoo
00Eoo`03001oogoo01ioo`004goo00@007ooOol000Uoo`03001oo`00009oo`04001oogoo0003Ool0
0`00Oomoo`0IOol60005Ool4000;Ool3000>Ool4000=Ool00`00Oomoo`02000;Ool00`00Oomoo`04
Ool20006Ool00`00Oomoo`08Ool20002Ool00`00Oomoo`0;Ool01000Oomoogoo0P0027oo00<007oo
Ool027oo0`001Goo00<007ooOol01Goo00<007ooOol00goo0P001goo00<007ooOol017oo0P007goo
000DOol2000;Ool20003Ool30003Ool00`00Oomoo`0EOol40005Ool7000;Ool30003Ool20009Ool4
0002Ool2000<Ool3000>Ool00`00Oomoo`030008Ool00`00Oomoo`06Ool20004Ool00`00Oomoo`0;
Ool01000Ool000002goo00<007ooOol01Goo0P000goo00<007ooOol00Woo00<007ooOol01Woo00D0
07ooOomoo`0000Uoo`03001oogoo00Ioo`03001oogoo01aoo`00;7oo0P004goo0`002Goo0`000goo
1@000goo100027oo0P000goo100027oo00<007ooOol01Woo0`000Woo0P003Goo0`0037oo00<007oo
Ool00goo0P001goo00<007ooOol02Woo0P003Goo00<007ooOol017oo00<007ooOol00goo00<007oo
Ool00goo00<007ooOol01Goo00@007oo000000]oo`03001oogoo00Eoo`03001oogoo01aoo`00;7oo
00<007ooOol03goo1P001Goo10002Woo10003Woo100037oo0P000Woo10001goo00<007ooOol027oo
0P000Woo00<007ooOol02goo00<007ooOol00P002Goo00<007ooOol027oo0P000Woo00<007ooOol0
2goo00@007ooOomoo`8000Moo`03001oogoo00=oo`03001oogoo00Eoo`03001oogoo00]oo`03001o
ogoo00Ioo`03001oogoo01]oo`00;7oo00<007ooOol02goo10001Goo1`002goo0`000goo0P002Goo
100000=oo`0000003Goo0`0037oo00<007ooOol017oo0`001Goo00<007ooOol02Woo0`0037oo00<0
07ooOol01Goo0P0017oo00<007ooOol02goo00@007oo000000Uoo`03001oogoo00Aoo`03001oogoo
009oo`<000ioo`03001oogoo00Ioo`03001oogoo01Yoo`000goo100097oo00<007ooOol02Goo0`00
1Woo1@0017oo1@000goo100027oo0P000goo1@001Woo0P0027oo0`000Woo00<007ooOol02goo0P00
0Woo0P002Goo00<007ooOol01goo0P000Woo00<007ooOol02goo00<007ooOol00Woo0P001goo00<0
07ooOol02goo00<007ooOol02Woo00<007ooOol00goo00<007ooOol00P000goo00<007ooOol02goo
00<007ooOol01Goo0P0077oo0004Ool00`00Ool0000TOol20006Ool=000=Ool4000=Ool6000;Ool2
0002Ool40006Ool2000<Ool3000<Ool00`00Oomoo`03Ool30005Ool00`00Oomoo`0:Ool00`00Oomo
o`020009Ool00`00Oomoo`09Ool3000<Ool00`00Oomoo`04Ool20005Ool00`00Oomoo`0<Ool00`00
Oomoo`06Ool00`00Oomoo`0IOol000=oo`03001oo`0002Eoo`03001oogoo00D000Aoo`@000Aoo`D0
00Aoo`D000Aoo`@000Aoo`X000eoo`<000]oo`8000Moo`@0009oo`03001oogoo00]oo`03001oogoo
008000Uoo`03001oogoo00Uoo`<000aoo`03001oogoo00Ioo`8000=oo`03001oogoo00Yoo`04001o
ogooOol200000goo001oo`05Ool00`00Oomoo`0;Ool00`00Oomoo`07Ool00`00Oomoo`0HOol000=o
o`@002Aoo`h000Qoo`L000aoo`D0009oo`<000Eoo`8000Qoo`<0009oo`03001oogoo00]oo`03001o
ogoo00@000Moo`03001oogoo00Yoo`<000]oo`03001oogoo00Moo`80009oo`03001oogoo00Yoo`03
001oogoo00Aoo`8000Eoo`03001oogoo00Yoo`04001oo`000003Ool00`00Oomoo`04Ool00`00Oomo
o`09Ool30009Ool20006Ool2000AOol002]oo`03001oogoo00]oo`P000Ioo`D000Aoo`@000Aoo`<0
00Yoo`80009oo`@000Ioo`8000]oo`<000aoo`03001oogoo00Moo`80009oo`03001oogoo00Yoo`03
001oogoo00=oo`<000Aoo`03001oogoo00]oo`04001oogooOol20008Ool00`00Oomoo`0:Ool00`00
Oomoo`04Ool00`00Oomoo`04Ool00`00Oomoo`07Ool01@00Oomoogoo000027oo0P001Goo00@007oo
Ool0011oo`00:goo00<007ooOol047oo10001goo1`001Woo0`0000=oo`0000001goo1P002goo0P00
1goo0`000Woo00<007ooOol02goo00<007ooOol00goo0`0017oo00<007ooOol02goo00<007ooOol0
0P0027oo00<007ooOol02Woo00@007oo000000Yoo`03001oogoo00Qoo`<000Moo`03001oogoo00=o
o`03001oogoo00Eoo`8000Aoo`03001oogoo00Qoo`03001oogoo009oo`04001oogoo000@Ool002Yo
o`8001Ioo`L000Ioo`80009oo`<000Eoo`P000Eoo`8000aoo`03001oogoo00@000Ioo`03001oogoo
00]oo`03001oogoo008000Qoo`03001oogoo00Yoo`<000]oo`03001oogoo00Uoo`8000eoo`03001o
ogoo00Eoo`8000=oo`03001oogoo00Eoo`03001oogoo00=oo`03001oogoo009oo`8000Moo`03001o
ogoo00Moo`03001oogoo009oo`04001oogoo000@Ool002Yoo`03001oogoo01Uoo`@000Ioo`<000Ao
o`@000=oo`8000Yoo`8000Uoo`@000]oo`03001oogoo00Yoo`<000]oo`03001oogoo00Ioo`<0009o
o`03001oogoo00Yoo`03001oogoo00Ioo`80009oo`03001oogoo00Yoo`03001oogoo00Aoo`03001o
ogoo00=oo`03001oogoo00Eoo`03001oogoo00=oo`05001oogooOol00009Ool00`00Oomoo`07Ool2
0003Ool01000Oomoo`0047oo000ZOol00`00Oomoo`0MOol=0009Ool2000:Ool20003Ool40003Ool2
000;Ool00`00Oomoo`05Ool40002Ool00`00Oomoo`0:Ool00`00Oomoo`03Ool20005Ool00`00Oomo
o`0:Ool00`00Oomoo`03Ool30004Ool00`00Oomoo`0;Ool01000Oomoogoo0P001goo00<007ooOol0
1Goo00<007ooOol00goo0`002goo00<007ooOol00goo1@0017oo0P004Goo000ZOol2000NOol50007
Ool2000:Ool20006Ool70009Ool00`00Oomoo`09Ool00`00Oomoo`040007Ool00`00Oomoo`09Ool0
0`00Oomoo`030008Ool00`00Oomoo`0:Ool00`00Oomoo`020008Ool00`00Oomoo`0:Ool01000Ool0
00002Goo00<007ooOol01Woo00<007ooOol00Woo00<007ooOol02goo00<007oo00000`0017oo00<0
07ooOol05Goo000ZOol00`00Oomoo`0JOol30005Ool20007Ool20006Ool:0005Ool00`00Oomoo`09
Ool00`00Oomoo`07Ool4000;Ool00`00Oomoo`09Ool3000;Ool00`00Oomoo`0:Ool3000:Ool00`00
Oomoo`0:Ool2000<Ool00`00Oomoo`06Ool00`00Ool00002000=Ool20008Ool00`00Oomoo`0EOol0
01Moo`8000Aoo`8000Aoo`8000Eoo`03001oogoo01Moo`<000Yoo`l000Eoo`03001oogoo00Qoo`80
00Yoo`8000=oo`@000=oo`03001oogoo00Yoo`03001oogoo00Eoo`<0009oo`03001oogoo00Yoo`03
001oogoo00Ioo`<0009oo`03001oogoo00Uoo`03001oogoo00Moo`800003Ool007oo00]oo`03001o
ogoo00Ioo`8000=oo`03001oogoo00]oo`03001oogoo00Moo`03001oogoo01Aoo`005Woo00@007oo
Ool000Qoo`04001oogoo0003Ool00`00Oomoo`0EOol3000?Ool20007Ool20008Ool20009Ool20009
Ool50008Ool00`00Oomoo`0:Ool00`00Oomoo`040005Ool00`00Oomoo`0;Ool00`00Oomoo`03Ool2
0005Ool00`00Oomoo`09Ool00`00Oomoo`05Ool20004Ool00`00Oomoo`0:Ool00`00Oomoo`03Ool2
00000goo001oo`02Ool00`00Oomoo`0;Ool00`00Oomoo`07Ool00`00Oomoo`0DOol001Ioo`04001o
ogoo000;Ool01000Oomoogoo0P004goo0`0057oo00<007ooOol01Woo00<007ooOol01Woo0P002Woo
00@007ooOomoo`D000=oo`03001oogoo00Qoo`03001oogoo00Uoo`<000Yoo`03001oogoo00Yoo`03
001oogoo00<000Qoo`03001oogoo00Uoo`03001oogoo009oo`8000Ioo`03001oogoo00Yoo`04001o
ogooOol20004Ool01@00Oomoogoo00003Goo00<007ooOol01Woo00<007ooOol057oo000FOol01000
Oomoo`002Goo0P0017oo00<007ooOol03goo0`0067oo0P001goo0P0027oo00<007ooOol017oo1`00
2Goo00<007ooOol027oo00<007ooOol01Goo0`000Woo00<007ooOol02Goo00<007ooOol02Goo0`00
2goo00<007ooOol02Goo00<007ooOol00P002Goo00<007ooOol02Woo0`001Woo00D007ooOomoo`00
00eoo`03001oogoo00Ioo`03001oogoo01Aoo`005Woo00@007ooOol000Uoo`03001oogoo00=oo`03
001oogoo00eoo`8001eooaP000Eoo`03001oogoo00Uoo`03001oogoo00Qoo`03001oogoo00@000Io
o`03001oogoo00Uoo`03001oogoo00Ioo`80009oo`03001oogoo00Yoo`03001oogoo00Qoo`<000]o
o`03001oogoo00Yoo`03001oogoo00Moo`05001oogooOol0000:Ool30008Ool00`00Oomoo`0DOol0
01Moo`8000Yoo`<000=oo`03001oogoo00Yoo`<0021oo`8000Moo`03001oogoo00Ioo`8000Uoo`80
00Yoo`03001oogoo00Ioo`@000]oo`03001oogoo00Qoo`03001oogoo00=oo`<000Aoo`03001oogoo
00Yoo`03001oogoo00Ioo`80009oo`03001oogoo00Yoo`03001oogoo00Moo`<000Yoo`04001oogoo
0008Ool20003Ool00`00Oomoo`05Ool00`00Oomoo`0DOol002Uoo`03001oogoo00Moo`<002Eoo`80
00Ioo`8000Qoo`8000Qoo`8000Yoo`04001oogooOol40003Ool00`00Oomoo`09Ool00`00Oomoo`09
Ool00`00Oomoo`020008Ool00`00Oomoo`0:Ool00`00Oomoo`02Ool30004Ool00`00Oomoo`0:Ool0
0`00Oomoo`05Ool20003Ool00`00Oomoo`08Ool01000Oomoo`001Goo0P001Goo00<007ooOol01Goo
00<007ooOol057oo000XOol30005Ool3000ZOol20005Ool20009Ool20008Ool00`00Oomoo`04Ool7
0008Ool00`00Oomoo`09Ool00`00Oomoo`07Ool4000:Ool00`00Oomoo`0:Ool00`00Oomoo`020008
Ool00`00Oomoo`0:Ool00`00Oomoo`02Ool20005Ool00`00Oomoo`08Ool01000Oomoo`000goo0P00
27oo00<007ooOol017oo00<007ooOol057oo000XOol00`00Oomoo`02Ool3000_Ool50002Ool00`00
Oomoo`04OolC0005Ool00`00Oomoo`08Ool00`00Oomoo`09Ool01000Oomoogoo1@000goo00<007oo
Ool02Goo00<007ooOol02Goo0`002Woo00<007ooOol02Woo00<007ooOol00P0027oo00<007ooOol0
27oo00<007ooOol00`002Woo00<007ooOol01Goo00<007ooOol04goo000XOol00`00Ool00002000c
Ool00`00Oomoo`0:0005Ool00`00Oomoo`06Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomo
o`07Ool50008Ool00`00Oomoo`09Ool00`00Oomoo`06Ool30002Ool00`00Oomoo`09Ool00`00Oomo
o`09Ool3000:Ool00`00Oomoo`09Ool00`00Ool0000=Ool00`00Oomoo`04Ool00`00Oomoo`0COol0
02Qoo`8003Moo`8000Eoo`03001oogoo00Qoo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo
00Uoo`04001oogooOol50003Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`03Ool20005
Ool00`00Oomoo`09Ool00`00Oomoo`06Ool300000goo001oo`0;Ool00`00Oomoo`08Ool3000=Ool0
0`00Oomoo`04Ool00`00Oomoo`0COol002Uoo`03001oogoo03Moo`03001oogoo00=oo`8000Qoo`03
001oogoo00Moo`03001oogoo00Moo`03001oogoo00Moo`D000Qoo`03001oogoo00Uoo`03001oogoo
00Qoo`03001oogoo00<000Qoo`03001oogoo00Uoo`03001oogoo00=oo`8000Eoo`03001oogoo00Uo
o`03001oogoo00Ioo`800004Ool007oo000<Ool00`00Oomoo`04Ool00`00Oomoo`0COol002Yoo`80
03Qoo`03001oogoo00=oo`8000Qoo`03001oogoo00Moo`03001oogoo00Moo`04001oogooOol50003
Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`06Ool4000;Ool00`00Oomoo`09Ool00`00
Oomoo`030007Ool00`00Oomoo`0:Ool00`00Oomoo`03Ool20004Ool2000=Ool00`00Oomoo`03Ool0
0`00Oomoo`0COol002aoo`03001oogoo03Ioo`D0009oo`03001oogoo00Ioo`03001oogoo00Moo`03
001oogoo00=oo`L000Qoo`03001oogoo00Qoo`03001oogoo00Uoo`04001oogooOol50003Ool00`00
Oomoo`0:Ool00`00Oomoo`08Ool3000;Ool00`00Oomoo`09Ool01000Oomoogoo0P001goo0P0037oo
00<007ooOol00goo00<007ooOol04goo000]Ool00`00Oomoo`0fOol00`00Oomoo`0L0005Ool00`00
Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool40008Ool00`00Oomoo`0:Ool00`00Oomoo`06
Ool200000goo001oo`0;Ool00`00Oomoo`0:Ool30009Ool2000=Ool00`00Oomoo`02Ool00`00Oomo
o`0COol002ioo`03001oogoo03Ioo`8000Eoo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo
00Moo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Aoo`<0009oo`03001oogoo00Qoo`03
001oogoo00Yoo`03001oogoo009oo`<000Aoo`03001oogoo00Yoo`03001oogoo00Uoo`03001oogoo
00Yoo`03001oogoo00]oo`03001oogoo00=oo`03001oogoo019oo`00;goo00<007ooOol0=goo00<0
07ooOol00goo0P001goo00<007ooOol01goo00<007ooOol01goo00<007ooOol027oo00<007ooOol0
1goo00<007ooOol010001Goo00<007ooOol02Goo00<007ooOol02Goo00<007ooOol00P001goo00<0
07ooOol02Woo00<007ooOol01goo0`003Goo00<007ooOol027oo0`001Goo00<007ooOol04Woo000`
Ool2000hOol00`00Oomoo`03Ool20007Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`08
Ool00`00Oomoo`04Ool6000:Ool00`00Oomoo`09Ool00`00Oomoo`08Ool3000:Ool00`00Oomoo`0:
Ool00`00Oomoo`04Ool20003Ool00`00Oomoo`0:Ool20007Ool20003Ool00`00Oomoo`02Ool00`00
Oomoo`0BOol0039oo`03001oogoo03Ioo`D0009oo`03001oogoo00Eoo`03001oogoo00Ioo`03001o
ogoo00Moo`03001oogoo00Ioo`P000Eoo`03001oogoo00Uoo`03001oogoo00Qoo`03001oogoo00Eo
o`<00003Ool007oo00Yoo`03001oogoo00Yoo`03001oogoo009oo`8000Eoo`03001oogoo00]oo`03
001oogoo00=oo`8000Eoo`03001oogoo009oo`03001oogoo019oo`00<goo00<007ooOol0=Woo00<0
07ooOol02@000goo00<007ooOol01Woo00<007ooOol01goo00@007ooOomoo`@000=oo`03001oogoo
00Moo`03001oogoo00Uoo`03001oogoo00Uoo`03001oogoo00@000Eoo`03001oogoo00Uoo`03001o
ogoo00Yoo`04001oo`000008Ool00`00Oomoo`0:Ool20002Ool20007Ool00`00Oomoo`02Ool00`00
Oomoo`0BOol003Aoo`03001oogoo03Ioo`03001oogoo00Aoo`04001oogooOolJ0007Ool00`00Oomo
o`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool30009Ool00`00Oomoo`09Ool00`00Oomoo`09Ool3
000:Ool00`00Oomoo`0;Ool30009Ool00`00Oomoo`02Ool00`00Oomoo`0BOol003Eoo`8003Ioo`03
001oogoo00Eoo`03001oogoo00Eoo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Moo`03
001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Aoo`<00003Ool007oo00Yoo`03001oogoo00Uo
o`03001oogoo00Eoo`<00003Ool007oo00]oo`03001oogoo00Uoo`<000]oo`05001oogooOol0000D
Ool003Moo`03001oogoo03Aoo`03001oogoo00Eoo`03001oogoo00Eoo`03001oogoo00Ioo`03001o
ogoo00Ioo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00@000Eoo`03
001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00=oo`8000Eoo`03001oogoo00Uoo`03001oogoo
00Moo`800004Ool007oo000:Ool00`00Oomoo`02Ool00`00Oomoo`0AOol003Qoo`03001oogoo03Ao
o`03001oogoo00Aoo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Qo
o`03001oogoo00Moo`03001oogoo00Moo`@000Uoo`03001oogoo00Uoo`03001oogoo00Uoo`03001o
o`00008000Moo`03001oogoo00Yoo`03001oogoo00Aoo`8000Aoo`03001oo`0000Uoo`03001oogoo
009oo`03001oogoo015oo`00>Goo00<007ooOol0=7oo0`0017oo00<007ooOol01Goo00<007ooOol0
1Woo00<007ooOol01goo00<007ooOol01goo00<007ooOol027oo00@007ooOomoo`D000=oo`03001o
ogoo00Qoo`03001oogoo00Qoo`03001oogoo00Qoo`<000]oo`03001oogoo00Uoo`03001oogoo009o
o`8000Ioo`04001oogoo0009Ool01@00Oomoogoo00004goo000jOol2000eOol00`00Ool000030000
0goo001oo`06Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`04Ool7
0008Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`04Ool300000goo001oo`0;Ool00`00
Oomoo`0:Ool01000Ool000002Goo00<007oo00002Goo00D007ooOomoo`0001=oo`00?7oo00<007oo
Ool0<goo00<007ooOol00Woo1`000goo00<007ooOol01Woo00<007ooOol01Woo00<007ooOol017oo
2P001Goo00<007ooOol027oo00<007ooOol027oo00<007ooOol027oo00<007ooOol010001Goo00<0
07ooOol02Woo00<007ooOol02Goo0P002goo00@007ooOol000Qoo`05001oogooOol0000COol003eo
o`03001oogoo03=oo`03001oogoo00Aoo`04001oogooOolL0005Ool00`00Oomoo`07Ool00`00Oomo
o`08Ool00`00Oomoo`08Ool00`00Oomoo`08Ool30009Ool00`00Oomoo`0:Ool00`00Oomoo`07Ool3
000<Ool01@00Oomoogoo00001goo00D007ooOomoo`0001=oo`00?Woo00<007ooOol0<Woo00<007oo
Ool017oo00<007ooOol01Woo00<007ooOol01Woo00<007ooOol01Woo00<007ooOol01goo00<007oo
Ool027oo00<007ooOol027oo00<007ooOol027oo00<007ooOol017oo0`0000=oo`00Ool02Woo00<0
07ooOol02Woo00<007ooOol017oo0P000goo00<007ooOol02Woo00D007ooOomoo`0000Moo`04001o
ogoo000COol003moo`03001oogoo039oo`03001oogoo00Aoo`03001oogoo00Eoo`03001oogoo00Io
o`03001oogoo00Ioo`03001oogoo00Qoo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Qo
o`03001oogoo00@000Eoo`03001oogoo00Qoo`03001oogoo00Yoo`03001oogoo009oo`8000Eoo`03
001oogoo00Yoo`05001oogooOol00007Ool01000Oomoo`004goo0010Ool2000cOol00`00Oomoo`04
Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`08
Ool00`00Oomoo`08Ool00`00Oomoo`06Ool40009Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool01000
Ool0000027oo00<007ooOol02Goo00<007ooOol00Woo00<007ooOol00Woo0`000goo00<007ooOol0
47oo0012Ool00`00Oomoo`0aOol30004Ool00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`07
Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`08Ool01000Oomoogoo1@000Woo00<007oo
Ool02Goo00<007ooOol027oo00<007ooOol02Goo0`002Woo00<007ooOol02Woo00<007ooOol00Woo
00@007oo000000=oo`04001oogoo000BOol004=oo`03001oogoo035oo`L000Qoo`03001oogoo00Eo
o`03001oogoo00Moo`03001oogoo00Ioo`03001oogoo00Qoo`03001oogoo00Ioo`D000Qoo`03001o
ogoo00Qoo`03001oogoo00Uoo`03001oogoo00Eoo`<00003Ool007oo00Yoo`03001oogoo00Yoo`03
001oogoo009oo`8000Eoo`04001oogoo000BOol004Aoo`03001oogoo031oo`03001oogoo009oo`L0
00=oo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Moo`04001oogoo
Ool50002Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomoo`03Ool20005
Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool01P00Ool00000Ool000Ioo`03001oo`00019oo`00AGoo
0P00<Goo00<007ooOol017oo00@007ooOomoo`T000=oo`03001oogoo00Moo`03001oogoo00Ioo`03
001oogoo00Aoo`L000Qoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Uoo`03001oo`00
008000Moo`03001oogoo00Uoo`03001oogoo00Uoo`<000Aoo`03001oogoo00=oo`03001oo`00019o
o`00Agoo00<007ooOol0;goo00<007ooOol017oo00<007ooOol01Goo00@007ooOomooad000Eoo`03
001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Moo`<000]oo`03001oogoo
00Qoo`03001oogoo00Moo`800003Ool007oo00Eoo`03001oogoo00=oo`80019oo`00B7oo00<007oo
Ool0;goo00<007ooOol017oo00<007ooOol017oo00<007ooOol01Woo00<007ooOol01goo00<007oo
Ool01Woo00<007ooOol01goo00<007ooOol027oo00<007ooOol027oo00<007ooOol027oo00<007oo
Ool017oo0`0000=oo`00Ool02goo00<007ooOol02Goo00<007ooOol017oo0P0017oo00<007ooOol0
0goo00<007ooOol00goo0P004Woo0019Ool00`00Oomoo`0^Ool00`00Oomoo`04Ool00`00Oomoo`05
Ool00`00Oomoo`05Ool00`00Oomoo`07Ool00`00Oomoo`06Ool00`00Oomoo`08Ool00`00Oomoo`07
Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Ool000030005Ool00`00Oomoo`09Ool00`00Oomo
o`09Ool00`00Oomoo`02Ool20006Ool00`00Oomoo`04Ool00`00Oomoo`03Ool2000AOol004Yoo`03
001oogoo02ioo`<000Aoo`03001oogoo00Aoo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo
00Ioo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Ioo`@000Uoo`03
001oogoo00Yoo`03001oogoo00Uoo`04001oo`000009Ool00`00Oomoo`04Ool00`00Oomoo`02Ool2
000AOol004]oo`8002moo`03001oo`0000<00003Ool007oo00Eoo`03001oogoo00Eoo`03001oogoo
00Moo`03001oogoo00Ioo`03001oogoo00Qoo`03001oogoo00Moo`03001oogoo00Qoo`04001oogoo
Ool50002Ool00`00Oomoo`09Ool00`00Oomoo`09Ool00`00Oomoo`08Ool3000;Ool00`00Oomoo`04
Ool00`00Oomoo`03Ool00`00Oomoo`0?Ool004eoo`03001oogoo02aoo`03001oogoo00=oo`D000Ao
o`03001oogoo00Eoo`03001oogoo00Moo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Qo
o`03001oogoo00Ioo`D000Qoo`03001oogoo00Qoo`03001oogoo00Yoo`03001oogoo00Eoo`800003
Ool007oo00]oo`03001oogoo00Eoo`03001oogoo009oo`03001oogoo00moo`00CWoo00<007ooOol0
;7oo00<007ooOol01Goo1`0027oo00<007ooOol01goo00<007ooOol01Woo00<007ooOol01goo00<0
07ooOol01goo00@007ooOomoo`D0009oo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Uo
o`03001oogoo009oo`<000Aoo`03001oogoo00Yoo`03001oogoo00Eoo`04001oogooOol2000@Ool0
04moo`03001oogoo02]oo`03001oogoo00Eoo`03001oogoo009oo`L000=oo`03001oogoo00Moo`03
001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Aoo`L000Qoo`03001oogoo00Qoo`03001oogoo
00Qoo`03001oogoo00Yoo`04001oo`000007Ool00`00Oomoo`0:Ool00`00Oomoo`06Ool00`00Oomo
o`02000@Ool0051oo`8002aoo`03001oogoo00Eoo`03001oogoo00Aoo`04001oogooOolW0004Ool0
0`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`09Ool00`00Oomoo`08Ool3000:Ool00`00Oomo
o`09Ool00`00Oomoo`06Ool4000AOol0059oo`03001oogoo02Yoo`03001oogoo00Aoo`03001oogoo
00Aoo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo
00Qoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Aoo`@00003Ool0
07oo00Yoo`03001oogoo00Yoo`03001oogoo00Aoo`<00003Ool007oo011oo`00Dgoo00<007ooOol0
:Goo00<007ooOol01Goo00<007ooOol017oo00<007ooOol01Goo00<007ooOol01goo00<007ooOol0
1Woo00<007ooOol01goo00<007ooOol01goo00<007ooOol027oo00<007ooOol027oo00<007ooOol0
2Goo00<007oo00000`001Woo00<007ooOol027oo00<007ooOol02Woo00<007ooOol00Woo0P000goo
00<007oo000047oo001DOol00`00Oomoo`0YOol30005Ool00`00Oomoo`04Ool00`00Oomoo`05Ool0
0`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`08Ool0
0`00Oomoo`08Ool00`00Oomoo`07Ool3000:Ool00`00Oomoo`09Ool00`00Oomoo`0:Ool01000Ool0
00001Woo0P0047oo001EOol00`00Oomoo`0YOol00`00Ool0000300000goo001oo`06Ool00`00Oomo
o`05Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomo
o`08Ool00`00Oomoo`08Ool00`00Oomoo`04Ool300000goo001oo`0;Ool00`00Oomoo`08Ool00`00
Oomoo`09Ool30008Ool2000@Ool005Ioo`8002Uoo`03001oogoo00=oo`D000Aoo`03001oogoo00Eo
o`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Qo
o`03001oogoo00Qoo`03001oo`0000<000Eoo`03001oogoo00Uoo`03001oogoo00Qoo`03001oogoo
00Moo`8000]oo`80011oo`00F7oo00<007ooOol09goo00<007ooOol01Goo00<007oo00000`0000=o
o`00Ool01Woo00<007ooOol01Woo00<007ooOol01Woo00<007ooOol01goo00<007ooOol01goo00<0
07ooOol027oo00<007ooOol01Woo10002Goo00<007ooOol02Goo00<007ooOol02Goo00<007ooOol0
17oo0P002Goo10004Woo001IOol00`00Oomoo`0VOol00`00Oomoo`05Ool00`00Oomoo`03Ool50005
Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`08
Ool00`00Oomoo`050002Ool00`00Oomoo`09Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00Oomo
o`02Ool20007Ool4000FOol005Yoo`03001oogoo02Ioo`03001oogoo00Eoo`03001oogoo00Eoo`03
001oo`0000<00003Ool007oo00Moo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Moo`03
001oogoo00Aoo`L000Qoo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Uoo`04001oo`00
0005Ool4000JOol005]oo`8002Moo`03001oogoo00Aoo`03001oogoo00Eoo`03001oogoo00=oo`L0
00=oo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Aoo`X000Aoo`03001oogoo00Qoo`03
001oogoo00Qoo`03001oogoo00Yoo`03001oogoo00Moo`<000Aoo`<001ioo`00GGoo00<007ooOol0
97oo00<007ooOol01Goo00<007ooOol01Goo00<007ooOol01Goo00@007ooOomooad000Aoo`03001o
ogoo00Qoo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Uoo`03001oogoo00Aoo`<000=o
o`@0025oo`00GWoo00<007ooOol097oo0P001Goo00<007ooOol01Goo00<007ooOol01Goo00<007oo
Ool01Woo00<007ooOol01Goo00<007ooOol01goo00<007ooOol027oo00<007ooOol01goo00<007oo
Ool027oo00<007ooOol027oo00<007ooOol02Woo00<007oo00000`000Woo10009Goo001OOol00`00
Oomoo`0SOol00`00Ool000020003Ool00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`05Ool0
0`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`08Ool0
0`00Oomoo`08Ool00`00Oomoo`08Ool30002Ool4000YOol0061oo`8002=oo`03001oogoo009oo`@0
00Moo`03001oogoo00Eoo`03001oogoo00Ioo`03001oogoo00Eoo`03001oogoo00Moo`03001oogoo
00Qoo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00Aoo`@0009oo`<0
02eoo`00HWoo00<007ooOol087oo00<007ooOol01Goo10001Goo00<007ooOol01Goo00<007ooOol0
1Goo00<007ooOol01Woo00<007ooOol01goo00<007ooOol01goo00<007ooOol027oo00<007ooOol0
27oo00<007ooOol027oo00<007oo00000`000Woo1000<7oo001SOol00`00Oomoo`0OOol00`00Oomo
o`09Ool400000goo001oo`06Ool00`00Oomoo`06Ool00`00Oomoo`05Ool00`00Oomoo`07Ool00`00
Oomoo`08Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`06Ool40002Ool4000dOol006Ao
o`03001oogoo01ioo`03001oogoo00eoo`D000Eoo`03001oogoo00Eoo`03001oogoo00Ioo`03001o
ogoo00Moo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Qoo`03001oogoo00D0009oo`@0
03Qoo`00IGoo00<007ooOol07Goo00<007ooOol04Woo100000=oo`00Ool01goo00<007ooOol01Goo
00<007ooOol01goo00<007ooOol027oo00<007ooOol01goo00<007ooOol01Woo1@0017oo0`00?7oo
001VOol2000MOol00`00Oomoo`0FOol70003Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomo
o`07Ool00`00Oomoo`08Ool00`00Oomoo`050005Ool4000oOol006Qoo`03001oogoo01Yoo`03001o
ogoo01eoo`T000=oo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00=oo`L000Ioo`@004=o
o`00JGoo00<007ooOol06Goo00<007ooOol09Woo7P002Goo1000Agoo001ZOol00`00Oomoo`0HOol0
0`00Oomoo`1:Ool3001;Ool006]oo`8001Qoo`03001oogoo04Ioo`@004ioo`00KGoo00<007ooOol0
5Goo00<007ooOol0@Woo1000DWoo001^Ool00`00Oomoo`0DOol00`00Oomoo`0nOol4001FOol006mo
o`03001oogoo01=oo`03001oogoo03Yoo`@005Yoo`00L7oo00<007ooOol04Woo00<007ooOol0=goo
0`00GWoo001aOol2000BOol00`00Oomoo`0cOol4001QOol007=oo`03001oogoo00moo`03001oogoo
02moo`@006Eoo`00M7oo00<007ooOol03Woo00<007ooOol0:goo1000JGoo001eOol00`00Oomoo`0<
Ool00`00Oomoo`0YOol3001]Ool007Ioo`8000aoo`03001oogoo02Eoo`@0071oo`00N7oo00<007oo
Ool02Goo00<007ooOol08Goo1000M7oo001iOol00`00Oomoo`08Ool00`00Oomoo`0MOol4001hOol0
07Yoo`03001oogoo00Moo`03001oogoo01Yoo`<007aoo`00Ngoo0P001goo00<007ooOol05Woo1000
Ogoo001mOol00`00Oomoo`04Ool00`00Oomoo`0BOol40023Ool007ioo`03001oogoo00=oo`03001o
ogoo00ioo`@008Moo`00Ogoo00<007ooOol00Woo00<007ooOol02goo0`00Rgoo0020Ool01@00Oomo
ogoo00002Goo1000SWoo0021Ool200000goo001oo`04Ool4002BOol008=oo`800003Ool000000080
09Ioo`00Q7oo0P00VWoo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo
8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol0
0?moob5oo`00ogoo8Goo003oOolQOol00001\
\>"],
  ImageRangeCache->{{{78.5625, 307.938}, {287.188, 101.375}} -> {-0.572623, 
  0.470956, 0.00412538, 0.00412538}}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(hess\  = \ Outer[D, derivs, vars]\  /. \ critpoint\)], "Input"],

Cell[BoxData[
    \({{\(-6.04200199290318`\), 3.0200091817769277`}, 
      {3.0200091817769277`, 7.788868287400845`}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \({evals, evecs}\  = \ Eigensystem[hess]\)], "Input"],

Cell[BoxData[
    \({{8.419538012387285`, \(-6.672671717889621`\)}, 
      {{\(-0.20442056099733508`\), \(-0.9788831565828144`\)}, 
        {\(-0.9788831565828144`\), 0.2044205609973351`}}}\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "The eigenvalues have different signs, so it is a saddle point. Not exactly \
a surprise, looking at the plot. We will use the eigensystem information to \
construct a simple quadratic function that shows very similar behavior near \
the critical point. Bear in mind that the matrix whose columns are ",
  Cell[BoxData[
      \(eigenvectors\)]],
  " gives a basis change. With respect to this new basis the Hessian at ",
  Cell[BoxData[
      \(critpoint\)]],
  " is a diagonal matrix with the corresponding ",
  Cell[BoxData[
      \(eigenvalues\)]],
  " on the diagonal. In math notation,\n",
  Cell[BoxData[
      \(Hessian = Transpose[eigenvectors] . dmatrix . eigenvectors\)]],
  "\nwhere ",
  Cell[BoxData[
      \(dmatrix\)]],
  " has ",
  Cell[BoxData[
      \(eigenvalues\)]],
  " (in order corresponding to ",
  Cell[BoxData[
      \(eigenvectors\)]],
  ") on the main diagonal\nHence ",
  Cell[BoxData[
      \({x - critpoint\[LeftDoubleBracket]1\[RightDoubleBracket], 
          y - critpoint\[LeftDoubleBracket]2\[RightDoubleBracket]} . Hessian 
        . {x - critpoint\[LeftDoubleBracket]1\[RightDoubleBracket], 
          y - critpoint\[LeftDoubleBracket]2\[RightDoubleBracket]}\)]],
  "\nis equal to\n",
  Cell[BoxData[
      \(\(({x - critpoint\[LeftDoubleBracket]1\[RightDoubleBracket], 
              y - critpoint\[LeftDoubleBracket]2\[RightDoubleBracket]} . 
            Transpose[eigenvectors])\) . dmatrix . 
        \((eigenvectors . 
            {x - critpoint\[LeftDoubleBracket]1\[RightDoubleBracket], 
              y - critpoint\[LeftDoubleBracket]2\[RightDoubleBracket]})\)\)]]
  ,
  "\nSo if we define",
  Cell[BoxData[
      \({newx, newy}\)]],
  " to be\n",
  Cell[BoxData[
      \(eigenvectors . 
        {x - critpoint\[LeftDoubleBracket]1\[RightDoubleBracket], 
          y - critpoint\[LeftDoubleBracket]2\[RightDoubleBracket]}\)]],
  "\nthen the mess above is equal to ",
  Cell[BoxData[
      \(eigenvalue1\ newx\^2 + eigenvalue2\ newy\^2\)]],
  ".\nTo approximate ",
  Cell[BoxData[
      \(f[x, y]\)]],
  " near the critical point, Taylor's formula says add ",
  Cell[BoxData[
      \(f[critpt]\)]],
  " to the above diagonalized quadratic form."
}], "Text"],

Cell["We put this all together in a procedure below.", "Text"],

Cell[BoxData[
    \(newf[x_, y_, \ critpt_, eigvals_, eigvecs_]\  := \ 
      Module[{newx, newy}, \[IndentingNewLine]{newx, newy}\  = \ 
          Expand[eigvecs\  . \ 
              {x - critpt[\([1]\)], 
                y - critpt[\([2]\)]}]; \[IndentingNewLine]f[critpt[\([1]\)], 
            critpt[\([2]\)]]\  + \ eigvals[\([1]\)]*newx^2\  + \ 
          eigvals[\([2]\)]*newy^2\[IndentingNewLine]]\)], "Input"],

Cell[CellGroupData[{

Cell[BoxData[
    \(quadraticfunc\  = \ newf[x, y, cpcoords, evals, evecs]\)], "Input"],

Cell[BoxData[
    \(\(0.4896476240693137`\[InvisibleSpace]\) + 
      8.419538012387285`\ \((\(-0.02638523735177307`\) - 0.20442056099733508`\
\ x - 0.9788831565828144`\ y)\)\^2 - 
      6.672671717889621`\ \((\(0.1539009998970051`\[InvisibleSpace]\) - \
0.9788831565828144`\ x + 0.2044205609973351`\ y)\)\^2\)], "Output"]
}, Open  ]],

Cell[TextData[{
  "To give you an idea of how well the quadratic surface approximates the \
original one I show both in the same plot. Bear in mind that I am no wizard \
with ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphics, so it may be a bit crude.\nI also show a pair of line segments \
emanating from the critical point on the surface, with ",
  Cell[BoxData[
      \(x\)]],
  " and ",
  Cell[BoxData[
      \(y\)]],
  " directions parallel to the eigenvectors. I had to tilt one of them \
upwards by messing with a ",
  Cell[BoxData[
      \(z\)]],
  "-coordinate, just so it would not be obscured by the surfaces, but you get \
the idea. It is clear that one segment points along the upward slope, the \
other along the downslope."
}], "Text"],

Cell[CellGroupData[{

Cell[BoxData[{
    \(\(zval\  = \ f[cpcoords[[1]], cpcoords[[2]]];\)\), "\n", 
    \(\(fullcoordinates\  = Append[cpcoords, zval];\)\), "\n", 
    \(\(l1\  = \ 
        Graphics3D[
          Line[{fullcoordinates, 
              fullcoordinates + 1/20*Append[evecs[[1]],  .4]}]];\)\), "\n", 
    \(\(l2\  = \ 
        Graphics3D[
          Line[{fullcoordinates, 
              fullcoordinates + 1/20*Append[evecs[[2]], 0]}]];\)\), "\n", 
    \(\(p1\  = \ 
        Plot3D[f[x, y], \ {x,  .1,  .2}, \ {y, \(- .1\), 0}, \ 
          AxesLabel \[Rule] {"\<x  \>", "\<    y\>", \ "\<z   \>"}, \ 
          Shading \[Rule] False, \ DisplayFunction \[Rule] Identity];\)\), 
  "\n", 
    \(\(p2\  = \ 
        Plot3D[quadraticfunc, \ {x,  .1,  .2}, \ {y, \(- .1\), 0}, \ 
          DisplayFunction \[Rule] Identity];\)\), "\n", 
    \(\(Show[p1, p2, \ l1, l2, DisplayFunction \[Rule] $DisplayFunction];\)\)}
  ], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics3D
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-1.11022e-016 1.04977 -0.0679587 1.04977 [
[.06115 .23296 -18 -8.79932 ]
[.06115 .23296 0 .20068 ]
[.19944 .17854 -29.2639 -9 ]
[.19944 .17854 .73612 0 ]
[.34555 .12107 -22.2621 -9 ]
[.34555 .12107 1.73792 0 ]
[.50015 .06029 -26.3913 -9 ]
[.50015 .06029 3.60867 0 ]
[.664 -0.00409 -14.973 -9 ]
[.664 -0.00409 3.02697 0 ]
[.29165 .07573 -20.4846 -12.5 ]
[.29165 .07573 1.51543 0 ]
[.7282 .03693 0 -6.22073 ]
[.7282 .03693 24 2.77927 ]
[.79962 .14629 0 -6.11423 ]
[.79962 .14629 36 2.88577 ]
[.86421 .2452 0 -6.02014 ]
[.86421 .2452 30 2.97986 ]
[.92291 .3351 0 -5.93642 ]
[.92291 .3351 36 3.06358 ]
[.97649 .41716 0 -5.86144 ]
[.97649 .41716 6 3.13856 ]
[.91861 .21225 0 -8.37676 ]
[.91861 .21225 34 4.12324 ]
[.01638 .3136 -24 -2.78331 ]
[.01638 .3136 0 6.21669 ]
[.00289 .39813 -18 -2.85508 ]
[.00289 .39813 0 6.14492 ]
[-0.01135 .48736 -24 -2.93125 ]
[-0.01135 .48736 0 6.06875 ]
[-0.05457 .39542 -28 -3.93527 ]
[-0.05457 .39542 0 8.56473 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.07064 .24203 m
.07538 .24656 L
s
[(0.1)] .06115 .23296 1 .9554 Mshowa
.20849 .18805 m
.21301 .19281 L
s
[(0.125)] .19944 .17854 .95093 1 Mshowa
.35408 .13104 m
.35834 .13603 L
s
[(0.15)] .34555 .12107 .85517 1 Mshowa
.50809 .07074 m
.51205 .07596 L
s
[(0.175)] .50015 .06029 .75942 1 Mshowa
.67126 .00684 m
.67489 .01231 L
s
[(0.2)] .664 -0.00409 .66367 1 Mshowa
.125 Mabswid
.09762 .23146 m
.10044 .23421 L
s
.12489 .22079 m
.12768 .22356 L
s
.15245 .20999 m
.15522 .21279 L
s
.18032 .19908 m
.18306 .20191 L
s
.23696 .1769 m
.23965 .17978 L
s
.26576 .16563 m
.26841 .16853 L
s
.29487 .15423 m
.29749 .15716 L
s
.32431 .1427 m
.3269 .14566 L
s
.38418 .11925 m
.38671 .12227 L
s
.41463 .10733 m
.41712 .11038 L
s
.44543 .09527 m
.44788 .09835 L
s
.47658 .08308 m
.479 .08618 L
s
.53996 .05826 m
.5423 .06142 L
s
.57221 .04563 m
.57451 .04882 L
s
.60484 .03285 m
.6071 .03607 L
s
.63785 .01992 m
.64007 .02318 L
s
.04394 .25248 m
.04681 .25517 L
s
gsave
.29165 .07573 -81.4846 -16.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
81.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.71594 .04162 m
.70982 .04396 L
s
[(-0.1)] .7282 .03693 -1 .38238 Mshowa
.78727 .15072 m
.78109 .15293 L
s
[(-0.075)] .79962 .14629 -1 .35872 Mshowa
.85178 .2494 m
.84556 .2515 L
s
[(-0.05)] .86421 .2452 -1 .33781 Mshowa
.91041 .33909 m
.90416 .34108 L
s
[(-0.025)] .92291 .3351 -1 .31921 Mshowa
.96393 .42096 m
.95765 .42286 L
s
[(0)] .97649 .41716 -1 .30254 Mshowa
.125 Mabswid
.7308 .06435 m
.72712 .06574 L
s
.74536 .08661 m
.74167 .08798 L
s
.75961 .10842 m
.75592 .10978 L
s
.77358 .12978 m
.76988 .13112 L
s
.80068 .17124 m
.79697 .17255 L
s
.81383 .19135 m
.81012 .19265 L
s
.82673 .21108 m
.82301 .21236 L
s
.83937 .23042 m
.83565 .2317 L
s
.86395 .26801 m
.86021 .26926 L
s
.87589 .28628 m
.87215 .28752 L
s
.88761 .30421 m
.88387 .30543 L
s
.89912 .32181 m
.89537 .32302 L
s
.9215 .35605 m
.91775 .35724 L
s
.9324 .37271 m
.92864 .37389 L
s
.9431 .38908 m
.93933 .39024 L
s
.95361 .40516 m
.94984 .40631 L
s
.70077 .0184 m
.6971 .01982 L
s
gsave
.91861 .21225 -61 -12.3768 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
87.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(y) show
93.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.02864 .30892 m
.03477 .30658 L
s
[(0.48)] .01638 .3136 1 -0.38149 Mshowa
.01521 .39363 m
.02138 .39137 L
s
[(0.5)] .00289 .39813 1 -0.36554 Mshowa
.00104 .48304 m
.00724 .48088 L
s
[(0.52)] -0.01135 .48736 1 -0.34861 Mshowa
.125 Mabswid
.02535 .32968 m
.02903 .32829 L
s
.02201 .35071 m
.0257 .34933 L
s
.01864 .37202 m
.02233 .37066 L
s
.01174 .41552 m
.01545 .41419 L
s
.00823 .43772 m
.01193 .4364 L
s
.00466 .46023 m
.00837 .45892 L
s
.03188 .28844 m
.03556 .28702 L
s
.03509 .26822 m
.03876 .26679 L
s
gsave
-0.05457 .39542 -89 -7.93527 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(z) show
87.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
.44051 .63811 m .44496 .64615 L p .42553 .65993 L 1 g
F P
0 g
s
.44135 .66611 m .42553 .65993 L p .44496 .64615 L 1 g
F P
0 g
s
.44046 .63998 m .44496 .64615 L p .42538 .66535 L 1 g
F P
0 g
s
.44496 .64615 m .46015 .66699 L .44135 .66611 L p 1 g
F P
0 g
s
.44135 .66611 m .42538 .66535 L p .44496 .64615 L 1 g
F P
0 g
s
.44496 .64615 m .46003 .6734 L .44135 .66611 L p 1 g
F P
0 g
s
.42553 .65993 m .40599 .6262 L .44051 .63811 L p 1 g
F P
0 g
s
.42538 .66535 m .40554 .63921 L .44046 .63998 L p 1 g
F P
0 g
s
.42062 .60542 m .44051 .63811 L .40599 .6262 L p 1 g
F P
0 g
s
.40599 .6262 m .38604 .59502 L .42062 .60542 L p 1 g
F P
0 g
s
.42036 .61407 m .44046 .63998 L .40554 .63921 L p 1 g
F P
0 g
s
.40554 .63921 m .38527 .61415 L .42036 .61407 L p 1 g
F P
0 g
s
.47609 .63961 m .49563 .6675 L .46015 .66699 L p 1 g
F P
0 g
s
.46015 .66699 m .44629 .64798 L p .47609 .63961 L 1 g
F P
0 g
s
.44629 .64798 m .44496 .64615 L p .44649 .64583 L 1 g
F P
0 g
s
.44629 .64798 m .44649 .64583 L .47609 .63961 L 1 g
F
.44705 .63991 m .47609 .63961 L p .44496 .64615 L F P
0 g
s
.44051 .63811 m .44838 .6258 L .4473 .6373 L 1 g
F
.44705 .63991 m .44667 .63981 L p .4473 .6373 L F P
0 g
s
.44667 .63981 m .44051 .63811 L p .4473 .6373 L 1 g
F P
0 g
s
.44449 .64529 m .44051 .63811 L .44705 .63991 L p 1 g
F P
0 g
s
.44838 .6258 m .45129 .62125 L .44705 .63991 L 1 g
F
.43593 .58201 m .45225 .60756 L p .42062 .60542 L F P
0 g
s
.45616 .61282 m .45848 .61593 L p .45375 .62225 L 1 g
F P
0 g
s
.45345 .61931 m .45616 .61282 L .45375 .62225 L 1 g
F
.4504 .62622 m .45616 .61282 L .45028 .62688 L F
.45129 .62125 m .45616 .61282 L .4499 .6274 L F
.44051 .63811 m .42062 .60542 L .4528 .61288 L p F P
0 g
s
.45301 .61293 m .45337 .61291 L p .45305 .61358 L 1 g
F P
0 g
s
.45225 .60756 m .45275 .60833 L p .4528 .61288 L 1 g
F P
0 g
s
.45275 .60833 m .45331 .60922 L p .45301 .61293 L 1 g
F P
0 g
s
.45331 .60922 m .45401 .6101 L p .45324 .61019 L 1 g
F P
0 g
s
.45401 .6101 m .45616 .61282 L .45301 .61293 L p 1 g
F P
0 g
s
.4528 .61288 m .45301 .61293 L p .45255 .61423 L 1 g
F P
0 g
s
.45301 .61293 m .4528 .61288 L p .45275 .60833 L 1 g
F P
0 g
s
.4528 .61288 m .42062 .60542 L p .45225 .60756 L 1 g
F P
0 g
s
.44051 .63811 m .4528 .61288 L .45129 .62125 L 1 g
F
.45301 .61293 m .45108 .61299 L p .45401 .6101 L F P
0 g
s
.45108 .61299 m .42036 .61407 L p .4527 .61025 L 1 g
F P
0 g
s
.4527 .61025 m .45401 .6101 L .45108 .61299 L 1 g
F
.45337 .61291 m .45616 .61282 L p .45129 .62125 L F P
0 g
s
.47609 .63961 m .45616 .61282 L .49266 .61035 L p 1 g
F P
0 g
s
.4725 .58375 m .49266 .61035 L .46166 .61244 L p 1 g
F P
0 g
s
.46166 .61244 m .45616 .61282 L p .46083 .60452 L 1 g
F P
0 g
s
.46083 .60452 m .4725 .58375 L .46166 .61244 L 1 g
F
.45193 .55585 m .45642 .56254 L p .43593 .58201 L F P
0 g
s
.45085 .60537 m .43593 .58201 L .45985 .58491 L p 1 g
F P
0 g
s
.45642 .56254 m .45891 .56626 L p .45875 .58478 L 1 g
F P
0 g
s
.45891 .56626 m .4616 .57025 L p .45985 .58491 L 1 g
F P
0 g
s
.4616 .57025 m .46464 .57401 L p .46095 .57569 L 1 g
F P
0 g
s
.46464 .57401 m .4725 .58375 L .45985 .58491 L p 1 g
F P
0 g
s
.45985 .58491 m .45875 .58478 L p .45891 .56626 L 1 g
F P
0 g
s
.45875 .58478 m .43593 .58201 L p .45642 .56254 L 1 g
F P
0 g
s
.45985 .58491 m .44866 .58593 L p .46464 .57401 L 1 g
F P
0 g
s
.44866 .58593 m .4358 .58711 L p .45482 .57847 L 1 g
F P
0 g
s
.45482 .57847 m .46464 .57401 L .44866 .58593 L 1 g
F
.45985 .58491 m .4725 .58375 L p .46083 .60452 L F P
0 g
s
.49266 .61035 m .4725 .58375 L .50988 .57912 L p 1 g
F P
0 g
s
.4895 .55269 m .50988 .57912 L .4725 .58375 L p 1 g
F P
0 g
s
.4725 .58375 m .46761 .57769 L p .4895 .55269 L 1 g
F P
0 g
s
.46761 .57769 m .46715 .56676 L .4895 .55269 L 1 g
F
.46671 .55604 m .4895 .55269 L p .4616 .57025 L F P
0 g
s
.46865 .52681 m .4712 .53042 L p .46671 .55604 L 1 g
F P
0 g
s
.46671 .55604 m .45193 .55585 L p .46571 .5319 L 1 g
F P
0 g
s
.46571 .5319 m .46865 .52681 L .46671 .55604 L 1 g
F
.4616 .57025 m .45193 .55585 L .46671 .55604 L p F P
0 g
s
.46864 .52731 m .4712 .53042 L p .45188 .55822 L 1 g
F P
0 g
s
.4712 .53042 m .4895 .55269 L .46671 .55604 L p 1 g
F P
0 g
s
.46671 .55604 m .45188 .55822 L p .4712 .53042 L 1 g
F P
0 g
s
.4712 .53042 m .48946 .55633 L .46671 .55604 L p 1 g
F P
0 g
s
.46761 .57769 m .4616 .57025 L p .46715 .56676 L 1 g
F P
0 g
s
.4616 .57025 m .45188 .55822 L .46671 .55604 L p 1 g
F P
0 g
s
.46671 .55604 m .47326 .55612 L p .4616 .57025 L 1 g
F P
0 g
s
.47246 .58644 m .4616 .57025 L p .48946 .55633 L 1 g
F P
0 g
s
.47326 .55612 m .48946 .55633 L p .4616 .57025 L 1 g
F P
0 g
s
.4616 .57025 m .47246 .58644 L .45985 .58491 L p 1 g
F P
0 g
s
.45188 .55822 m .4616 .57025 L p .4358 .58711 L 1 g
F P
0 g
s
.4358 .58711 m .4616 .57025 L .46095 .57569 L 1 g
F
.45331 .60922 m .45085 .60537 L p .45985 .58491 L F P
0 g
s
.45616 .61282 m .45331 .60922 L p .45985 .58491 L 1 g
F P
0 g
s
.45616 .61282 m .45985 .58491 L .46083 .60452 L 1 g
F
.45985 .58491 m .46715 .58579 L p .45331 .60922 L F P
0 g
s
.45331 .60922 m .4358 .58711 L .45985 .58491 L p 1 g
F P
0 g
s
.45615 .61365 m .45331 .60922 L p .47246 .58644 L 1 g
F P
0 g
s
.46715 .58579 m .47246 .58644 L p .45331 .60922 L 1 g
F P
0 g
s
.45129 .62125 m .45255 .61423 L .45301 .61293 L 1 g
F
.44046 .63998 m .45337 .61291 L .45129 .62125 L F
.45331 .60922 m .45615 .61365 L .45301 .61293 L p F P
0 g
s
.4358 .58711 m .45331 .60922 L p .42036 .61407 L 1 g
F P
0 g
s
.42036 .61407 m .45331 .60922 L .45324 .61019 L 1 g
F
.45275 .61422 m .45301 .61293 L .45305 .61358 L F
.45301 .61293 m .45615 .61365 L p .45129 .62125 L F P
0 g
s
.44046 .63998 m .45301 .61293 L .45275 .61422 L 1 g
F
.44046 .63998 m .42036 .61407 L .45301 .61293 L p F P
0 g
s
.45028 .62688 m .45345 .61931 L .45375 .62225 L 1 g
F
.4499 .6274 m .4504 .62622 L .45028 .62688 L F
.44705 .63991 m .44046 .63998 L p .45848 .61593 L F P
0 g
s
.45129 .62125 m .45615 .61365 L .44705 .63991 L 1 g
F
.44046 .63998 m .45129 .62125 L .4499 .6274 L F
.45848 .61593 m .47609 .63961 L .44705 .63991 L p F P
0 g
s
.44496 .64615 m .44449 .64529 L p .44705 .63991 L 1 g
F P
0 g
s
.45615 .61365 m .47598 .64789 L .44705 .63991 L p 1 g
F P
0 g
s
.46003 .6734 m .44496 .64615 L p .47598 .64789 L 1 g
F P
0 g
s
.44705 .63991 m .47598 .64789 L p .44496 .64615 L 1 g
F P
0 g
s
.44496 .64615 m .44046 .63998 L .44705 .63991 L p 1 g
F P
0 g
s
.47598 .64789 m .49548 .68477 L .46003 .6734 L p 1 g
F P
0 g
s
.40031 .57529 m .42062 .60542 L .38604 .59502 L p 1 g
F P
0 g
s
.38604 .59502 m .36563 .56636 L .40031 .57529 L p 1 g
F P
0 g
s
.39979 .58923 m .42036 .61407 L .38527 .61415 L p 1 g
F P
0 g
s
.38527 .61415 m .36451 .59017 L .39979 .58923 L p 1 g
F P
0 g
s
.51242 .63803 m .53179 .66682 L .49563 .6675 L p 1 g
F P
0 g
s
.49563 .6675 m .47609 .63961 L .51242 .63803 L p 1 g
F P
0 g
s
.37949 .54771 m .40031 .57529 L .36563 .56636 L p 1 g
F P
0 g
s
.36563 .56636 m .34467 .54022 L .37949 .54771 L p 1 g
F P
0 g
s
.51235 .65545 m .53185 .69396 L .49548 .68477 L p 1 g
F P
0 g
s
.49548 .68477 m .47598 .64789 L .51235 .65545 L p 1 g
F P
0 g
s
.42062 .60542 m .40031 .57529 L .43593 .58201 L p 1 g
F P
0 g
s
.49266 .61035 m .51242 .63803 L .47609 .63961 L p 1 g
F P
0 g
s
.54941 .63522 m .56863 .66492 L .53179 .66682 L p 1 g
F P
0 g
s
.53179 .66682 m .51242 .63803 L .54941 .63522 L p 1 g
F P
0 g
s
.42036 .61407 m .39979 .58923 L .4358 .58711 L p 1 g
F P
0 g
s
.37872 .56547 m .39979 .58923 L .36451 .59017 L p 1 g
F P
0 g
s
.36451 .59017 m .34322 .56726 L .37872 .56547 L p 1 g
F P
0 g
s
.49258 .61963 m .51235 .65545 L .47598 .64789 L p 1 g
F P
0 g
s
.47598 .64789 m .45615 .61365 L .49258 .61963 L p 1 g
F P
0 g
s
.35811 .52268 m .37949 .54771 L .34467 .54022 L p 1 g
F P
0 g
s
.34467 .54022 m .3231 .51661 L .35811 .52268 L p 1 g
F P
0 g
s
.41526 .55295 m .43593 .58201 L .40031 .57529 L p 1 g
F P
0 g
s
.40031 .57529 m .37949 .54771 L .41526 .55295 L p 1 g
F P
0 g
s
.54959 .66069 m .56908 .70085 L .53185 .69396 L p 1 g
F P
0 g
s
.53185 .69396 m .51235 .65545 L .54959 .66069 L p 1 g
F P
0 g
s
.58705 .63113 m .60611 .66175 L .56863 .66492 L p 1 g
F P
0 g
s
.56863 .66492 m .54941 .63522 L .58705 .63113 L p 1 g
F P
0 g
s
.52984 .60663 m .54941 .63522 L .51242 .63803 L p 1 g
F P
0 g
s
.51242 .63803 m .49266 .61035 L .52984 .60663 L p 1 g
F P
0 g
s
.47246 .58644 m .49258 .61963 L .45615 .61365 L p 1 g
F P
0 g
s
.41495 .56247 m .4358 .58711 L .39979 .58923 L p 1 g
F P
0 g
s
.39979 .58923 m .37872 .56547 L .41495 .56247 L p 1 g
F P
0 g
s
.3571 .54278 m .37872 .56547 L .34322 .56726 L p 1 g
F P
0 g
s
.34322 .56726 m .32136 .54542 L .3571 .54278 L p 1 g
F P
0 g
s
.52986 .62325 m .54959 .66069 L .51235 .65545 L p 1 g
F P
0 g
s
.51235 .65545 m .49258 .61963 L .52986 .62325 L p 1 g
F P
0 g
s
.33607 .50022 m .35811 .52268 L .3231 .51661 L p 1 g
F P
0 g
s
.3231 .51661 m .30084 .49554 L .33607 .50022 L p 1 g
F P
0 g
s
.39406 .52646 m .41526 .55295 L .37949 .54771 L p 1 g
F P
0 g
s
.37949 .54771 m .35811 .52268 L .39406 .52646 L p 1 g
F P
0 g
s
.62532 .62571 m .64422 .65727 L .60611 .66175 L p 1 g
F P
0 g
s
.60611 .66175 m .58705 .63113 L .62532 .62571 L p 1 g
F P
0 g
s
.56767 .60161 m .58705 .63113 L .54941 .63522 L p 1 g
F P
0 g
s
.54941 .63522 m .52984 .60663 L .56767 .60161 L p 1 g
F P
0 g
s
.43593 .58201 m .41526 .55295 L .45193 .55585 L p 1 g
F P
0 g
s
.50988 .57912 m .52984 .60663 L .49266 .61035 L p 1 g
F P
0 g
s
.58763 .66352 m .60713 .70536 L .56908 .70085 L p 1 g
F P
0 g
s
.56908 .70085 m .54959 .66069 L .58763 .66352 L p 1 g
F P
0 g
s
.4358 .58711 m .41495 .56247 L .45188 .55822 L p 1 g
F P
0 g
s
.50984 .58847 m .52986 .62325 L .49258 .61963 L p 1 g
F P
0 g
s
.49258 .61963 m .47246 .58644 L .50984 .58847 L p 1 g
F P
0 g
s
.39358 .5389 m .41495 .56247 L .37872 .56547 L p 1 g
F P
0 g
s
.37872 .56547 m .3571 .54278 L .39358 .5389 L p 1 g
F P
0 g
s
.33488 .52114 m .3571 .54278 L .32136 .54542 L p 1 g
F P
0 g
s
.32136 .54542 m .29887 .52465 L .33488 .52114 L p 1 g
F P
0 g
s
.56795 .62443 m .58763 .66352 L .54959 .66069 L p 1 g
F P
0 g
s
.54959 .66069 m .52986 .62325 L .56795 .62443 L p 1 g
F P
0 g
s
.66419 .61893 m .68293 .65144 L .64422 .65727 L p 1 g
F P
0 g
s
.64422 .65727 m .62532 .62571 L .66419 .61893 L p 1 g
F P
0 g
s
.60612 .59526 m .62532 .62571 L .58705 .63113 L p 1 g
F P
0 g
s
.58705 .63113 m .56767 .60161 L .60612 .59526 L p 1 g
F P
0 g
s
.37226 .50256 m .39406 .52646 L .35811 .52268 L p 1 g
F P
0 g
s
.35811 .52268 m .33607 .50022 L .37226 .50256 L p 1 g
F P
0 g
s
.3133 .48035 m .33607 .50022 L .30084 .49554 L p 1 g
F P
0 g
s
.30084 .49554 m .27779 .47705 L .3133 .48035 L p 1 g
F P
0 g
s
.54792 .57318 m .56767 .60161 L .52984 .60663 L p 1 g
F P
0 g
s
.52984 .60663 m .50988 .57912 L .54792 .57318 L p 1 g
F P
0 g
s
.43093 .52787 m .45193 .55585 L .41526 .55295 L p 1 g
F P
0 g
s
.41526 .55295 m .39406 .52646 L .43093 .52787 L p 1 g
F P
0 g
s
.48946 .55633 m .50984 .58847 L .47246 .58644 L p 1 g
F P
0 g
s
.62643 .66384 m .64594 .7074 L .60713 .70536 L p 1 g
F P
0 g
s
.60713 .70536 m .58763 .66352 L .62643 .66384 L p 1 g
F P
0 g
s
.43076 .53376 m .45188 .55822 L .41495 .56247 L p 1 g
F P
0 g
s
.41495 .56247 m .39358 .5389 L .43076 .53376 L p 1 g
F P
0 g
s
.54802 .58804 m .56795 .62443 L .52986 .62325 L p 1 g
F P
0 g
s
.52986 .62325 m .50984 .58847 L .54802 .58804 L p 1 g
F P
0 g
s
.37163 .51639 m .39358 .5389 L .3571 .54278 L p 1 g
F P
0 g
s
.3571 .54278 m .33488 .52114 L .37163 .51639 L p 1 g
F P
0 g
s
.70363 .61075 m .72221 .64422 L .68293 .65144 L p 1 g
F P
0 g
s
.68293 .65144 m .66419 .61893 L .70363 .61075 L p 1 g
F P
0 g
s
.64518 .58753 m .66419 .61893 L .62532 .62571 L p 1 g
F P
0 g
s
.62532 .62571 m .60612 .59526 L .64518 .58753 L p 1 g
F P
0 g
s
.312 .50058 m .33488 .52114 L .29887 .52465 L p 1 g
F P
0 g
s
.29887 .52465 m .2757 .50494 L .312 .50058 L p 1 g
F P
0 g
s
.60678 .62309 m .62643 .66384 L .58763 .66352 L p 1 g
F P
0 g
s
.58763 .66352 m .56795 .62443 L .60678 .62309 L p 1 g
F P
0 g
s
.58659 .56589 m .60612 .59526 L .56767 .60161 L p 1 g
F P
0 g
s
.56767 .60161 m .54792 .57318 L .58659 .56589 L p 1 g
F P
0 g
s
.52777 .54582 m .54792 .57318 L .50988 .57912 L p 1 g
F P
0 g
s
.50988 .57912 m .4895 .55269 L .52777 .54582 L p 1 g
F P
0 g
s
.40937 .50248 m .43093 .52787 L .39406 .52646 L p 1 g
F P
0 g
s
.39406 .52646 m .37226 .50256 L .40937 .50248 L p 1 g
F P
0 g
s
.45193 .55585 m .43093 .52787 L .46865 .52681 L p 1 g
F P
0 g
s
.34978 .48126 m .37226 .50256 L .33607 .50022 L p 1 g
F P
0 g
s
.33607 .50022 m .3133 .48035 L .34978 .48126 L p 1 g
F P
0 g
s
.45188 .55822 m .43076 .53376 L .46864 .52731 L p 1 g
F P
0 g
s
.52778 .55431 m .54802 .58804 L .50984 .58847 L p 1 g
F P
0 g
s
.50984 .58847 m .48946 .55633 L .52778 .55431 L p 1 g
F P
0 g
s
.2897 .46312 m .3133 .48035 L .27779 .47705 L p 1 g
F P
0 g
s
.27779 .47705 m .25386 .46119 L .2897 .46312 L p 1 g
F P
0 g
s
.66592 .6616 m .68545 .70687 L .64594 .7074 L p 1 g
F P
0 g
s
.64594 .7074 m .62643 .66384 L .66592 .6616 L p 1 g
F P
0 g
s
.4091 .51035 m .43076 .53376 L .39358 .5389 L p 1 g
F P
0 g
s
.39358 .5389 m .37163 .51639 L .4091 .51035 L p 1 g
F P
0 g
s
.74361 .60113 m .76203 .63557 L .72221 .64422 L p 1 g
F P
0 g
s
.72221 .64422 m .70363 .61075 L .74361 .60113 L p 1 g
F P
0 g
s
.58694 .58506 m .60678 .62309 L .56795 .62443 L p 1 g
F P
0 g
s
.56795 .62443 m .54802 .58804 L .58694 .58506 L p 1 g
F P
0 g
s
.68481 .57839 m .70363 .61075 L .66419 .61893 L p 1 g
F P
0 g
s
.66419 .61893 m .64518 .58753 L .68481 .57839 L p 1 g
F P
0 g
s
.62587 .55721 m .64518 .58753 L .60612 .59526 L p 1 g
F P
0 g
s
.60612 .59526 m .58659 .56589 L .62587 .55721 L p 1 g
F P
0 g
s
.34905 .49494 m .37163 .51639 L .33488 .52114 L p 1 g
F P
0 g
s
.33488 .52114 m .312 .50058 L .34905 .49494 L p 1 g
F P
0 g
s
.6463 .61915 m .66592 .6616 L .62643 .66384 L p 1 g
F P
0 g
s
.62643 .66384 m .60678 .62309 L .6463 .61915 L p 1 g
F P
0 g
s
.56668 .53759 m .58659 .56589 L .54792 .57318 L p 1 g
F P
0 g
s
.54792 .57318 m .52777 .54582 L .56668 .53759 L p 1 g
F P
0 g
s
.50718 .51952 m .52777 .54582 L .4895 .55269 L p 1 g
F P
0 g
s
.4895 .55269 m .47348 .53321 L p .50718 .51952 L 1 g
F P
0 g
s
.47348 .53321 m .4712 .53042 L p .47337 .52976 L 1 g
F P
0 g
s
.47348 .53321 m .47337 .52976 L .50718 .51952 L 1 g
F
.47327 .52638 m .50718 .51952 L p .4712 .53042 L F P
0 g
s
.46865 .52681 m .47303 .51875 L .47321 .52463 L 1 g
F
.47327 .52638 m .47266 .52643 L p .47321 .52463 L F P
0 g
s
.47266 .52643 m .46865 .52681 L p .47321 .52463 L 1 g
F P
0 g
s
.4707 .52972 m .46865 .52681 L .47327 .52638 L p 1 g
F P
0 g
s
.47303 .51875 m .47755 .51044 L .47327 .52638 L 1 g
F
.48372 .50025 m .4861 .49426 L .48842 .50578 L F
.47669 .51623 m .4861 .49426 L .47628 .519 L F
.47755 .51044 m .4861 .49426 L .47482 .52058 L F
.44733 .49989 m .47976 .49558 L p .47781 .50106 L F P
0 g
s
.46865 .52681 m .46576 .52316 L p .47781 .50106 L 1 g
F P
0 g
s
.46576 .52316 m .44733 .49989 L p .47781 .50106 L 1 g
F P
0 g
s
.46449 .46889 m .46933 .47467 L p .47976 .49558 L 1 g
F P
0 g
s
.46933 .47467 m .47975 .48715 L p .48087 .49543 L 1 g
F P
0 g
s
.47975 .48715 m .48325 .49107 L p .48041 .49201 L 1 g
F P
0 g
s
.48325 .49107 m .4861 .49426 L .48087 .49543 L p 1 g
F P
0 g
s
.48087 .49543 m .47976 .49558 L p .46933 .47467 L 1 g
F P
0 g
s
.47976 .49558 m .44733 .49989 L p .46157 .47417 L 1 g
F P
0 g
s
.46157 .47417 m .46449 .46889 L .47976 .49558 L 1 g
F
.48087 .49543 m .48223 .49513 L p .48157 .49668 L F P
0 g
s
.47976 .49558 m .48087 .49543 L p .47755 .51044 L 1 g
F P
0 g
s
.47952 .50154 m .48087 .49543 L .48157 .49668 L 1 g
F
.48223 .49513 m .4861 .49426 L p .47755 .51044 L F P
0 g
s
.46865 .52681 m .47976 .49558 L .47755 .51044 L 1 g
F
.46864 .52731 m .48223 .49513 L .47755 .51044 L F
.48087 .49543 m .48609 .49474 L p .47755 .51044 L F P
0 g
s
.46864 .52731 m .48087 .49543 L .47952 .50154 L 1 g
F
.44733 .49989 m .46865 .52681 L .43093 .52787 L p F P
0 g
s
.48087 .49543 m .44726 .50299 L p .48325 .49107 L 1 g
F P
0 g
s
.47975 .48715 m .48609 .49474 L .48087 .49543 L p 1 g
F P
0 g
s
.46447 .47003 m .47975 .48715 L p .44726 .50299 L 1 g
F P
0 g
s
.44726 .50299 m .47975 .48715 L .48041 .49201 L 1 g
F
.46864 .52731 m .44726 .50299 L .48087 .49543 L p F P
0 g
s
.4861 .49426 m .49224 .50162 L p .48842 .50578 L 1 g
F P
0 g
s
.47628 .519 m .48372 .50025 L .48842 .50578 L 1 g
F
.47482 .52058 m .47669 .51623 L .47628 .519 L F
.47327 .52638 m .46864 .52731 L p .49224 .50162 L F P
0 g
s
.47755 .51044 m .48609 .49474 L .47327 .52638 L 1 g
F
.46864 .52731 m .47755 .51044 L .47482 .52058 L F
.49224 .50162 m .50718 .51952 L .47327 .52638 L p F P
0 g
s
.4712 .53042 m .4707 .52972 L p .47327 .52638 L 1 g
F P
0 g
s
.48609 .49474 m .50716 .52321 L .47327 .52638 L p 1 g
F P
0 g
s
.48946 .55633 m .4712 .53042 L p .50716 .52321 L 1 g
F P
0 g
s
.47327 .52638 m .50716 .52321 L p .4712 .53042 L 1 g
F P
0 g
s
.4712 .53042 m .46864 .52731 L .47327 .52638 L p 1 g
F P
0 g
s
.28841 .48108 m .312 .50058 L .2757 .50494 L p 1 g
F P
0 g
s
.2757 .50494 m .25179 .48631 L .28841 .48108 L p 1 g
F P
0 g
s
.43093 .52787 m .40937 .50248 L .44733 .49989 L p 1 g
F P
0 g
s
.50716 .52321 m .52778 .55431 L .48946 .55633 L p 1 g
F P
0 g
s
.7841 .59003 m .80235 .62545 L .76203 .63557 L p 1 g
F P
0 g
s
.76203 .63557 m .74361 .60113 L .7841 .59003 L p 1 g
F P
0 g
s
.38718 .4797 m .40937 .50248 L .37226 .50256 L p 1 g
F P
0 g
s
.37226 .50256 m .34978 .48126 L .38718 .4797 L p 1 g
F P
0 g
s
.70602 .6567 m .72559 .70371 L .68545 .70687 L p 1 g
F P
0 g
s
.68545 .70687 m .66592 .6616 L .70602 .6567 L p 1 g
F P
0 g
s
.56684 .54971 m .58694 .58506 L .54802 .58804 L p 1 g
F P
0 g
s
.54802 .58804 m .52778 .55431 L .56684 .54971 L p 1 g
F P
0 g
s
.72498 .5678 m .74361 .60113 L .70363 .61075 L p 1 g
F P
0 g
s
.70363 .61075 m .68481 .57839 L .72498 .5678 L p 1 g
F P
0 g
s
.44726 .50299 m .46864 .52731 L .43076 .53376 L p 1 g
F P
0 g
s
.43076 .53376 m .4091 .51035 L .44726 .50299 L p 1 g
F P
0 g
s
.32651 .46261 m .34978 .48126 L .3133 .48035 L p 1 g
F P
0 g
s
.3133 .48035 m .2897 .46312 L .32651 .46261 L p 1 g
F P
0 g
s
.66571 .54711 m .68481 .57839 L .64518 .58753 L p 1 g
F P
0 g
s
.64518 .58753 m .62587 .55721 L .66571 .54711 L p 1 g
F P
0 g
s
.62653 .57947 m .6463 .61915 L .60678 .62309 L p 1 g
F P
0 g
s
.60678 .62309 m .58694 .58506 L .62653 .57947 L p 1 g
F P
0 g
s
.6062 .52796 m .62587 .55721 L .58659 .56589 L p 1 g
F P
0 g
s
.58659 .56589 m .56668 .53759 L .6062 .52796 L p 1 g
F P
0 g
s
.38683 .488 m .4091 .51035 L .37163 .51639 L p 1 g
F P
0 g
s
.37163 .51639 m .34905 .49494 L .38683 .488 L p 1 g
F P
0 g
s
.26516 .44859 m .2897 .46312 L .25386 .46119 L p 1 g
F P
0 g
s
.25386 .46119 m .22895 .44803 L .26516 .44859 L p 1 g
F P
0 g
s
.68642 .61256 m .70602 .6567 L .66592 .6616 L p 1 g
F P
0 g
s
.66592 .6616 m .6463 .61915 L .68642 .61256 L p 1 g
F P
0 g
s
.54636 .51034 m .56668 .53759 L .52777 .54582 L p 1 g
F P
0 g
s
.52777 .54582 m .50718 .51952 L .54636 .51034 L p 1 g
F P
0 g
s
.82505 .57743 m .84314 .61383 L .80235 .62545 L p 1 g
F P
0 g
s
.80235 .62545 m .7841 .59003 L .82505 .57743 L p 1 g
F P
0 g
s
.32579 .47455 m .34905 .49494 L .312 .50058 L p 1 g
F P
0 g
s
.312 .50058 m .28841 .48108 L .32579 .47455 L p 1 g
F P
0 g
s
.76566 .55572 m .7841 .59003 L .74361 .60113 L p 1 g
F P
0 g
s
.74361 .60113 m .72498 .5678 L .76566 .55572 L p 1 g
F P
0 g
s
.54641 .51702 m .56684 .54971 L .52778 .55431 L p 1 g
F P
0 g
s
.52778 .55431 m .50716 .52321 L .54641 .51702 L p 1 g
F P
0 g
s
.74668 .64911 m .7663 .69785 L .72559 .70371 L p 1 g
F P
0 g
s
.72559 .70371 m .70602 .6567 L .74668 .64911 L p 1 g
F P
0 g
s
.60656 .54248 m .62653 .57947 L .58694 .58506 L p 1 g
F P
0 g
s
.58694 .58506 m .56684 .54971 L .60656 .54248 L p 1 g
F P
0 g
s
.7061 .53555 m .72498 .5678 L .68481 .57839 L p 1 g
F P
0 g
s
.68481 .57839 m .66571 .54711 L .7061 .53555 L p 1 g
F P
0 g
s
.42544 .4756 m .44733 .49989 L .40937 .50248 L p 1 g
F P
0 g
s
.40937 .50248 m .38718 .4797 L .42544 .4756 L p 1 g
F P
0 g
s
.66672 .5712 m .68642 .61256 L .6463 .61915 L p 1 g
F P
0 g
s
.6463 .61915 m .62653 .57947 L .66672 .5712 L p 1 g
F P
0 g
s
.6463 .51689 m .66571 .54711 L .62587 .55721 L p 1 g
F P
0 g
s
.62587 .55721 m .6062 .52796 L .6463 .51689 L p 1 g
F P
0 g
s
.26404 .46265 m .28841 .48108 L .25179 .48631 L p 1 g
F P
0 g
s
.25179 .48631 m .22707 .46876 L .26404 .46265 L p 1 g
F P
0 g
s
.36426 .45958 m .38718 .4797 L .34978 .48126 L p 1 g
F P
0 g
s
.34978 .48126 m .32651 .46261 L .36426 .45958 L p 1 g
F P
0 g
s
.42531 .4797 m .44726 .50299 L .4091 .51035 L p 1 g
F P
0 g
s
.4091 .51035 m .38683 .488 L .42531 .4797 L p 1 g
F P
0 g
s
.86643 .5633 m .88436 .60068 L .84314 .61383 L p 1 g
F P
0 g
s
.84314 .61383 m .82505 .57743 L .86643 .5633 L p 1 g
F P
0 g
s
.72708 .60326 m .74668 .64911 L .70602 .6567 L p 1 g
F P
0 g
s
.70602 .6567 m .68642 .61256 L .72708 .60326 L p 1 g
F P
0 g
s
.58616 .49975 m .6062 .52796 L .56668 .53759 L p 1 g
F P
0 g
s
.56668 .53759 m .54636 .51034 L .58616 .49975 L p 1 g
F P
0 g
s
.8068 .54214 m .82505 .57743 L .7841 .59003 L p 1 g
F P
0 g
s
.7841 .59003 m .76566 .55572 L .8068 .54214 L p 1 g
F P
0 g
s
.30237 .44666 m .32651 .46261 L .2897 .46312 L p 1 g
F P
0 g
s
.2897 .46312 m .26516 .44859 L .30237 .44666 L p 1 g
F P
0 g
s
.36391 .46669 m .38683 .488 L .34905 .49494 L p 1 g
F P
0 g
s
.34905 .49494 m .32579 .47455 L .36391 .46669 L p 1 g
F P
0 g
s
.747 .52249 m .76566 .55572 L .72498 .5678 L p 1 g
F P
0 g
s
.72498 .5678 m .7061 .53555 L .747 .52249 L p 1 g
F P
0 g
s
.7878 .63876 m .80748 .68924 L .7663 .69785 L p 1 g
F P
0 g
s
.7663 .69785 m .74668 .64911 L .7878 .63876 L p 1 g
F P
0 g
s
.52558 .48413 m .54636 .51034 L .50718 .51952 L p 1 g
F P
0 g
s
.50718 .51952 m .4861 .49426 L .52558 .48413 L p 1 g
F P
0 g
s
.58631 .50816 m .60656 .54248 L .56684 .54971 L p 1 g
F P
0 g
s
.56684 .54971 m .54641 .51702 L .58631 .50816 L p 1 g
F P
0 g
s
.64687 .53257 m .66672 .5712 L .62653 .57947 L p 1 g
F P
0 g
s
.62653 .57947 m .60656 .54248 L .64687 .53257 L p 1 g
F P
0 g
s
.52558 .48695 m .54641 .51702 L .50716 .52321 L p 1 g
F P
0 g
s
.50716 .52321 m .48609 .49474 L .52558 .48695 L p 1 g
F P
0 g
s
.68694 .50435 m .7061 .53555 L .66571 .54711 L p 1 g
F P
0 g
s
.66571 .54711 m .6463 .51689 L .68694 .50435 L p 1 g
F P
0 g
s
.70744 .56023 m .72708 .60326 L .68642 .61256 L p 1 g
F P
0 g
s
.68642 .61256 m .66672 .5712 L .70744 .56023 L p 1 g
F P
0 g
s
.23957 .43686 m .26516 .44859 L .22895 .44803 L p 1 g
F P
0 g
s
.22895 .44803 m .20293 .43766 L .23957 .43686 L p 1 g
F P
0 g
s
.44733 .49989 m .42544 .4756 L .46449 .46889 L p 1 g
F P
0 g
s
.90819 .54759 m .92597 .58597 L .88436 .60068 L p 1 g
F P
0 g
s
.88436 .60068 m .86643 .5633 L .90819 .54759 L p 1 g
F P
0 g
s
.30178 .45522 m .32579 .47455 L .28841 .48108 L p 1 g
F P
0 g
s
.28841 .48108 m .26404 .46265 L .30178 .45522 L p 1 g
F P
0 g
s
.44726 .50299 m .42531 .4797 L .46447 .47003 L p 1 g
F P
0 g
s
.76819 .59121 m .7878 .63876 L .74668 .64911 L p 1 g
F P
0 g
s
.74668 .64911 m .72708 .60326 L .76819 .59121 L p 1 g
F P
0 g
s
.84836 .52701 m .86643 .5633 L .82505 .57743 L p 1 g
F P
0 g
s
.82505 .57743 m .8068 .54214 L .84836 .52701 L p 1 g
F P
0 g
s
.62653 .48771 m .6463 .51689 L .6062 .52796 L p 1 g
F P
0 g
s
.6062 .52796 m .58616 .49975 L .62653 .48771 L p 1 g
F P
0 g
s
.40287 .45396 m .42544 .4756 L .38718 .4797 L p 1 g
F P
0 g
s
.38718 .4797 m .36426 .45958 L .40287 .45396 L p 1 g
F P
0 g
s
.78836 .50792 m .8068 .54214 L .76566 .55572 L p 1 g
F P
0 g
s
.76566 .55572 m .747 .52249 L .78836 .50792 L p 1 g
F P
0 g
s
.8293 .62562 m .84907 .67782 L .80748 .68924 L p 1 g
F P
0 g
s
.80748 .68924 m .7878 .63876 L .8293 .62562 L p 1 g
F P
0 g
s
.40274 .45746 m .42531 .4797 L .38683 .488 L p 1 g
F P
0 g
s
.38683 .488 m .36391 .46669 L .40274 .45746 L p 1 g
F P
0 g
s
.2644 .54716 m
.3906 .48316 L
s
.56568 .47257 m .58616 .49975 L .54636 .51034 L p 1 g
F P
0 g
s
.54636 .51034 m .52558 .48413 L .56568 .47257 L p 1 g
F P
0 g
s
.23883 .44531 m .26404 .46265 L .22707 .46876 L p 1 g
F P
0 g
s
.22707 .46876 m .20148 .45231 L .23883 .44531 L p 1 g
F P
0 g
s
.34051 .44215 m .36426 .45958 L .32651 .46261 L p 1 g
F P
0 g
s
.32651 .46261 m .30237 .44666 L .34051 .44215 L p 1 g
F P
0 g
s
.6877 .51993 m .70744 .56023 L .66672 .5712 L p 1 g
F P
0 g
s
.66672 .5712 m .64687 .53257 L .6877 .51993 L p 1 g
F P
0 g
s
.95028 .5303 m .96791 .56967 L .92597 .58597 L p 1 g
F P
0 g
s
.92597 .58597 m .90819 .54759 L .95028 .5303 L p 1 g
F P
0 g
s
.6268 .49661 m .64687 .53257 L .60656 .54248 L p 1 g
F P
0 g
s
.60656 .54248 m .58631 .50816 L .6268 .49661 L p 1 g
F P
0 g
s
.72808 .49031 m .747 .52249 L .7061 .53555 L p 1 g
F P
0 g
s
.7061 .53555 m .68694 .50435 L .72808 .49031 L p 1 g
F P
0 g
s
.74859 .5465 m .76819 .59121 L .72708 .60326 L p 1 g
F P
0 g
s
.72708 .60326 m .70744 .56023 L .74859 .5465 L p 1 g
F P
0 g
s
.56573 .47648 m .58631 .50816 L .54641 .51702 L p 1 g
F P
0 g
s
.54641 .51702 m .52558 .48695 L .56573 .47648 L p 1 g
F P
0 g
s
.89031 .51032 m .90819 .54759 L .86643 .5633 L p 1 g
F P
0 g
s
.86643 .5633 m .84836 .52701 L .89031 .51032 L p 1 g
F P
0 g
s
.80967 .57638 m .8293 .62562 L .7878 .63876 L p 1 g
F P
0 g
s
.7878 .63876 m .76819 .59121 L .80967 .57638 L p 1 g
F P
0 g
s
.50429 .45894 m .51622 .47305 L p .49931 .46861 L 1 g
F P
0 g
s
.4861 .49426 m .47975 .48715 L p .49116 .4626 L 1 g
F P
0 g
s
.49116 .4626 m .50429 .45894 L p .49931 .46861 L 1 g
F P
0 g
s
.4861 .49426 m .49116 .4626 L .49931 .46861 L 1 g
F
.52558 .48413 m .4861 .49426 L p .49931 .46861 L F P
0 g
s
.51622 .47305 m .52558 .48413 L p .49931 .46861 L 1 g
F P
0 g
s
.48609 .49474 m .47975 .48715 L p .50429 .45951 L 1 g
F P
0 g
s
.47396 .48022 m .46449 .46889 L .49116 .4626 L p 1 g
F P
0 g
s
.48244 .4347 m .48506 .43768 L p .49116 .4626 L 1 g
F P
0 g
s
.48506 .43768 m .49969 .45386 L p .4901 .45826 L 1 g
F P
0 g
s
.49969 .45386 m .50429 .45894 L .49116 .4626 L p 1 g
F P
0 g
s
.49116 .4626 m .4834 .46443 L p .48315 .45669 L 1 g
F P
0 g
s
.48315 .45669 m .48244 .4347 L .49116 .4626 L 1 g
F
.4834 .46443 m .46449 .46889 L p .46884 .4606 L F P
0 g
s
.46884 .4606 m .48244 .4347 L .4834 .46443 L 1 g
F
.49116 .4626 m .46447 .47003 L p .49969 .45386 L F P
0 g
s
.48506 .43768 m .50429 .45951 L .49116 .4626 L p 1 g
F P
0 g
s
.48244 .43477 m .48506 .43768 L p .46447 .47003 L 1 g
F P
0 g
s
.46447 .47003 m .48506 .43768 L .4901 .45826 L 1 g
F
.47975 .48715 m .47396 .48022 L p .49116 .4626 L F P
0 g
s
.49116 .4626 m .50429 .45951 L p .47975 .48715 L 1 g
F P
0 g
s
.47975 .48715 m .46447 .47003 L .49116 .4626 L p 1 g
F P
0 g
s
.50429 .45951 m .52558 .48695 L .48609 .49474 L p 1 g
F P
0 g
s
.66745 .47418 m .68694 .50435 L .6463 .51689 L p 1 g
F P
0 g
s
.6463 .51689 m .62653 .48771 L .66745 .47418 L p 1 g
F P
0 g
s
.34027 .44644 m .36391 .46669 L .32579 .47455 L p 1 g
F P
0 g
s
.32579 .47455 m .30178 .45522 L .34027 .44644 L p 1 g
F P
0 g
s
.83014 .4918 m .84836 .52701 L .8068 .54214 L p 1 g
F P
0 g
s
.8068 .54214 m .78836 .50792 L .83014 .4918 L p 1 g
F P
0 g
s
.27722 .43349 m .30237 .44666 L .26516 .44859 L p 1 g
F P
0 g
s
.26516 .44859 m .23957 .43686 L .27722 .43349 L p 1 g
F P
0 g
s
.8711 .60966 m .89097 .66357 L .84907 .67782 L p 1 g
F P
0 g
s
.84907 .67782 m .8293 .62562 L .8711 .60966 L p 1 g
F P
0 g
s
.44229 .44568 m .46449 .46889 L .42544 .4756 L p 1 g
F P
0 g
s
.42544 .4756 m .40287 .45396 L .44229 .44568 L p 1 g
F P
0 g
s
.60636 .45955 m .62653 .48771 L .58616 .49975 L p 1 g
F P
0 g
s
.58616 .49975 m .56568 .47257 L .60636 .45955 L p 1 g
F P
0 g
s
.76969 .47474 m .78836 .50792 L .747 .52249 L p 1 g
F P
0 g
s
.747 .52249 m .72808 .49031 L .76969 .47474 L p 1 g
F P
0 g
s
.44226 .44684 m .46447 .47003 L .42531 .4797 L p 1 g
F P
0 g
s
.42531 .4797 m .40274 .45746 L .44226 .44684 L p 1 g
F P
0 g
s
.3906 .48316 m
.41388 .47136 L
s
.72896 .50453 m .74859 .5465 L .70744 .56023 L p 1 g
F P
0 g
s
.70744 .56023 m .6877 .51993 L .72896 .50453 L p 1 g
F P
0 g
s
.6678 .48231 m .6877 .51993 L .64687 .53257 L p 1 g
F P
0 g
s
.64687 .53257 m .6268 .49661 L .6678 .48231 L p 1 g
F P
0 g
s
.93258 .49204 m .95028 .5303 L .90819 .54759 L p 1 g
F P
0 g
s
.90819 .54759 m .89031 .51032 L .93258 .49204 L p 1 g
F P
0 g
s
.7901 .52999 m .80967 .57638 L .76819 .59121 L p 1 g
F P
0 g
s
.76819 .59121 m .74859 .5465 L .7901 .52999 L p 1 g
F P
0 g
s
.27696 .43696 m .30178 .45522 L .26404 .46265 L p 1 g
F P
0 g
s
.26404 .46265 m .23883 .44531 L .27696 .43696 L p 1 g
F P
0 g
s
.60646 .46328 m .6268 .49661 L .58631 .50816 L p 1 g
F P
0 g
s
.58631 .50816 m .56573 .47648 L .60646 .46328 L p 1 g
F P
0 g
s
.85143 .55874 m .8711 .60966 L .8293 .62562 L p 1 g
F P
0 g
s
.8293 .62562 m .80967 .57638 L .85143 .55874 L p 1 g
F P
0 g
s
.37954 .435 m .40287 .45396 L .36426 .45958 L p 1 g
F P
0 g
s
.36426 .45958 m .34051 .44215 L .37954 .435 L p 1 g
F P
0 g
s
.21279 .42801 m .23957 .43686 L .20293 .43766 L p 1 g
F P
0 g
s
.20293 .43766 m .17566 .43019 L .21279 .42801 L p 1 g
F P
0 g
s
.70888 .45915 m .72808 .49031 L .68694 .50435 L p 1 g
F P
0 g
s
.68694 .50435 m .66745 .47418 L .70888 .45915 L p 1 g
F P
0 g
s
.8723 .47411 m .89031 .51032 L .84836 .52701 L p 1 g
F P
0 g
s
.84836 .52701 m .83014 .4918 L .8723 .47411 L p 1 g
F P
0 g
s
.54473 .4464 m .56568 .47257 L .52558 .48413 L p 1 g
F P
0 g
s
.52558 .48413 m .50429 .45894 L .54473 .4464 L p 1 g
F P
0 g
s
.54473 .44741 m .56573 .47648 L .52558 .48695 L p 1 g
F P
0 g
s
.52558 .48695 m .50429 .45951 L .54473 .44741 L p 1 g
F P
0 g
s
.91311 .59087 m .93309 .64647 L .89097 .66357 L p 1 g
F P
0 g
s
.89097 .66357 m .8711 .60966 L .91311 .59087 L p 1 g
F P
0 g
s
.37948 .43626 m .40274 .45746 L .36391 .46669 L p 1 g
F P
0 g
s
.36391 .46669 m .34027 .44644 L .37948 .43626 L p 1 g
F P
0 g
s
.81172 .45762 m .83014 .4918 L .78836 .50792 L p 1 g
F P
0 g
s
.78836 .50792 m .76969 .47474 L .81172 .45762 L p 1 g
F P
0 g
s
.64759 .44503 m .66745 .47418 L .62653 .48771 L p 1 g
F P
0 g
s
.62653 .48771 m .60636 .45955 L .64759 .44503 L p 1 g
F P
0 g
s
.77056 .48636 m .7901 .52999 L .74859 .5465 L p 1 g
F P
0 g
s
.74859 .5465 m .72896 .50453 L .77056 .48636 L p 1 g
F P
0 g
s
.83188 .51069 m .85143 .55874 L .80967 .57638 L p 1 g
F P
0 g
s
.80967 .57638 m .7901 .52999 L .83188 .51069 L p 1 g
F P
0 g
s
.21269 .42907 m .23883 .44531 L .20148 .45231 L p 1 g
F P
0 g
s
.20148 .45231 m .17493 .43698 L .21269 .42907 L p 1 g
F P
0 g
s
.70922 .46526 m .72896 .50453 L .6877 .51993 L p 1 g
F P
0 g
s
.6877 .51993 m .6678 .48231 L .70922 .46526 L p 1 g
F P
0 g
s
.34051 .44215 m .30237 .44666 L p .31544 .42805 L 1 g
F P
0 g
s
.34027 .44644 m .30178 .45522 L p .31544 .42805 L 1 g
F P
0 g
s
.25124 .41979 m .29057 .40912 L p .28284 .42493 L 1 g
F P
0 g
s
.30237 .44666 m .27722 .43349 L .30786 .42873 L p 1 g
F P
0 g
s
.27722 .43349 m .23957 .43686 L p .24766 .42715 L 1 g
F P
0 g
s
.27696 .43696 m .23883 .44531 L p .24766 .42715 L 1 g
F P
0 g
s
.23957 .43686 m .21279 .42801 L .21686 .4275 L p 1 g
F P
0 g
s
.24766 .42715 m .24818 .42609 L .26202 .42699 L 1 g
F
.25124 .41979 m .25989 .42557 L p .24818 .42609 L F P
0 g
s
.25989 .42557 m .26202 .42699 L p .24818 .42609 L 1 g
F P
0 g
s
.22179 .42688 m .25124 .41979 L p .24818 .42609 L 1 g
F P
0 g
s
.22186 .42632 m .22456 .40373 L .22548 .42599 L 1 g
F
.21686 .4275 m .22179 .42688 L p .2377 .42726 L F P
0 g
s
.22179 .42688 m .21686 .4275 L p .21713 .42702 L 1 g
F P
0 g
s
.21713 .42702 m .21716 .42698 L .22179 .42688 L 1 g
F
.21686 .4275 m .21279 .42801 L p .21309 .42767 L F P
0 g
s
.21363 .42706 m .2138 .42669 L .22179 .42688 L 1 g
F
.2138 .42669 m .22456 .40373 L .22179 .42688 L F
.21309 .42767 m .21363 .42706 L .21686 .4275 L F
.21363 .42706 m .22342 .41592 L .21712 .42704 L F
.21363 .42706 m .21712 .42704 L .21686 .4275 L F
.18554 .41395 m .22456 .40373 L p .21363 .42706 L F P
0 g
s
.21279 .42801 m .20982 .4274 L p .21363 .42706 L 1 g
F P
0 g
s
.20982 .4274 m .20961 .42736 L p .21363 .42706 L 1 g
F P
0 g
s
.20961 .42736 m .20911 .42708 L p .21363 .42706 L 1 g
F P
0 g
s
.20911 .42708 m .18554 .41395 L p .21363 .42706 L 1 g
F P
0 g
s
.21269 .42907 m .17493 .43698 L p .17774 .43088 L 1 g
F P
0 g
s
.21279 .42801 m .17566 .43019 L p .17971 .4266 L 1 g
F P
0 g
s
.18554 .41395 m .20671 .42574 L p .17971 .4266 L 1 g
F P
0 g
s
.20671 .42574 m .20961 .42736 L p .17971 .4266 L 1 g
F P
0 g
s
.17566 .43019 m .15704 .42732 L p .17971 .4266 L 1 g
F P
0 g
s
.14734 .42277 m .18554 .41395 L p .17971 .4266 L 1 g
F P
0 g
s
.15704 .42732 m .15579 .42713 L p .17971 .4266 L 1 g
F P
0 g
s
.17493 .43698 m .15579 .42713 L p .17971 .4266 L 1 g
F P
0 g
s
.15579 .42713 m .14734 .42277 L p .17971 .4266 L 1 g
F P
0 g
s
.15579 .42713 m .14699 .42577 L p .17971 .4266 L 1 g
F P
0 g
s
.14699 .42577 m .18468 .4222 L p .17971 .4266 L 1 g
F P
0 g
s
.18468 .4222 m .20961 .42736 L p .17971 .4266 L 1 g
F P
0 g
s
.20961 .42736 m .21279 .42801 L p .17971 .4266 L 1 g
F P
0 g
s
.17774 .43088 m .17971 .4266 L .20961 .42736 L 1 g
F
.20961 .42736 m .21269 .42907 L p .17774 .43088 L F P
0 g
s
.20961 .42736 m .18468 .4222 L p .21311 .42697 L 1 g
F P
0 g
s
.20961 .42736 m .21311 .42697 L .21363 .42706 L 1 g
F
.21269 .42907 m .20961 .42736 L p .21363 .42706 L F P
0 g
s
.23957 .43686 m .21686 .4275 L .24766 .42715 L 1 g
F
.22179 .42688 m .21269 .42907 L p .21363 .42706 L F P
0 g
s
.22548 .42599 m .22179 .42688 L p .22186 .42632 L 1 g
F P
0 g
s
.21716 .42698 m .22342 .41592 L .22179 .42688 L 1 g
F
.18468 .4222 m .22342 .41592 L p .21363 .42706 L F P
0 g
s
.22456 .40373 m .25124 .41979 L .22548 .42599 L p 1 g
F P
0 g
s
.24766 .42715 m .2377 .42726 L .22179 .42688 L 1 g
F
.24766 .42715 m .22179 .42688 L .24818 .42609 L F
.22179 .42688 m .25095 .42321 L p .24766 .42715 L F P
0 g
s
.23883 .44531 m .22179 .42688 L .24766 .42715 L 1 g
F
.23883 .44531 m .21269 .42907 L .22179 .42688 L p F P
0 g
s
.22342 .41592 m .25095 .42321 L .22179 .42688 L p 1 g
F P
0 g
s
.26202 .42699 m .26401 .42832 L p .24766 .42715 L 1 g
F P
0 g
s
.24766 .42715 m .25152 .42743 L .27722 .43349 L 1 g
F
.26401 .42832 m .27722 .43349 L p .2481 .42662 L F P
0 g
s
.26401 .42832 m .27696 .43696 L p .24766 .42715 L 1 g
F P
0 g
s
.24766 .42715 m .2481 .42662 L .25152 .42743 L 1 g
F
.25095 .42321 m .26401 .42832 L p .2481 .42662 L F P
0 g
s
.31544 .42805 m .31552 .4279 L .31692 .42808 L 1 g
F
.31585 .42724 m .31665 .42787 L p .31552 .4279 L F P
0 g
s
.31665 .42787 m .31692 .42808 L p .31552 .4279 L 1 g
F P
0 g
s
.30786 .42873 m .31264 .42799 L p .3119 .42935 L 1 g
F P
0 g
s
.30237 .44666 m .30786 .42873 L .3119 .42935 L 1 g
F
.31319 .4279 m .31585 .42724 L p .31552 .4279 L F P
0 g
s
.29057 .40912 m .31585 .42724 L .31319 .4279 L p 1 g
F P
0 g
s
.31264 .42799 m .31319 .4279 L p .31448 .42803 L 1 g
F P
0 g
s
.31319 .4279 m .31264 .42799 L p .31249 .42791 L 1 g
F P
0 g
s
.31249 .42791 m .31246 .42789 L .31319 .4279 L 1 g
F
.28164 .42738 m .28284 .42493 L .29579 .4262 L F
.28164 .42738 m .29579 .4262 L .31319 .4279 L F
.28284 .42493 m .29057 .40912 L .31319 .4279 L F
.28164 .42738 m .25189 .42023 L .28284 .42493 L F
.27101 .42684 m .25189 .42023 L .28164 .42738 L F
.27722 .43349 m .26515 .42877 L p .28164 .42738 L F P
0 g
s
.26401 .42832 m .25189 .42023 L p .2738 .4278 L 1 g
F P
0 g
s
.26515 .42877 m .26401 .42832 L p .28164 .42738 L 1 g
F P
0 g
s
.30786 .42873 m .27722 .43349 L p .28164 .42738 L 1 g
F P
0 g
s
.26401 .42832 m .2738 .4278 L .28033 .43006 L 1 g
F
.28033 .43006 m .27101 .42684 L .28164 .42738 L F
.26401 .42832 m .25095 .42321 L p .28164 .42738 L F P
0 g
s
.25189 .42023 m .25124 .41979 L p .28284 .42493 L 1 g
F P
0 g
s
.25095 .42321 m .29009 .4157 L p .28164 .42738 L 1 g
F P
0 g
s
.31246 .42789 m .29009 .4157 L .31319 .4279 L 1 g
F
.31264 .42799 m .30786 .42873 L p .30708 .42816 L F P
0 g
s
.30708 .42816 m .29009 .4157 L .31264 .42799 L 1 g
F
.28164 .42738 m .29009 .4157 L .30786 .42873 L F
.30237 .44666 m .31264 .42799 L .31544 .42805 L F
.31319 .4279 m .27696 .43696 L p .28164 .42738 L F P
0 g
s
.31544 .42805 m .31448 .42803 L .31319 .4279 L 1 g
F
.31544 .42805 m .31319 .4279 L .31552 .4279 L F
.31319 .4279 m .31583 .4275 L p .31544 .42805 L F P
0 g
s
.30178 .45522 m .31319 .4279 L .31544 .42805 L 1 g
F
.30178 .45522 m .27696 .43696 L .31319 .4279 L p F P
0 g
s
.29009 .4157 m .31583 .4275 L .31319 .4279 L p 1 g
F P
0 g
s
.27696 .43696 m .26401 .42832 L p .28033 .43006 L 1 g
F P
0 g
s
.31692 .42808 m .31722 .42832 L p .31544 .42805 L 1 g
F P
0 g
s
.31722 .42832 m .34051 .44215 L p .31544 .42805 L 1 g
F P
0 g
s
.31722 .42832 m .34027 .44644 L p .31544 .42805 L 1 g
F P
0 g
s
.31583 .4275 m .31722 .42832 L p .31544 .42805 L 1 g
F P
0 g
s
.46449 .46889 m .44229 .44568 L .48244 .4347 L p 1 g
F P
0 g
s
.89337 .5383 m .91311 .59087 L .8711 .60966 L p 1 g
F P
0 g
s
.8711 .60966 m .85143 .55874 L .89337 .5383 L p 1 g
F P
0 g
s
.46447 .47003 m .44226 .44684 L .48244 .43477 L p 1 g
F P
0 g
s
.46165 .44714 m
.46167 .44712 L
s
.4433 .45644 m
.46165 .44714 L
s
.91479 .45482 m .93258 .49204 L .89031 .51032 L p 1 g
F P
0 g
s
.89031 .51032 m .8723 .47411 L .91479 .45482 L p 1 g
F P
0 g
s
.64768 .44733 m .6678 .48231 L .6268 .49661 L p 1 g
F P
0 g
s
.6268 .49661 m .60646 .46328 L .64768 .44733 L p 1 g
F P
0 g
s
.75077 .44257 m .76969 .47474 L .72808 .49031 L p 1 g
F P
0 g
s
.72808 .49031 m .70888 .45915 L .75077 .44257 L p 1 g
F P
0 g
s
.95522 .56925 m .97533 .6265 L .93309 .64647 L p 1 g
F P
0 g
s
.93309 .64647 m .91311 .59087 L .95522 .56925 L p 1 g
F P
0 g
s
.58575 .43239 m .60636 .45955 L .56568 .47257 L p 1 g
F P
0 g
s
.56568 .47257 m .54473 .4464 L .56433 .43971 L p 1 g
F P
0 g
s
.54473 .4464 m .53161 .45047 L p .517 .43366 L 1 g
F P
0 g
s
.53161 .45047 m .50429 .45894 L p .517 .43366 L 1 g
F P
0 g
s
.48244 .4347 m .50988 .42547 L p .50835 .43475 L 1 g
F P
0 g
s
.48506 .43768 m .48244 .4347 L p .50835 .43475 L 1 g
F P
0 g
s
.50988 .42547 m .52324 .42097 L p .517 .43366 L 1 g
F P
0 g
s
.50429 .45894 m .48506 .43768 L p .517 .43366 L 1 g
F P
0 g
s
.50429 .45951 m .50988 .42547 L .517 .43366 L 1 g
F
.52324 .42097 m .52786 .42665 L p .517 .43366 L F P
0 g
s
.48244 .43477 m .52324 .42124 L p .517 .43366 L 1 g
F P
0 g
s
.48506 .43768 m .48244 .43477 L p .517 .43366 L 1 g
F P
0 g
s
.50429 .45951 m .48506 .43768 L p .50835 .43475 L 1 g
F P
0 g
s
.54473 .44741 m .50429 .45951 L p .517 .43366 L 1 g
F P
0 g
s
.517 .43366 m .52626 .42768 L .54473 .4464 L 1 g
F
.52786 .42665 m .54473 .4464 L p .52212 .42348 L F P
0 g
s
.52786 .42665 m .54473 .44741 L p .517 .43366 L 1 g
F P
0 g
s
.517 .43366 m .52212 .42348 L .52626 .42768 L 1 g
F
.52324 .42124 m .52786 .42665 L p .52212 .42348 L F P
0 g
s
.52324 .42097 m .55044 .41013 L p .54727 .43083 L 1 g
F P
0 g
s
.53317 .42779 m .52324 .42097 L .54727 .43083 L 1 g
F
.54473 .4464 m .52786 .42665 L p .55195 .43184 L F P
0 g
s
.52786 .42665 m .52324 .42097 L p .54635 .43684 L 1 g
F P
0 g
s
.55044 .41013 m .56462 .40447 L p .55626 .42252 L 1 g
F P
0 g
s
.55195 .43184 m .55044 .41013 L .55626 .42252 L 1 g
F
.56433 .43971 m .54473 .4464 L p .55195 .43184 L F P
0 g
s
.54473 .44741 m .54754 .42904 L .55195 .43184 L 1 g
F
.54754 .42904 m .55044 .41013 L .55195 .43184 L F
.58575 .43239 m .56433 .43971 L p .56441 .43108 L F P
0 g
s
.56462 .40447 m .58375 .42991 L p .55195 .43184 L 1 g
F P
0 g
s
.55195 .43184 m .56465 .40622 L .56433 .43971 L 1 g
F
.56433 .43971 m .58575 .43239 L p .57294 .45804 L F P
0 g
s
.56568 .47257 m .56433 .43971 L .57294 .45804 L 1 g
F
.58575 .43257 m .54473 .44741 L p .55195 .43184 L F P
0 g
s
.58375 .42991 m .58575 .43239 L p .56441 .43108 L 1 g
F P
0 g
s
.58375 .42991 m .58575 .43257 L p .55195 .43184 L 1 g
F P
0 g
s
.56465 .40622 m .58375 .42991 L p .56441 .43108 L 1 g
F P
0 g
s
.52324 .42124 m .56465 .40622 L p .55195 .43184 L 1 g
F P
0 g
s
.52786 .42665 m .52324 .42124 L p .55195 .43184 L 1 g
F P
0 g
s
.54635 .43684 m .53317 .42779 L .54727 .43083 L 1 g
F
.54473 .44741 m .52786 .42665 L p .54635 .43684 L F P
0 g
s
.44229 .44568 m .40287 .45396 L p .41736 .42866 L 1 g
F P
0 g
s
.44226 .44684 m .40274 .45746 L p .41736 .42866 L 1 g
F P
0 g
s
.37954 .435 m .34051 .44215 L p .34958 .42787 L 1 g
F P
0 g
s
.37948 .43626 m .34027 .44644 L p .34958 .42787 L 1 g
F P
0 g
s
.35547 .41611 m .36787 .42651 L p .34958 .42787 L 1 g
F P
0 g
s
.34051 .44215 m .32013 .43005 L p .34958 .42787 L 1 g
F P
0 g
s
.31585 .42724 m .35547 .41611 L p .34958 .42787 L 1 g
F P
0 g
s
.32013 .43005 m .31722 .42832 L p .34958 .42787 L 1 g
F P
0 g
s
.34027 .44644 m .31722 .42832 L p .34958 .42787 L 1 g
F P
0 g
s
.31722 .42832 m .31585 .42724 L p .34958 .42787 L 1 g
F P
0 g
s
.31722 .42832 m .31583 .4275 L p .34958 .42787 L 1 g
F P
0 g
s
.31583 .4275 m .35533 .4188 L p .34958 .42787 L 1 g
F P
0 g
s
.36787 .42651 m .37185 .42985 L p .34958 .42787 L 1 g
F P
0 g
s
.34958 .42787 m .35976 .42877 L .37954 .435 L 1 g
F
.37185 .42985 m .37954 .435 L p .35079 .42595 L F P
0 g
s
.37185 .42985 m .37948 .43626 L p .34958 .42787 L 1 g
F P
0 g
s
.34958 .42787 m .35079 .42595 L .35976 .42877 L 1 g
F
.35533 .4188 m .37185 .42985 L p .35079 .42595 L F P
0 g
s
.35547 .41611 m .36594 .41284 L p .37548 .42934 L 1 g
F P
0 g
s
.37454 .42947 m .35547 .41611 L .37548 .42934 L 1 g
F
.37954 .435 m .37185 .42985 L p .3835 .4282 L F P
0 g
s
.37185 .42985 m .35547 .41611 L p .37625 .43067 L 1 g
F P
0 g
s
.36594 .41284 m .39581 .40352 L p .38621 .42277 L 1 g
F P
0 g
s
.40287 .45396 m .37954 .435 L .40055 .4298 L p 1 g
F P
0 g
s
.3835 .4282 m .36594 .41284 L .38621 .42277 L 1 g
F
.40055 .4298 m .37954 .435 L p .3835 .4282 L F P
0 g
s
.37948 .43626 m .37432 .42733 L .3835 .4282 L 1 g
F
.37432 .42733 m .36594 .41284 L .3835 .4282 L F
.41736 .42866 m .41804 .42733 L .4232 .42836 L F
.41939 .42467 m .42164 .42685 L p .41804 .42733 L F P
0 g
s
.42164 .42685 m .4232 .42836 L p .41804 .42733 L 1 g
F P
0 g
s
.40055 .4298 m .40192 .42947 L p .40196 .43049 L 1 g
F P
0 g
s
.40287 .45396 m .40055 .4298 L .40196 .43049 L 1 g
F
.40833 .42788 m .41939 .42467 L p .41804 .42733 L F P
0 g
s
.39581 .40352 m .41939 .42467 L .40833 .42788 L p 1 g
F P
0 g
s
.40192 .42947 m .40833 .42788 L p .41275 .4289 L 1 g
F P
0 g
s
.40833 .42788 m .40192 .42947 L p .40158 .42826 L 1 g
F P
0 g
s
.40158 .42826 m .40149 .42797 L .40833 .42788 L 1 g
F
.3835 .4282 m .38621 .42277 L .40833 .42788 L F
.38621 .42277 m .39581 .40352 L .40833 .42788 L F
.40149 .42797 m .39566 .40735 L .40833 .42788 L F
.40192 .42947 m .40055 .4298 L p .40052 .42965 L F P
0 g
s
.40052 .42965 m .39566 .40735 L .40192 .42947 L 1 g
F
.3835 .4282 m .39566 .40735 L .40055 .4298 L F
.40287 .45396 m .40192 .42947 L .41736 .42866 L F
.40833 .42788 m .37948 .43626 L p .3835 .4282 L F P
0 g
s
.41736 .42866 m .41275 .4289 L .40833 .42788 L 1 g
F
.41736 .42866 m .40833 .42788 L .41804 .42733 L F
.40833 .42788 m .41937 .42515 L p .41736 .42866 L F P
0 g
s
.40274 .45746 m .40833 .42788 L .41736 .42866 L 1 g
F
.40274 .45746 m .37948 .43626 L .40833 .42788 L p F P
0 g
s
.39566 .40735 m .41937 .42515 L .40833 .42788 L p 1 g
F P
0 g
s
.35533 .4188 m .39566 .40735 L p .3835 .4282 L 1 g
F P
0 g
s
.37185 .42985 m .35533 .4188 L p .3835 .4282 L 1 g
F P
0 g
s
.37625 .43067 m .37454 .42947 L .37548 .42934 L 1 g
F
.37948 .43626 m .37185 .42985 L p .37625 .43067 L F P
0 g
s
.4232 .42836 m .42615 .43122 L p .41736 .42866 L 1 g
F P
0 g
s
.42615 .43122 m .44229 .44568 L p .41736 .42866 L 1 g
F P
0 g
s
.42615 .43122 m .44226 .44684 L p .41736 .42866 L 1 g
F P
0 g
s
.41937 .42515 m .42615 .43122 L p .41736 .42866 L 1 g
F P
0 g
s
.58575 .43257 m .60646 .46328 L .56573 .47648 L p 1 g
F P
0 g
s
.56573 .47648 m .54473 .44741 L .58575 .43257 L p 1 g
F P
0 g
s
.85413 .43892 m .8723 .47411 L .83014 .4918 L p 1 g
F P
0 g
s
.83014 .4918 m .81172 .45762 L .85413 .43892 L p 1 g
F P
0 g
s
.87383 .48861 m .89337 .5383 L .85143 .55874 L p 1 g
F P
0 g
s
.85143 .55874 m .83188 .51069 L .87383 .48861 L p 1 g
F P
0 g
s
.81241 .46542 m .83188 .51069 L .7901 .52999 L p 1 g
F P
0 g
s
.7901 .52999 m .77056 .48636 L .81241 .46542 L p 1 g
F P
0 g
s
.41388 .47136 m
.4433 .45644 L
s
.93541 .51505 m .95522 .56925 L .91311 .59087 L p 1 g
F P
0 g
s
.91311 .59087 m .89337 .5383 L .93541 .51505 L p 1 g
F P
0 g
s
.70888 .45915 m .66745 .47418 L p .68837 .43097 L 1 g
F P
0 g
s
.70922 .46526 m .6678 .48231 L p .68837 .43097 L 1 g
F P
0 g
s
.62727 .41496 m .66918 .39457 L p .65525 .42877 L 1 g
F P
0 g
s
.66745 .47418 m .64759 .44503 L .67189 .43569 L p 1 g
F P
0 g
s
.64759 .44503 m .60636 .45955 L p .62019 .43139 L 1 g
F P
0 g
s
.64768 .44733 m .60646 .46328 L p .62019 .43139 L 1 g
F P
0 g
s
.62727 .41496 m .63532 .42773 L p .62019 .43139 L 1 g
F P
0 g
s
.62019 .43139 m .59309 .43217 L .5894 .43102 L 1 g
F
.56462 .40447 m .60649 .38518 L p .58679 .43021 L F P
0 g
s
.58575 .43239 m .58801 .43154 L p .58871 .4323 L 1 g
F P
0 g
s
.59046 .43859 m .58575 .43239 L p .62019 .43139 L 1 g
F P
0 g
s
.60649 .38518 m .62727 .41496 L .59101 .43034 L p 1 g
F P
0 g
s
.59101 .43034 m .5894 .43102 L p .58961 .43046 L 1 g
F P
0 g
s
.58961 .43046 m .60649 .38518 L .59101 .43034 L 1 g
F
.58656 .43073 m .57109 .41307 L .58679 .43021 L F
.58277 .4286 m .57109 .41307 L p .58656 .43073 L F P
0 g
s
.57109 .41307 m .56462 .40447 L p .58679 .43021 L 1 g
F P
0 g
s
.5894 .43102 m .58801 .43154 L p .58829 .43091 L 1 g
F P
0 g
s
.58801 .43154 m .58575 .43239 L p .58656 .43073 L 1 g
F P
0 g
s
.58656 .43073 m .58679 .43021 L .5894 .43102 L 1 g
F
.58679 .43021 m .60649 .38518 L .5894 .43102 L F
.58656 .43073 m .6066 .38967 L .58801 .43154 L F
.5894 .43102 m .58575 .43257 L p .58656 .43073 L F P
0 g
s
.5894 .43102 m .62727 .41496 L p .62019 .43139 L 1 g
F P
0 g
s
.6066 .38967 m .62733 .41686 L .5894 .43102 L p 1 g
F P
0 g
s
.58829 .43091 m .6066 .38967 L .5894 .43102 L 1 g
F
.56465 .40622 m .6066 .38967 L p .58656 .43073 L F P
0 g
s
.58575 .43239 m .58455 .4309 L p .58656 .43073 L 1 g
F P
0 g
s
.58455 .4309 m .58375 .42991 L p .58656 .43073 L 1 g
F P
0 g
s
.58575 .43257 m .58375 .42991 L p .58656 .43073 L 1 g
F P
0 g
s
.58375 .42991 m .58277 .4286 L p .58656 .43073 L 1 g
F P
0 g
s
.58375 .42991 m .56465 .40622 L p .58656 .43073 L 1 g
F P
0 g
s
.59309 .43217 m .58871 .4323 L .5894 .43102 L 1 g
F
.58801 .43154 m .5894 .43102 L p .58871 .4323 L F P
0 g
s
.60636 .45955 m .59046 .43859 L p .62019 .43139 L 1 g
F P
0 g
s
.60646 .46328 m .58575 .43257 L .5894 .43102 L p 1 g
F P
0 g
s
.60646 .46328 m .5894 .43102 L .62019 .43139 L 1 g
F
.5894 .43102 m .62733 .41686 L p .62019 .43139 L F P
0 g
s
.63532 .42773 m .63653 .42965 L p .62019 .43139 L 1 g
F P
0 g
s
.62019 .43139 m .62923 .43043 L .64759 .44503 L 1 g
F
.63653 .42965 m .64759 .44503 L p .62307 .42553 L F P
0 g
s
.63653 .42965 m .64768 .44733 L p .62019 .43139 L 1 g
F P
0 g
s
.62019 .43139 m .62307 .42553 L .62923 .43043 L 1 g
F
.62733 .41686 m .63653 .42965 L p .62307 .42553 L F P
0 g
s
.68837 .43097 m .68847 .43072 L .69035 .43053 L 1 g
F
.68931 .42862 m .69022 .43029 L p .68847 .43072 L F P
0 g
s
.69022 .43029 m .69035 .43053 L p .68847 .43072 L 1 g
F P
0 g
s
.67189 .43569 m .67829 .43323 L p .67707 .43784 L 1 g
F P
0 g
s
.66745 .47418 m .67189 .43569 L .67707 .43784 L 1 g
F
.68356 .4312 m .68931 .42862 L p .68847 .43072 L F P
0 g
s
.66918 .39457 m .68931 .42862 L .68356 .4312 L p 1 g
F P
0 g
s
.67829 .43323 m .68356 .4312 L p .68629 .43144 L 1 g
F P
0 g
s
.68356 .4312 m .67829 .43323 L p .6779 .43176 L 1 g
F P
0 g
s
.6779 .43176 m .67775 .43123 L .68356 .4312 L 1 g
F
.6542 .43133 m .65525 .42877 L .66665 .42975 L F
.6542 .43133 m .66665 .42975 L .68356 .4312 L F
.65525 .42877 m .66918 .39457 L .68356 .4312 L F
.6542 .43133 m .62912 .4179 L .65525 .42877 L F
.62912 .4179 m .62727 .41496 L p .65525 .42877 L F P
0 g
s
.62733 .41686 m .66941 .3998 L p .6542 .43133 L 1 g
F P
0 g
s
.64759 .44503 m .63912 .43325 L p .6542 .43133 L 1 g
F P
0 g
s
.63912 .43325 m .63653 .42965 L p .6542 .43133 L 1 g
F P
0 g
s
.67189 .43569 m .64759 .44503 L p .6542 .43133 L 1 g
F P
0 g
s
.65285 .43464 m .64722 .43067 L .6542 .43133 L 1 g
F
.64722 .43067 m .64492 .42905 L .6542 .43133 L F
.64492 .42905 m .62912 .4179 L .6542 .43133 L F
.63653 .42965 m .62912 .4179 L p .65285 .43464 L F P
0 g
s
.63653 .42965 m .62733 .41686 L p .6542 .43133 L 1 g
F P
0 g
s
.67775 .43123 m .66941 .3998 L .68356 .4312 L 1 g
F
.67829 .43323 m .67189 .43569 L p .67183 .43482 L F P
0 g
s
.67183 .43482 m .66941 .3998 L .67829 .43323 L 1 g
F
.6542 .43133 m .66941 .3998 L .67189 .43569 L F
.66745 .47418 m .67829 .43323 L .68837 .43097 L F
.68356 .4312 m .64768 .44733 L p .6542 .43133 L F P
0 g
s
.68837 .43097 m .68629 .43144 L .68356 .4312 L 1 g
F
.68837 .43097 m .68356 .4312 L .68847 .43072 L F
.68356 .4312 m .68933 .42898 L p .68837 .43097 L F P
0 g
s
.6678 .48231 m .68356 .4312 L .68837 .43097 L 1 g
F
.6678 .48231 m .64768 .44733 L .68356 .4312 L p F P
0 g
s
.66941 .3998 m .68933 .42898 L .68356 .4312 L p 1 g
F P
0 g
s
.64768 .44733 m .63653 .42965 L p .65285 .43464 L 1 g
F P
0 g
s
.69035 .43053 m .69046 .43072 L p .68837 .43097 L 1 g
F P
0 g
s
.69046 .43072 m .70888 .45915 L p .68837 .43097 L 1 g
F P
0 g
s
.69046 .43072 m .70922 .46526 L p .68837 .43097 L 1 g
F P
0 g
s
.68933 .42898 m .69046 .43072 L p .68837 .43097 L 1 g
F P
0 g
s
.75096 .44543 m .77056 .48636 L .72896 .50453 L p 1 g
F P
0 g
s
.72896 .50453 m .70922 .46526 L .75096 .44543 L p 1 g
F P
0 g
s
.48454 .43553 m
.48454 .43553 L
s
.81172 .45762 m .76969 .47474 L p .79019 .43066 L 1 g
F P
0 g
s
.81241 .46542 m .77056 .48636 L p .79019 .43066 L 1 g
F P
0 g
s
.76969 .47474 m .75077 .44257 L .77165 .43363 L p 1 g
F P
0 g
s
.79019 .43066 m .79039 .43008 L .79609 .42971 L 1 g
F
.79295 .42283 m .79579 .42906 L p .79039 .43008 L F P
0 g
s
.79579 .42906 m .79609 .42971 L p .79039 .43008 L 1 g
F P
0 g
s
.77777 .431 m .79295 .42283 L p .79039 .43008 L 1 g
F P
0 g
s
.77344 .38288 m .79295 .42283 L .77777 .431 L p 1 g
F P
0 g
s
.77165 .43363 m .77777 .431 L p .78507 .43148 L 1 g
F P
0 g
s
.77777 .431 m .77165 .43363 L p .77178 .43144 L 1 g
F P
0 g
s
.77178 .43144 m .77181 .431 L .77777 .431 L 1 g
F
.75615 .43098 m .75664 .42963 L .76509 .43018 L F
.75615 .43098 m .76509 .43018 L .77777 .431 L F
.75664 .42963 m .77344 .38288 L .77777 .431 L F
.75077 .44257 m .70888 .45915 L p .72212 .43124 L F P
0 g
s
.73127 .40713 m .74215 .4283 L p .72212 .43124 L 1 g
F P
0 g
s
.74215 .4283 m .74311 .43017 L p .72212 .43124 L 1 g
F P
0 g
s
.70888 .45915 m .69352 .43545 L p .72212 .43124 L 1 g
F P
0 g
s
.68931 .42862 m .73127 .40713 L p .72212 .43124 L 1 g
F P
0 g
s
.69352 .43545 m .69046 .43072 L p .72212 .43124 L 1 g
F P
0 g
s
.70922 .46526 m .69046 .43072 L p .72212 .43124 L 1 g
F P
0 g
s
.69046 .43072 m .68931 .42862 L p .72212 .43124 L 1 g
F P
0 g
s
.69046 .43072 m .68933 .42898 L p .72212 .43124 L 1 g
F P
0 g
s
.68933 .42898 m .73154 .4114 L p .72212 .43124 L 1 g
F P
0 g
s
.73154 .4114 m .74311 .43017 L p .72212 .43124 L 1 g
F P
0 g
s
.74311 .43017 m .75077 .44257 L p .72212 .43124 L 1 g
F P
0 g
s
.71998 .43688 m .72212 .43124 L .74311 .43017 L 1 g
F
.75096 .44543 m .70922 .46526 L p .71998 .43688 L F P
0 g
s
.74311 .43017 m .75096 .44543 L p .71998 .43688 L 1 g
F P
0 g
s
.75077 .44257 m .74428 .43206 L p .75615 .43098 L 1 g
F P
0 g
s
.73127 .40713 m .77344 .38288 L p .75615 .43098 L 1 g
F P
0 g
s
.74428 .43206 m .74311 .43017 L p .75615 .43098 L 1 g
F P
0 g
s
.77165 .43363 m .75077 .44257 L p .75332 .43709 L 1 g
F P
0 g
s
.75332 .43709 m .77417 .39224 L .77165 .43363 L 1 g
F
.75096 .44543 m .74311 .43017 L p .75615 .43098 L F P
0 g
s
.74311 .43017 m .73127 .40713 L p .75615 .43098 L 1 g
F P
0 g
s
.74311 .43017 m .73154 .4114 L p .75615 .43098 L 1 g
F P
0 g
s
.73154 .4114 m .77417 .39224 L p .75863 .42566 L 1 g
F P
0 g
s
.75615 .43098 m .75192 .42761 L .75863 .42566 L 1 g
F
.75192 .42761 m .73154 .4114 L .75863 .42566 L F
.77181 .431 m .77417 .39224 L .77777 .431 L F
.76969 .47474 m .77165 .43363 L .79019 .43066 L F
.77777 .431 m .75096 .44543 L p .75615 .43098 L F P
0 g
s
.79019 .43066 m .78507 .43148 L .77777 .431 L 1 g
F
.79019 .43066 m .77777 .431 L .79039 .43008 L F
.77777 .431 m .79308 .42444 L p .79019 .43066 L F P
0 g
s
.77056 .48636 m .77777 .431 L .79019 .43066 L 1 g
F
.77056 .48636 m .75096 .44543 L .77777 .431 L p F P
0 g
s
.77417 .39224 m .79308 .42444 L .77777 .431 L p 1 g
F P
0 g
s
.79609 .42971 m .7963 .43017 L p .79019 .43066 L 1 g
F P
0 g
s
.7963 .43017 m .81172 .45762 L p .79019 .43066 L 1 g
F P
0 g
s
.7963 .43017 m .81241 .46542 L p .79019 .43066 L 1 g
F P
0 g
s
.79308 .42444 m .7963 .43017 L p .79019 .43066 L 1 g
F P
0 g
s
.91585 .46375 m .93541 .51505 L .89337 .5383 L p 1 g
F P
0 g
s
.89337 .5383 m .87383 .48861 L .91585 .46375 L p 1 g
F P
0 g
s
.85441 .4417 m .87383 .48861 L .83188 .51069 L p 1 g
F P
0 g
s
.83188 .51069 m .81241 .46542 L .85441 .4417 L p 1 g
F P
0 g
s
.91479 .45482 m .8723 .47411 L p .89211 .42936 L 1 g
F P
0 g
s
.91585 .46375 m .87383 .48861 L p .89211 .42936 L 1 g
F P
0 g
s
.8723 .47411 m .85413 .43892 L .86713 .43275 L p 1 g
F P
0 g
s
.89211 .42936 m .89236 .42857 L .9016 .42807 L 1 g
F
.89647 .41523 m .90121 .4271 L p .89236 .42857 L F P
0 g
s
.90121 .4271 m .9016 .42807 L p .89236 .42857 L 1 g
F P
0 g
s
.87326 .42984 m .89647 .41523 L p .89236 .42857 L 1 g
F P
0 g
s
.87723 .36941 m .89647 .41523 L .87539 .4285 L p 1 g
F P
0 g
s
.87334 .42867 m .87723 .36941 L .87539 .4285 L 1 g
F
.86713 .43275 m .87326 .42984 L p .88455 .43039 L F P
0 g
s
.87326 .42984 m .86713 .43275 L p .86773 .43021 L 1 g
F P
0 g
s
.86773 .43021 m .86781 .42986 L .87326 .42984 L 1 g
F
.86713 .43275 m .85413 .43892 L p .85606 .43458 L F P
0 g
s
.85813 .42991 m .85838 .42912 L .87326 .42984 L 1 g
F
.85838 .42912 m .87723 .36941 L .87326 .42984 L F
.85606 .43458 m .85813 .42991 L .86713 .43275 L F
.85813 .42991 m .87878 .38342 L .86753 .43109 L F
.85813 .42991 m .86753 .43109 L .86713 .43275 L F
.83507 .39749 m .87723 .36941 L p .85813 .42991 L F P
0 g
s
.85413 .43892 m .84966 .4306 L p .85813 .42991 L 1 g
F P
0 g
s
.84966 .4306 m .84909 .42954 L p .85813 .42991 L 1 g
F P
0 g
s
.84909 .42954 m .84874 .42873 L p .85813 .42991 L 1 g
F P
0 g
s
.84874 .42873 m .83507 .39749 L p .85813 .42991 L 1 g
F P
0 g
s
.85413 .43892 m .81172 .45762 L p .82406 .43048 L 1 g
F P
0 g
s
.83507 .39749 m .84827 .42766 L p .82406 .43048 L 1 g
F P
0 g
s
.84827 .42766 m .84909 .42954 L p .82406 .43048 L 1 g
F P
0 g
s
.81172 .45762 m .79816 .43349 L p .82406 .43048 L 1 g
F P
0 g
s
.79295 .42283 m .83507 .39749 L p .82406 .43048 L 1 g
F P
0 g
s
.79816 .43349 m .7963 .43017 L p .82406 .43048 L 1 g
F P
0 g
s
.81241 .46542 m .7963 .43017 L p .82406 .43048 L 1 g
F P
0 g
s
.7963 .43017 m .79295 .42283 L p .82406 .43048 L 1 g
F P
0 g
s
.7963 .43017 m .79308 .42444 L p .82406 .43048 L 1 g
F P
0 g
s
.79308 .42444 m .83577 .40473 L p .82406 .43048 L 1 g
F P
0 g
s
.83577 .40473 m .84909 .42954 L p .82406 .43048 L 1 g
F P
0 g
s
.84909 .42954 m .85413 .43892 L p .82406 .43048 L 1 g
F P
0 g
s
.82238 .43552 m .82406 .43048 L .84909 .42954 L 1 g
F
.85441 .4417 m .81241 .46542 L p .82238 .43552 L F P
0 g
s
.84909 .42954 m .85441 .4417 L p .82238 .43552 L 1 g
F P
0 g
s
.84909 .42954 m .83577 .40473 L p .85636 .42791 L 1 g
F P
0 g
s
.84909 .42954 m .85636 .42791 L .85813 .42991 L 1 g
F
.85441 .4417 m .84909 .42954 L p .85813 .42991 L F P
0 g
s
.8723 .47411 m .86713 .43275 L .89211 .42936 L 1 g
F
.87326 .42984 m .85441 .4417 L p .85813 .42991 L F P
0 g
s
.87539 .4285 m .87326 .42984 L p .87334 .42867 L 1 g
F P
0 g
s
.86781 .42986 m .87878 .38342 L .87326 .42984 L 1 g
F
.83577 .40473 m .87878 .38342 L p .85813 .42991 L F P
0 g
s
.89211 .42936 m .88455 .43039 L .87326 .42984 L 1 g
F
.89211 .42936 m .87326 .42984 L .89236 .42857 L F
.87326 .42984 m .89687 .41863 L p .89211 .42936 L F P
0 g
s
.87383 .48861 m .87326 .42984 L .89211 .42936 L 1 g
F
.87383 .48861 m .85441 .4417 L .87326 .42984 L p F P
0 g
s
.87878 .38342 m .89687 .41863 L .87326 .42984 L p 1 g
F P
0 g
s
.9016 .42807 m .90186 .42873 L p .89211 .42936 L 1 g
F P
0 g
s
.90186 .42873 m .91479 .45482 L p .89211 .42936 L 1 g
F P
0 g
s
.90186 .42873 m .91585 .46375 L p .89211 .42936 L 1 g
F P
0 g
s
.89687 .41863 m .90186 .42873 L p .89211 .42936 L 1 g
F P
0 g
s
.48244 .4347 m .46925 .4383 L p .452 .42745 L 1 g
F P
0 g
s
.46925 .4383 m .44229 .44568 L p .452 .42745 L 1 g
F P
0 g
s
.41939 .42467 m .43804 .41867 L p .43962 .42926 L 1 g
F P
0 g
s
.42615 .43122 m .41939 .42467 L p .43962 .42926 L 1 g
F P
0 g
s
.43804 .41867 m .45997 .41162 L p .452 .42745 L 1 g
F P
0 g
s
.44229 .44568 m .42615 .43122 L p .452 .42745 L 1 g
F P
0 g
s
.44226 .44684 m .43804 .41867 L .452 .42745 L 1 g
F
.45997 .41162 m .48075 .43304 L p .452 .42745 L F P
0 g
s
.41937 .42515 m .45995 .41255 L p .452 .42745 L 1 g
F P
0 g
s
.42615 .43122 m .41937 .42515 L p .452 .42745 L 1 g
F P
0 g
s
.44226 .44684 m .42615 .43122 L p .43962 .42926 L 1 g
F P
0 g
s
.48244 .43477 m .44226 .44684 L p .452 .42745 L 1 g
F P
0 g
s
.48075 .43304 m .48244 .4347 L p .452 .42745 L 1 g
F P
0 g
s
.48075 .43304 m .48244 .43477 L p .452 .42745 L 1 g
F P
0 g
s
.45995 .41255 m .48075 .43304 L p .452 .42745 L 1 g
F P
0 g
s
.47361 .44107 m
.47878 .43845 L
s
.47878 .43845 m
.48454 .43553 L
s
.47222 .44177 m
.47361 .44107 L
s
.46167 .44712 m
.47222 .44177 L
s
.52324 .42097 m .48943 .43234 L p .49071 .41814 L 1 g
F P
0 g
s
.48943 .43234 m .48244 .4347 L p .49071 .41814 L 1 g
F P
0 g
s
.45997 .41162 m .49232 .4002 L p .48456 .42734 L 1 g
F P
0 g
s
.48075 .43304 m .45997 .41162 L p .48456 .42734 L 1 g
F P
0 g
s
.49232 .4002 m .50118 .39708 L p .49071 .41814 L 1 g
F P
0 g
s
.48244 .4347 m .48075 .43304 L p .49071 .41814 L 1 g
F P
0 g
s
.48244 .43477 m .49232 .4002 L .49071 .41814 L 1 g
F
.50118 .39708 m .50693 .40337 L p .49071 .41814 L F P
0 g
s
.45995 .41255 m .50118 .39717 L p .49071 .41814 L 1 g
F P
0 g
s
.48075 .43304 m .45995 .41255 L p .49071 .41814 L 1 g
F P
0 g
s
.48244 .43477 m .48075 .43304 L p .48456 .42734 L 1 g
F P
0 g
s
.52324 .42124 m .48244 .43477 L p .49071 .41814 L 1 g
F P
0 g
s
.50693 .40337 m .52324 .42097 L p .49071 .41814 L 1 g
F P
0 g
s
.50693 .40337 m .52324 .42124 L p .49071 .41814 L 1 g
F P
0 g
s
.50118 .39717 m .50693 .40337 L p .49071 .41814 L 1 g
F P
0 g
s
.48898 .43327 m
.49563 .4299 L
s
.48454 .43553 m
.48898 .43327 L
s
.7114 .37141 m .73127 .40713 L .68931 .42862 L p 1 g
F P
0 g
s
.68931 .42862 m .66918 .39457 L .7114 .37141 L p 1 g
F P
0 g
s
.71197 .38118 m .73154 .4114 L .68933 .42898 L p 1 g
F P
0 g
s
.68933 .42898 m .66941 .3998 L .71197 .38118 L p 1 g
F P
0 g
s
.48286 .42612 m
.49563 .4299 L
s
.81574 .35589 m .83507 .39749 L .79295 .42283 L p 1 g
F P
0 g
s
.79295 .42283 m .77344 .38288 L .81574 .35589 L p 1 g
F P
0 g
s
.81717 .3715 m .83577 .40473 L .79308 .42444 L p 1 g
F P
0 g
s
.79308 .42444 m .77417 .39224 L .81717 .3715 L p 1 g
F P
0 g
s
.33065 .397 m .35547 .41611 L .31585 .42724 L p 1 g
F P
0 g
s
.31585 .42724 m .29057 .40912 L .33065 .397 L p 1 g
F P
0 g
s
.33014 .40543 m .35533 .4188 L .31583 .4275 L p 1 g
F P
0 g
s
.31583 .4275 m .29009 .4157 L .33014 .40543 L p 1 g
F P
0 g
s
.43682 .38947 m .45997 .41162 L .41939 .42467 L p 1 g
F P
0 g
s
.41939 .42467 m .39581 .40352 L .43682 .38947 L p 1 g
F P
0 g
s
.43672 .39311 m .45995 .41255 L .41937 .42515 L p 1 g
F P
0 g
s
.41937 .42515 m .39566 .40735 L .43672 .39311 L p 1 g
F P
0 g
s
.85807 .32619 m .87723 .36941 L .83507 .39749 L p 1 g
F P
0 g
s
.83507 .39749 m .81574 .35589 L .85807 .32619 L p 1 g
F P
0 g
s
.54298 .37899 m .56462 .40447 L .52324 .42097 L p 1 g
F P
0 g
s
.52324 .42097 m .50693 .40337 L p .511 .40062 L 1 g
F P
0 g
s
.52843 .40993 m .511 .40062 L .54298 .37899 L 1 g
F
.52324 .42097 m .511 .40062 L .52843 .40993 L F
.50304 .39636 m .54298 .37899 L p .50693 .40337 L F P
0 g
s
.50118 .39708 m .50184 .39572 L .5025 .39607 L 1 g
F
.50304 .39636 m .50234 .39663 L p .5025 .39607 L F P
0 g
s
.50234 .39663 m .50118 .39708 L p .5025 .39607 L 1 g
F P
0 g
s
.50207 .39805 m .50118 .39708 L .50304 .39636 L p 1 g
F P
0 g
s
.50184 .39572 m .50368 .39194 L .50304 .39636 L 1 g
F
.52074 .35616 m .52425 .35976 L p .51602 .37311 L F P
0 g
s
.51365 .37154 m .52074 .35616 L .51602 .37311 L 1 g
F
.50372 .39292 m .52074 .35616 L .50372 .39305 L F
.50368 .39194 m .52074 .35616 L .50346 .39348 L F
.47848 .3739 m .50249 .36419 L p .5022 .3716 L F P
0 g
s
.50118 .39708 m .49046 .38613 L p .5022 .3716 L 1 g
F P
0 g
s
.49046 .38613 m .47848 .3739 L p .5022 .3716 L 1 g
F P
0 g
s
.49783 .33355 m .50667 .34253 L p .50249 .36419 L 1 g
F P
0 g
s
.50667 .34253 m .51603 .35203 L p .51096 .36076 L 1 g
F P
0 g
s
.51603 .35203 m .5172 .35305 L p .51451 .35464 L 1 g
F P
0 g
s
.5172 .35305 m .52074 .35616 L .51096 .36076 L p 1 g
F P
0 g
s
.51096 .36076 m .50249 .36419 L p .50667 .34253 L 1 g
F P
0 g
s
.50249 .36419 m .47848 .3739 L p .48779 .35449 L 1 g
F P
0 g
s
.48779 .35449 m .49783 .33355 L .50249 .36419 L 1 g
F
.51096 .36076 m .51279 .3599 L p .51163 .36363 L F P
0 g
s
.50249 .36419 m .51096 .36076 L p .50368 .39194 L 1 g
F P
0 g
s
.50627 .38084 m .51096 .36076 L .51163 .36363 L 1 g
F
.51279 .3599 m .52074 .35616 L p .50368 .39194 L F P
0 g
s
.50118 .39708 m .50249 .36419 L .50368 .39194 L 1 g
F
.50118 .39717 m .51279 .3599 L .50368 .39194 L F
.51096 .36076 m .52074 .35681 L p .50368 .39194 L F P
0 g
s
.50118 .39717 m .51096 .36076 L .50627 .38084 L 1 g
F
.47848 .3739 m .50118 .39708 L .45997 .41162 L p F P
0 g
s
.51096 .36076 m .48814 .3715 L p .5172 .35305 L 1 g
F P
0 g
s
.48814 .3715 m .47845 .37606 L p .4996 .3635 L 1 g
F P
0 g
s
.4996 .3635 m .5172 .35305 L .48814 .3715 L 1 g
F
.54114 .31151 m .56352 .33343 L .52526 .35376 L p F P
0 g
s
.52526 .35376 m .52074 .35616 L p .5235 .35013 L 1 g
F P
0 g
s
.5235 .35013 m .54114 .31151 L .52526 .35376 L 1 g
F
.52074 .35616 m .51603 .35203 L p .52018 .32337 L F P
0 g
s
.52018 .32337 m .54114 .31151 L p .5235 .35013 L 1 g
F P
0 g
s
.52074 .35616 m .52018 .32337 L .5235 .35013 L 1 g
F
.52648 .34474 m .51759 .34127 L .54116 .31382 L F
.49783 .33355 m .52018 .32337 L p .51185 .33903 L F P
0 g
s
.51805 .29041 m .52849 .30098 L p .52018 .32337 L 1 g
F P
0 g
s
.52849 .30098 m .53099 .30306 L p .52581 .30821 L 1 g
F P
0 g
s
.53099 .30306 m .54114 .31151 L .52018 .32337 L p 1 g
F P
0 g
s
.52018 .32337 m .51228 .32697 L p .51695 .29737 L 1 g
F P
0 g
s
.51695 .29737 m .51805 .29041 L .52018 .32337 L 1 g
F
.51228 .32697 m .49783 .33355 L p .5065 .31504 L F P
0 g
s
.5065 .31504 m .51805 .29041 L .51228 .32697 L 1 g
F
.51603 .35203 m .49783 .33355 L p .52648 .34474 L F P
0 g
s
.52018 .32337 m .50533 .33177 L p .53099 .30306 L 1 g
F P
0 g
s
.50533 .33177 m .4978 .33603 L p .51491 .31903 L 1 g
F P
0 g
s
.51491 .31903 m .53099 .30306 L .50533 .33177 L 1 g
F
.56352 .33343 m .54114 .31151 L .58486 .28415 L p F P
0 g
s
.5624 .26307 m .58486 .28415 L .54114 .31151 L p 1 g
F P
0 g
s
.54114 .31151 m .5318 .30374 L p .5624 .26307 L 1 g
F P
0 g
s
.5318 .30374 m .53135 .29778 L .5624 .26307 L 1 g
F
.53034 .2842 m .5624 .26307 L p .52849 .30098 L F P
0 g
s
.53919 .24432 m .54089 .24604 L p .53034 .2842 L 1 g
F P
0 g
s
.53034 .2842 m .51805 .29041 L p .52902 .26649 L 1 g
F P
0 g
s
.52902 .26649 m .53919 .24432 L .53034 .2842 L 1 g
F
.52849 .30098 m .51805 .29041 L .53034 .2842 L p F P
0 g
s
.53919 .2447 m .54089 .24604 L p .51805 .2923 L 1 g
F P
0 g
s
.54089 .24604 m .5624 .26307 L .53034 .2842 L p 1 g
F P
0 g
s
.53034 .2842 m .51805 .2923 L p .54089 .24604 L 1 g
F P
0 g
s
.54089 .24604 m .56246 .26795 L .53034 .2842 L p 1 g
F P
0 g
s
.5318 .30374 m .52849 .30098 L p .53135 .29778 L 1 g
F P
0 g
s
.52849 .30098 m .51805 .2923 L .53034 .2842 L p 1 g
F P
0 g
s
.53034 .2842 m .54773 .2754 L p .52849 .30098 L 1 g
F P
0 g
s
.54116 .31382 m .52849 .30098 L p .56246 .26795 L 1 g
F P
0 g
s
.54773 .2754 m .56246 .26795 L p .52849 .30098 L 1 g
F P
0 g
s
.52849 .30098 m .54116 .31382 L .52018 .32337 L p 1 g
F P
0 g
s
.51805 .2923 m .52849 .30098 L p .4978 .33603 L 1 g
F P
0 g
s
.4978 .33603 m .52849 .30098 L .52581 .30821 L 1 g
F
.51759 .34127 m .51185 .33903 L .52018 .32337 L F
.52018 .32337 m .54116 .31382 L p .51759 .34127 L F P
0 g
s
.51603 .35203 m .4978 .33603 L .52018 .32337 L p 1 g
F P
0 g
s
.54298 .37899 m .52074 .35616 L .56352 .33343 L p 1 g
F P
0 g
s
.52074 .35681 m .51603 .35203 L p .52648 .34474 L 1 g
F P
0 g
s
.51603 .35203 m .52074 .35681 L .51096 .36076 L p 1 g
F P
0 g
s
.4978 .33603 m .51603 .35203 L p .47845 .37606 L 1 g
F P
0 g
s
.47845 .37606 m .51603 .35203 L .51451 .35464 L 1 g
F
.50118 .39717 m .47845 .37606 L .51096 .36076 L p F P
0 g
s
.50372 .39305 m .51365 .37154 L .51602 .37311 L 1 g
F
.50346 .39348 m .50372 .39292 L .50372 .39305 L F
.50304 .39636 m .50118 .39717 L p .52425 .35976 L F P
0 g
s
.50368 .39194 m .52074 .35681 L .50304 .39636 L 1 g
F
.50118 .39717 m .50368 .39194 L .50346 .39348 L F
.52425 .35976 m .54298 .37899 L .50304 .39636 L p F P
0 g
s
.50693 .40337 m .50207 .39805 L p .50304 .39636 L 1 g
F P
0 g
s
.52074 .35681 m .543 .38103 L .50304 .39636 L p 1 g
F P
0 g
s
.52324 .42124 m .50693 .40337 L p .543 .38103 L 1 g
F P
0 g
s
.50304 .39636 m .543 .38103 L p .50693 .40337 L 1 g
F P
0 g
s
.50693 .40337 m .50118 .39717 L .50304 .39636 L p 1 g
F P
0 g
s
.543 .38103 m .56465 .40622 L .52324 .42124 L p 1 g
F P
0 g
s
.75383 .34549 m .77344 .38288 L .73127 .40713 L p 1 g
F P
0 g
s
.73127 .40713 m .7114 .37141 L .75383 .34549 L p 1 g
F P
0 g
s
.64875 .3631 m .66918 .39457 L .62727 .41496 L p 1 g
F P
0 g
s
.62727 .41496 m .60649 .38518 L .64875 .3631 L p 1 g
F P
0 g
s
.64908 .37157 m .66941 .3998 L .62733 .41686 L p 1 g
F P
0 g
s
.62733 .41686 m .6066 .38967 L .64908 .37157 L p 1 g
F P
0 g
s
.75495 .36099 m .77417 .39224 L .73154 .4114 L p 1 g
F P
0 g
s
.73154 .4114 m .71197 .38118 L .75495 .36099 L p 1 g
F P
0 g
s
.8605 .34916 m .87878 .38342 L .83577 .40473 L p 1 g
F P
0 g
s
.83577 .40473 m .81717 .3715 L .8605 .34916 L p 1 g
F P
0 g
s
.1573 .39997 m .18554 .41395 L .14734 .42277 L p 1 g
F P
0 g
s
.14734 .42277 m .11861 .40973 L .1573 .39997 L p 1 g
F P
0 g
s
.15506 .41957 m .18468 .4222 L .14699 .42577 L p 1 g
F P
0 g
s
.14699 .42577 m .11675 .42455 L .15506 .41957 L p 1 g
F P
0 g
s
.26436 .39207 m .29057 .40912 L .25124 .41979 L p 1 g
F P
0 g
s
.25124 .41979 m .22456 .40373 L .26436 .39207 L p 1 g
F P
0 g
s
.79637 .31684 m .81574 .35589 L .77344 .38288 L p 1 g
F P
0 g
s
.77344 .38288 m .75383 .34549 L .79637 .31684 L p 1 g
F P
0 g
s
.26316 .40688 m .29009 .4157 L .25095 .42321 L p 1 g
F P
0 g
s
.25095 .42321 m .22342 .41592 L .26316 .40688 L p 1 g
F P
0 g
s
.37144 .38341 m .39581 .40352 L .35547 .41611 L p 1 g
F P
0 g
s
.35547 .41611 m .33065 .397 L .37144 .38341 L p 1 g
F P
0 g
s
.6913 .33824 m .7114 .37141 L .66918 .39457 L p 1 g
F P
0 g
s
.66918 .39457 m .64875 .3631 L .6913 .33824 L p 1 g
F P
0 g
s
.37102 .39236 m .39566 .40735 L .35533 .4188 L p 1 g
F P
0 g
s
.35533 .4188 m .33014 .40543 L .37102 .39236 L p 1 g
F P
0 g
s
.45997 .41162 m .43682 .38947 L .47848 .3739 L p 1 g
F P
0 g
s
.58527 .358 m .60649 .38518 L .56462 .40447 L p 1 g
F P
0 g
s
.56462 .40447 m .54298 .37899 L .58527 .358 L p 1 g
F P
0 g
s
.47845 .37606 m .50118 .39717 L .45995 .41255 L p 1 g
F P
0 g
s
.45995 .41255 m .43672 .39311 L .47845 .37606 L p 1 g
F P
0 g
s
.42119 .40784 m
.48286 .42612 L
s
.58537 .36344 m .6066 .38967 L .56465 .40622 L p 1 g
F P
0 g
s
.56465 .40622 m .543 .38103 L .58537 .36344 L p 1 g
F P
0 g
s
.79831 .3392 m .81717 .3715 L .77417 .39224 L p 1 g
F P
0 g
s
.77417 .39224 m .75495 .36099 L .79831 .3392 L p 1 g
F P
0 g
s
.69202 .35191 m .71197 .38118 L .66941 .3998 L p 1 g
F P
0 g
s
.66941 .3998 m .64908 .37157 L .69202 .35191 L p 1 g
F P
0 g
s
.83892 .28551 m .85807 .32619 L .81574 .35589 L p 1 g
F P
0 g
s
.81574 .35589 m .79637 .31684 L .83892 .28551 L p 1 g
F P
0 g
s
.73405 .31063 m .75383 .34549 L .7114 .37141 L p 1 g
F P
0 g
s
.7114 .37141 m .6913 .33824 L .73405 .31063 L p 1 g
F P
0 g
s
.19681 .38878 m .22456 .40373 L .18554 .41395 L p 1 g
F P
0 g
s
.18554 .41395 m .1573 .39997 L .19681 .38878 L p 1 g
F P
0 g
s
.62794 .3342 m .64875 .3631 L .60649 .38518 L p 1 g
F P
0 g
s
.60649 .38518 m .58527 .358 L .62794 .3342 L p 1 g
F P
0 g
s
.30493 .37896 m .33065 .397 L .29057 .40912 L p 1 g
F P
0 g
s
.29057 .40912 m .26436 .39207 L .30493 .37896 L p 1 g
F P
0 g
s
.84198 .31581 m .8605 .34916 L .81717 .3715 L p 1 g
F P
0 g
s
.81717 .3715 m .79831 .3392 L .84198 .31581 L p 1 g
F P
0 g
s
.41293 .36833 m .43682 .38947 L .39581 .40352 L p 1 g
F P
0 g
s
.39581 .40352 m .37144 .38341 L .41293 .36833 L p 1 g
F P
0 g
s
.73539 .33066 m .75495 .36099 L .71197 .38118 L p 1 g
F P
0 g
s
.71197 .38118 m .69202 .35191 L .73539 .33066 L p 1 g
F P
0 g
s
.62827 .34429 m .64908 .37157 L .6066 .38967 L p 1 g
F P
0 g
s
.6066 .38967 m .58537 .36344 L .62827 .34429 L p 1 g
F P
0 g
s
.77689 .2803 m .79637 .31684 L .75383 .34549 L p 1 g
F P
0 g
s
.75383 .34549 m .73405 .31063 L .77689 .2803 L p 1 g
F P
0 g
s
.19446 .4118 m .22342 .41592 L .18468 .4222 L p 1 g
F P
0 g
s
.18468 .4222 m .15506 .41957 L .19446 .4118 L p 1 g
F P
0 g
s
.41265 .37645 m .43672 .39311 L .39566 .40735 L p 1 g
F P
0 g
s
.39566 .40735 m .37102 .39236 L .41265 .37645 L p 1 g
F P
0 g
s
.30382 .39501 m .33014 .40543 L .29009 .4157 L p 1 g
F P
0 g
s
.29009 .4157 m .26316 .40688 L .30382 .39501 L p 1 g
F P
0 g
s
.81973 .2473 m .83892 .28551 L .79637 .31684 L p 1 g
F P
0 g
s
.79637 .31684 m .77689 .2803 L .81973 .2473 L p 1 g
F P
0 g
s
.67089 .30762 m .6913 .33824 L .64875 .3631 L p 1 g
F P
0 g
s
.64875 .3631 m .62794 .3342 L .67089 .30762 L p 1 g
F P
0 g
s
.77913 .30781 m .79831 .3392 L .75495 .36099 L p 1 g
F P
0 g
s
.75495 .36099 m .73539 .33066 L .77913 .30781 L p 1 g
F P
0 g
s
.56352 .33343 m .58527 .358 L .54298 .37899 L p 1 g
F P
0 g
s
.67163 .32356 m .69202 .35191 L .64908 .37157 L p 1 g
F P
0 g
s
.64908 .37157 m .62827 .34429 L .67163 .32356 L p 1 g
F P
0 g
s
.12785 .38717 m .1573 .39997 L .11861 .40973 L p 1 g
F P
0 g
s
.11861 .40973 m .08865 .39789 L .12785 .38717 L p 1 g
F P
0 g
s
.23713 .37613 m .26436 .39207 L .22456 .40373 L p 1 g
F P
0 g
s
.22456 .40373 m .19681 .38878 L .23713 .37613 L p 1 g
F P
0 g
s
.56358 .33816 m .58537 .36344 L .543 .38103 L p 1 g
F P
0 g
s
.543 .38103 m .52074 .35681 L .56358 .33816 L p 1 g
F P
0 g
s
.34622 .36435 m .37144 .38341 L .33065 .397 L p 1 g
F P
0 g
s
.33065 .397 m .30493 .37896 L .34622 .36435 L p 1 g
F P
0 g
s
.45507 .35172 m .47848 .3739 L .43682 .38947 L p 1 g
F P
0 g
s
.43682 .38947 m .41293 .36833 L .45507 .35172 L p 1 g
F P
0 g
s
.71403 .27827 m .73405 .31063 L .6913 .33824 L p 1 g
F P
0 g
s
.6913 .33824 m .67089 .30762 L .71403 .27827 L p 1 g
F P
0 g
s
.45494 .35768 m .47845 .37606 L .43672 .39311 L p 1 g
F P
0 g
s
.43672 .39311 m .41265 .37645 L .45494 .35768 L p 1 g
F P
0 g
s
.34533 .38028 m .37102 .39236 L .33014 .40543 L p 1 g
F P
0 g
s
.33014 .40543 m .30382 .39501 L .34533 .38028 L p 1 g
F P
0 g
s
.23487 .40118 m .26316 .40688 L .22342 .41592 L p 1 g
F P
0 g
s
.22342 .41592 m .19446 .4118 L .23487 .40118 L p 1 g
F P
0 g
s
.82319 .28335 m .84198 .31581 L .79831 .3392 L p 1 g
F P
0 g
s
.79831 .3392 m .77913 .30781 L .82319 .28335 L p 1 g
F P
0 g
s
.12375 .42032 m .15506 .41957 L .11675 .42455 L p 1 g
F P
0 g
s
.11675 .42455 m .08474 .42676 L .12375 .42032 L p 1 g
F P
0 g
s
.75725 .24622 m .77689 .2803 L .73405 .31063 L p 1 g
F P
0 g
s
.73405 .31063 m .71403 .27827 L .75725 .24622 L p 1 g
F P
0 g
s
.60667 .30788 m .62794 .3342 L .58527 .358 L p 1 g
F P
0 g
s
.58527 .358 m .56352 .33343 L .60667 .30788 L p 1 g
F P
0 g
s
.71543 .30123 m .73539 .33066 L .69202 .35191 L p 1 g
F P
0 g
s
.69202 .35191 m .67163 .32356 L .71543 .30123 L p 1 g
F P
0 g
s
.60693 .31793 m .62827 .34429 L .58537 .36344 L p 1 g
F P
0 g
s
.58537 .36344 m .56358 .33816 L .60693 .31793 L p 1 g
F P
0 g
s
.80044 .21152 m .81973 .2473 L .77689 .2803 L p 1 g
F P
0 g
s
.77689 .2803 m .75725 .24622 L .80044 .21152 L p 1 g
F P
0 g
s
.47848 .3739 m .45507 .35172 L .49783 .33355 L p 1 g
F P
0 g
s
.38822 .34821 m .41293 .36833 L .37144 .38341 L p 1 g
F P
0 g
s
.37144 .38341 m .34622 .36435 L .38822 .34821 L p 1 g
F P
0 g
s
.47845 .37606 m .45494 .35768 L .4978 .33603 L p 1 g
F P
0 g
s
.27822 .36199 m .30493 .37896 L .26436 .39207 L p 1 g
F P
0 g
s
.26436 .39207 m .23713 .37613 L .27822 .36199 L p 1 g
F P
0 g
s
.1679 .37499 m .19681 .38878 L .1573 .39997 L p 1 g
F P
0 g
s
.1573 .39997 m .12785 .38717 L .1679 .37499 L p 1 g
F P
0 g
s
.6501 .27952 m .67089 .30762 L .62794 .3342 L p 1 g
F P
0 g
s
.62794 .3342 m .60667 .30788 L .6501 .27952 L p 1 g
F P
0 g
s
.3876 .36265 m .41265 .37645 L .37102 .39236 L p 1 g
F P
0 g
s
.37102 .39236 m .34533 .38028 L .3876 .36265 L p 1 g
F P
0 g
s
.7596 .27729 m .77913 .30781 L .73539 .33066 L p 1 g
F P
0 g
s
.73539 .33066 m .71543 .30123 L .7596 .27729 L p 1 g
F P
0 g
s
.65077 .29611 m .67163 .32356 L .62827 .34429 L p 1 g
F P
0 g
s
.62827 .34429 m .60693 .31793 L .65077 .29611 L p 1 g
F P
0 g
s
.27623 .38767 m .30382 .39501 L .26316 .40688 L p 1 g
F P
0 g
s
.26316 .40688 m .23487 .40118 L .27623 .38767 L p 1 g
F P
0 g
s
.6937 .24841 m .71403 .27827 L .67089 .30762 L p 1 g
F P
0 g
s
.67089 .30762 m .6501 .27952 L .6937 .24841 L p 1 g
F P
0 g
s
.54116 .31382 m .56358 .33816 L .52074 .35681 L p 1 g
F P
0 g
s
.16388 .41101 m .19446 .4118 L .15506 .41957 L p 1 g
F P
0 g
s
.15506 .41957 m .12375 .42032 L .16388 .41101 L p 1 g
F P
0 g
s
.43089 .33052 m .45507 .35172 L .41293 .36833 L p 1 g
F P
0 g
s
.41293 .36833 m .38822 .34821 L .43089 .33052 L p 1 g
F P
0 g
s
.80409 .25175 m .82319 .28335 L .77913 .30781 L p 1 g
F P
0 g
s
.77913 .30781 m .7596 .27729 L .80409 .25175 L p 1 g
F P
0 g
s
.32006 .34633 m .34622 .36435 L .30493 .37896 L p 1 g
F P
0 g
s
.30493 .37896 m .27822 .36199 L .32006 .34633 L p 1 g
F P
0 g
s
.73736 .21459 m .75725 .24622 L .71403 .27827 L p 1 g
F P
0 g
s
.71403 .27827 m .6937 .24841 L .73736 .21459 L p 1 g
F P
0 g
s
.20877 .36132 m .23713 .37613 L .19681 .38878 L p 1 g
F P
0 g
s
.19681 .38878 m .1679 .37499 L .20877 .36132 L p 1 g
F P
0 g
s
.43054 .34211 m .45494 .35768 L .41265 .37645 L p 1 g
F P
0 g
s
.41265 .37645 m .3876 .36265 L .43054 .34211 L p 1 g
F P
0 g
s
.69503 .27268 m .71543 .30123 L .67163 .32356 L p 1 g
F P
0 g
s
.67163 .32356 m .65077 .29611 L .69503 .27268 L p 1 g
F P
0 g
s
.78098 .17813 m .80044 .21152 L .75725 .24622 L p 1 g
F P
0 g
s
.75725 .24622 m .73736 .21459 L .78098 .17813 L p 1 g
F P
0 g
s
.58486 .28415 m .60667 .30788 L .56352 .33343 L p 1 g
F P
0 g
s
.58502 .29249 m .60693 .31793 L .56358 .33816 L p 1 g
F P
0 g
s
.56358 .33816 m .54116 .31382 L .58502 .29249 L p 1 g
F P
0 g
s
.31845 .37123 m .34533 .38028 L .30382 .39501 L p 1 g
F P
0 g
s
.30382 .39501 m .27623 .38767 L .31845 .37123 L p 1 g
F P
0 g
s
.47418 .31127 m .49783 .33355 L .45507 .35172 L p 1 g
F P
0 g
s
.45507 .35172 m .43089 .33052 L .47418 .31127 L p 1 g
F P
0 g
s
.62884 .25397 m .6501 .27952 L .60667 .30788 L p 1 g
F P
0 g
s
.60667 .30788 m .58486 .28415 L .62884 .25397 L p 1 g
F P
0 g
s
.73967 .24763 m .7596 .27729 L .71543 .30123 L p 1 g
F P
0 g
s
.71543 .30123 m .69503 .27268 L .73967 .24763 L p 1 g
F P
0 g
s
.36262 .32911 m .38822 .34821 L .34622 .36435 L p 1 g
F P
0 g
s
.34622 .36435 m .32006 .34633 L .36262 .32911 L p 1 g
F P
0 g
s
.20505 .39878 m .23487 .40118 L .19446 .4118 L p 1 g
F P
0 g
s
.19446 .4118 m .16388 .41101 L .20505 .39878 L p 1 g
F P
0 g
s
.47406 .31866 m .4978 .33603 L .45494 .35768 L p 1 g
F P
0 g
s
.45494 .35768 m .43054 .34211 L .47406 .31866 L p 1 g
F P
0 g
s
.25044 .34612 m .27822 .36199 L .23713 .37613 L p 1 g
F P
0 g
s
.23713 .37613 m .20877 .36132 L .25044 .34612 L p 1 g
F P
0 g
s
.62936 .26955 m .65077 .29611 L .60693 .31793 L p 1 g
F P
0 g
s
.60693 .31793 m .58502 .29249 L .62936 .26955 L p 1 g
F P
0 g
s
.67297 .22103 m .6937 .24841 L .6501 .27952 L p 1 g
F P
0 g
s
.6501 .27952 m .62884 .25397 L .67297 .22103 L p 1 g
F P
0 g
s
.78463 .22097 m .80409 .25175 L .7596 .27729 L p 1 g
F P
0 g
s
.7596 .27729 m .73967 .24763 L .78463 .22097 L p 1 g
F P
0 g
s
.36145 .35183 m .3876 .36265 L .34533 .38028 L p 1 g
F P
0 g
s
.34533 .38028 m .31845 .37123 L .36145 .35183 L p 1 g
F P
0 g
s
.49783 .33355 m .47418 .31127 L .51805 .29041 L p 1 g
F P
0 g
s
.71716 .18539 m .73736 .21459 L .6937 .24841 L p 1 g
F P
0 g
s
.6937 .24841 m .67297 .22103 L .71716 .18539 L p 1 g
F P
0 g
s
.4978 .33603 m .47406 .31866 L .51805 .2923 L p 1 g
F P
0 g
s
.40585 .31033 m .43089 .33052 L .38822 .34821 L p 1 g
F P
0 g
s
.38822 .34821 m .36262 .32911 L .40585 .31033 L p 1 g
F P
0 g
s
.76127 .14713 m .78098 .17813 L .73736 .21459 L p 1 g
F P
0 g
s
.73736 .21459 m .71716 .18539 L .76127 .14713 L p 1 g
F P
0 g
s
.67414 .24499 m .69503 .27268 L .65077 .29611 L p 1 g
F P
0 g
s
.65077 .29611 m .62936 .26955 L .67414 .24499 L p 1 g
F P
0 g
s
.2472 .38358 m .27623 .38767 L .23487 .40118 L p 1 g
F P
0 g
s
.23487 .40118 m .20505 .39878 L .2472 .38358 L p 1 g
F P
0 g
s
.29287 .32938 m .32006 .34633 L .27822 .36199 L p 1 g
F P
0 g
s
.27822 .36199 m .25044 .34612 L .29287 .32938 L p 1 g
F P
0 g
s
.56246 .26795 m .58502 .29249 L .54116 .31382 L p 1 g
F P
0 g
s
.40513 .32946 m .43054 .34211 L .3876 .36265 L p 1 g
F P
0 g
s
.3876 .36265 m .36145 .35183 L .40513 .32946 L p 1 g
F P
0 g
s
.71929 .21881 m .73967 .24763 L .69503 .27268 L p 1 g
F P
0 g
s
.69503 .27268 m .67414 .24499 L .71929 .21881 L p 1 g
F P
0 g
s
.44972 .28994 m .47418 .31127 L .43089 .33052 L p 1 g
F P
0 g
s
.43089 .33052 m .40585 .31033 L .44972 .28994 L p 1 g
F P
0 g
s
.60701 .23101 m .62884 .25397 L .58486 .28415 L p 1 g
F P
0 g
s
.58486 .28415 m .5624 .26307 L .60701 .23101 L p 1 g
F P
0 g
s
.33603 .31106 m .36262 .32911 L .32006 .34633 L p 1 g
F P
0 g
s
.32006 .34633 m .29287 .32938 L .33603 .31106 L p 1 g
F P
0 g
s
.29023 .36536 m .31845 .37123 L .27623 .38767 L p 1 g
F P
0 g
s
.26055 .36023 m
.26685 .3621 L
s
.30057 .37209 m
.42119 .40784 L
s
.27623 .38767 m .2472 .38358 L .29023 .36536 L p 1 g
F P
0 g
s
.26685 .3621 m
.27077 .36326 L
s
.60736 .24387 m .62936 .26955 L .58502 .29249 L p 1 g
F P
0 g
s
.58502 .29249 m .56246 .26795 L .60736 .24387 L p 1 g
F P
0 g
s
.44938 .30414 m .47406 .31866 L .43054 .34211 L p 1 g
F P
0 g
s
.43054 .34211 m .40513 .32946 L .44938 .30414 L p 1 g
F P
0 g
s
.65177 .19618 m .67297 .22103 L .62884 .25397 L p 1 g
F P
0 g
s
.62884 .25397 m .60701 .23101 L .65177 .19618 L p 1 g
F P
0 g
s
.76476 .19099 m .78463 .22097 L .73967 .24763 L p 1 g
F P
0 g
s
.73967 .24763 m .71929 .21881 L .76476 .19099 L p 1 g
F P
0 g
s
.69656 .15865 m .71716 .18539 L .67297 .22103 L p 1 g
F P
0 g
s
.67297 .22103 m .65177 .19618 L .69656 .15865 L p 1 g
F P
0 g
s
.49419 .26795 m .51805 .29041 L .47418 .31127 L p 1 g
F P
0 g
s
.47418 .31127 m .44972 .28994 L .49419 .26795 L p 1 g
F P
0 g
s
.74126 .11851 m .76127 .14713 L .71716 .18539 L p 1 g
F P
0 g
s
.71716 .18539 m .69656 .15865 L .74126 .11851 L p 1 g
F P
0 g
s
.65269 .21815 m .67414 .24499 L .62936 .26955 L p 1 g
F P
0 g
s
.62936 .26955 m .60736 .24387 L .65269 .21815 L p 1 g
F P
0 g
s
.49411 .27587 m .51805 .2923 L .47406 .31866 L p 1 g
F P
0 g
s
.47406 .31866 m .44938 .30414 L .49411 .27587 L p 1 g
F P
0 g
s
.37988 .29113 m .40585 .31033 L .36262 .32911 L p 1 g
F P
0 g
s
.36262 .32911 m .33603 .31106 L .37988 .29113 L p 1 g
F P
0 g
s
.33404 .34412 m .36145 .35183 L .31845 .37123 L p 1 g
F P
0 g
s
.31845 .37123 m .29023 .36536 L .33404 .34412 L p 1 g
F P
0 g
s
.27077 .36326 m
.30057 .37209 L
s
.51805 .29041 m .49419 .26795 L .53919 .24432 L p 1 g
F P
0 g
s
.51805 .2923 m .49411 .27587 L .53919 .2447 L p 1 g
F P
0 g
s
.69841 .19079 m .71929 .21881 L .67414 .24499 L p 1 g
F P
0 g
s
.67414 .24499 m .65269 .21815 L .69841 .19079 L p 1 g
F P
0 g
s
.42438 .26959 m .44972 .28994 L .40585 .31033 L p 1 g
F P
0 g
s
.40585 .31033 m .37988 .29113 L .42438 .26959 L p 1 g
F P
0 g
s
.37855 .31986 m .40513 .32946 L .36145 .35183 L p 1 g
F P
0 g
s
.36145 .35183 m .33404 .34412 L .37855 .31986 L p 1 g
F P
0 g
s
.58453 .21068 m .60701 .23101 L .5624 .26307 L p 1 g
F P
0 g
s
.5624 .26307 m .54129 .24636 L p .58453 .21068 L 1 g
F P
0 g
s
.54129 .24636 m .54089 .24604 L p .54127 .24574 L 1 g
F P
0 g
s
.54129 .24636 m .54127 .24574 L .58453 .21068 L 1 g
F
.54116 .24323 m .58453 .21068 L p .54089 .24604 L F P
0 g
s
.53919 .24432 m .54102 .24024 L .54112 .2425 L 1 g
F
.54116 .24323 m .5409 .24337 L p .54112 .2425 L F P
0 g
s
.5409 .24337 m .53919 .24432 L p .54112 .2425 L 1 g
F P
0 g
s
.53919 .24432 m .51514 .22158 L .54977 .20171 L p 1 g
F P
0 g
s
.56127 .19306 m .58453 .21068 L .54131 .24311 L p 1 g
F P
0 g
s
.54143 .24255 m .56127 .19306 L .54131 .24311 L 1 g
F
.54041 .24555 m .53919 .24432 L .54116 .24323 L p F P
0 g
s
.54102 .24024 m .54263 .23666 L .54116 .24323 L 1 g
F
.54116 .24323 m .53919 .2447 L p .54263 .23666 L F P
0 g
s
.54263 .23666 m .54518 .23069 L .54116 .24323 L 1 g
F
.54518 .23069 m .56127 .19306 L .54116 .24323 L F
.54263 .23666 m .56129 .1951 L .54116 .24323 L F
.54977 .20171 m .55139 .20078 L p .54263 .23666 L F P
0 g
s
.53919 .24432 m .54977 .20171 L .54263 .23666 L 1 g
F
.51514 .22158 m .53919 .24432 L .49419 .26795 L p F P
0 g
s
.53919 .2447 m .51511 .22913 L .55139 .20078 L p 1 g
F P
0 g
s
.55139 .20078 m .55203 .20028 L p .54871 .21178 L 1 g
F P
0 g
s
.53919 .2447 m .55139 .20078 L .54871 .21178 L 1 g
F
.55203 .20028 m .56127 .19306 L p .54263 .23666 L F P
0 g
s
.53919 .2447 m .55203 .20028 L .54263 .23666 L 1 g
F
.55139 .20078 m .56129 .1951 L p .54263 .23666 L F P
0 g
s
.54131 .24311 m .54116 .24323 L p .54143 .24255 L 1 g
F P
0 g
s
.54089 .24604 m .54041 .24555 L p .54116 .24323 L 1 g
F P
0 g
s
.56129 .1951 m .58469 .21905 L .54116 .24323 L p 1 g
F P
0 g
s
.56246 .26795 m .54089 .24604 L p .58469 .21905 L 1 g
F P
0 g
s
.54116 .24323 m .58469 .21905 L p .54089 .24604 L 1 g
F P
0 g
s
.54089 .24604 m .53919 .2447 L .54116 .24323 L p 1 g
F P
0 g
s
.74444 .1618 m .76476 .19099 L .71929 .21881 L p 1 g
F P
0 g
s
.71929 .21881 m .69841 .19079 L .74444 .1618 L p 1 g
F P
0 g
s
.58469 .21905 m .60736 .24387 L .56246 .26795 L p 1 g
F P
0 g
s
.62999 .17388 m .65177 .19618 L .60701 .23101 L p 1 g
F P
0 g
s
.60701 .23101 m .58453 .21068 L .62999 .17388 L p 1 g
F P
0 g
s
.67547 .13438 m .69656 .15865 L .65177 .19618 L p 1 g
F P
0 g
s
.65177 .19618 m .62999 .17388 L .67547 .13438 L p 1 g
F P
0 g
s
.42364 .29258 m .44938 .30414 L .40513 .32946 L p 1 g
F P
0 g
s
.40513 .32946 m .37855 .31986 L .42364 .29258 L p 1 g
F P
0 g
s
.72084 .09228 m .74126 .11851 L .69656 .15865 L p 1 g
F P
0 g
s
.69656 .15865 m .67547 .13438 L .72084 .09228 L p 1 g
F P
0 g
s
.46948 .24641 m .49419 .26795 L .44972 .28994 L p 1 g
F P
0 g
s
.44972 .28994 m .42438 .26959 L .46948 .24641 L p 1 g
F P
0 g
s
.63063 .19214 m .65269 .21815 L .60736 .24387 L p 1 g
F P
0 g
s
.60736 .24387 m .58469 .21905 L .63063 .19214 L p 1 g
F P
0 g
s
.4692 .26232 m .49411 .27587 L .44938 .30414 L p 1 g
F P
0 g
s
.44938 .30414 m .42364 .29258 L .4692 .26232 L p 1 g
F P
0 g
s
.67696 .16357 m .69841 .19079 L .65269 .21815 L p 1 g
F P
0 g
s
.65269 .21815 m .63063 .19214 L .67696 .16357 L p 1 g
F P
0 g
s
.49419 .26795 m .46948 .24641 L .51514 .22158 L p 1 g
F P
0 g
s
.51511 .22913 m .53919 .2447 L .49411 .27587 L p 1 g
F P
0 g
s
.49411 .27587 m .4692 .26232 L .51511 .22913 L p 1 g
F P
0 g
s
.7236 .13336 m .74444 .1618 L .69841 .19079 L p 1 g
F P
0 g
s
.69841 .19079 m .67696 .16357 L .7236 .13336 L p 1 g
F P
0 g
s
.60754 .15419 m .62999 .17388 L .58453 .21068 L p 1 g
F P
0 g
s
.58453 .21068 m .56127 .19306 L .60754 .15419 L p 1 g
F P
0 g
s
.65381 .11263 m .67547 .13438 L .62999 .17388 L p 1 g
F P
0 g
s
.62999 .17388 m .60754 .15419 L .65381 .11263 L p 1 g
F P
0 g
s
.69994 .06847 m .72084 .09228 L .67547 .13438 L p 1 g
F P
0 g
s
.67547 .13438 m .65381 .11263 L .69994 .06847 L p 1 g
F P
0 g
s
.6079 .16695 m .63063 .19214 L .58469 .21905 L p 1 g
F P
0 g
s
.58469 .21905 m .56129 .1951 L .6079 .16695 L p 1 g
F P
0 g
s
.65488 .13713 m .67696 .16357 L .63063 .19214 L p 1 g
F P
0 g
s
.63063 .19214 m .6079 .16695 L .65488 .13713 L p 1 g
F P
0 g
s
.70219 .10566 m .7236 .13336 L .67696 .16357 L p 1 g
F P
0 g
s
.67696 .16357 m .65488 .13713 L .70219 .10566 L p 1 g
F P
0 g
s
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.07064 .24203 m
.07538 .24656 L
s
[(0.1)] .06115 .23296 1 .9554 Mshowa
.20849 .18805 m
.21301 .19281 L
s
[(0.125)] .19944 .17854 .95093 1 Mshowa
.35408 .13104 m
.35834 .13603 L
s
[(0.15)] .34555 .12107 .85517 1 Mshowa
.50809 .07074 m
.51205 .07596 L
s
[(0.175)] .50015 .06029 .75942 1 Mshowa
.67126 .00684 m
.67489 .01231 L
s
[(0.2)] .664 -0.00409 .66367 1 Mshowa
.125 Mabswid
.09762 .23146 m
.10044 .23421 L
s
.12489 .22079 m
.12768 .22356 L
s
.15245 .20999 m
.15522 .21279 L
s
.18032 .19908 m
.18306 .20191 L
s
.23696 .1769 m
.23965 .17978 L
s
.26576 .16563 m
.26841 .16853 L
s
.29487 .15423 m
.29749 .15716 L
s
.32431 .1427 m
.3269 .14566 L
s
.38418 .11925 m
.38671 .12227 L
s
.41463 .10733 m
.41712 .11038 L
s
.44543 .09527 m
.44788 .09835 L
s
.47658 .08308 m
.479 .08618 L
s
.53996 .05826 m
.5423 .06142 L
s
.57221 .04563 m
.57451 .04882 L
s
.60484 .03285 m
.6071 .03607 L
s
.63785 .01992 m
.64007 .02318 L
s
.04394 .25248 m
.04681 .25517 L
s
gsave
.29165 .07573 -81.4846 -16.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
81.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 233.563},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jA000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00
ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00;9oo`8000Aoo`8000=oo`@005moo`00/Goo00@007oo
Ool000Qoo`03001oogoo061oo`00/Goo00@007ooOol000Uoo`03001oogoo05moo`00/Goo00@007oo
Ool000Yoo`03001oogoo05ioo`00/Goo00@007ooOol000Qoo`04001oogoo001OOol00;9oo`8000Yo
o`80061oo`00ogoo8Goo003oOolQOol00?moob5oo`00a7oo0P00FWoo0031Ool300000goo0000001I
Ool00;moo`@0009oo`8005Uoo`00_Goo0P001Woo00<007oo0000F7oo002iOol40008Ool01000Ool0
00003Goo0P0017oo0P000goo1@00>Woo0022Ool20004Ool20003Ool50003Ool00`00Oomoo`02Ool2
000KOol4000:Ool01000Oomoo`0037oo00@007ooOol000Yoo`03001oogoo03Yoo`00PGoo00@007oo
Ool000Yoo`03001oogoo00=oo`05001oogooOol00002Ool00`00Oomoo`0FOol2000>Ool01@00Oomo
ogoo00002goo00@007ooOol000Yoo`03001oogoo03Yoo`00PGoo00@007ooOol000Yoo`03001oogoo
00=oo`03001oogoo00Aoo`03001oogoo01=oo`<0011oo`03001oogoo009oo`03001oogoo009oo`@0
009oo`04001oogoo000:Ool00`00Oomoo`0jOol0085oo`04001oogoo000:Ool00`00Oomoo`04Ool0
1000Oomoogoo0P0057oo0`004Woo00<007ooOol00Woo00<007ooOol027oo00@007ooOol000Yoo`03
001oogoo03Yoo`00F7oo0P0000=oo`00000097oo00@007ooOol000Yoo`05001oogooOol00002Ool0
1@00Oomoogoo00004Woo0`005Goo00<007ooOol00goo00<007ooOol027oo0P002Woo0P00?7oo001I
Ool00`00Ool0000VOol2000:Ool20003Ool40003Ool3000=Ool3000IOol01`00Oomoogoo001oo`00
05=oo`00FWoo00<007ooOol0Bgoo10006Woo00<007ooOol00goo00<007ooOol0DGoo001HOol20000
0goo00000018Ool3000NOol00`00Oomoo`04Ool00`00Oomoo`1@Ool00:=oo`80025oo`03001oogoo
00Aoo`03001oogoo051oo`00X7oo0`0000=oo`00Ool087oo00<007ooOol01Goo0P00D7oo002NOol2
000VOol00`00Ool00005Ool2001@Ool009]oo`<002Qoo`800003Ool007oo00Aoo`03001oogoo04io
o`00VGoo0P000Woo00<007ooOol09Woo00@007ooOol000Ioo`03001oogoo04eoo`00UWoo0`00;7oo
0P000goo00<007ooOol017oo00<007ooOol0C7oo001ROol20004Ool20003Ool50002Ool2000MOol3
000^Ool00`00Ool00004Ool00`00Oomoo`03Ool2001=Ool0065oo`04001oogoo000:Ool01@00Oomo
ogoo00000Woo00<007ooOol067oo0P00;goo0P000Woo00<007ooOol00Woo00<007ooOol00goo0P00
CGoo001QOol01000Oomoo`002Woo00<007ooOol017oo00<007ooOol05Goo0`00<7oo00<007ooOol0
0Woo00<007ooOol00goo00<007ooOol017oo00<007ooOol0BWoo001QOol01000Oomoo`002Woo00<0
07ooOol00Woo0P005Woo0`00<Goo00<007ooOol00goo00<007ooOol00Woo00<007oo00001Goo00<0
07ooOol0BWoo001QOol01000Oomoo`002Woo00<007ooOol00Woo00<007ooOol04Woo0`00<goo00<0
07ooOol017oo00D007ooOomoo`0000=oo`03001oogoo00=oo`03001oogoo04Uoo`00HWoo0P002Woo
0P0017oo0`0047oo0P00=Goo00<007ooOol01Goo00@007oo000000Eoo`03001oogoo00=oo`03001o
ogoo04Qoo`00Q7oo1000=Goo00<007ooOol01Woo0P0000=oo`00Ool017oo00<007ooOol00Woo0P00
BWoo0021Ool3000hOol00`00Ool00006Ool20003Ool00`00Oomoo`03Ool00`00Oomoo`03Ool00`00
Oomoo`17Ool007moo`8003Yoo`05001oogooOol00003Ool200000goo001oo`03Ool00`00Oomoo`03
Ool00`00Oomoo`03Ool00`00Oomoo`16Ool007aoo`<00003Ool007oo03Qoo`03001oogoo00=oo`<0
00=oo`03001oogoo009oo`03001oogoo009oo`03001oo`0000Aoo`03001oogoo04Ioo`00NWoo0P00
?Goo00<007ooOol017oo00<007ooOol017oo00<007ooOol00Woo00@007ooOol000=oo`03001oogoo
009oo`03001oogoo00]oo`8000Aoo`8000Aoo`8000Eoo`03001oogoo009oo`80021oo`00Mgoo0`00
?Woo00<007ooOol00goo0P001goo00<007ooOol00goo0P001Goo00@007ooOomoo`8000]oo`04001o
ogoo0008Ool01000Oomoo`0017oo00D007ooOomoo`00009oo`03001oogoo01eoo`00MGoo0P000goo
00<007ooOol0>Woo00<007ooOol00goo00<007ooOol01goo00@007ooOomoo`800003Ool007oo00Ao
o`03001oogoo009oo`03001oogoo00Uoo`04001oogoo0008Ool01000Oomoo`0017oo00<007ooOol0
17oo00<007ooOol07Goo000jOol20004Ool20003Ool500000goo000000020003Ool2000NOol30011
Ool00`00Oomoo`02Ool200000goo001oo`07Ool01000Oomoo`000goo00<007ooOol017oo00<007oo
Ool00Woo00<007ooOol00Woo10000Woo00@007ooOol000Qoo`04001oogoo0005Ool01000Oomoogoo
0P0087oo000iOol01000Oomoo`002Woo00D007ooOomoo`0000Eoo`04001oogoo000JOol300000goo
001oo`10Ool00`00Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`05Ool30005Ool00`00Oomoo`04
Ool00`00Oomoo`02Ool00`00Oomoo`07Ool01000Oomoo`0027oo00@007ooOol0009oo`04001oogoo
0003Ool00`00Oomoo`0OOol003Uoo`04001oogoo000:Ool00`00Oomoo`02Ool00`00Oomoo`05Ool0
0`00Oomoo`0FOol20015Ool01000Oomoogoo0P001Goo00<007ooOol017oo0P0027oo00<007ooOol0
0Woo00<007oo00000goo0P002Goo0P002Woo0P000goo10000goo0`007goo000iOol01000Oomoo`00
2Woo00<007ooOol00goo00<007ooOol00Woo0P005Woo0`00AWoo00D007ooOomoo`0000Qoo`03001o
ogoo009oo`03001oo`0000Uoo`04001oogoo0002Ool00`00Oomoo`030013Ool003Uoo`04001oogoo
000:Ool01@00Oomoogoo00000Woo00D007ooOomoo`0001Eoo`80009oo`03001oogoo04=oo`05001o
o`00Ool0000:Ool01000Ool000000Woo00<007ooOol01goo00<007oo000017oo00<007ooOol00Woo
00<007ooOol0@7oo000jOol2000:Ool20004Ool20004Ool3000@Ool30018Ool20002Ool2000<Ool0
0`00Oomoo`02Ool00`00Oomoo`08Ool00`00Oomoo`03Ool00`00Oomoo`02Ool00`00Oomoo`10Ool0
06=oo`<004Uoo`05001oogooOol0000<Ool200000goo001oo`02Ool00`00Oomoo`06Ool200000goo
001oo`04Ool00`00Oomoo`02Ool00`00Oomoo`0oOol0061oo`<004]oo`03001oogoo00800003Ool0
07oo00Qoo`8000Aoo`04001oogoo0006Ool20003Ool00`00Oomoo`03Ool00`00Oomoo`02Ool20010
Ool005eoo`<004eoo`04001oo`000004Ool00`00Oomoo`05Ool00`00Oomoo`04Ool01000Oomoo`00
17oo0P001Woo00<007ooOol00goo00<007ooOol00Woo00<007ooOol0?Woo001KOol20002Ool00`00
Oomoo`1:Ool30007Ool00`00Oomoo`02Ool20008Ool00`00Ool00002Ool20009Ool01`00Oomoogoo
001oo`0000Aoo`03001oogoo03eoo`00F7oo0`00Cgoo0`002Woo00@007oo000000]oo`@000]oo`04
001oogoo0002Ool00`00Oomoo`02Ool00`00Oomoo`0bOol20009Ool005Ioo`<0051oo`8000eoo`03
001oogoo00aoo`03001oogoo00aoo`8000Aoo`03001oogoo009oo`8003=oo`03001oogoo00Moo`00
Dgoo0`000goo00<007ooOol0Bgoo0P003Goo0P0000=oo`00Ool02Woo10002goo0P0000=oo`00Ool0
17oo00@007ooOomoo`8003Aoo`03001oogoo00Ioo`009Woo0P0017oo0P000goo1@006goo0P00Dgoo
00@007ooOol000]oo`03001oogoo009oo`03001oogoo00Ioo`80009oo`8000Yoo`05001oogooOol0
0005Ool00`00Oomoo`02Ool00`00Oomoo`0aOol00`00Ool00007Ool002Eoo`04001oogoo000:Ool0
0`00Oomoo`0HOol300000goo001oo`1@Ool20004Ool00`00Oomoo`06Ool20005Ool00`00Oomoo`05
Ool00`00Oomoo`02Ool00`00Ool00008Ool00`00Oomoo`03Ool00`00Oomoo`03Ool00`00Oomoo`02
Ool00`00Oomoo`0_Ool01000Oomoo`001goo000UOol01000Oomoo`002Woo00<007ooOol05Goo0`00
E7oo0`001Goo00<007ooOol017oo0P0027oo00<007ooOol00Woo0P001Woo00<007oo00001Goo0P00
1goo00<007ooOol00Woo00<007ooOol00goo00<007ooOol0;Goo0`0000=oo`0000001Woo000UOol0
1000Oomoo`002Woo00<007ooOol04goo0`00E7oo0`0027oo00<007ooOol00Woo00<007ooOol02Goo
00@007oo000000Qoo`04001oogoo0003Ool00`00Oomoo`08Ool00`00Oomoo`02Ool00`00Oomoo`02
Ool2000jOol002Eoo`04001oogoo000:Ool00`00Oomoo`0@Ool3001EOol200000goo001oo`09Ool0
1000Ool000003Goo00<007ooOol027oo00D007ooOol007oo008000]oo`07001oogooOol007oo0000
17oo00<007ooOol0>7oo000VOol2000:Ool2000@Ool2001GOol01000Ool000002goo0P003Goo0P00
00=oo`00Ool027oo00D007ooOomoo`0000ioo`<0009oo`03001oogoo00=oo`03001oogoo03Moo`00
@Goo0`0000=oo`00Ool0E7oo0P0000=oo`00Ool02Woo0P000Woo00<007ooOol027oo0P000goo00<0
07ooOol01goo00H007oo000007oo000;Ool200000goo001oo`03Ool00`00Oomoo`02Ool00`00Oomo
o`0gOol003ioo`<005Qoo`80009oo`8000Uoo`8000Eoo`8000Moo`03001oogoo00Aoo`03001oogoo
00Ioo`8000Aoo`03001oogoo00Ioo`8000=oo`03001oogoo009oo`03001oogoo00=oo`03001oogoo
00]oo`8000Aoo`8000Aoo`8000Aoo`8001Moo`00?7oo1000F7oo00H007ooOomoo`00Ool30005Ool2
0008Ool20004Ool20008Ool00`00Oomoo`04Ool20005Ool00`00Oomoo`05Ool00`00Oomoo`04Ool0
0`00Oomoo`02Ool00`00Oomoo`02Ool2000;Ool01000Oomoo`0027oo00@007ooOol0009oo`04001o
ogoo000FOol003Uoo`<005Yoo`80009oo`8000Aoo`<00003Ool0000000aoo`04001oo`00000;Ool0
1000Oomoogoo0P0000=oo`00Ool01Goo00<007ooOol00Woo0P001goo00<007ooOol00Woo00<007oo
Ool00goo00<007ooOol02Goo00@007ooOol000Qoo`04001oogoo0005Ool00`00Oomoo`0DOol003Mo
o`8005]oo`8000=oo`03001oogoo00Ioo`8000moo`03001oogoo00]oo`04001oo`000003Ool00`00
Oomoo`05Ool01000Ool000002Woo00<007ooOol00Woo00<007ooOol00goo00<007ooOol00Woo1000
0Woo00@007ooOol000Qoo`04001oogoo0003Ool2000GOol003Aoo`<0009oo`03001oogoo05Ioo`80
00Aoo`03001oogoo00Moo`8000eoo`800003Ool007oo00aoo`03001oogoo009oo`8000Qoo`03001o
ogoo00Yoo`03001oogoo009oo`03001oogoo00Aoo`03001oogoo00Moo`04001oogoo0008Ool01000
Oomoo`000goo00<007ooOol05Woo000bOol2001LOol20002Ool01000Oomoo`0027oo0P000Woo00<0
07ooOol027oo0P0017oo00<007ooOol027oo0P0000=oo`0000000P000Woo00<007ooOol00goo0P00
00=oo`00Ool02goo00<007ooOol00P001Goo0P002Woo0P002Woo0P0017oo0`005Woo000`Ool20000
0goo001oo`1IOol20005Ool20008Ool00`00Oomoo`03Ool00`00Oomoo`05Ool20006Ool00`00Oomo
o`06Ool200000goo000000020004Ool01000Oomoogoo0P000goo00<007ooOol027oo0P0000=oo`00
00000Woo00<007ooOol017oo00<007ooOol0<Woo000`Ool2001JOol20006Ool00`00Oomoo`07Ool0
0`00Oomoo`05Ool00`00Oomoo`02Ool20009Ool00`00Oomoo`03Ool50003Ool00`00Oomoo`03Ool0
0`00Ool00006Ool00`00Oomoo`05Ool20003Ool01@00Oomoogoo00001goo00<007ooOol0<Goo000`
Ool00`00Ool0001GOol20007Ool20008Ool00`00Oomoo`07Ool01000Ool0000037oo00<007ooOol0
10001goo00<007ooOol00Woo0P001goo00<007ooOol017oo00<007ooOol017oo00D007ooOomoo`00
00Ioo`03001oogoo035oo`00<7oo0P0000=oo`00Ool0Dgoo0P001goo0P0000Aoo`00000000Aoo`80
00Yoo`8000ioo`@000]oo`04001oo`000002Ool00`00Oomoo`05Ool01000Oomoogoo0P0027oo00@0
07ooOol000Ioo`80039oo`00;goo00<007ooOol00Woo00<007ooOol0Cgoo0P000Woo00<007ooOol0
0Woo0P001Goo0`0000=oo`00Ool027oo0`000Woo00<007ooOol02Goo0`0000=oo`00Ool02goo0P00
17oo00<007ooOol01Goo00@007oo000000]oo`04001oogoo0006Ool00`00Oomoo`0`Ool002moo`03
001oogoo00=oo`03001oogoo04aoo`8000Eoo`800003Ool007oo00Qoo`8000Qoo`8000Ioo`03001o
ogoo00Ioo`<000=oo`03001oogoo00Qoo`80009oo`03001oogoo009oo`03001oogoo00Eoo`03001o
ogoo00]oo`04001oogoo0007Ool00`00Oomoo`0_Ool002moo`03001oogoo00Aoo`8004Yoo`8000Qo
o`8000Uoo`800003Ool0000000Eoo`03001oogoo00Moo`03001oogoo009oo`<000Moo`03001oogoo
00Eoo`8000Aoo`03001oogoo00=oo`03001oogoo009oo`800003Ool007oo00aoo`04001oogoo0007
Ool00`00Oomoo`0^Ool002moo`8000Moo`03001oogoo04Eoo`8000Qoo`8000Uoo`8000Eoo`800003
Ool0000000]oo`D000Yoo`03001oogoo009oo`8000Moo`03001oogoo00=oo`04001oo`000003Ool0
0`00Oomoo`09Ool2000017oo001oo`001goo0P00;goo000_Ool00`00Oomoo`07Ool00`00Oomoo`12
Ool20009Ool00`00Oomoo`07Ool20009Ool00`00Oomoo`09Ool5000<Ool00`00Oomoo`02000:Ool0
0`00Oomoo`02Ool20006Ool00`00Oomoo`06Ool20003Ool200000goo001oo`06Ool00`00Oomoo`0]
Ool002moo`03001oogoo00Qoo`03001oogoo03moo`80009oo`8000Eoo`800003Ool0000000Eoo`80
00Uoo`800003Ool0000000Ioo`<00005Ool00000Ool0000=Ool2000<Ool00`00Oomoo`0200000goo
001oo`06Ool00`00Oomoo`04Ool00`00Oomoo`04Ool00`00Ool00008Ool00`00Oomoo`0/Ool002io
o`03001oogoo00Yoo`03001oogoo03aoo`8000Ioo`05001oogooOol00005Ool200000goo00000009
Ool20005Ool00`00Oomoo`030002Ool20004Ool00`00Oomoo`08Ool200000goo001oo`0<Ool20004
Ool00`00Oomoo`05Ool01000Oomoogoo0P001goo00@007ooOol000Moo`03001oogoo02aoo`00;Woo
00<007ooOol02goo00<007ooOol0>Goo0P002Goo0`0027oo0P002Goo00<007ooOol01Goo0`000goo
0P001Woo00<007ooOol01Woo0P0017oo00<007ooOol027oo0P0000=oo`00Ool017oo00<007ooOol0
17oo00@007oo000000Yoo`03001oo`0000Qoo`03001oogoo02]oo`00;Woo0`0037oo00<007ooOol0
=Woo0P002Woo0P0027oo0P0000Aoo`00000000Eoo`8000Eoo`<00004Ool007ooOol20009Ool00`00
Oomoo`03Ool20006Ool00`00Oomoo`06Ool20003Ool00`00Oomoo`03Ool00`00Oomoo`05Ool00`00
Oomoo`0:Ool01000Oomoo`0027oo00<007ooOol0:Woo000^Ool00`00Oomoo`0=Ool00`00Oomoo`0c
Ool2000;Ool00`00Oomoo`06Ool20005Ool300000goo00000004Ool30004Ool3000<Ool00`00Oomo
o`020009Ool00`00Oomoo`03Ool20006Ool00`00Oomoo`03Ool00`00Oomoo`02Ool200000goo001o
o`0;Ool00`00Ool00008Ool00`00Oomoo`0ZOol001Ioo`8000Aoo`8000Eoo`<0009oo`8000Aoo`03
001oogoo00ioo`03001oogoo031oo`80009oo`8000Moo`8000Qoo`03001oogoo00Moo`8000=oo`<0
00Eoo`80009oo`03001oogoo00]oo`8000aoo`03001oogoo008000Uoo`03001oogoo00=oo`04001o
o`000003Ool00`00Oomoo`0:Ool01000Oomoo`0027oo00<007ooOol037oo0P0017oo0P0017oo0P00
0goo10000goo0P000goo000EOol01000Oomoo`002goo00@007ooOol0009oo`05001oogooOol0000A
Ool00`00Oomoo`0/Ool30006Ool00`00Oomoo`02Ool30007Ool20009Ool2000017oo000000001goo
00<007ooOol00goo00<007ooOol027oo0P0000=oo`00Ool037oo0P0037oo00<007ooOol00Woo0P00
1Woo00<007ooOol01goo0P0000Aoo`00Ool000Qoo`8000aoo`04001oogoo0008Ool01000Oomoo`00
0Woo00<007ooOol00goo00@007ooOol0009oo`005Goo00@007ooOol000Qoo`D00005Ool007ooOol0
0003Ool00`00Oomoo`0@Ool00`00Oomoo`0YOol2000:Ool40003Ool30002Ool20009Ool200001Woo
00000000Ool000Eoo`8000Moo`03001oogoo00Eoo`8000Aoo`03001oogoo00Qoo`800003Ool007oo
00]oo`03001oogoo00800003Ool007oo00Eoo`03001oogoo00Eoo`8000=oo`04001oogoo0008Ool0
0`00Oomoo`0:Ool01000Oomoo`0027oo00@007ooOol000=oo`03001oogoo00Eoo`40005oo`5oo`00
5Goo00@007ooOol000Qoo`04001oogoo0003Ool20003Ool3000BOol00`00Oomoo`0UOol3000<Ool2
0008Ool30009Ool40005Ool200000goo0000000:Ool00`00Oomoo`02Ool20006Ool00`00Oomoo`06
Ool20003Ool00`00Oomoo`0;Ool20004Ool00`00Oomoo`04Ool00`00Oomoo`02Ool20006Ool30009
Ool00`00Oomoo`03Ool40002Ool01000Oomoo`0027oo00@007ooOol000Aoo`03001oogoo009oo`80
00=oo`005Goo00@007ooOol000Uoo`03001oo`00009oo`04001oogoo0002Ool00`00Oomoo`0COol0
0`00Oomoo`0ROol20002Ool20009Ool20009Ool2000017oo000000001Goo0`002Woo00<007ooOol0
2goo00@007oo000000Uoo`03001oogoo00=oo`8000Ioo`03001oogoo00Qoo`800003Ool007oo00Ao
o`03001oogoo00=oo`03001oogoo008000Qoo`04001oogoo0009Ool00`00Oomoo`08Ool01000Oomo
o`0027oo00@007ooOol0009oo`04001oogoo0003Ool00`00Oomoo`02Ool001Ioo`8000]oo`8000=o
o`8000=oo`03001oogoo01Aoo`8001moo`<000Ioo`03001oogoo00Aoo`8000Uoo`8000Eoo`H000]o
o`800003Ool007oo00]oo`8000aoo`03001oogoo008000Uoo`03001oogoo00Eoo`8000=oo`03001o
ogoo00=oo`03001oogoo00Aoo`8000]oo`03001oo`0000Qoo`<000Yoo`8000Yoo`8000Aoo`8000Ao
o`<0009oo`00;Goo00<007ooOol05Woo00<007ooOol06Woo0P002Woo0P000Woo0`0027oo0P001goo
1@002Woo0P0017oo00<007ooOol01goo0P000Woo00<007ooOol02Woo0P0037oo00<007ooOol00Woo
0P001Woo00<007ooOol00goo00@007ooOomoo`800003Ool007oo00Yoo`04001oogoo0009Ool00`00
Oomoo`0UOol002eoo`8001Qoo`03001oogoo01Ioo`<000eoo`<000=oo`<000=oo`8000Ioo`D0009o
o`<000Ioo`8000Moo`03001oogoo00=oo`<000Aoo`03001oogoo00Qoo`800003Ool007oo00]oo`03
001oogoo008000Qoo`03001oogoo00Aoo`<000=oo`03001oogoo00Yoo`03001oo`0000Yoo`03001o
ogoo02Aoo`00;Goo00<007ooOol067oo00<007ooOol04Woo0`000Woo0`0027oo100027oo0`001Goo
1P001Woo0`000Woo0P002Woo00<007ooOol00P0027oo00<007ooOol01Goo0P0017oo00<007ooOol0
2Goo0`002goo00<007ooOol00Woo0`001Goo00<007ooOol01goo0P0000=oo`0000002Woo00<007oo
Ool097oo000]Ool00`00Oomoo`0IOol00`00Oomoo`0>Ool30007Ool30002Ool60008Ool20003Ool5
00000goo0000000;Ool3000=Ool2000;Ool00`00Oomoo`02Ool20006Ool00`00Oomoo`07Ool20002
Ool00`00Oomoo`09Ool00`00Oomoo`020002Ool00`00Oomoo`03Ool00`00Oomoo`05Ool20003Ool0
0`00Ool00009Ool2000UOol002aoo`03001oogoo01]oo`03001oogoo00Yoo`<000Uoo`H0009oo`80
00Qoo`8000Aoo`<00004Ool00000000<Ool200000goo001oo`09Ool300000goo001oo`0:Ool00`00
Oomoo`020009Ool00`00Oomoo`03Ool30004Ool00`00Oomoo`0:Ool20005Ool00`00Oomoo`03Ool0
0`00Oomoo`02Ool20006Ool2000;Ool00`00Oomoo`0ROol002aoo`03001oogoo01aoo`03001oogoo
00Eoo`@0009oo`<000Moo`/000Eoo`8000=oo`<0009oo`80009oo`8000Uoo`8000Aoo`8000Eoo`<0
00Eoo`03001oogoo00Qoo`<000aoo`03001oogoo008000Qoo`03001oogoo00Moo`800003Ool007oo
00Eoo`03001oogoo009oo`03001oogoo008000Qoo`03001oo`0000Yoo`03001oogoo029oo`00;7oo
0P007Woo00@007ooOomoo`<000Qoo`<0009oo`<000Aoo`D0009oo`D000=oo`8000=oo`8000Ioo`03
001oogoo00Aoo`8000Qoo`03001oogoo008000Qoo`03001oogoo00Ioo`80009oo`03001oogoo00]o
o`8000Yoo`03001oogoo00Eoo`8000=oo`03001oogoo00Aoo`03001oogoo00=oo`8000]oo`8000]o
o`03001oogoo025oo`00;7oo00<007ooOol07Woo0`003Goo0`0017oo0`001Goo10000goo0`000goo
0P002Goo0P000Woo0P002Woo0`002goo00<007ooOol00goo0P001Goo00<007ooOol027oo0P0000=o
o`00Ool02Woo00<007ooOol00Woo0P001Woo00<007ooOol017oo00<007ooOol00P0000=oo`00Ool0
2Woo0P002goo0P008Woo000/Ool00`00Oomoo`0KOol30003Ool20008Ool30002Ool200000goo0000
0007Ool300000goo000000020003Ool3000<Ool3000:Ool20002Ool00`00Oomoo`0:Ool00`00Oomo
o`020007Ool00`00Oomoo`06Ool20003Ool00`00Oomoo`09Ool00`00Oomoo`020009Ool00`00Oomo
o`04Ool20003Ool00`00Oomoo`0:Ool2000;Ool00`00Oomoo`0POol002aoo`03001oogoo01Qoo`<0
00Qoo`8000=oo`<000Eoo`D000Eoo`80009oo`<000=oo`<0009oo`<000Moo`<0009oo`<000Ioo`80
00Eoo`03001oogoo00Uoo`<000Yoo`03001oogoo00=oo`8000Ioo`03001oogoo00Qoo`<000aoo`04
001oogooOol200000goo001oo`03Ool00`00Oomoo`09Ool2000<Ool00`00Oomoo`0OOol002]oo`03
001oogoo01Eoo`@000eoo`<000Ioo`8000Eoo`D00004Ool000000004Ool30002Ool3000017oo0000
00000Woo0`001goo0`000Woo0P0027oo00<007ooOol01Goo0`000Woo00<007ooOol02Goo00<007oo
Ool00P002Goo00<007ooOol01Goo0P000Woo00<007ooOol02Woo00@007oo000000=oo`03001oogoo
009oo`03001oogoo00Moo`800003Ool0000000]oo`80021oo`00:goo0`004Goo100017oo0P0027oo
0`000Woo0P000goo0P001Woo0`0000=oo`0000000`000goo0P0027oo1P0037oo0`002goo00@007oo
Oomoo`<000Ioo`03001oogoo00Qoo`<000aoo`03001oogoo009oo`8000Aoo`03001oogoo00]oo`03
001oogoo00Aoo`03001oogoo009oo`03001oogoo00Aoo`8000=oo`8000]oo`80021oo`00:goo00<0
07ooOol03Woo0`002Woo0P000goo0`001goo0`001Goo0`0000Aoo`00000000Aoo`@000Yoo`<00004
Ool000000009Ool20002Ool00`00Oomoo`09Ool4000:Ool00`00Oomoo`04Ool30002Ool00`00Oomo
o`0:Ool00`00Oomoo`020007Ool00`00Oomoo`08Ool200000goo001oo`05Ool00`00Oomoo`02Ool0
1000Oomoogoo0P001Woo00<007ooOol02goo00<007ooOol07Goo0003Ool4000BOol20004Ool20004
Ool20004Ool00`00Oomoo`0@Ool2000:Ool40007Ool?0003Ool40004Ool30003Ool40006Ool40003
Ool30005Ool00`00Oomoo`06Ool20002Ool00`00Oomoo`0:Ool00`00Oomoo`030006Ool00`00Oomo
o`09Ool3000:Ool00`00Oomoo`05Ool20003Ool00`00Oomoo`05Ool01P00Oomoogoo001oo`8000Uo
o`03001oogoo00Yoo`03001oogoo00Ioo`8001Eoo`0017oo00<007oo00004Goo00@007ooOol000Qo
o`04001oogoo0003Ool00`00Oomoo`0BOol20004Ool40003Ool80005Ool300000goo00000007000:
Ool4000<Ool00`00Ool000030008Ool00`00Oomoo`03Ool20005Ool00`00Oomoo`09Ool30009Ool0
0`00Oomoo`07Ool20002Ool00`00Oomoo`09Ool00`00Oomoo`02Ool20006Ool00`00Oomoo`04Ool0
1000Oomoogoo0P002goo00<007ooOol02goo0P001Goo00@007ooOol001Aoo`000goo00<007oo0000
4Woo00@007ooOol000]oo`05001oogooOol0000FOol40006Ool50005Ool4000017oo000000000Woo
0`000goo0`0017oo10000goo0`002Goo10000goo0`001Woo00<007ooOol00P0027oo00<007ooOol0
1Woo0P0000=oo`0000002Woo00<007ooOol00goo0`0017oo00<007ooOol02Goo00<007ooOol00P00
27oo00<007ooOol01Goo0`0000=oo`00Ool02goo00<007ooOol02goo00<007ooOol00goo00@007oo
Ool001Aoo`000goo10004Goo00@007ooOol000Uoo`8000Aoo`03001oogoo01Moo`80009oo`<000Ao
o`/0009oo`<000Qoo`D000Uoo`<000=oo`@000=oo`8000Eoo`@0009oo`<000Yoo`03001oogoo00=o
o`<000Aoo`03001oogoo00Uoo`03001oogoo008000Qoo`03001oogoo00Qoo`<000]oo`03001oogoo
009oo`@000=oo`03001oogoo00Uoo`03001oogoo00]oo`03001oogoo00=oo`04001oogoo000DOol0
01Qoo`04001oogoo0009Ool00`00Oomoo`02Ool00`00Oomoo`0IOol:0006Ool90005Ool40003Ool3
0009Ool50009Ool00`00Oomoo`05Ool400000goo001oo`0:Ool00`00Oomoo`020008Ool00`00Oomo
o`08Ool3000:Ool00`00Oomoo`06Ool200000goo001oo`0;Ool00`00Oomoo`020004Ool01000Oomo
o`0037oo00<007ooOol02goo00<007ooOol00Woo00@007ooOol001Aoo`006Goo0P002Woo0`000Woo
00<007ooOol04goo2P0017oo2@001goo1P002Goo0`000goo10000goo0P002Woo0P000Woo0`0017oo
10002Goo0`002goo00<007ooOol017oo0`000Woo00<007ooOol02Goo00<007ooOol00Woo0`0017oo
00<007ooOol02Woo0P001Woo00D007ooOomoo`0000Uoo`800003Ool007oo00Yoo`<000Aoo`8001Eo
o`00:Woo00<007ooOol03Woo4`000Woo0`000goo0`0000=oo`0000000`0017oo0P0027oo1P0027oo
0P001goo1P002goo0`000goo0`000Woo00<007ooOol02Goo00@007ooOomoo`<000Eoo`03001oogoo
00Uoo`03001oogoo008000Moo`03001oogoo00Qoo`<000Qoo`04001oogoo0007Ool20003Ool00`00
Oomoo`0;Ool00`00Oomoo`0IOol002Yoo`03001oogoo021oo`<000Qoo`8000]oo`80009oo`@000Ao
o`03001oogoo00Uoo`80009oo`<000Eoo`03001oogoo00Yoo`03001oo`00008000Ioo`03001oogoo
00Uoo`<000Uoo`03001oogoo00Moo`@000Yoo`04001oogooool3Ool30003Ool00`00Oomoo`06Ool0
1`00Oomoo`00Ooooogoo00<000Eoo`8000aoo`8001Yoo`00:Woo0P007Woo0`000Woo1P0017oo0`00
0goo00<007ooOol00goo10002Goo0P002Woo0`002Goo00<007ooOol027oo10002Goo00<007ooOol0
1Woo0P0000=oo`00Ool02Woo00<007ooOol017oo0P000goo00<007ooOol02Goo00@007ooOomoo`80
00Ioo`03001oogoo00Ioo`05001oogoo001oo`020009Ool00`00Oomoo`06Ool400000goo001oo`0I
Ool002Yoo`03001oogoo01Yoo`<000Moo`8000Qoo`<000=oo`@000=oo`03001oogoo00Qoo`8000Io
o`<0009oo`03001oogoo00Qoo`03001oogoo00Aoo`@00003Ool007oo00Uoo`03001oogoo00=oo`<0
00Aoo`03001oogoo00Qoo`03001oogoo009oo`8000Eoo`03001oogoo00Uoo`04001oo`000009Ool0
0`00Oomoo`06Ool01000Oomoo`002goo0P0017oo0`001Goo00<007ooOol067oo000ZOol00`00Oomo
o`0GOol3000<Ool00`00Oomoo`07Ool40008Ool2000:Ool00`00Oomoo`030006Ool00`00Oomoo`08
Ool00`00Oomoo`050004Ool00`00Oomoo`08Ool00`00Oomoo`02000COol01000Ool0000027oo00<0
07ooOol02Goo00<007ooOol02Goo00<007ooOol01goo0`0037oo00<007oo00000P0027oo00<007oo
Ool067oo000YOol00`00Oomoo`0EOol3000@Ool20003Ool40002Ool20009Ool20008Ool4000:Ool0
0`00Oomoo`06Ool400000goo00000007Ool00`00Oomoo`06Ool4000:Ool00`00Oomoo`08Ool2000:
Ool00`00Oomoo`07Ool3000;Ool00`00Oomoo`05Ool2000017oo001oo`002goo0P0037oo00<007oo
Ool05goo000YOol00`00Oomoo`0BOol3000EOol30008Ool00`00Oomoo`08Ool01000Oomoogoo1000
0goo00<007ooOol02Goo00@007ooOomoo`@000=oo`8000Yoo`04001oogooOol40003Ool00`00Oomo
o`0@Ool300000goo001oo`0:Ool00`00Oomoo`04Ool20003Ool00`00Oomoo`09Ool00`00Oomoo`03
Ool01`00Oomoogoo001oo`0000]oo`8000aoo`03001oogoo01Moo`00:Goo0P0047oo0`006Goo0P00
2Goo0P001Goo1`0027oo0P0027oo1@001Goo0P0000=oo`00Ool02Goo100027oo00<007ooOol027oo
00<007ooOol010001Goo00<007ooOol02Goo00@007ooOomoo`8000Eoo`03001oogoo00Uoo`04001o
ogooOol20005Ool2000<Ool00`00Oomoo`0:Ool00`00Oomoo`0GOol002Uoo`03001oogoo00aoo`<0
01ioo`03001oogoo00Moo`H000Eoo`03001oogoo00Uoo`04001oogooOol40003Ool00`00Oomoo`02
Ool20004Ool00`00Oomoo`04Ool30002Ool00`00Oomoo`07Ool00`00Oomoo`08Ool30009Ool00`00
Oomoo`09Ool01000Ool0000027oo00<007ooOol02Goo0`0027oo0P002goo0P002goo00<007ooOol0
5goo000YOol00`00Oomoo`0:Ool2000ROol20003Ool40002Ool20008Ool20008Ool50008Ool01000
Ool000001goo00<007ooOol00`001Goo00<007ooOol027oo00<007ooOol017oo0`000Woo00<007oo
Ool027oo00<007ooOol027oo0P002Woo00<007ooOol02Goo00<007ooOol02Goo00<007ooOol01goo
10002goo00<007ooOol05goo000YOol00`00Oomoo`07Ool3000VOol30008Ool00`00Oomoo`07Ool0
1000Oomoogoo10000goo00<007ooOol01goo0P0027oo10002Goo00<007ooOol027oo00<007ooOol0
0`001Goo00<007ooOol027oo00<007ooOol01Woo0P0000=oo`00Ool02Woo00<007ooOol01Woo0`00
2goo0P001Woo0P000goo0P002Woo00<007ooOol05goo000XOol00`00Oomoo`05Ool3000ZOol00`00
Oomoo`08Ool20005Ool60008Ool00`00Oomoo`04Ool20002Ool01000Oomoogoo10000goo00<007oo
Ool027oo00<007ooOol01goo0`002Goo00<007ooOol027oo00<007ooOol00Woo0`0017oo00<007oo
Ool027oo00<007ooOol017oo0P000goo00<007ooOol02Goo00<007ooOol00goo00<007ooOol00goo
0P002Woo00<007ooOol05goo000XOol30002Ool3000^Ool20008Ool70004Ool00`00Oomoo`08Ool0
1000Oomoogoo0P000goo1@001goo00<007ooOol02Goo00<007ooOol00goo0`000Woo00<007ooOol0
1goo00<007ooOol027oo00<007ooOol00P001goo00<007ooOol02Goo00@007ooOomoo`8000Eoo`03
001oogoo00Uoo`80009oo`8000Ioo`03001oo`0000Yoo`03001oogoo01Ioo`00:7oo00<007oo0000
0P00<goo00@007ooOomoo`@000=oo`03001oogoo00Ioo`8000Uoo`<00003Ool00000008000=oo`03
001oogoo00Moo`03001oogoo00Qoo`03001oogoo00<000Eoo`03001oogoo00Qoo`03001oogoo00Mo
o`<000Yoo`03001oogoo00Qoo`04001oo`000008Ool00`00Oomoo`09Ool30009Ool2000:Ool00`00
Oomoo`0FOol000moo`8000Aoo`8000Aoo`8000=oo`@000Aoo`8003Moo`<000Qoo`03001oogoo00Io
o`8000Moo`D000Qoo`03001oogoo00Moo`03001oogoo00Ioo`@000Uoo`03001oogoo00Qoo`03001o
ogoo00=oo`<0009oo`03001oogoo00Qoo`03001oogoo00Qoo`8000Yoo`03001oogoo00Uoo`8000Yo
o`03001oo`0000Uoo`03001oogoo01Ioo`003Woo00@007ooOol000Qoo`04001oogoo0002Ool00`00
Oomoo`06Ool00`00Oomoo`0fOol20009Ool00`00Oomoo`07Ool01000Oomoogoo10000Woo00<007oo
Ool027oo00<007ooOol01goo00@007ooOomoo`@000=oo`03001oogoo00Moo`03001oogoo00Qoo`03
001oogoo00<000Eoo`03001oogoo00Uoo`03001oogoo00Eoo`800003Ool007oo00Yoo`03001oogoo
00Ioo`<00003Ool007oo00Qoo`<000Uoo`03001oogoo01Ioo`003Woo00@007ooOol000]oo`05001o
ogooOol00008Ool2000hOol00`00Oomoo`07Ool00`00Oomoo`03Ool80006Ool00`00Oomoo`08Ool0
0`00Oomoo`02Ool80007Ool00`00Oomoo`08Ool00`00Oomoo`07Ool30009Ool00`00Oomoo`08Ool0
0`00Oomoo`02Ool30004Ool00`00Oomoo`08Ool00`00Oomoo`04Ool20003Ool00`00Ool00009Ool2
0009Ool00`00Oomoo`0FOol000ioo`04001oogoo0009Ool20005Ool00`00Oomoo`07Ool00`00Oomo
o`0fOol20005Ool80004Ool20009Ool00`00Oomoo`06Ool60006Ool00`00Oomoo`07Ool00`00Oomo
o`08Ool00`00Oomoo`03Ool30002Ool00`00Oomoo`07Ool00`00Oomoo`09Ool01000Ool000001goo
00<007ooOol02Goo00@007ooOomoo`8000Eoo`03001oo`0000Uoo`03001oo`0000Qoo`03001oogoo
01Ioo`003Woo00@007ooOol000Uoo`03001oogoo009oo`04001oogoo0009Ool00`00Oomoo`0gOol5
0004Ool00`00Oomoo`03Ool20002Ool20008Ool01000Oomoogoo10000goo00<007ooOol01goo00<0
07ooOol01goo00<007ooOol01goo00<007ooOol00`001Goo00<007ooOol027oo00<007ooOol01goo
0`002Woo00<007ooOol027oo00@007oo000000Qoo`03001oo`0000Qoo`03001oo`0000Qoo`03001o
ogoo01Ioo`003goo0P002Woo0`000goo0P002goo00<007ooOol0=goo00<007ooOol01Woo00<007oo
Ool00P001Woo00<007ooOol00goo1P0027oo00<007ooOol01Woo00<007ooOol027oo00<007ooOol0
1Goo10002Goo00<007ooOol027oo00<007ooOol00goo0`000Woo00<007ooOol027oo00<007ooOol0
2Goo00<007ooOol027oo00@007ooOol000Qoo`03001oo`0000Moo`03001oogoo01Ioo`00;goo00<0
07ooOol0=goo0P001goo0P001goo1`0017oo00<007ooOol027oo00<007ooOol01Woo00<007ooOol0
1goo00@007ooOomoo`@000=oo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00<000Eoo`03
001oogoo00Qoo`03001oogoo00Moo`<000]oo`04001oogoo0007Ool00`00Ool00008Ool00`00Oomo
o`0EOol0031oo`8003Uoo`03001oogoo009oo`800003Ool007oo009oo`@000=oo`03001oogoo00Io
o`03001oogoo00Moo`03001oogoo00Moo`03001oogoo00=oo`L000Moo`03001oogoo00Qoo`03001o
ogoo00Moo`<000Uoo`03001oogoo00Qoo`03001oogoo00Aoo`8000=oo`03001oogoo00Qoo`04001o
ogoo0005Ool30002Ool00`00Oomoo`05Ool00`00Oomoo`0EOol0039oo`03001oogoo03Moo`/000Qo
o`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00=oo`T000Eoo`03001oogoo00Moo`03001o
ogoo00Qoo`03001oogoo00=oo`<00003Ool007oo00Uoo`03001oogoo00Qoo`03001oogoo009oo`80
00Eoo`03001oogoo00Uoo`04001oogoo0002Ool20003Ool00`00Ool00007Ool00`00Oomoo`0EOol0
03=oo`03001oogoo03Ioo`8000Moo`03001oogoo00Moo`03001oogoo00Moo`03001oogoo00=oo`T0
00Eoo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00<000Eoo`03001o
ogoo00Qoo`03001oogoo00Qoo`04001oo`000007Ool00`00Oomoo`09Ool01000Oomoogoo0P001Goo
00@007ooOol000Ioo`03001oogoo01Eoo`00=7oo00<007ooOol0<goo0P000Woo00<007ooOol01Goo
0P0027oo00<007ooOol01Woo1P0017oo00<007ooOol01goo00<007ooOol01goo00<007ooOol01goo
00<007ooOol01Woo100027oo00<007ooOol027oo00<007ooOol01goo0`002Woo00<007ooOol02Goo
0`0000=oo`00Ool01Goo0`001Woo00<007ooOol05Goo000eOol2000aOol20005Ool00`00Oomoo`06
Ool00`00Oomoo`06Ool01000Oomoogoo10000Woo00<007ooOol01Woo00<007ooOol01goo00<007oo
Ool01Woo00<007ooOol01goo00@007ooOomoo`D0009oo`03001oogoo00Qoo`03001oogoo00Qoo`03
001oogoo00=oo`<00003Ool007oo00Yoo`03001oogoo00Uoo`05001oogooOol00006Ool00`00Ool0
0006Ool00`00Oomoo`0EOol003Moo`03001oogoo02aoo`8000Qoo`03001oogoo00=oo`l000Moo`03
001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Eoo`D000Qoo`03001oogoo
00Moo`03001oogoo00Qoo`03001oogoo00<000Eoo`03001oogoo00Uoo`03001oogoo00Ioo`<000Ao
o`03001oogoo00=oo`04001oogoo0005Ool00`00Oomoo`0EOol003Qoo`03001oogoo02Uoo`8000]o
o`D000Aoo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001o
ogoo00Ioo`04001oogooOol40003Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`07Ool3
0008Ool00`00Oomoo`09Ool00`00Oomoo`04Ool20003Ool00`00Oomoo`02Ool00`00Oomoo`03Ool0
0`00Ool00006Ool00`00Oomoo`0DOol003Uoo`03001oogoo03Ioo`03001oogoo00Ioo`03001oogoo
00Ioo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00=oo`L000Moo`03
001oogoo00Qoo`03001oogoo00Moo`03001oogoo00Aoo`<00003Ool007oo00Uoo`03001oogoo00Qo
o`03001oogoo009oo`8000Eoo`03001oogoo009oo`03001oogoo00=oo`04001oogoo0005Ool00`00
Oomoo`0DOol003Yoo`8003Moo`03001oogoo00Eoo`03001oogoo00Moo`03001oogoo00Ioo`03001o
ogoo00Ioo`03001oogoo00=oo`P000Eoo`03001oogoo00Moo`03001oogoo00Moo`03001oogoo00Qo
o`03001oo`0000<000Eoo`03001oogoo00Moo`03001oogoo00Uoo`04001oo`000007Ool00`00Oomo
o`03Ool00`00Oomoo`02Ool01000Oomoo`001Goo00<007ooOol057oo000lOol00`00Oomoo`0eOol0
0`00Oomoo`05Ool00`00Oomoo`07Ool00`00Oomoo`03OolA0004Ool00`00Oomoo`06Ool00`00Oomo
o`07Ool00`00Oomoo`08Ool00`00Oomoo`06Ool30009Ool00`00Oomoo`08Ool00`00Oomoo`07Ool3
000:Ool00`00Oomoo`03Ool00`00Oomoo`02Ool01000Oomoo`0017oo00<007ooOol057oo000mOol0
0`00Oomoo`0eOol00`00Oomoo`05Ool00`00Oomoo`03Ool90004Ool00`00Oomoo`06Ool00`00Oomo
o`05Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`03Ool300000goo
001oo`09Ool00`00Oomoo`08Ool00`00Oomoo`05Ool200000goo001oo`0:Ool00`00Oomoo`04Ool0
1@00Oomoogoo00000Woo00<007ooOol00Woo00<007ooOol057oo000nOol00`00Oomoo`0eOol=0005
Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`07
Ool00`00Oomoo`08Ool00`00Ool000020005Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomo
o`02Ool20004Ool00`00Oomoo`08Ool00`00Oomoo`04Ool00`00Oomoo`02Ool01000Oomoo`000goo
00<007ooOol057oo000oOol00`00Oomoo`0eOol00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomo
o`06Ool00`00Oomoo`06Ool00`00Oomoo`05Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomo
o`05Ool40008Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`020006Ool00`00Oomoo`09
Ool00`00Oomoo`04Ool00`00Oomoo`0200000goo00000003Ool00`00Oomoo`0DOol0041oo`8003Io
o`03001oogoo00Eoo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Eoo`03001oogoo00Io
o`03001oogoo00Moo`03001oogoo00Ioo`04001oogooOol40003Ool00`00Oomoo`07Ool00`00Oomo
o`08Ool00`00Oomoo`07Ool30009Ool00`00Oomoo`08Ool00`00Oomoo`05Ool200000goo00000006
Ool00`00Oomoo`0COol0049oo`03001oogoo03Aoo`03001oogoo00Eoo`03001oogoo00Ioo`03001o
ogoo00Ioo`03001oogoo00Eoo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00=oo`L000Mo
o`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Eoo`800003Ool007oo00Uoo`03001oogoo
00Qoo`03001oogoo009oo`<00005Ool007ooOol00006Ool00`00Oomoo`0COol004=oo`03001oogoo
03Aoo`03001oogoo00Eoo`03001oogoo00Eoo`03001oogoo00`0009oo`03001oogoo00Ioo`03001o
ogoo00=oo`T000Aoo`03001oogoo00Qoo`03001oogoo00Moo`03001oogoo00Moo`03001oogoo009o
o`<000Aoo`03001oogoo00Moo`03001oogoo00Uoo`04001oo`000004Ool01000Oomoo`001Woo00<0
07ooOol04goo0014Ool00`00Oomoo`0dOolC0006Ool01000Oomoogoo4P0017oo00<007ooOol01goo
00<007ooOol01goo00<007ooOol01goo00<007ooOol027oo00@007oo000000Moo`03001oogoo00Qo
o`03001oogoo00Qoo`8000Moo`03001oo`0000Ioo`03001oogoo01=oo`00AGoo0P00=Goo00<007oo
Ool01Goo00<007ooOol01Goo00<007ooOol01Woo00<007ooOol01Goo00<007ooOol01Woo00<007oo
Ool01Woo00<007ooOol01Woo00<007ooOol027oo00<007ooOol01goo00<007ooOol01Goo10002Goo
00<007ooOol027oo00<007ooOol01Woo0`002Goo00<007oo00001Goo00<007ooOol04goo0017Ool0
0`00Oomoo`0cOol00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`05Ool0
0`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`07Ool0
1000Oomoogoo10000Woo00<007ooOol02Goo00<007ooOol027oo00<007ooOol00goo0P000goo00<0
07ooOol01Woo00<007oo00001Goo00<007ooOol04goo0018Ool00`00Oomoo`0bOol00`00Oomoo`05
Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06
Ool00`00Oomoo`06Ool00`00Oomoo`08Ool00`00Oomoo`05Ool50007Ool00`00Oomoo`08Ool00`00
Oomoo`08Ool01000Oomoogoo0P001Goo00<007ooOol01goo0P001Goo00<007ooOol04goo0019Ool0
0`00Oomoo`0bOol00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`05Ool0
0`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`07Ool00`00Oomoo`07Ool01000Oomoogoo1000
0Woo00<007ooOol01goo00<007ooOol02Goo00<007ooOol027oo0`0027oo00<007ooOol01goo00<0
07ooOol00goo00<007ooOol04goo001:Ool2000cOol00`00Oomoo`05Ool00`00Oomoo`02Ool40007
Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`06
Ool50007Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`07Ool2000:Ool00`00Oomoo`08
Ool00`00Oomoo`03Ool00`00Oomoo`0BOol004aoo`03001oogoo035oo``000Aoo`D0009oo`03001o
ogoo00Ioo`03001oogoo00Ioo`03001oogoo00Eoo`03001oogoo00Moo`03001oogoo00D0009oo`03
001oogoo00Moo`03001oogoo00Moo`03001oogoo00Qoo`03001oogoo00Aoo`<00003Ool007oo00Uo
o`03001oogoo00Qoo`03001oogoo00=oo`03001oogoo019oo`00CGoo00<007ooOol0<Goo00<007oo
Ool01Goo00<007ooOol01Goo00<007ooOol02@000goo00<007ooOol01Woo00<007ooOol01Woo00<0
07ooOol00Woo1`0027oo00<007ooOol01Woo00<007ooOol027oo00<007ooOol027oo00<007oo0000
0`001Goo00<007ooOol027oo00<007ooOol01goo0P0017oo00<007ooOol04Woo001>Ool00`00Oomo
o`0aOol00`00Oomoo`04Ool00`00Oomoo`05Ool00`00Oomoo`05Ool01000Oomoogoo6P0017oo00<0
07ooOol027oo00<007ooOol01goo00<007ooOol01goo00<007ooOol01goo0`002Goo00<007ooOol0
27oo00<007ooOol027oo0P000goo00<007ooOol04Woo001?Ool00`00Oomoo`0`Ool00`00Oomoo`05
Ool00`00Oomoo`05Ool00`00Oomoo`04Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06
Ool00`00Oomoo`06Ool00`00Oomoo`08Ool00`00Oomoo`06Ool00`00Oomoo`08Ool00`00Oomoo`03
Ool300000goo001oo`09Ool00`00Oomoo`08Ool00`00Oomoo`07Ool200001Goo001oogoo000057oo
001@Ool2000aOol00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`04Ool00`00Oomoo`05Ool0
0`00Oomoo`06Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`08Ool00`00Oomoo`07Ool0
0`00Oomoo`07Ool00`00Oomoo`030005Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`04
Ool20003Ool01000Oomoo`0057oo001BOol00`00Oomoo`0_Ool50002Ool00`00Oomoo`06Ool00`00
Oomoo`03Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00
Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`07Ool30008Ool00`00Oomoo`08Ool00`00Oomoo`08
Ool00`00Oomoo`02Ool20006Ool00`00Ool0000DOol005=oo`03001oogoo02ioo`80009oo`X0009o
o`03001oogoo00Aoo`03001oogoo00Eoo`03001oogoo00Ioo`03001oogoo00Eoo`03001oogoo00Io
o`03001oogoo00Qoo`03001oogoo00Ioo`03001oogoo00Aoo`<00003Ool007oo00Uoo`03001oogoo
00Moo`03001oogoo00Uoo`04001oo`000009Ool00`00Ool0000COol005Aoo`03001oogoo02eoo`03
001oo`0000Ioo`03001oogoo009oo`H000Aoo`03001oogoo00Aoo`03001oogoo00Ioo`03001oogoo
00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Moo`03001oo`0000<000Eoo`03001o
ogoo00Moo`03001oogoo00Qoo`03001oogoo00Moo`<000aoo`8001=oo`00EGoo0P00;Goo00<007oo
00001goo00<007ooOol017oo00<007oo00001@001goo00<007ooOol01Woo00<007ooOol01Goo00<0
07ooOol01Woo00<007ooOol027oo00<007ooOol00Woo1P002Goo00<007ooOol01goo00<007ooOol0
27oo00<007ooOol01Goo0P003goo0P004goo001GOol00`00Oomoo`0ZOol01000Oomoo`001Woo00<0
07ooOol01Goo00<007ooOol00Woo1`000Woo00<007ooOol01Woo00<007ooOol01Woo00<007ooOol0
1Woo00<007ooOol01Goo1`0017oo00<007ooOol027oo00<007ooOol027oo00<007ooOol01goo00<0
07ooOol00Woo0`004Goo0P004goo001HOol00`00Oomoo`0YOol01@00Oomoogoo00001Woo00<007oo
Ool017oo00<007ooOol017oo00@007ooOomoo`T000Aoo`03001oogoo00Eoo`03001oogoo00Ioo`04
001oogooOol40002Ool00`00Oomoo`07Ool00`00Oomoo`08Ool00`00Oomoo`07Ool00`00Oomoo`08
Ool01000Ool0000047oo10005Goo001IOol00`00Oomoo`0XOol00`00Oomoo`02Ool30004Ool00`00
Oomoo`04Ool00`00Oomoo`04Ool00`00Oomoo`04Ool00`00Oomoo`02OolI0007Ool00`00Oomoo`06
Ool00`00Oomoo`08Ool00`00Oomoo`08Ool00`00Oomoo`06Ool3000>Ool4000IOol005Yoo`8002Qo
o`03001oogoo00Eoo`D000Ioo`03001oogoo00Aoo`03001oogoo00Eoo`03001oogoo00Eoo`03001o
ogoo00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Moo`03001oogoo00Moo`03001o
ogoo00Qoo`03001oogoo00=oo`<000eoo`@001eoo`00G7oo00<007ooOol09Goo00<007ooOol02Goo
0`001Goo00<007ooOol017oo00<007ooOol017oo00<007ooOol01Woo00<007ooOol01Woo00<007oo
Ool01Woo00<007ooOol01Woo00<007ooOol01goo00<007ooOol027oo00<007ooOol027oo00<007oo
00000P0037oo10008Goo001MOol00`00Oomoo`0TOol00`00Oomoo`0<Ool30002Ool00`00Oomoo`04
Ool00`00Oomoo`05Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`07
Ool00`00Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`07Ool3000<Ool3000UOol005ioo`03001o
ogoo02=oo`03001oogoo00moo`800003Ool007oo00Eoo`03001oogoo00Aoo`03001oogoo00Ioo`03
001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Ioo`03001oogoo00Moo`03001oogoo00Moo`03
001oogoo00Aoo`<000]oo`@002Qoo`00Ggoo00<007ooOol08Woo00<007ooOol04Goo100017oo00<0
07ooOol01Goo00<007ooOol01Goo00<007ooOol01Woo00<007ooOol01Woo00<007ooOol01goo00<0
07ooOol01goo00<007ooOol01goo00<007oo00000`002Woo1000;7oo001POol2000ROol00`00Oomo
o`0EOol400000goo001oo`05Ool00`00Oomoo`05Ool00`00Oomoo`06Ool00`00Oomoo`06Ool00`00
Oomoo`07Ool00`00Oomoo`07Ool00`00Oomoo`05Ool4000:Ool4000`Ool0069oo`03001oogoo01mo
o`03001oogoo01Uoo`@000Eoo`03001oogoo00Eoo`03001oogoo00Ioo`03001oogoo00Ioo`03001o
ogoo00Ioo`03001oogoo00Qoo`03001oogoo00@000Yoo`@003Aoo`00Hgoo00<007ooOol07Woo00<0
07ooOol07Goo100000=oo`00Ool01Woo00<007ooOol01Woo00<007ooOol01Woo00<007ooOol01goo
00<007ooOol00goo1`002goo0`00>7oo001TOol00`00Oomoo`0MOol00`00Oomoo`0QOol70003Ool0
0`00Oomoo`06Ool00`00Oomoo`06Ool00`00Oomoo`02Ool:000>Ool4000kOol006Eoo`8001aoo`03
001oogoo02UooaX001Aoo`@003moo`00Igoo00<007ooOol06Goo00<007ooOol0Dgoo1000@goo001X
Ool00`00Oomoo`0HOol00`00Oomoo`1?Ool40017Ool006Uoo`03001oogoo01Moo`03001oogoo04ao
o`<004]oo`00JWoo0P005goo00<007ooOol0B7oo1000CWoo001/Ool00`00Oomoo`0DOol00`00Oomo
o`14Ool4001BOol006eoo`03001oogoo01=oo`03001oogoo041oo`@005Ioo`00KWoo00<007ooOol0
4Woo00<007ooOol0?7oo1000FWoo001_Ool00`00Oomoo`0AOol00`00Oomoo`0hOol4001NOol0071o
o`80015oo`03001oogoo03Eoo`<0069oo`00LWoo00<007ooOol03Woo00<007ooOol0<Goo1000IGoo
001cOol00`00Oomoo`0=Ool00`00Oomoo`0]Ool4001YOol007Aoo`03001oogoo00aoo`03001oogoo
02Uoo`@006eoo`00MGoo0P0037oo00<007ooOol09Goo1000LGoo001gOol00`00Oomoo`09Ool00`00
Oomoo`0ROol3001eOol007Qoo`03001oogoo00Qoo`03001oogoo01ioo`@007Qoo`00NGoo00<007oo
Ool01goo00<007ooOol06Woo1000O7oo001jOol20007Ool00`00Oomoo`0FOol40020Ool007aoo`03
001oogoo00Aoo`03001oogoo019oo`@008Aoo`00OGoo00<007ooOol00goo00<007ooOol03goo0`00
R7oo001nOol00`00Oomoo`02Ool00`00Oomoo`0;Ool4002;Ool007moo`05001oogooOol00009Ool4
002?Ool0081oo`800003Ool007oo00Aoo`@009=oo`00PWoo0P0000=oo`0000000P00Ugoo0023Ool2
002KOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5o
o`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003o
OolQOol00?moob5oo`00ogoo8Goo003oOolQOol00001\
\>"],
  ImageRangeCache->{{{78.5625, 307.938}, {530.688, 344.875}} -> {-0.57962, 
  1.74711, 0.00418082, 0.00418082}}]
}, Open  ]],

Cell["\<\
Okay, now let's take a look at the surface near the other critical point we \
found.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(Plot3D[f[x, y], \ {x, \(-1.7\), \(-1.5\)}, \ 
        {y, \(-5.7\), \(-5.3\)}, \ PlotPoints \[Rule] 30, \ 
        AxesLabel \[Rule] {"\<x  \>", "\<    y\>", \ "\<z   \>"}, 
        Shading \[Rule] False];\)\)], "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .81114 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% SurfaceGraphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
-2.44249e-015 1.04977 -0.0679587 1.04977 [
[.17646 .18758 -29.4972 -9 ]
[.17646 .18758 .5028 0 ]
[.3345 .12542 -22.3468 -9 ]
[.3345 .12542 1.65319 0 ]
[.50253 .05935 -26.3698 -9 ]
[.50253 .05935 3.63018 0 ]
[.68156 -0.01098 -19.8449 -9 ]
[.68156 -0.01098 4.1551 0 ]
[.29165 .07573 -20.4846 -12.5 ]
[.29165 .07573 1.51543 0 ]
[.70096 -0.00478 0 -6.26206 ]
[.70096 -0.00478 24 2.73794 ]
[.78313 .12104 0 -6.13858 ]
[.78313 .12104 24 2.86142 ]
[.8565 .23339 0 -6.03127 ]
[.8565 .23339 24 2.96873 ]
[.9224 .33431 0 -5.93715 ]
[.9224 .33431 24 3.06285 ]
[.98191 .42546 0 -5.85393 ]
[.98191 .42546 24 3.14607 ]
[.91861 .21225 0 -8.37676 ]
[.91861 .21225 34 4.12324 ]
[.01123 .34586 -18 -2.81065 ]
[.01123 .34586 0 6.18935 ]
[-0.0035 .43816 -18 -2.8892 ]
[-0.0035 .43816 0 6.1108 ]
[-0.05457 .39542 -28 -3.93527 ]
[-0.05457 .39542 0 8.56473 ]
[ 0 0 0 0 ]
[ 1 .81114 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.03716 .25514 m
.68874 0 L
s
.18558 .19702 m
.19014 .20174 L
s
[(-1.65)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(-1.6)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(-1.55)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(-1.5)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.1552 .20892 m
.15797 .21172 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.27893 .16047 m
.28157 .16339 L
s
.3108 .14799 m
.31341 .15094 L
s
.37572 .12257 m
.37826 .12558 L
s
.40878 .10962 m
.41128 .11266 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.54521 .0562 m
.54755 .05937 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
gsave
.29165 .07573 -81.4846 -16.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
81.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.68874 0 m
.68263 .00239 L
s
[(-5.7)] .70096 -0.00478 -1 .39157 Mshowa
.7708 .12553 m
.76464 .12778 L
s
[(-5.6)] .78313 .12104 -1 .36413 Mshowa
.84407 .23761 m
.83786 .23973 L
s
[(-5.5)] .8565 .23339 -1 .34028 Mshowa
.9099 .3383 m
.90365 .34029 L
s
[(-5.4)] .9224 .33431 -1 .31937 Mshowa
.96935 .42924 m
.96306 .43113 L
s
[(-5.3)] .98191 .42546 -1 .30087 Mshowa
.125 Mabswid
.70593 .0263 m
.70226 .02771 L
s
.72272 .05198 m
.71904 .05338 L
s
.73912 .07706 m
.73543 .07845 L
s
.75514 .10158 m
.75145 .10294 L
s
.78611 .14895 m
.78241 .15028 L
s
.80108 .17185 m
.79737 .17317 L
s
.81573 .19425 m
.81201 .19555 L
s
.83006 .21617 m
.82633 .21745 L
s
.8578 .2586 m
.85407 .25986 L
s
.87123 .27915 m
.8675 .28039 L
s
.88439 .29928 m
.88065 .3005 L
s
.89727 .31899 m
.89353 .3202 L
s
.92227 .35722 m
.91851 .3584 L
s
.93439 .37576 m
.93063 .37693 L
s
.94627 .39394 m
.94251 .3951 L
s
.95792 .41176 m
.95416 .41291 L
s
gsave
.91861 .21225 -61 -12.3768 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
87.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(y) show
93.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
.02351 .34125 m
.02966 .33894 L
s
[(150)] .01123 .34586 1 -0.37541 Mshowa
.00886 .43374 m
.01503 .43153 L
s
[(160)] -0.0035 .43816 1 -0.35796 Mshowa
.125 Mabswid
.03465 .27095 m
.03832 .26952 L
s
.03192 .28823 m
.03559 .28681 L
s
.02915 .3057 m
.03282 .3043 L
s
.02635 .32337 m
.03003 .32198 L
s
.02065 .35932 m
.02434 .35795 L
s
.01775 .37761 m
.02145 .37624 L
s
.01482 .3961 m
.01852 .39475 L
s
.01186 .41481 m
.01556 .41347 L
s
.00582 .45289 m
.00953 .45158 L
s
.00275 .47227 m
.00647 .47097 L
s
gsave
-0.05457 .39542 -89 -7.93527 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(z) show
87.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
.25 Mabswid
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
.40529 .59895 m
.03716 .25514 L
s
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.40529 .59895 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.39787 .81114 L
s
.39787 .81114 m
.40529 .59895 L
s
0 0 m
1 0 L
1 .81114 L
0 .81114 L
closepath
clip
newpath
.5 Mabswid
1 g
.39357 .62009 .40409 .63325 .42192 .62832 .41144 .61541 Metetra
1 g
.41144 .61541 .42192 .62832 .43987 .62321 .42944 .61074 Metetra
1 g
.42944 .61074 .43987 .62321 .45794 .61791 .44757 .60604 Metetra
1 g
.44757 .60604 .45794 .61791 .47615 .61241 .46582 .60129 Metetra
1 g
.46582 .60129 .47615 .61241 .49447 .6067 .48421 .59647 Metetra
1 g
.48421 .59647 .49447 .6067 .51291 .60079 .50272 .59156 Metetra
1 g
.50272 .59156 .51291 .60079 .53147 .59466 .52138 .58653 Metetra
1 g
.52138 .58653 .53147 .59466 .55016 .58833 .54016 .58136 Metetra
1 g
.54016 .58136 .55016 .58833 .56895 .5818 .55907 .57604 Metetra
1 g
.55907 .57604 .56895 .5818 .58786 .57508 .57812 .57054 Metetra
1 g
.57812 .57054 .58786 .57508 .60689 .56818 .5973 .56486 Metetra
1 g
.5973 .56486 .60689 .56818 .62602 .56111 .6166 .55899 Metetra
1 g
.6166 .55899 .62602 .56111 .64527 .5539 .63603 .5529 Metetra
1 g
.63603 .5529 .64527 .5539 .66464 .54656 .65559 .54661 Metetra
1 g
.65559 .54661 .66464 .54656 .68411 .53911 .67527 .54011 Metetra
1 g
.67527 .54011 .68411 .53911 .7037 .53158 .69508 .53339 Metetra
1 g
.69508 .53339 .7037 .53158 .7234 .52397 .715 .52647 Metetra
1 g
.715 .52647 .7234 .52397 .74323 .51632 .73505 .51935 Metetra
1 g
.73505 .51935 .74323 .51632 .76317 .50865 .75521 .51204 Metetra
1 g
.75521 .51204 .76317 .50865 .78323 .50098 .77549 .50456 Metetra
1 g
.77549 .50456 .78323 .50098 .80342 .49331 .79589 .49691 Metetra
1 g
.79589 .49691 .80342 .49331 .82375 .48569 .8164 .48912 Metetra
1 g
.8164 .48912 .82375 .48569 .8442 .4781 .83704 .48121 Metetra
1 g
.83704 .48121 .8442 .4781 .8648 .47059 .8578 .47319 Metetra
1 g
.8578 .47319 .8648 .47059 .88554 .46314 .87868 .46508 Metetra
1 g
.87868 .46508 .88554 .46314 .90643 .45578 .89969 .4569 Metetra
1 g
.89969 .4569 .90643 .45578 .92747 .44851 .92083 .44868 Metetra
1 g
.92083 .44868 .92747 .44851 .94867 .44133 .9421 .44044 Metetra
1 g
.9421 .44044 .94867 .44133 .97003 .43424 .96351 .43219 Metetra
1 g
.38277 .61104 .39357 .62009 .41144 .61541 .40076 .60523 Metetra
1 g
.40076 .60523 .41144 .61541 .42944 .61074 .41886 .59953 Metetra
1 g
.41886 .59953 .42944 .61074 .44757 .60604 .43707 .59394 Metetra
1 g
.43707 .59394 .44757 .60604 .46582 .60129 .45541 .58846 Metetra
1 g
.45541 .58846 .46582 .60129 .48421 .59647 .47386 .58307 Metetra
1 g
.47386 .58307 .48421 .59647 .50272 .59156 .49245 .57776 Metetra
1 g
.49245 .57776 .50272 .59156 .52138 .58653 .51116 .57252 Metetra
1 g
.51116 .57252 .52138 .58653 .54016 .58136 .53 .56732 Metetra
1 g
.53 .56732 .54016 .58136 .55907 .57604 .54899 .56216 Metetra
1 g
.54899 .56216 .55907 .57604 .57812 .57054 .5681 .557 Metetra
1 g
.5681 .557 .57812 .57054 .5973 .56486 .58736 .55182 Metetra
1 g
.58736 .55182 .5973 .56486 .6166 .55899 .60676 .54661 Metetra
1 g
.60676 .54661 .6166 .55899 .63603 .5529 .6263 .54133 Metetra
1 g
.6263 .54133 .63603 .5529 .65559 .54661 .64599 .53597 Metetra
1 g
.64599 .53597 .65559 .54661 .67527 .54011 .66581 .53051 Metetra
1 g
.66581 .53051 .67527 .54011 .69508 .53339 .68578 .52491 Metetra
1 g
.68578 .52491 .69508 .53339 .715 .52647 .70589 .51917 Metetra
1 g
.70589 .51917 .715 .52647 .73505 .51935 .72614 .51327 Metetra
1 g
.72614 .51327 .73505 .51935 .75521 .51204 .74652 .50719 Metetra
1 g
.74652 .50719 .75521 .51204 .77549 .50456 .76704 .50092 Metetra
1 g
.76704 .50092 .77549 .50456 .79589 .49691 .78769 .49444 Metetra
1 g
.78769 .49444 .79589 .49691 .8164 .48912 .80848 .48776 Metetra
1 g
.80848 .48776 .8164 .48912 .83704 .48121 .82939 .48086 Metetra
1 g
.82939 .48086 .83704 .48121 .8578 .47319 .85044 .47375 Metetra
1 g
.85044 .47375 .8578 .47319 .87868 .46508 .87161 .46643 Metetra
1 g
.87161 .46643 .87868 .46508 .89969 .4569 .8929 .4589 Metetra
1 g
.8929 .4589 .89969 .4569 .92083 .44868 .91432 .45117 Metetra
1 g
.91432 .45117 .92083 .44868 .9421 .44044 .93586 .44325 Metetra
1 g
.93586 .44325 .9421 .44044 .96351 .43219 .95753 .43516 Metetra
1 g
.37136 .61246 .38277 .61104 .40076 .60523 .38952 .60549 Metetra
1 g
.38952 .60549 .40076 .60523 .41886 .59953 .4078 .59854 Metetra
1 g
.4078 .59854 .41886 .59953 .43707 .59394 .42617 .59164 Metetra
1 g
.42617 .59164 .43707 .59394 .45541 .58846 .44466 .58478 Metetra
1 g
.44466 .58478 .45541 .58846 .47386 .58307 .46325 .57801 Metetra
1 g
.46325 .57801 .47386 .58307 .49245 .57776 .48196 .57131 Metetra
1 g
.48196 .57131 .49245 .57776 .51116 .57252 .50079 .56472 Metetra
1 g
.50079 .56472 .51116 .57252 .53 .56732 .51973 .55823 Metetra
1 g
.51973 .55823 .53 .56732 .54899 .56216 .5388 .55185 Metetra
1 g
.5388 .55185 .54899 .56216 .5681 .557 .558 .54558 Metetra
1 g
.558 .54558 .5681 .557 .58736 .55182 .57733 .53942 Metetra
1 g
.57733 .53942 .58736 .55182 .60676 .54661 .59679 .53337 Metetra
1 g
.59679 .53337 .60676 .54661 .6263 .54133 .61639 .5274 Metetra
1 g
.61639 .5274 .6263 .54133 .64599 .53597 .63613 .52153 Metetra
1 g
.63613 .52153 .64599 .53597 .66581 .53051 .65602 .51572 Metetra
1 g
.65602 .51572 .66581 .53051 .68578 .52491 .67606 .50997 Metetra
1 g
.67606 .50997 .68578 .52491 .70589 .51917 .69625 .50425 Metetra
1 g
.69625 .50425 .70589 .51917 .72614 .51327 .71659 .49855 Metetra
1 g
.71659 .49855 .72614 .51327 .74652 .50719 .73708 .49285 Metetra
1 g
.73708 .49285 .74652 .50719 .76704 .50092 .75773 .48712 Metetra
1 g
.75773 .48712 .76704 .50092 .78769 .49444 .77853 .48134 Metetra
1 g
.77853 .48134 .78769 .49444 .80848 .48776 .79949 .47549 Metetra
1 g
.79949 .47549 .80848 .48776 .82939 .48086 .82061 .46954 Metetra
1 g
.82061 .46954 .82939 .48086 .85044 .47375 .84187 .46348 Metetra
1 g
.84187 .46348 .85044 .47375 .87161 .46643 .86329 .45729 Metetra
1 g
.86329 .45729 .87161 .46643 .8929 .4589 .88486 .45095 Metetra
1 g
.88486 .45095 .8929 .4589 .91432 .45117 .90658 .44443 Metetra
1 g
.90658 .44443 .91432 .45117 .93586 .44325 .92844 .43774 Metetra
1 g
.92844 .43774 .93586 .44325 .95753 .43516 .95045 .43084 Metetra
1 g
.35958 .6169 .37136 .61246 .38952 .60549 .37789 .61022 Metetra
1 g
.37789 .61022 .38952 .60549 .4078 .59854 .39631 .60338 Metetra
1 g
.39631 .60338 .4078 .59854 .42617 .59164 .41486 .59637 Metetra
1 g
.41486 .59637 .42617 .59164 .44466 .58478 .43351 .58923 Metetra
1 g
.43351 .58923 .44466 .58478 .46325 .57801 .45228 .58198 Metetra
1 g
.45228 .58198 .46325 .57801 .48196 .57131 .47117 .57464 Metetra
1 g
.47117 .57464 .48196 .57131 .50079 .56472 .49016 .56723 Metetra
1 g
.49016 .56723 .50079 .56472 .51973 .55823 .50927 .55977 Metetra
1 g
.50927 .55977 .51973 .55823 .5388 .55185 .5285 .55229 Metetra
1 g
.5285 .55229 .5388 .55185 .558 .54558 .54784 .54481 Metetra
1 g
.54784 .54481 .558 .54558 .57733 .53942 .5673 .53736 Metetra
1 g
.5673 .53736 .57733 .53942 .59679 .53337 .58689 .52994 Metetra
1 g
.58689 .52994 .59679 .53337 .61639 .5274 .60659 .52259 Metetra
1 g
.60659 .52259 .61639 .5274 .63613 .52153 .62643 .5153 Metetra
1 g
.62643 .5153 .63613 .52153 .65602 .51572 .64639 .5081 Metetra
1 g
.64639 .5081 .65602 .51572 .67606 .50997 .66649 .50099 Metetra
1 g
.66649 .50099 .67606 .50997 .69625 .50425 .68673 .49399 Metetra
1 g
.68673 .49399 .69625 .50425 .71659 .49855 .70712 .48709 Metetra
1 g
.70712 .48709 .71659 .49855 .73708 .49285 .72765 .48029 Metetra
1 g
.72765 .48029 .73708 .49285 .75773 .48712 .74834 .47359 Metetra
1 g
.74834 .47359 .75773 .48712 .77853 .48134 .76918 .46699 Metetra
1 g
.76918 .46699 .77853 .48134 .79949 .47549 .79018 .46047 Metetra
1 g
.79018 .46047 .79949 .47549 .82061 .46954 .81135 .45403 Metetra
1 g
.81135 .45403 .82061 .46954 .84187 .46348 .83268 .44765 Metetra
1 g
.83268 .44765 .84187 .46348 .86329 .45729 .85418 .44132 Metetra
1 g
.85418 .44132 .86329 .45729 .88486 .45095 .87585 .43501 Metetra
1 g
.87585 .43501 .88486 .45095 .90658 .44443 .89769 .42871 Metetra
1 g
.89769 .42871 .90658 .44443 .92844 .43774 .91971 .4224 Metetra
1 g
.91971 .4224 .92844 .43774 .95045 .43084 .9419 .41606 Metetra
1 g
.34801 .61247 .35958 .6169 .37789 .61022 .36637 .60709 Metetra
1 g
.36637 .60709 .37789 .61022 .39631 .60338 .38486 .60151 Metetra
1 g
.38486 .60151 .39631 .60338 .41486 .59637 .40349 .59573 Metetra
1 g
.40349 .59573 .41486 .59637 .43351 .58923 .42225 .58972 Metetra
1 g
.42225 .58972 .43351 .58923 .45228 .58198 .44114 .58349 Metetra
1 g
.44114 .58349 .45228 .58198 .47117 .57464 .46016 .57704 Metetra
1 g
.46016 .57704 .47117 .57464 .49016 .56723 .47931 .57037 Metetra
1 g
.47931 .57037 .49016 .56723 .50927 .55977 .49859 .56349 Metetra
1 g
.49859 .56349 .50927 .55977 .5285 .55229 .51799 .55641 Metetra
1 g
.51799 .55641 .5285 .55229 .54784 .54481 .53751 .54915 Metetra
1 g
.53751 .54915 .54784 .54481 .5673 .53736 .55716 .54171 Metetra
1 g
.55716 .54171 .5673 .53736 .58689 .52994 .57692 .53413 Metetra
1 g
.57692 .53413 .58689 .52994 .60659 .52259 .59681 .52641 Metetra
1 g
.59681 .52641 .60659 .52259 .62643 .5153 .61682 .51859 Metetra
1 g
.61682 .51859 .62643 .5153 .64639 .5081 .63696 .51067 Metetra
1 g
.63696 .51067 .64639 .5081 .66649 .50099 .65721 .50269 Metetra
1 g
.65721 .50269 .66649 .50099 .68673 .49399 .6776 .49467 Metetra
1 g
.6776 .49467 .68673 .49399 .70712 .48709 .69811 .48662 Metetra
1 g
.69811 .48662 .70712 .48709 .72765 .48029 .71875 .47858 Metetra
1 g
.71875 .47858 .72765 .48029 .74834 .47359 .73952 .47055 Metetra
1 g
.73952 .47055 .74834 .47359 .76918 .46699 .76043 .46256 Metetra
1 g
.76043 .46256 .76918 .46699 .79018 .46047 .78149 .45463 Metetra
1 g
.78149 .45463 .79018 .46047 .81135 .45403 .80269 .44677 Metetra
1 g
.80269 .44677 .81135 .45403 .83268 .44765 .82405 .43898 Metetra
1 g
.82405 .43898 .83268 .44765 .85418 .44132 .84556 .43128 Metetra
1 g
.84556 .43128 .85418 .44132 .87585 .43501 .86724 .42368 Metetra
1 g
.86724 .42368 .87585 .43501 .89769 .42871 .88908 .41617 Metetra
1 g
.88908 .41617 .89769 .42871 .91971 .4224 .91109 .40876 Metetra
1 g
.91109 .40876 .91971 .4224 .9419 .41606 .93328 .40144 Metetra
1 g
.33676 .59888 .34801 .61247 .36637 .60709 .35515 .59397 Metetra
1 g
.35515 .59397 .36637 .60709 .38486 .60151 .37367 .58903 Metetra
1 g
.37367 .58903 .38486 .60151 .40349 .59573 .39234 .58405 Metetra
1 g
.39234 .58405 .40349 .59573 .42225 .58972 .41114 .57899 Metetra
1 g
.41114 .57899 .42225 .58972 .44114 .58349 .43008 .57383 Metetra
1 g
.43008 .57383 .44114 .58349 .46016 .57704 .44917 .56855 Metetra
1 g
.44917 .56855 .46016 .57704 .47931 .57037 .4684 .56312 Metetra
1 g
.4684 .56312 .47931 .57037 .49859 .56349 .48777 .55753 Metetra
1 g
.48777 .55753 .49859 .56349 .51799 .55641 .50728 .55176 Metetra
1 g
.50728 .55176 .51799 .55641 .53751 .54915 .52694 .54579 Metetra
1 g
.52694 .54579 .53751 .54915 .55716 .54171 .54674 .53961 Metetra
1 g
.54674 .53961 .55716 .54171 .57692 .53413 .56667 .53322 Metetra
1 g
.56667 .53322 .57692 .53413 .59681 .52641 .58675 .5266 Metetra
1 g
.58675 .5266 .59681 .52641 .61682 .51859 .60696 .51976 Metetra
1 g
.60696 .51976 .61682 .51859 .63696 .51067 .6273 .5127 Metetra
1 g
.6273 .5127 .63696 .51067 .65721 .50269 .64777 .50542 Metetra
1 g
.64777 .50542 .65721 .50269 .6776 .49467 .66838 .49792 Metetra
1 g
.66838 .49792 .6776 .49467 .69811 .48662 .68911 .49023 Metetra
1 g
.68911 .49023 .69811 .48662 .71875 .47858 .70998 .48236 Metetra
1 g
.70998 .48236 .71875 .47858 .73952 .47055 .73097 .47431 Metetra
1 g
.73097 .47431 .73952 .47055 .76043 .46256 .75209 .46611 Metetra
1 g
.75209 .46611 .76043 .46256 .78149 .45463 .77334 .45778 Metetra
1 g
.77334 .45778 .78149 .45463 .80269 .44677 .79472 .44934 Metetra
1 g
.79472 .44934 .80269 .44677 .82405 .43898 .81623 .44081 Metetra
1 g
.81623 .44081 .82405 .43898 .84556 .43128 .83788 .4322 Metetra
1 g
.83788 .4322 .84556 .43128 .86724 .42368 .85967 .42356 Metetra
1 g
.85967 .42356 .86724 .42368 .88908 .41617 .88161 .41488 Metetra
1 g
.88161 .41488 .88908 .41617 .91109 .40876 .90369 .4062 Metetra
1 g
.90369 .4062 .91109 .40876 .93328 .40144 .92592 .39754 Metetra
1 g
.3252 .58836 .33676 .59888 .35515 .59397 .34372 .58233 Metetra
1 g
.34372 .58233 .35515 .59397 .37367 .58903 .36235 .57642 Metetra
1 g
.36235 .57642 .37367 .58903 .39234 .58405 .38111 .57062 Metetra
1 g
.38111 .57062 .39234 .58405 .41114 .57899 .4 .56492 Metetra
1 g
.4 .56492 .41114 .57899 .43008 .57383 .41902 .55931 Metetra
1 g
.41902 .55931 .43008 .57383 .44917 .56855 .43817 .55377 Metetra
1 g
.43817 .55377 .44917 .56855 .4684 .56312 .45747 .54829 Metetra
1 g
.45747 .54829 .4684 .56312 .48777 .55753 .4769 .54283 Metetra
1 g
.4769 .54283 .48777 .55753 .50728 .55176 .49648 .53739 Metetra
1 g
.49648 .53739 .50728 .55176 .52694 .54579 .51621 .53194 Metetra
1 g
.51621 .53194 .52694 .54579 .54674 .53961 .53609 .52645 Metetra
1 g
.53609 .52645 .54674 .53961 .56667 .53322 .55612 .5209 Metetra
1 g
.55612 .5209 .56667 .53322 .58675 .5266 .5763 .51526 Metetra
1 g
.5763 .51526 .58675 .5266 .60696 .51976 .59663 .50951 Metetra
1 g
.59663 .50951 .60696 .51976 .6273 .5127 .61712 .50364 Metetra
1 g
.61712 .50364 .6273 .5127 .64777 .50542 .63776 .49761 Metetra
1 g
.63776 .49761 .64777 .50542 .66838 .49792 .65855 .49141 Metetra
1 g
.65855 .49141 .66838 .49792 .68911 .49023 .67949 .48502 Metetra
1 g
.67949 .48502 .68911 .49023 .70998 .48236 .70059 .47843 Metetra
1 g
.70059 .47843 .70998 .48236 .73097 .47431 .72182 .47163 Metetra
1 g
.72182 .47163 .73097 .47431 .75209 .46611 .7432 .46461 Metetra
1 g
.7432 .46461 .75209 .46611 .77334 .45778 .76473 .45736 Metetra
1 g
.76473 .45736 .77334 .45778 .79472 .44934 .78639 .44989 Metetra
1 g
.78639 .44989 .79472 .44934 .81623 .44081 .8082 .44219 Metetra
1 g
.8082 .44219 .81623 .44081 .83788 .4322 .83014 .43427 Metetra
1 g
.83014 .43427 .83788 .4322 .85967 .42356 .85222 .42614 Metetra
1 g
.85222 .42614 .85967 .42356 .88161 .41488 .87443 .4178 Metetra
1 g
.87443 .4178 .88161 .41488 .90369 .4062 .89678 .40928 Metetra
1 g
.89678 .40928 .90369 .4062 .92592 .39754 .91926 .40058 Metetra
1 g
.31277 .58914 .3252 .58836 .34372 .58233 .3315 .58182 Metetra
1 g
.3315 .58182 .34372 .58233 .36235 .57642 .35034 .57453 Metetra
1 g
.35034 .57453 .36235 .57642 .38111 .57062 .3693 .56729 Metetra
1 g
.3693 .56729 .38111 .57062 .4 .56492 .38836 .56011 Metetra
1 g
.38836 .56011 .4 .56492 .41902 .55931 .40755 .55302 Metetra
1 g
.40755 .55302 .41902 .55931 .43817 .55377 .42685 .54603 Metetra
1 g
.42685 .54603 .43817 .55377 .45747 .54829 .44628 .53914 Metetra
1 g
.44628 .53914 .45747 .54829 .4769 .54283 .46584 .53236 Metetra
1 g
.46584 .53236 .4769 .54283 .49648 .53739 .48552 .52569 Metetra
1 g
.48552 .52569 .49648 .53739 .51621 .53194 .50534 .51914 Metetra
1 g
.50534 .51914 .51621 .53194 .53609 .52645 .5253 .5127 Metetra
1 g
.5253 .5127 .53609 .52645 .55612 .5209 .54541 .50636 Metetra
1 g
.54541 .50636 .55612 .5209 .5763 .51526 .56566 .50011 Metetra
1 g
.56566 .50011 .5763 .51526 .59663 .50951 .58607 .49394 Metetra
1 g
.58607 .49394 .59663 .50951 .61712 .50364 .60662 .48784 Metetra
1 g
.60662 .48784 .61712 .50364 .63776 .49761 .62734 .48177 Metetra
1 g
.62734 .48177 .63776 .49761 .65855 .49141 .64822 .47573 Metetra
1 g
.64822 .47573 .65855 .49141 .67949 .48502 .66927 .4697 Metetra
1 g
.66927 .4697 .67949 .48502 .70059 .47843 .69047 .46364 Metetra
1 g
.69047 .46364 .70059 .47843 .72182 .47163 .71185 .45753 Metetra
1 g
.71185 .45753 .72182 .47163 .7432 .46461 .73339 .45136 Metetra
1 g
.73339 .45136 .7432 .46461 .76473 .45736 .7551 .4451 Metetra
1 g
.7551 .4451 .76473 .45736 .78639 .44989 .77698 .43872 Metetra
1 g
.77698 .43872 .78639 .44989 .8082 .44219 .79902 .43221 Metetra
1 g
.79902 .43221 .8082 .44219 .83014 .43427 .82122 .42554 Metetra
1 g
.82122 .42554 .83014 .43427 .85222 .42614 .84359 .41869 Metetra
1 g
.84359 .41869 .85222 .42614 .87443 .4178 .86612 .41165 Metetra
1 g
.86612 .41165 .87443 .4178 .89678 .40928 .8888 .40441 Metetra
1 g
.8888 .40441 .89678 .40928 .91926 .40058 .91164 .39695 Metetra
1 g
.29984 .59393 .31277 .58914 .3315 .58182 .31872 .58685 Metetra
1 g
.31872 .58685 .3315 .58182 .35034 .57453 .33773 .57958 Metetra
1 g
.33773 .57958 .35034 .57453 .3693 .56729 .35686 .57215 Metetra
1 g
.35686 .57215 .3693 .56729 .38836 .56011 .37612 .56459 Metetra
1 g
.37612 .56459 .38836 .56011 .40755 .55302 .3955 .55692 Metetra
1 g
.3955 .55692 .40755 .55302 .42685 .54603 .415 .54915 Metetra
1 g
.415 .54915 .42685 .54603 .44628 .53914 .43463 .54132 Metetra
1 g
.43463 .54132 .44628 .53914 .46584 .53236 .45437 .53344 Metetra
1 g
.45437 .53344 .46584 .53236 .48552 .52569 .47424 .52555 Metetra
1 g
.47424 .52555 .48552 .52569 .50534 .51914 .49424 .51767 Metetra
1 g
.49424 .51767 .50534 .51914 .5253 .5127 .51436 .50981 Metetra
1 g
.51436 .50981 .5253 .5127 .54541 .50636 .53461 .502 Metetra
1 g
.53461 .502 .54541 .50636 .56566 .50011 .55499 .49426 Metetra
1 g
.55499 .49426 .56566 .50011 .58607 .49394 .57551 .48659 Metetra
1 g
.57551 .48659 .58607 .49394 .60662 .48784 .59617 .47901 Metetra
1 g
.59617 .47901 .60662 .48784 .62734 .48177 .61697 .47154 Metetra
1 g
.61697 .47154 .62734 .48177 .64822 .47573 .63792 .46417 Metetra
1 g
.63792 .46417 .64822 .47573 .66927 .4697 .65903 .4569 Metetra
1 g
.65903 .4569 .66927 .4697 .69047 .46364 .68029 .44974 Metetra
1 g
.68029 .44974 .69047 .46364 .71185 .45753 .70172 .44269 Metetra
1 g
.70172 .44269 .71185 .45753 .73339 .45136 .72331 .43572 Metetra
1 g
.72331 .43572 .73339 .45136 .7551 .4451 .74508 .42884 Metetra
1 g
.74508 .42884 .7551 .4451 .77698 .43872 .76702 .42203 Metetra
1 g
.76702 .42203 .77698 .43872 .79902 .43221 .78914 .41528 Metetra
1 g
.78914 .41528 .79902 .43221 .82122 .42554 .81144 .40856 Metetra
1 g
.81144 .40856 .82122 .42554 .84359 .41869 .83393 .40186 Metetra
1 g
.83393 .40186 .84359 .41869 .86612 .41165 .85659 .39516 Metetra
1 g
.85659 .39516 .86612 .41165 .8888 .40441 .87945 .38842 Metetra
1 g
.87945 .38842 .8888 .40441 .91164 .39695 .90249 .38164 Metetra
1 g
.28727 .58942 .29984 .59393 .31872 .58685 .30618 .58371 Metetra
1 g
.30618 .58371 .31872 .58685 .33773 .57958 .32523 .57779 Metetra
1 g
.32523 .57779 .33773 .57958 .35686 .57215 .34443 .57164 Metetra
1 g
.34443 .57164 .35686 .57215 .37612 .56459 .36378 .56526 Metetra
1 g
.36378 .56526 .37612 .56459 .3955 .55692 .38327 .55864 Metetra
1 g
.38327 .55864 .3955 .55692 .415 .54915 .40291 .55179 Metetra
1 g
.40291 .55179 .415 .54915 .43463 .54132 .42268 .54471 Metetra
1 g
.42268 .54471 .43463 .54132 .45437 .53344 .4426 .53741 Metetra
1 g
.4426 .53741 .45437 .53344 .47424 .52555 .46265 .5299 Metetra
1 g
.46265 .5299 .47424 .52555 .49424 .51767 .48283 .52219 Metetra
1 g
.48283 .52219 .49424 .51767 .51436 .50981 .50315 .51431 Metetra
1 g
.50315 .51431 .51436 .50981 .53461 .502 .5236 .50627 Metetra
1 g
.5236 .50627 .53461 .502 .55499 .49426 .54419 .49809 Metetra
1 g
.54419 .49809 .55499 .49426 .57551 .48659 .5649 .48979 Metetra
1 g
.5649 .48979 .57551 .48659 .59617 .47901 .58575 .48141 Metetra
1 g
.58575 .48141 .59617 .47901 .61697 .47154 .60673 .47295 Metetra
1 g
.60673 .47295 .61697 .47154 .63792 .46417 .62785 .46445 Metetra
1 g
.62785 .46445 .63792 .46417 .65903 .4569 .6491 .45594 Metetra
1 g
.6491 .45594 .65903 .4569 .68029 .44974 .6705 .44742 Metetra
1 g
.6705 .44742 .68029 .44974 .70172 .44269 .69204 .43892 Metetra
1 g
.69204 .43892 .70172 .44269 .72331 .43572 .71373 .43047 Metetra
1 g
.71373 .43047 .72331 .43572 .74508 .42884 .73557 .42207 Metetra
1 g
.73557 .42207 .74508 .42884 .76702 .42203 .75757 .41375 Metetra
1 g
.75757 .41375 .76702 .42203 .78914 .41528 .77973 .40551 Metetra
1 g
.77973 .40551 .78914 .41528 .81144 .40856 .80206 .39735 Metetra
1 g
.80206 .39735 .81144 .40856 .83393 .40186 .82456 .3893 Metetra
1 g
.82456 .3893 .83393 .40186 .85659 .39516 .84724 .38134 Metetra
1 g
.84724 .38134 .85659 .39516 .87945 .38842 .87011 .37348 Metetra
1 g
.87011 .37348 .87945 .38842 .90249 .38164 .89317 .36571 Metetra
1 g
.27521 .57497 .28727 .58942 .30618 .58371 .29414 .56974 Metetra
1 g
.29414 .56974 .30618 .58371 .32523 .57779 .31322 .56449 Metetra
1 g
.31322 .56449 .32523 .57779 .34443 .57164 .33244 .55919 Metetra
1 g
.33244 .55919 .34443 .57164 .36378 .56526 .35181 .55381 Metetra
1 g
.35181 .55381 .36378 .56526 .38327 .55864 .37134 .54833 Metetra
1 g
.37134 .54833 .38327 .55864 .40291 .55179 .39102 .54272 Metetra
1 g
.39102 .54272 .40291 .55179 .42268 .54471 .41085 .53695 Metetra
1 g
.41085 .53695 .42268 .54471 .4426 .53741 .43084 .53102 Metetra
1 g
.43084 .53102 .4426 .53741 .46265 .5299 .45099 .5249 Metetra
1 g
.45099 .5249 .46265 .5299 .48283 .52219 .47129 .51856 Metetra
1 g
.47129 .51856 .48283 .52219 .50315 .51431 .49175 .51201 Metetra
1 g
.49175 .51201 .50315 .51431 .5236 .50627 .51236 .50523 Metetra
1 g
.51236 .50523 .5236 .50627 .54419 .49809 .53312 .49822 Metetra
1 g
.53312 .49822 .54419 .49809 .5649 .48979 .55403 .49096 Metetra
1 g
.55403 .49096 .5649 .48979 .58575 .48141 .57509 .48347 Metetra
1 g
.57509 .48347 .58575 .48141 .60673 .47295 .59629 .47575 Metetra
1 g
.59629 .47575 .60673 .47295 .62785 .46445 .61764 .46781 Metetra
1 g
.61764 .46781 .62785 .46445 .6491 .45594 .63913 .45965 Metetra
1 g
.63913 .45965 .6491 .45594 .6705 .44742 .66076 .45129 Metetra
1 g
.66076 .45129 .6705 .44742 .69204 .43892 .68254 .44275 Metetra
1 g
.68254 .44275 .69204 .43892 .71373 .43047 .70445 .43405 Metetra
1 g
.70445 .43405 .71373 .43047 .73557 .42207 .7265 .42521 Metetra
1 g
.7265 .42521 .73557 .42207 .75757 .41375 .7487 .41624 Metetra
1 g
.7487 .41624 .75757 .41375 .77973 .40551 .77104 .40718 Metetra
1 g
.77104 .40718 .77973 .40551 .80206 .39735 .79353 .39805 Metetra
1 g
.79353 .39805 .80206 .39735 .82456 .3893 .81617 .38886 Metetra
1 g
.81617 .38886 .82456 .3893 .84724 .38134 .83897 .37964 Metetra
1 g
.83897 .37964 .84724 .38134 .87011 .37348 .86192 .37041 Metetra
1 g
.86192 .37041 .87011 .37348 .89317 .36571 .88504 .3612 Metetra
1 g
.26273 .5641 .27521 .57497 .29414 .56974 .28182 .55764 Metetra
1 g
.28182 .55764 .29414 .56974 .31322 .56449 .30104 .55131 Metetra
1 g
.30104 .55131 .31322 .56449 .33244 .55919 .32038 .54509 Metetra
1 g
.32038 .54509 .33244 .55919 .35181 .55381 .33986 .53898 Metetra
1 g
.33986 .53898 .35181 .55381 .37134 .54833 .35947 .53297 Metetra
1 g
.35947 .53297 .37134 .54833 .39102 .54272 .37923 .52703 Metetra
1 g
.37923 .52703 .39102 .54272 .41085 .53695 .39913 .52116 Metetra
1 g
.39913 .52116 .41085 .53695 .43084 .53102 .41918 .51532 Metetra
1 g
.41918 .51532 .43084 .53102 .45099 .5249 .43939 .5095 Metetra
1 g
.43939 .5095 .45099 .5249 .47129 .51856 .45976 .50367 Metetra
1 g
.45976 .50367 .47129 .51856 .49175 .51201 .48028 .49781 Metetra
1 g
.48028 .49781 .49175 .51201 .51236 .50523 .50098 .49189 Metetra
1 g
.50098 .49189 .51236 .50523 .53312 .49822 .52183 .48589 Metetra
1 g
.52183 .48589 .53312 .49822 .55403 .49096 .54285 .47978 Metetra
1 g
.54285 .47978 .55403 .49096 .57509 .48347 .56404 .47353 Metetra
1 g
.56404 .47353 .57509 .48347 .59629 .47575 .5854 .46713 Metetra
1 g
.5854 .46713 .59629 .47575 .61764 .46781 .60692 .46055 Metetra
1 g
.60692 .46055 .61764 .46781 .63913 .45965 .62861 .45377 Metetra
1 g
.62861 .45377 .63913 .45965 .66076 .45129 .65046 .44679 Metetra
1 g
.65046 .44679 .66076 .45129 .68254 .44275 .67248 .43958 Metetra
1 g
.67248 .43958 .68254 .44275 .70445 .43405 .69465 .43214 Metetra
1 g
.69465 .43214 .70445 .43405 .7265 .42521 .71698 .42446 Metetra
1 g
.71698 .42446 .7265 .42521 .7487 .41624 .73947 .41653 Metetra
1 g
.73947 .41653 .7487 .41624 .77104 .40718 .76211 .40837 Metetra
1 g
.76211 .40837 .77104 .40718 .79353 .39805 .78491 .39997 Metetra
1 g
.78491 .39997 .79353 .39805 .81617 .38886 .80785 .39135 Metetra
1 g
.80785 .39135 .81617 .38886 .83897 .37964 .83095 .3825 Metetra
1 g
.83095 .3825 .83897 .37964 .86192 .37041 .8542 .37345 Metetra
1 g
.8542 .37345 .86192 .37041 .88504 .3612 .87759 .36422 Metetra
1 g
.24907 .56543 .26273 .5641 .28182 .55764 .26841 .55765 Metetra
1 g
.26841 .55765 .28182 .55764 .30104 .55131 .28787 .54989 Metetra
1 g
.28787 .54989 .30104 .55131 .32038 .54509 .30745 .54216 Metetra
1 g
.30745 .54216 .32038 .54509 .33986 .53898 .32715 .53449 Metetra
1 g
.32715 .53449 .33986 .53898 .35947 .53297 .34697 .5269 Metetra
1 g
.34697 .5269 .35947 .53297 .37923 .52703 .36692 .51939 Metetra
1 g
.36692 .51939 .37923 .52703 .39913 .52116 .38699 .51199 Metetra
1 g
.38699 .51199 .39913 .52116 .41918 .51532 .4072 .5047 Metetra
1 g
.4072 .5047 .41918 .51532 .43939 .5095 .42755 .49753 Metetra
1 g
.42755 .49753 .43939 .5095 .45976 .50367 .44804 .49047 Metetra
1 g
.44804 .49047 .45976 .50367 .48028 .49781 .46867 .48352 Metetra
1 g
.46867 .48352 .48028 .49781 .50098 .49189 .48946 .47669 Metetra
1 g
.48946 .47669 .50098 .49189 .52183 .48589 .5104 .46995 Metetra
1 g
.5104 .46995 .52183 .48589 .54285 .47978 .5315 .4633 Metetra
1 g
.5315 .4633 .54285 .47978 .56404 .47353 .55277 .45672 Metetra
1 g
.55277 .45672 .56404 .47353 .5854 .46713 .5742 .4502 Metetra
1 g
.5742 .4502 .5854 .46713 .60692 .46055 .59581 .44371 Metetra
1 g
.59581 .44371 .60692 .46055 .62861 .45377 .61759 .43722 Metetra
1 g
.61759 .43722 .62861 .45377 .65046 .44679 .63956 .43073 Metetra
1 g
.63956 .43073 .65046 .44679 .67248 .43958 .6617 .42419 Metetra
1 g
.6617 .42419 .67248 .43958 .69465 .43214 .68403 .41759 Metetra
1 g
.68403 .41759 .69465 .43214 .71698 .42446 .70653 .41091 Metetra
1 g
.70653 .41091 .71698 .42446 .73947 .41653 .72922 .40411 Metetra
1 g
.72922 .40411 .73947 .41653 .76211 .40837 .75209 .39717 Metetra
1 g
.75209 .39717 .76211 .40837 .78491 .39997 .77515 .39008 Metetra
1 g
.77515 .39008 .78491 .39997 .80785 .39135 .79838 .38281 Metetra
1 g
.79838 .38281 .80785 .39135 .83095 .3825 .82179 .37534 Metetra
1 g
.82179 .37534 .83095 .3825 .8542 .37345 .84537 .36765 Metetra
1 g
.84537 .36765 .8542 .37345 .87759 .36422 .86913 .35974 Metetra
1 g
.23484 .57009 .24907 .56543 .26841 .55765 .25433 .56271 Metetra
1 g
.25433 .56271 .26841 .55765 .28787 .54989 .27395 .55511 Metetra
1 g
.27395 .55511 .28787 .54989 .30745 .54216 .29371 .54734 Metetra
1 g
.29371 .54734 .30745 .54216 .32715 .53449 .31361 .5394 Metetra
1 g
.31361 .5394 .32715 .53449 .34697 .5269 .33365 .53131 Metetra
1 g
.33365 .53131 .34697 .5269 .36692 .51939 .35382 .52311 Metetra
1 g
.35382 .52311 .36692 .51939 .38699 .51199 .37412 .51482 Metetra
1 g
.37412 .51482 .38699 .51199 .4072 .5047 .39455 .50646 Metetra
1 g
.39455 .50646 .4072 .5047 .42755 .49753 .41512 .49807 Metetra
1 g
.41512 .49807 .42755 .49753 .44804 .49047 .43582 .48966 Metetra
1 g
.43582 .48966 .44804 .49047 .46867 .48352 .45666 .48125 Metetra
1 g
.45666 .48125 .46867 .48352 .48946 .47669 .47763 .47288 Metetra
1 g
.47763 .47288 .48946 .47669 .5104 .46995 .49874 .46456 Metetra
1 g
.49874 .46456 .5104 .46995 .5315 .4633 .52 .45631 Metetra
1 g
.52 .45631 .5315 .4633 .55277 .45672 .5414 .44815 Metetra
1 g
.5414 .44815 .55277 .45672 .5742 .4502 .56295 .44007 Metetra
1 g
.56295 .44007 .5742 .4502 .59581 .44371 .58466 .4321 Metetra
1 g
.58466 .4321 .59581 .44371 .61759 .43722 .60654 .42424 Metetra
1 g
.60654 .42424 .61759 .43722 .63956 .43073 .62858 .41648 Metetra
1 g
.62858 .41648 .63956 .43073 .6617 .42419 .65079 .40883 Metetra
1 g
.65079 .40883 .6617 .42419 .68403 .41759 .67318 .40128 Metetra
1 g
.67318 .40128 .68403 .41759 .70653 .41091 .69575 .39383 Metetra
1 g
.69575 .39383 .70653 .41091 .72922 .40411 .71851 .38645 Metetra
1 g
.71851 .38645 .72922 .40411 .75209 .39717 .74146 .37914 Metetra
1 g
.74146 .37914 .75209 .39717 .77515 .39008 .76461 .37187 Metetra
1 g
.76461 .37187 .77515 .39008 .79838 .38281 .78795 .36464 Metetra
1 g
.78795 .36464 .79838 .38281 .82179 .37534 .81149 .3574 Metetra
1 g
.81149 .3574 .82179 .37534 .84537 .36765 .83524 .35016 Metetra
1 g
.83524 .35016 .84537 .36765 .86913 .35974 .85919 .34287 Metetra
1 g
.22127 .56384 .23484 .57009 .25433 .56271 .24074 .55792 Metetra
1 g
.24074 .55792 .25433 .56271 .27395 .55511 .26037 .55179 Metetra
1 g
.26037 .55179 .27395 .55511 .29371 .54734 .28017 .54543 Metetra
1 g
.28017 .54543 .29371 .54734 .31361 .5394 .30013 .53884 Metetra
1 g
.30013 .53884 .31361 .5394 .33365 .53131 .32024 .53199 Metetra
1 g
.32024 .53199 .33365 .53131 .35382 .52311 .34052 .5249 Metetra
1 g
.34052 .5249 .35382 .52311 .37412 .51482 .36096 .51756 Metetra
1 g
.36096 .51756 .37412 .51482 .39455 .50646 .38155 .50997 Metetra
1 g
.38155 .50997 .39455 .50646 .41512 .49807 .40229 .50215 Metetra
1 g
.40229 .50215 .41512 .49807 .43582 .48966 .42318 .49411 Metetra
1 g
.42318 .49411 .43582 .48966 .45666 .48125 .44423 .48586 Metetra
1 g
.44423 .48586 .45666 .48125 .47763 .47288 .46542 .47743 Metetra
1 g
.46542 .47743 .47763 .47288 .49874 .46456 .48675 .46882 Metetra
1 g
.48675 .46882 .49874 .46456 .52 .45631 .50823 .46007 Metetra
1 g
.50823 .46007 .52 .45631 .5414 .44815 .52986 .45119 Metetra
1 g
.52986 .45119 .5414 .44815 .56295 .44007 .55162 .44222 Metetra
1 g
.55162 .44222 .56295 .44007 .58466 .4321 .57354 .43317 Metetra
1 g
.57354 .43317 .58466 .4321 .60654 .42424 .5956 .42408 Metetra
1 g
.5956 .42408 .60654 .42424 .62858 .41648 .61782 .41496 Metetra
1 g
.61782 .41496 .62858 .41648 .65079 .40883 .64019 .40584 Metetra
1 g
.64019 .40584 .65079 .40883 .67318 .40128 .66271 .39674 Metetra
1 g
.66271 .39674 .67318 .40128 .69575 .39383 .6854 .38768 Metetra
1 g
.6854 .38768 .69575 .39383 .71851 .38645 .70825 .37868 Metetra
1 g
.70825 .37868 .71851 .38645 .74146 .37914 .73127 .36975 Metetra
1 g
.73127 .36975 .74146 .37914 .76461 .37187 .75448 .36091 Metetra
1 g
.75448 .36091 .76461 .37187 .78795 .36464 .77786 .35216 Metetra
1 g
.77786 .35216 .78795 .36464 .81149 .3574 .80144 .3435 Metetra
1 g
.80144 .3435 .81149 .3574 .83524 .35016 .82521 .33494 Metetra
1 g
.82521 .33494 .83524 .35016 .85919 .34287 .84918 .32648 Metetra
1 g
.20835 .54805 .22127 .56384 .24074 .55792 .22785 .5424 Metetra
1 g
.22785 .5424 .24074 .55792 .26037 .55179 .2475 .53675 Metetra
1 g
.2475 .53675 .26037 .55179 .28017 .54543 .26731 .53108 Metetra
1 g
.26731 .53108 .28017 .54543 .30013 .53884 .28727 .52536 Metetra
1 g
.28727 .52536 .30013 .53884 .32024 .53199 .3074 .51956 Metetra
1 g
.3074 .51956 .32024 .53199 .34052 .5249 .3277 .51366 Metetra
1 g
.3277 .51366 .34052 .5249 .36096 .51756 .34816 .50762 Metetra
1 g
.34816 .50762 .36096 .51756 .38155 .50997 .3688 .50143 Metetra
1 g
.3688 .50143 .38155 .50997 .40229 .50215 .38961 .49506 Metetra
1 g
.38961 .49506 .40229 .50215 .42318 .49411 .41059 .48849 Metetra
1 g
.41059 .48849 .42318 .49411 .44423 .48586 .43174 .4817 Metetra
1 g
.43174 .4817 .44423 .48586 .46542 .47743 .45306 .47468 Metetra
1 g
.45306 .47468 .46542 .47743 .48675 .46882 .47455 .46741 Metetra
1 g
.47455 .46741 .48675 .46882 .50823 .46007 .49621 .4599 Metetra
1 g
.49621 .4599 .50823 .46007 .52986 .45119 .51803 .45213 Metetra
1 g
.51803 .45213 .52986 .45119 .55162 .44222 .54002 .44412 Metetra
1 g
.54002 .44412 .55162 .44222 .57354 .43317 .56217 .43585 Metetra
1 g
.56217 .43585 .57354 .43317 .5956 .42408 .58448 .42735 Metetra
1 g
.58448 .42735 .5956 .42408 .61782 .41496 .60695 .41861 Metetra
1 g
.60695 .41861 .61782 .41496 .64019 .40584 .62957 .40966 Metetra
1 g
.62957 .40966 .64019 .40584 .66271 .39674 .65235 .40052 Metetra
1 g
.65235 .40052 .66271 .39674 .6854 .38768 .67529 .39119 Metetra
1 g
.67529 .39119 .6854 .38768 .70825 .37868 .69838 .38171 Metetra
1 g
.69838 .38171 .70825 .37868 .73127 .36975 .72164 .3721 Metetra
1 g
.72164 .3721 .73127 .36975 .75448 .36091 .74505 .36238 Metetra
1 g
.74505 .36238 .75448 .36091 .77786 .35216 .76862 .35257 Metetra
1 g
.76862 .35257 .77786 .35216 .80144 .3435 .79235 .3427 Metetra
1 g
.79235 .3427 .80144 .3435 .82521 .33494 .81626 .33279 Metetra
1 g
.81626 .33279 .82521 .33494 .84918 .32648 .84034 .32287 Metetra
1 g
.19468 .53827 .20835 .54805 .22785 .5424 .2144 .53114 Metetra
1 g
.2144 .53114 .22785 .5424 .2475 .53675 .23425 .52414 Metetra
1 g
.23425 .52414 .2475 .53675 .26731 .53108 .25423 .51726 Metetra
1 g
.25423 .51726 .26731 .53108 .28727 .52536 .27434 .5105 Metetra
1 g
.27434 .5105 .28727 .52536 .3074 .51956 .2946 .50386 Metetra
1 g
.2946 .50386 .3074 .51956 .3277 .51366 .315 .49731 Metetra
1 g
.315 .49731 .3277 .51366 .34816 .50762 .33555 .49086 Metetra
1 g
.33555 .49086 .34816 .50762 .3688 .50143 .35626 .48447 Metetra
1 g
.35626 .48447 .3688 .50143 .38961 .49506 .37713 .47813 Metetra
1 g
.37713 .47813 .38961 .49506 .41059 .48849 .39817 .47181 Metetra
1 g
.39817 .47181 .41059 .48849 .43174 .4817 .41938 .46549 Metetra
1 g
.41938 .46549 .43174 .4817 .45306 .47468 .44077 .45914 Metetra
1 g
.44077 .45914 .45306 .47468 .47455 .46741 .46234 .45274 Metetra
1 g
.46234 .45274 .47455 .46741 .49621 .4599 .48408 .44626 Metetra
1 g
.48408 .44626 .49621 .4599 .51803 .45213 .50601 .43966 Metetra
1 g
.50601 .43966 .51803 .45213 .54002 .44412 .52812 .43294 Metetra
1 g
.52812 .43294 .54002 .44412 .56217 .43585 .55042 .42605 Metetra
1 g
.55042 .42605 .56217 .43585 .58448 .42735 .5729 .41898 Metetra
1 g
.5729 .41898 .58448 .42735 .60695 .41861 .59557 .41171 Metetra
1 g
.59557 .41171 .60695 .41861 .62957 .40966 .61841 .40422 Metetra
1 g
.61841 .40422 .62957 .40966 .65235 .40052 .64144 .3965 Metetra
1 g
.64144 .3965 .65235 .40052 .67529 .39119 .66465 .38853 Metetra
1 g
.66465 .38853 .67529 .39119 .69838 .38171 .68803 .3803 Metetra
1 g
.68803 .3803 .69838 .38171 .72164 .3721 .71159 .37182 Metetra
1 g
.71159 .37182 .72164 .3721 .74505 .36238 .73532 .36309 Metetra
1 g
.73532 .36309 .74505 .36238 .76862 .35257 .75922 .3541 Metetra
1 g
.75922 .3541 .76862 .35257 .79235 .3427 .78329 .34486 Metetra
1 g
.78329 .34486 .79235 .3427 .81626 .33279 .80753 .33538 Metetra
1 g
.80753 .33538 .81626 .33279 .84034 .32287 .83194 .32568 Metetra
1 g
.17947 .54135 .19468 .53827 .2144 .53114 .19947 .53304 Metetra
1 g
.19947 .53304 .2144 .53114 .23425 .52414 .2196 .5247 Metetra
1 g
.2196 .5247 .23425 .52414 .25423 .51726 .23986 .51635 Metetra
1 g
.23986 .51635 .25423 .51726 .27434 .5105 .26025 .50802 Metetra
1 g
.26025 .50802 .27434 .5105 .2946 .50386 .28077 .49973 Metetra
1 g
.28077 .49973 .2946 .50386 .315 .49731 .30142 .49151 Metetra
1 g
.30142 .49151 .315 .49731 .33555 .49086 .32221 .48337 Metetra
1 g
.32221 .48337 .33555 .49086 .35626 .48447 .34313 .47532 Metetra
1 g
.34313 .47532 .35626 .48447 .37713 .47813 .3642 .46739 Metetra
1 g
.3642 .46739 .37713 .47813 .39817 .47181 .38541 .45956 Metetra
1 g
.38541 .45956 .39817 .47181 .41938 .46549 .40678 .45185 Metetra
1 g
.40678 .45185 .41938 .46549 .44077 .45914 .4283 .44427 Metetra
1 g
.4283 .44427 .44077 .45914 .46234 .45274 .44998 .43679 Metetra
1 g
.44998 .43679 .46234 .45274 .48408 .44626 .47183 .42942 Metetra
1 g
.47183 .42942 .48408 .44626 .50601 .43966 .49385 .42215 Metetra
1 g
.49385 .42215 .50601 .43966 .52812 .43294 .51605 .41495 Metetra
1 g
.51605 .41495 .52812 .43294 .55042 .42605 .53843 .40782 Metetra
1 g
.53843 .40782 .55042 .42605 .5729 .41898 .561 .40073 Metetra
1 g
.561 .40073 .5729 .41898 .59557 .41171 .58376 .39366 Metetra
1 g
.58376 .39366 .59557 .41171 .61841 .40422 .60672 .38658 Metetra
1 g
.60672 .38658 .61841 .40422 .64144 .3965 .62987 .37948 Metetra
1 g
.62987 .37948 .64144 .3965 .66465 .38853 .65323 .37231 Metetra
1 g
.65323 .37231 .66465 .38853 .68803 .3803 .67678 .36507 Metetra
1 g
.67678 .36507 .68803 .3803 .71159 .37182 .70054 .35772 Metetra
1 g
.70054 .35772 .71159 .37182 .73532 .36309 .7245 .35023 Metetra
1 g
.7245 .35023 .73532 .36309 .75922 .3541 .74866 .34258 Metetra
1 g
.74866 .34258 .75922 .3541 .78329 .34486 .77302 .33476 Metetra
1 g
.77302 .33476 .78329 .34486 .80753 .33538 .79759 .32673 Metetra
1 g
.79759 .32673 .80753 .33538 .83194 .32568 .82234 .31848 Metetra
1 g
.16386 .54481 .17947 .54135 .19947 .53304 .18396 .53729 Metetra
1 g
.18396 .53729 .19947 .53304 .2196 .5247 .20422 .52952 Metetra
1 g
.20422 .52952 .2196 .5247 .23986 .51635 .22464 .52153 Metetra
1 g
.22464 .52153 .23986 .51635 .26025 .50802 .24521 .51332 Metetra
1 g
.24521 .51332 .26025 .50802 .28077 .49973 .26594 .50491 Metetra
1 g
.26594 .50491 .28077 .49973 .30142 .49151 .28682 .49633 Metetra
1 g
.28682 .49633 .30142 .49151 .32221 .48337 .30785 .4876 Metetra
1 g
.30785 .4876 .32221 .48337 .34313 .47532 .32903 .47875 Metetra
1 g
.32903 .47875 .34313 .47532 .3642 .46739 .35035 .46981 Metetra
1 g
.35035 .46981 .3642 .46739 .38541 .45956 .37182 .4608 Metetra
1 g
.37182 .4608 .38541 .45956 .40678 .45185 .39344 .45174 Metetra
1 g
.39344 .45174 .40678 .45185 .4283 .44427 .4152 .44267 Metetra
1 g
.4152 .44267 .4283 .44427 .44998 .43679 .43711 .43361 Metetra
1 g
.43711 .43361 .44998 .43679 .47183 .42942 .45917 .42458 Metetra
1 g
.45917 .42458 .47183 .42942 .49385 .42215 .48139 .41561 Metetra
1 g
.48139 .41561 .49385 .42215 .51605 .41495 .50376 .4067 Metetra
1 g
.50376 .4067 .51605 .41495 .53843 .40782 .5263 .39788 Metetra
1 g
.5263 .39788 .53843 .40782 .561 .40073 .54901 .38915 Metetra
1 g
.54901 .38915 .561 .40073 .58376 .39366 .57188 .38053 Metetra
1 g
.57188 .38053 .58376 .39366 .60672 .38658 .59494 .37201 Metetra
1 g
.59494 .37201 .60672 .38658 .62987 .37948 .61819 .3636 Metetra
1 g
.61819 .3636 .62987 .37948 .65323 .37231 .64162 .3553 Metetra
1 g
.64162 .3553 .65323 .37231 .67678 .36507 .66525 .34709 Metetra
1 g
.66525 .34709 .67678 .36507 .70054 .35772 .68908 .33898 Metetra
1 g
.68908 .33898 .70054 .35772 .7245 .35023 .71312 .33093 Metetra
1 g
.71312 .33093 .7245 .35023 .74866 .34258 .73738 .32295 Metetra
1 g
.73738 .32295 .74866 .34258 .77302 .33476 .76185 .315 Metetra
1 g
.76185 .315 .77302 .33476 .79759 .32673 .78654 .30707 Metetra
1 g
.78654 .30707 .79759 .32673 .82234 .31848 .81145 .29914 Metetra
1 g
.14939 .53495 .16386 .54481 .18396 .53729 .16942 .5289 Metetra
1 g
.16942 .5289 .18396 .53729 .20422 .52952 .18964 .52267 Metetra
1 g
.18964 .52267 .20422 .52952 .22464 .52153 .21003 .51623 Metetra
1 g
.21003 .51623 .22464 .52153 .24521 .51332 .2306 .50957 Metetra
1 g
.2306 .50957 .24521 .51332 .26594 .50491 .25136 .50266 Metetra
1 g
.25136 .50266 .26594 .50491 .28682 .49633 .27229 .4955 Metetra
1 g
.27229 .4955 .28682 .49633 .30785 .4876 .29341 .48808 Metetra
1 g
.29341 .48808 .30785 .4876 .32903 .47875 .3147 .4804 Metetra
1 g
.3147 .4804 .32903 .47875 .35035 .46981 .33616 .47245 Metetra
1 g
.33616 .47245 .35035 .46981 .37182 .4608 .3578 .46424 Metetra
1 g
.3578 .46424 .37182 .4608 .39344 .45174 .37961 .45578 Metetra
1 g
.37961 .45578 .39344 .45174 .4152 .44267 .40158 .44708 Metetra
1 g
.40158 .44708 .4152 .44267 .43711 .43361 .42372 .43816 Metetra
1 g
.42372 .43816 .43711 .43361 .45917 .42458 .44603 .42903 Metetra
1 g
.44603 .42903 .45917 .42458 .48139 .41561 .4685 .41973 Metetra
1 g
.4685 .41973 .48139 .41561 .50376 .4067 .49112 .41026 Metetra
1 g
.49112 .41026 .50376 .4067 .5263 .39788 .51391 .40066 Metetra
1 g
.51391 .40066 .5263 .39788 .54901 .38915 .53686 .39095 Metetra
1 g
.53686 .39095 .54901 .38915 .57188 .38053 .55997 .38116 Metetra
1 g
.55997 .38116 .57188 .38053 .59494 .37201 .58325 .37131 Metetra
1 g
.58325 .37131 .59494 .37201 .61819 .3636 .60669 .36143 Metetra
1 g
.60669 .36143 .61819 .3636 .64162 .3553 .6303 .35154 Metetra
1 g
.6303 .35154 .64162 .3553 .66525 .34709 .65409 .34167 Metetra
1 g
.65409 .34167 .66525 .34709 .68908 .33898 .67805 .33184 Metetra
1 g
.67805 .33184 .68908 .33898 .71312 .33093 .7022 .32206 Metetra
1 g
.7022 .32206 .71312 .33093 .73738 .32295 .72655 .31235 Metetra
1 g
.72655 .31235 .73738 .32295 .76185 .315 .75109 .30272 Metetra
1 g
.75109 .30272 .76185 .315 .78654 .30707 .77583 .29318 Metetra
1 g
.77583 .29318 .78654 .30707 .81145 .29914 .80078 .28373 Metetra
1 g
.13548 .51811 .14939 .53495 .16942 .5289 .15559 .51181 Metetra
1 g
.15559 .51181 .16942 .5289 .18964 .52267 .17585 .50557 Metetra
1 g
.17585 .50557 .18964 .52267 .21003 .51623 .19628 .49937 Metetra
1 g
.19628 .49937 .21003 .51623 .2306 .50957 .21687 .49317 Metetra
1 g
.21687 .49317 .2306 .50957 .25136 .50266 .23763 .48696 Metetra
1 g
.23763 .48696 .25136 .50266 .27229 .4955 .25857 .48069 Metetra
1 g
.25857 .48069 .27229 .4955 .29341 .48808 .27968 .47435 Metetra
1 g
.27968 .47435 .29341 .48808 .3147 .4804 .30098 .4679 Metetra
1 g
.30098 .4679 .3147 .4804 .33616 .47245 .32247 .46131 Metetra
1 g
.32247 .46131 .33616 .47245 .3578 .46424 .34415 .45457 Metetra
1 g
.34415 .45457 .3578 .46424 .37961 .45578 .36601 .44763 Metetra
1 g
.36601 .44763 .37961 .45578 .40158 .44708 .38807 .44049 Metetra
1 g
.38807 .44049 .40158 .44708 .42372 .43816 .41032 .43311 Metetra
1 g
.41032 .43311 .42372 .43816 .44603 .42903 .43276 .4255 Metetra
1 g
.43276 .4255 .44603 .42903 .4685 .41973 .45539 .41763 Metetra
1 g
.45539 .41763 .4685 .41973 .49112 .41026 .47821 .40949 Metetra
1 g
.47821 .40949 .49112 .41026 .51391 .40066 .50121 .40108 Metetra
1 g
.50121 .40108 .51391 .40066 .53686 .39095 .5244 .3924 Metetra
1 g
.5244 .3924 .53686 .39095 .55997 .38116 .54777 .38346 Metetra
1 g
.54777 .38346 .55997 .38116 .58325 .37131 .57132 .37426 Metetra
1 g
.57132 .37426 .58325 .37131 .60669 .36143 .59504 .36481 Metetra
1 g
.59504 .36481 .60669 .36143 .6303 .35154 .61894 .35512 Metetra
1 g
.61894 .35512 .6303 .35154 .65409 .34167 .64302 .34522 Metetra
1 g
.64302 .34522 .65409 .34167 .67805 .33184 .66727 .33512 Metetra
1 g
.66727 .33512 .67805 .33184 .7022 .32206 .6917 .32485 Metetra
1 g
.6917 .32485 .7022 .32206 .72655 .31235 .7163 .31442 Metetra
1 g
.7163 .31442 .72655 .31235 .75109 .30272 .74108 .30387 Metetra
1 g
.74108 .30387 .75109 .30272 .77583 .29318 .76604 .29321 Metetra
1 g
.76604 .29321 .77583 .29318 .80078 .28373 .79118 .28248 Metetra
1 g
.12013 .51151 .13548 .51811 .15559 .51181 .14056 .50345 Metetra
1 g
.14056 .50345 .15559 .51181 .17585 .50557 .16112 .4955 Metetra
1 g
.16112 .4955 .17585 .50557 .19628 .49937 .18181 .48767 Metetra
1 g
.18181 .48767 .19628 .49937 .21687 .49317 .20263 .47997 Metetra
1 g
.20263 .47997 .21687 .49317 .23763 .48696 .2236 .4724 Metetra
1 g
.2236 .4724 .23763 .48696 .25857 .48069 .24471 .46495 Metetra
1 g
.24471 .46495 .25857 .48069 .27968 .47435 .26598 .45761 Metetra
1 g
.26598 .45761 .27968 .47435 .30098 .4679 .28741 .45039 Metetra
1 g
.28741 .45039 .30098 .4679 .32247 .46131 .309 .44326 Metetra
1 g
.309 .44326 .32247 .46131 .34415 .45457 .33076 .4362 Metetra
1 g
.33076 .4362 .34415 .45457 .36601 .44763 .35271 .4292 Metetra
1 g
.35271 .4292 .36601 .44763 .38807 .44049 .37483 .42223 Metetra
1 g
.37483 .42223 .38807 .44049 .41032 .43311 .39715 .41527 Metetra
1 g
.39715 .41527 .41032 .43311 .43276 .4255 .41965 .40828 Metetra
1 g
.41965 .40828 .43276 .4255 .45539 .41763 .44236 .40124 Metetra
1 g
.44236 .40124 .45539 .41763 .47821 .40949 .46526 .39413 Metetra
1 g
.46526 .39413 .47821 .40949 .50121 .40108 .48837 .38691 Metetra
1 g
.48837 .38691 .50121 .40108 .5244 .3924 .51168 .37955 Metetra
1 g
.51168 .37955 .5244 .3924 .54777 .38346 .5352 .37204 Metetra
1 g
.5352 .37204 .54777 .38346 .57132 .37426 .55893 .36433 Metetra
1 g
.55893 .36433 .57132 .37426 .59504 .36481 .58286 .35642 Metetra
1 g
.58286 .35642 .59504 .36481 .61894 .35512 .607 .34828 Metetra
1 g
.607 .34828 .61894 .35512 .64302 .34522 .63134 .3399 Metetra
1 g
.63134 .3399 .64302 .34522 .66727 .33512 .65588 .33126 Metetra
1 g
.65588 .33126 .66727 .33512 .6917 .32485 .68063 .32235 Metetra
1 g
.68063 .32235 .6917 .32485 .7163 .31442 .70557 .31317 Metetra
1 g
.70557 .31317 .7163 .31442 .74108 .30387 .73071 .30371 Metetra
1 g
.73071 .30371 .74108 .30387 .76604 .29321 .75604 .29397 Metetra
1 g
.75604 .29397 .76604 .29321 .79118 .28248 .78156 .28397 Metetra
1 g
.103 .51674 .12013 .51151 .14056 .50345 .1237 .50797 Metetra
1 g
.1237 .50797 .14056 .50345 .16112 .4955 .14455 .49909 Metetra
1 g
.14455 .49909 .16112 .4955 .18181 .48767 .16555 .49011 Metetra
1 g
.16555 .49011 .18181 .48767 .20263 .47997 .1867 .48107 Metetra
1 g
.1867 .48107 .20263 .47997 .2236 .4724 .20799 .47201 Metetra
1 g
.20799 .47201 .2236 .4724 .24471 .46495 .22942 .46293 Metetra
1 g
.22942 .46293 .24471 .46495 .26598 .45761 .251 .45389 Metetra
1 g
.251 .45389 .26598 .45761 .28741 .45039 .27272 .44489 Metetra
1 g
.27272 .44489 .28741 .45039 .309 .44326 .29459 .43595 Metetra
1 g
.29459 .43595 .309 .44326 .33076 .4362 .31661 .4271 Metetra
1 g
.31661 .4271 .33076 .4362 .35271 .4292 .33879 .41835 Metetra
1 g
.33879 .41835 .35271 .4292 .37483 .42223 .36112 .4097 Metetra
1 g
.36112 .4097 .37483 .42223 .39715 .41527 .38362 .40117 Metetra
1 g
.38362 .40117 .39715 .41527 .41965 .40828 .40629 .39276 Metetra
1 g
.40629 .39276 .41965 .40828 .44236 .40124 .42914 .38446 Metetra
1 g
.42914 .38446 .44236 .40124 .46526 .39413 .45217 .37628 Metetra
1 g
.45217 .37628 .46526 .39413 .48837 .38691 .47539 .36819 Metetra
1 g
.47539 .36819 .48837 .38691 .51168 .37955 .4988 .36019 Metetra
1 g
.4988 .36019 .51168 .37955 .5352 .37204 .52242 .35227 Metetra
1 g
.52242 .35227 .5352 .37204 .55893 .36433 .54624 .34439 Metetra
1 g
.54624 .34439 .55893 .36433 .58286 .35642 .57027 .33655 Metetra
1 g
.57027 .33655 .58286 .35642 .607 .34828 .59452 .32871 Metetra
1 g
.59452 .32871 .607 .34828 .63134 .3399 .61899 .32085 Metetra
1 g
.61899 .32085 .63134 .3399 .65588 .33126 .64369 .31294 Metetra
1 g
.64369 .31294 .65588 .33126 .68063 .32235 .66861 .30496 Metetra
1 g
.66861 .30496 .68063 .32235 .70557 .31317 .69375 .29688 Metetra
1 g
.69375 .29688 .70557 .31317 .73071 .30371 .71913 .28866 Metetra
1 g
.71913 .28866 .73071 .30371 .75604 .29397 .74473 .28029 Metetra
1 g
.74473 .28029 .75604 .29397 .78156 .28397 .77057 .27174 Metetra
1 g
.08614 .51666 .103 .51674 .1237 .50797 .10684 .50924 Metetra
1 g
.10684 .50924 .1237 .50797 .14455 .49909 .12772 .50154 Metetra
1 g
.12772 .50154 .14455 .49909 .16555 .49011 .14879 .49357 Metetra
1 g
.14879 .49357 .16555 .49011 .1867 .48107 .17005 .48533 Metetra
1 g
.17005 .48533 .1867 .48107 .20799 .47201 .19149 .47682 Metetra
1 g
.19149 .47682 .20799 .47201 .22942 .46293 .21311 .46807 Metetra
1 g
.21311 .46807 .22942 .46293 .251 .45389 .2349 .45909 Metetra
1 g
.2349 .45909 .251 .45389 .27272 .44489 .25686 .4499 Metetra
1 g
.25686 .4499 .27272 .44489 .29459 .43595 .27899 .44053 Metetra
1 g
.27899 .44053 .29459 .43595 .31661 .4271 .30129 .431 Metetra
1 g
.30129 .431 .31661 .4271 .33879 .41835 .32375 .42134 Metetra
1 g
.32375 .42134 .33879 .41835 .36112 .4097 .34638 .41157 Metetra
1 g
.34638 .41157 .36112 .4097 .38362 .40117 .36917 .40173 Metetra
1 g
.36917 .40173 .38362 .40117 .40629 .39276 .39212 .39184 Metetra
1 g
.39212 .39184 .40629 .39276 .42914 .38446 .41524 .38194 Metetra
1 g
.41524 .38194 .42914 .38446 .45217 .37628 .43852 .37203 Metetra
1 g
.43852 .37203 .45217 .37628 .47539 .36819 .46198 .36216 Metetra
1 g
.46198 .36216 .47539 .36819 .4988 .36019 .48561 .35233 Metetra
1 g
.48561 .35233 .4988 .36019 .52242 .35227 .50941 .34258 Metetra
1 g
.50941 .34258 .52242 .35227 .54624 .34439 .53341 .3329 Metetra
1 g
.53341 .3329 .54624 .34439 .57027 .33655 .55759 .32332 Metetra
1 g
.55759 .32332 .57027 .33655 .59452 .32871 .58197 .31384 Metetra
1 g
.58197 .31384 .59452 .32871 .61899 .32085 .60655 .30446 Metetra
1 g
.60655 .30446 .61899 .32085 .64369 .31294 .63134 .29519 Metetra
1 g
.63134 .29519 .64369 .31294 .66861 .30496 .65634 .28601 Metetra
1 g
.65634 .28601 .66861 .30496 .69375 .29688 .68158 .27694 Metetra
1 g
.68158 .27694 .69375 .29688 .71913 .28866 .70704 .26794 Metetra
1 g
.70704 .26794 .71913 .28866 .74473 .28029 .73273 .25901 Metetra
1 g
.73273 .25901 .74473 .28029 .77057 .27174 .75867 .25013 Metetra
1 g
.07092 .50176 .08614 .51666 .10684 .50924 .09153 .49549 Metetra
1 g
.09153 .49549 .10684 .50924 .12772 .50154 .11233 .48912 Metetra
1 g
.11233 .48912 .12772 .50154 .14879 .49357 .13333 .4826 Metetra
1 g
.13333 .4826 .14879 .49357 .17005 .48533 .15451 .47592 Metetra
1 g
.15451 .47592 .17005 .48533 .19149 .47682 .1759 .46903 Metetra
1 g
.1759 .46903 .19149 .47682 .21311 .46807 .19749 .46193 Metetra
1 g
.19749 .46193 .21311 .46807 .2349 .45909 .21928 .45459 Metetra
1 g
.21928 .45459 .2349 .45909 .25686 .4499 .24127 .44699 Metetra
1 g
.24127 .44699 .25686 .4499 .27899 .44053 .26346 .43913 Metetra
1 g
.26346 .43913 .27899 .44053 .30129 .431 .28585 .43098 Metetra
1 g
.28585 .43098 .30129 .431 .32375 .42134 .30844 .42255 Metetra
1 g
.30844 .42255 .32375 .42134 .34638 .41157 .33123 .41384 Metetra
1 g
.33123 .41384 .34638 .41157 .36917 .40173 .35421 .40485 Metetra
1 g
.35421 .40485 .36917 .40173 .39212 .39184 .37739 .39558 Metetra
1 g
.37739 .39558 .39212 .39184 .41524 .38194 .40075 .38606 Metetra
1 g
.40075 .38606 .41524 .38194 .43852 .37203 .4243 .3763 Metetra
1 g
.4243 .3763 .43852 .37203 .46198 .36216 .44803 .36632 Metetra
1 g
.44803 .36632 .46198 .36216 .48561 .35233 .47194 .35613 Metetra
1 g
.47194 .35613 .48561 .35233 .50941 .34258 .49604 .34577 Metetra
1 g
.49604 .34577 .50941 .34258 .53341 .3329 .52031 .33526 Metetra
1 g
.52031 .33526 .53341 .3329 .55759 .32332 .54477 .32462 Metetra
1 g
.54477 .32462 .55759 .32332 .58197 .31384 .56942 .31388 Metetra
1 g
.56942 .31388 .58197 .31384 .60655 .30446 .59424 .30308 Metetra
1 g
.59424 .30308 .60655 .30446 .63134 .29519 .61926 .29223 Metetra
1 g
.61926 .29223 .63134 .29519 .65634 .28601 .64447 .28136 Metetra
1 g
.64447 .28136 .65634 .28601 .68158 .27694 .66987 .2705 Metetra
1 g
.66987 .2705 .68158 .27694 .70704 .26794 .69548 .25966 Metetra
1 g
.69548 .25966 .70704 .26794 .73273 .25901 .7213 .24887 Metetra
1 g
.7213 .24887 .73273 .25901 .75867 .25013 .74734 .23814 Metetra
1 g
.05558 .48598 .07092 .50176 .09153 .49549 .0764 .47865 Metetra
1 g
.0764 .47865 .09153 .49549 .11233 .48912 .09736 .47143 Metetra
1 g
.09736 .47143 .11233 .48912 .13333 .4826 .11848 .46432 Metetra
1 g
.11848 .46432 .13333 .4826 .15451 .47592 .13976 .45729 Metetra
1 g
.13976 .45729 .15451 .47592 .1759 .46903 .16121 .45033 Metetra
1 g
.16121 .45033 .1759 .46903 .19749 .46193 .18284 .4434 Metetra
1 g
.18284 .4434 .19749 .46193 .21928 .45459 .20465 .43649 Metetra
1 g
.20465 .43649 .21928 .45459 .24127 .44699 .22665 .42956 Metetra
1 g
.22665 .42956 .24127 .44699 .26346 .43913 .24885 .42258 Metetra
1 g
.24885 .42258 .26346 .43913 .28585 .43098 .27124 .41553 Metetra
1 g
.27124 .41553 .28585 .43098 .30844 .42255 .29385 .40837 Metetra
1 g
.29385 .40837 .30844 .42255 .33123 .41384 .31666 .40106 Metetra
1 g
.31666 .40106 .33123 .41384 .35421 .40485 .33969 .39359 Metetra
1 g
.33969 .39359 .35421 .40485 .37739 .39558 .36293 .38593 Metetra
1 g
.36293 .38593 .37739 .39558 .40075 .38606 .38638 .37805 Metetra
1 g
.38638 .37805 .40075 .38606 .4243 .3763 .41005 .36993 Metetra
1 g
.41005 .36993 .4243 .3763 .44803 .36632 .43394 .36155 Metetra
1 g
.43394 .36155 .44803 .36632 .47194 .35613 .45804 .3529 Metetra
1 g
.45804 .3529 .47194 .35613 .49604 .34577 .48235 .34397 Metetra
1 g
.48235 .34397 .49604 .34577 .52031 .33526 .50687 .33475 Metetra
1 g
.50687 .33475 .52031 .33526 .54477 .32462 .53161 .32524 Metetra
1 g
.53161 .32524 .54477 .32462 .56942 .31388 .55654 .31544 Metetra
1 g
.55654 .31544 .56942 .31388 .59424 .30308 .58168 .30536 Metetra
1 g
.58168 .30536 .59424 .30308 .61926 .29223 .60703 .295 Metetra
1 g
.60703 .295 .61926 .29223 .64447 .28136 .63257 .28439 Metetra
1 g
.63257 .28439 .64447 .28136 .66987 .2705 .65831 .27354 Metetra
1 g
.65831 .27354 .66987 .2705 .69548 .25966 .68425 .26246 Metetra
1 g
.68425 .26246 .69548 .25966 .7213 .24887 .71039 .25118 Metetra
1 g
.71039 .25118 .7213 .24887 .74734 .23814 .73673 .23973 Metetra
1 g
.03784 .48486 .05558 .48598 .0764 .47865 .05906 .47575 Metetra
1 g
.05906 .47575 .0764 .47865 .09736 .47143 .08042 .46667 Metetra
1 g
.08042 .46667 .09736 .47143 .11848 .46432 .10192 .45766 Metetra
1 g
.10192 .45766 .11848 .46432 .13976 .45729 .12356 .44874 Metetra
1 g
.12356 .44874 .13976 .45729 .16121 .45033 .14535 .43991 Metetra
1 g
.14535 .43991 .16121 .45033 .18284 .4434 .16728 .4312 Metetra
1 g
.16728 .4312 .18284 .4434 .20465 .43649 .18937 .42261 Metetra
1 g
.18937 .42261 .20465 .43649 .22665 .42956 .21162 .41414 Metetra
1 g
.21162 .41414 .22665 .42956 .24885 .42258 .23402 .40581 Metetra
1 g
.23402 .40581 .24885 .42258 .27124 .41553 .2566 .39759 Metetra
1 g
.2566 .39759 .27124 .41553 .29385 .40837 .27936 .38948 Metetra
1 g
.27936 .38948 .29385 .40837 .31666 .40106 .3023 .38147 Metetra
1 g
.3023 .38147 .31666 .40106 .33969 .39359 .32543 .37354 Metetra
1 g
.32543 .37354 .33969 .39359 .36293 .38593 .34876 .36568 Metetra
1 g
.34876 .36568 .36293 .38593 .38638 .37805 .37229 .35785 Metetra
1 g
.37229 .35785 .38638 .37805 .41005 .36993 .39604 .35004 Metetra
1 g
.39604 .35004 .41005 .36993 .43394 .36155 .42 .34221 Metetra
1 g
.42 .34221 .43394 .36155 .45804 .3529 .44419 .33434 Metetra
1 g
.44419 .33434 .45804 .3529 .48235 .34397 .4686 .32639 Metetra
1 g
.4686 .32639 .48235 .34397 .50687 .33475 .49324 .31834 Metetra
1 g
.49324 .31834 .50687 .33475 .53161 .32524 .51811 .31016 Metetra
1 g
.51811 .31016 .53161 .32524 .55654 .31544 .54322 .30182 Metetra
1 g
.54322 .30182 .55654 .31544 .58168 .30536 .56856 .29328 Metetra
1 g
.56856 .29328 .58168 .30536 .60703 .295 .59414 .28454 Metetra
1 g
.59414 .28454 .60703 .295 .63257 .28439 .61995 .27556 Metetra
1 g
.61995 .27556 .63257 .28439 .65831 .27354 .646 .26632 Metetra
1 g
.646 .26632 .65831 .27354 .68425 .26246 .67227 .25681 Metetra
1 g
.67227 .25681 .68425 .26246 .71039 .25118 .69878 .24701 Metetra
1 g
.69878 .24701 .71039 .25118 .73673 .23973 .72551 .23692 Metetra
1 g
.01866 .49047 .03784 .48486 .05906 .47575 .04005 .48156 Metetra
1 g
.04005 .48156 .05906 .47575 .08042 .46667 .06163 .47243 Metetra
1 g
.06163 .47243 .08042 .46667 .10192 .45766 .08339 .46308 Metetra
1 g
.08339 .46308 .10192 .45766 .12356 .44874 .10532 .45356 Metetra
1 g
.10532 .45356 .12356 .44874 .14535 .43991 .12743 .44388 Metetra
1 g
.12743 .44388 .14535 .43991 .16728 .4312 .1497 .43408 Metetra
1 g
.1497 .43408 .16728 .4312 .18937 .42261 .17214 .42418 Metetra
1 g
.17214 .42418 .18937 .42261 .21162 .41414 .19475 .41423 Metetra
1 g
.19475 .41423 .21162 .41414 .23402 .40581 .21751 .40424 Metetra
1 g
.21751 .40424 .23402 .40581 .2566 .39759 .24044 .39424 Metetra
1 g
.24044 .39424 .2566 .39759 .27936 .38948 .26354 .38427 Metetra
1 g
.26354 .38427 .27936 .38948 .3023 .38147 .2868 .37434 Metetra
1 g
.2868 .37434 .3023 .38147 .32543 .37354 .31023 .36448 Metetra
1 g
.31023 .36448 .32543 .37354 .34876 .36568 .33383 .3547 Metetra
1 g
.33383 .3547 .34876 .36568 .37229 .35785 .35762 .34502 Metetra
1 g
.35762 .34502 .37229 .35785 .39604 .35004 .38158 .33544 Metetra
1 g
.38158 .33544 .39604 .35004 .42 .34221 .40574 .32598 Metetra
1 g
.40574 .32598 .42 .34221 .44419 .33434 .4301 .31663 Metetra
1 g
.4301 .31663 .44419 .33434 .4686 .32639 .45465 .30739 Metetra
1 g
.45465 .30739 .4686 .32639 .49324 .31834 .47943 .29825 Metetra
1 g
.47943 .29825 .49324 .31834 .51811 .31016 .50442 .28921 Metetra
1 g
.50442 .28921 .51811 .31016 .54322 .30182 .52963 .28024 Metetra
1 g
.52963 .28024 .54322 .30182 .56856 .29328 .55508 .27134 Metetra
1 g
.55508 .27134 .56856 .29328 .59414 .28454 .58077 .26247 Metetra
1 g
.58077 .26247 .59414 .28454 .61995 .27556 .60671 .25362 Metetra
1 g
.60671 .25362 .61995 .27556 .646 .26632 .63289 .24476 Metetra
1 g
.63289 .24476 .646 .26632 .67227 .25681 .65933 .23586 Metetra
1 g
.65933 .23586 .67227 .25681 .69878 .24701 .68603 .22689 Metetra
1 g
.68603 .22689 .69878 .24701 .72551 .23692 .71299 .21783 Metetra
1 g
.001 .48329 .01866 .49047 .04005 .48156 .02226 .47614 Metetra
1 g
.02226 .47614 .04005 .48156 .06163 .47243 .04373 .46872 Metetra
1 g
.04373 .46872 .06163 .47243 .08339 .46308 .06541 .46104 Metetra
1 g
.06541 .46104 .08339 .46308 .10532 .45356 .08731 .45306 Metetra
1 g
.08731 .45306 .10532 .45356 .12743 .44388 .10942 .4448 Metetra
1 g
.10942 .4448 .12743 .44388 .1497 .43408 .13175 .43623 Metetra
1 g
.13175 .43623 .1497 .43408 .17214 .42418 .15429 .42737 Metetra
1 g
.15429 .42737 .17214 .42418 .19475 .41423 .17703 .41821 Metetra
1 g
.17703 .41821 .19475 .41423 .21751 .40424 .19998 .40878 Metetra
1 g
.19998 .40878 .21751 .40424 .24044 .39424 .22313 .39908 Metetra
1 g
.22313 .39908 .24044 .39424 .26354 .38427 .24648 .38914 Metetra
1 g
.24648 .38914 .26354 .38427 .2868 .37434 .27002 .37897 Metetra
1 g
.27002 .37897 .2868 .37434 .31023 .36448 .29375 .3686 Metetra
1 g
.29375 .3686 .31023 .36448 .33383 .3547 .31767 .35805 Metetra
1 g
.31767 .35805 .33383 .3547 .35762 .34502 .34178 .34736 Metetra
1 g
.34178 .34736 .35762 .34502 .38158 .33544 .36608 .33655 Metetra
1 g
.36608 .33655 .38158 .33544 .40574 .32598 .39056 .32566 Metetra
1 g
.39056 .32566 .40574 .32598 .4301 .31663 .41522 .3147 Metetra
1 g
.41522 .3147 .4301 .31663 .45465 .30739 .44008 .30372 Metetra
1 g
.44008 .30372 .45465 .30739 .47943 .29825 .46513 .29273 Metetra
1 g
.46513 .29273 .47943 .29825 .50442 .28921 .49038 .28176 Metetra
1 g
.49038 .28176 .50442 .28921 .52963 .28024 .51583 .27083 Metetra
1 g
.51583 .27083 .52963 .28024 .55508 .27134 .54148 .25996 Metetra
1 g
.54148 .25996 .55508 .27134 .58077 .26247 .56735 .24916 Metetra
1 g
.56735 .24916 .58077 .26247 .60671 .25362 .59343 .23845 Metetra
1 g
.59343 .23845 .60671 .25362 .63289 .24476 .61975 .22784 Metetra
1 g
.61975 .22784 .63289 .24476 .65933 .23586 .6463 .21732 Metetra
1 g
.6463 .21732 .65933 .23586 .68603 .22689 .6731 .2069 Metetra
1 g
.6731 .2069 .68603 .22689 .71299 .21783 .70015 .19657 Metetra
0 g
.25 Mabswid
.68874 0 m
.96935 .42924 L
s
.96935 .42924 m
1 .6535 L
s
1 .6535 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.03716 .25514 m
0 .48963 L
s
0 .48963 m
.70298 .24544 L
s
.70298 .24544 m
.68874 0 L
s
.68874 0 m
.03716 .25514 L
s
.03716 .25514 m
.68874 0 L
s
.18558 .19702 m
.19014 .20174 L
s
[(-1.65)] .17646 .18758 .96648 1 Mshowa
.34306 .13535 m
.34735 .14032 L
s
[(-1.6)] .3345 .12542 .86223 1 Mshowa
.51046 .06981 m
.51442 .07504 L
s
[(-1.55)] .50253 .05935 .75799 1 Mshowa
.68874 0 m
.69233 .00549 L
s
[(-1.5)] .68156 -0.01098 .65374 1 Mshowa
.125 Mabswid
.06616 .24378 m
.06901 .2465 L
s
.09549 .2323 m
.09832 .23504 L
s
.12517 .22067 m
.12797 .22345 L
s
.1552 .20892 m
.15797 .21172 L
s
.21633 .18498 m
.21903 .18784 L
s
.24744 .1728 m
.25012 .17569 L
s
.27893 .16047 m
.28157 .16339 L
s
.3108 .14799 m
.31341 .15094 L
s
.37572 .12257 m
.37826 .12558 L
s
.40878 .10962 m
.41128 .11266 L
s
.44225 .09652 m
.44471 .09959 L
s
.47614 .08324 m
.47856 .08635 L
s
.54521 .0562 m
.54755 .05937 L
s
.58041 .04242 m
.5827 .04562 L
s
.61605 .02846 m
.6183 .03169 L
s
.65216 .01432 m
.65436 .01759 L
s
gsave
.29165 .07573 -81.4846 -16.5 Mabsadd m
1 1 Mabs scale
currentpoint translate
/MISOfy
{
    /newfontname exch def
    /oldfontname exch def

    oldfontname findfont
    dup length dict begin
        {1 index /FID ne {def} {pop pop} ifelse} forall
        /Encoding ISOLatin1Encoding def
        currentdict
    end

    newfontname exch definefont pop
} def

0 20.5 translate 1 -1 scale
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
63.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
(x) show
81.000 12.750 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000 0.000 0.000 setrgbcolor
0.000 0.000 rmoveto
1.000 setlinewidth
grestore
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 233.563},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000jA000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00
ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo003oOolQOol00;Ioo`D0009oo`8000Aoo`8005]oo`00^7oo00<007oo
Ool01goo00@007ooOol005Yoo`00^7oo00<007ooOol02Woo00<007ooOol0F7oo002`Ool40004Ool0
0`00Oomoo`08Ool2001KOol00;Qoo`03001oogoo00Qoo`03001oogoo05Yoo`00]goo0P002Woo0`00
2Goo0P0017oo0P001Goo00<007ooOol0@Goo003>Ool01000Oomoo`002Woo00<007ooOol0@Goo003A
Ool00`00Oomoo`08Ool00`00Oomoo`11Ool00<Qoo`@000=oo`8000aoo`03001oogoo041oo`00cgoo
00<007ooOol027oo00@007ooOol0049oo`00`goo0P002Woo0`0027oo1000@Woo0031Ool5001JOol0
0;ioo`<000=oo`8005Yoo`00_7oo0P001Woo00<007oo0000FGoo002iOol40007Ool01000Oomoo`00
F7oo0026Ool50002Ool20004Ool20004Ool2000KOol3000;Ool01000Oomoo`00F7oo0028Ool00`00
Oomoo`07Ool01000Oomoo`000Woo00@007ooOol001Qoo`8000ioo`03001oogoo008005Moo`00R7oo
00<007ooOol02Woo00<007ooOol00goo00<007ooOol04goo10003goo00<007ooOol00Woo00<007oo
Ool0E7oo0020Ool40004Ool00`00Oomoo`08Ool20004Ool2000DOol2000COol00`00Oomoo`02Ool0
0`00Oomoo`1DOol005Qoo`800003Ool0000002]oo`03001oogoo00Qoo`03001oogoo00=oo`03001o
ogoo011oo`<001Eoo`03001oogoo00=oo`03001oogoo05=oo`00FGoo00<007oo0000:goo0P002Woo
0`000goo0`003Woo0`0067oo00<007ooOol00goo00<007ooOol0DWoo001JOol00`00Oomoo`1:Ool3
000KOol00`00Oomoo`03Ool00`00Oomoo`1BOol005Qoo`800003Ool0000004Qoo`8001ioo`03001o
ogoo00=oo`8005=oo`00XWoo0`0087oo00<007ooOol01Goo00<007ooOol0D7oo002OOol3000SOol0
0`00Oomoo`05Ool00`00Oomoo`1@Ool009eoo`<002Eoo`03001oogoo00Ioo`03001oogoo04moo`00
VWoo0`00:7oo00<007ooOol01goo00<007ooOol0CWoo002HOol20003Ool00`00Oomoo`0UOol00`00
Oomoo`07Ool00`00Oomoo`1>Ool009Eoo`<002eoo`03001oogoo00Moo`8004moo`00Tgoo0P00<7oo
00<007ooOol02Goo00<007ooOol0C7oo001TOol50002Ool20003Ool3000MOol300000goo001oo`0_
Ool00`00Oomoo`09Ool00`00Oomoo`1<Ool006Ioo`03001oogoo00Moo`04001oogoo000JOol2000e
Ool00`00Oomoo`0:Ool00`00Oomoo`1;Ool006Ioo`03001oogoo00Moo`04001oogoo000GOol3000g
Ool00`00Oomoo`0;Ool00`00Oomoo`1:Ool005ioo`@000Aoo`03001oogoo00Moo`<001Ioo`80009o
o`03001oogoo03Eoo`03001oogoo00Yoo`8000eoo`8000Aoo`8000=oo`<0035oo`00IWoo00<007oo
Ool027oo00<007ooOol04Woo0`00?7oo00<007ooOol037oo00<007ooOol02Goo00@007ooOol000Qo
o`04001oogoo000`Ool006Eoo`8000Yoo`<000moo`<003moo`03001oogoo00aoo`03001oogoo00ao
o`03001oogoo00Ioo`04001oogoo000`Ool0085oo`80009oo`03001oogoo03eoo`03001oogoo00eo
o`03001oogoo009oo`@000=oo`8000Uoo`<0035oo`00OWoo0`00A7oo00<007ooOol03Woo00<007oo
Ool027oo00<007ooOol02Goo00<007ooOol0<7oo001lOol30016Ool00`00Oomoo`0>Ool20009Ool3
0009Ool3000`Ool007Uoo`<004Yoo`03001oogoo00aoo`<004Qoo`00Mgoo0P00CGoo00<007ooOol0
3goo00<007ooOol0AGoo001dOol3001?Ool00`00Oomoo`0?Ool00`00Oomoo`15Ool0079oo`8000=o
o`03001oogoo04aoo`03001oogoo011oo`03001oogoo04Aoo`00Kgoo0`00E7oo00<007ooOol04Goo
00<007ooOol0@goo001/Ool4001FOol00`00Oomoo`0@Ool20015Ool003Uoo`D0009oo`8000=oo`<0
00Aoo`8001aoo`8005Yoo`03001oogoo019oo`03001oogoo049oo`00>goo00<007ooOol01goo00@0
07ooOol0009oo`04001oogoo000HOol3001LOol00`00Oomoo`0COol00`00Oomoo`11Ool003]oo`03
001oogoo00Moo`04001oogoo0005Ool00`00Oomoo`0DOol200000goo001oo`1LOol00`00Oomoo`0C
Ool00`00Oomoo`11Ool003=oo`@000Aoo`03001oogoo00Moo`<000Aoo`8001Aoo`<0065oo`03001o
ogoo01Aoo`03001oogoo041oo`00>goo00<007ooOol027oo00<007ooOol00goo00<007ooOol04Goo
0P00I7oo00<007ooOol057oo0P00@Goo000jOol2000:Ool30003Ool3000>Ool300000goo001oo`1S
Ool00`00Oomoo`0EOol00`00Oomoo`0oOol005]oo`8006Uoo`03001oogoo01Ioo`03001oogoo03io
o`00F7oo0`00Jgoo00<007ooOol05goo00<007ooOol0?Goo001EOol30002Ool00`00Oomoo`1YOol0
0`00Oomoo`0GOol00`00Oomoo`0`Ool2000;Ool005=oo`<0071oo`03001oogoo01Moo`80039oo`03
001oogoo00Uoo`00D7oo0`00Lgoo0P006Woo00<007ooOol0<7oo00<007ooOol027oo001>Ool20003
Ool00`00Oomoo`1/Ool500000goo001oo`0HOol00`00Oomoo`0_Ool00`00Ool00009Ool004]oo`<0
075oo`<000Eoo`03001oo`0001Uoo`03001oogoo02eoo`04001oogoo0009Ool004Uoo`800003Ool0
07oo06ioo`<0009oo`03001oogoo00=oo`03001oo`0001Yoo`03001oogoo02]oo`<00003Ool00000
00Qoo`00AWoo0`00L7oo0`001Goo00<007ooOol00goo00@007oo000001Qoo`8000eoo`8000Aoo`80
00Aoo`80021oo`00A7oo0P00LGoo0P000Woo00<007ooOol017oo00<007ooOol0100000=oo`00Ool0
6Goo00<007ooOol02Goo00@007ooOol000Qoo`04001oogoo000OOol0045oo`<00003Ool007oo06io
o`8000Aoo`03001oogoo00Aoo`<0009oo`05001oogooOol0000JOol00`00Oomoo`0;Ool00`00Oomo
o`09Ool00`00Oomoo`0MOol003moo`80079oo`8000Moo`03001oogoo00<0009oo`04001oogoo0003
Ool00`00Oomoo`0HOol00`00Oomoo`02Ool40003Ool2000:Ool2000POol003aoo`<0075oo`<0009o
o`03001oogoo00Eoo`8000Eoo`04001oogoo0004Ool00`00Oomoo`0HOol00`00Oomoo`08Ool00`00
Oomoo`09Ool00`00Oomoo`0OOol003Uoo`<0075oo`<000Eoo`03001oogoo00=oo`800003Ool007oo
00Eoo`03001oo`0000=oo`L001Aoo`<000Uoo`<000Uoo`<001moo`00=goo0P00LWoo0P000Woo00<0
07ooOol017oo00<007ooOol00P000goo00<007ooOol017oo00<007oo00000`000Woo0`000Woo00<0
07ooOol04goo00<007ooOol0=Goo000dOol300000goo001oo`1^Ool30004Ool00`00Oomoo`04Ool3
0006Ool00`00Oomoo`03Ool30003Ool20006Ool00`00Oomoo`0COol00`00Oomoo`0dOol0039oo`80
079oo`80009oo`03001oogoo00=oo`03001oogoo00<0009oo`03001oogoo00Aoo`03001oogoo00@0
009oo`<00003Ool007oo00Eoo`8001Eoo`03001oogoo03=oo`00<7oo0P00LWoo0P0017oo00<007oo
Ool00goo0`001Goo00<007ooOol017oo100017oo0P0017oo00<007ooOol00`000Woo00<007ooOol0
4Woo00<007ooOol0<goo000`Ool2001`Ool20007Ool00`00Oomoo`020002Ool00`00Oomoo`04Ool0
0`00Oomoo`0400000goo00000002Ool20002Ool01000Oomoogoo0`001Goo00<007ooOol04Woo0P00
=7oo000`Ool00`00Ool0001/Ool30002Ool00`00Oomoo`04Ool30004Ool00`00Oomoo`04Ool40005
Ool30002Ool00`00Oomoo`030002Ool00`00Oomoo`04Ool00`00Oomoo`0COol00`00Oomoo`0aOol0
031oo`800003Ool007oo06Moo`<000Ioo`03001oogoo00<0009oo`03001oogoo00=oo`03001oogoo
00@00003Ool00000009oo`<000Eoo`<00003Ool007oo009oo`03001oogoo00Aoo`03001oogoo01=o
o`03001oogoo035oo`00;goo00<007ooOol00Woo00<007ooOol0Hgoo0P000Woo00<007ooOol017oo
0`001Goo00<007ooOol017oo100017oo100000Aoo`00Oomoo`<0009oo`03001oo`0000Aoo`03001o
ogoo00=oo`8001Eoo`03001oogoo031oo`00;goo00<007ooOol00goo00<007ooOol0H7oo0P001Goo
00<007ooOol00`000Woo00<007ooOol017oo00@007ooOomoo`D0009oo`D000=oo`<000Eoo`04001o
ogoo0003Ool00`00Oomoo`030002Ool00`00Oomoo`0BOol2000aOol002moo`03001oogoo00Aoo`80
05ioo`8000Moo`<000Eoo`03001oogoo00Aoo`@000Eoo`D000=oo`80009oo`03001oogoo00Aoo`05
001oogoo001oo`040006Ool5000?Ool2000aOol002moo`8000Moo`03001oogoo05Qoo`<0009oo`80
009oo`<0009oo`03001oogoo00Aoo`04001oogooOol40005Ool40003Ool30004Ool00`00Oomoo`04
Ool50003Ool00`00Oomoo`02Ool50002Ool00`00Oomoo`0>Ool00`00Oomoo`0^Ool002moo`03001o
ogoo00Moo`03001oogoo05Eoo`8000Ioo`<000Eoo`03001oogoo00Aoo`H0009oo`800004Ool00000
0003Ool300000goo001oo`05Ool00`00Oomoo`030002Ool00`00Ool00003Ool00`00Oomoo`050005
Ool00`00Oomoo`0>Ool00`00Oomoo`0]Ool002moo`03001oogoo00Qoo`03001oogoo059oo`800004
Ool000000002Ool20002Ool00`00Oomoo`04Ool01000Oomoogoo1@001Woo0`000Woo0`001Goo00<0
07ooOol00goo0`001Goo00@007ooOol0009oo`L000Ioo`03001oogoo00ioo`03001oogoo02eoo`00
;Woo00<007ooOol02Woo00<007ooOol0Cgoo0P001Goo0`0017oo00<007ooOol017oo1@000goo0P00
0Woo100000Aoo`000000009oo`03001oogoo00=oo`03001oogoo00<0009oo`03001oogoo00Aoo`03
001oogoo00H000=oo`03001oogoo00=oo`8000moo`8002ioo`00;Woo0P0037oo00<007ooOol0C7oo
0`000goo0`000Woo00<007ooOol00goo00<007ooOol0100000=oo`00Ool01Woo1`001Goo00<007oo
Ool00goo0`001Goo00<007ooOol017oo1`001Goo00<007ooOol00`000Woo00<007ooOol03Woo00<0
07ooOol0:goo000^Ool00`00Oomoo`0<Ool00`00Oomoo`18Ool30003Ool30005Ool00`00Oomoo`03
Ool40004Ool00`00Oomoo`03Ool60002Ool00`00Oomoo`04Ool00`00Oomoo`020002Ool00`00Oomo
o`04Ool00`00Oomoo`0800000goo00000003Ool40005Ool00`00Oomoo`0>Ool00`00Oomoo`0[Ool0
02ioo`03001oogoo00eoo`03001oogoo04Aoo`<000=oo`<0009oo`03001oogoo00Aoo`03001oogoo
00D000Ioo`03001oogoo00H000Aoo`03001oogoo00Aoo`<000Aoo`03001oogoo00Aoo`P000=oo`03
001oo`0000<000=oo`03001oogoo00Aoo`03001oogoo00ioo`03001oogoo02Yoo`00;Woo00<007oo
Ool03Woo00<007ooOol0@Goo0P000goo0`001Goo00<007ooOol017oo10000goo00<007ooOol017oo
1P000Woo00<007ooOol00goo00<007ooOol00`000Woo00<007ooOol00goo00<007ooOol00`0000=o
o`0000000P000Woo00<007ooOol00`0000=oo`00Ool017oo00<007ooOol00goo0P0047oo0P0037oo
0P0017oo0P001Goo0`003goo000^Ool2000@Ool00`00Oomoo`0nOol20003Ool20002Ool00`00Oomo
o`04Ool00`00Oomoo`0400000goo001oo`04Ool00`00Oomoo`060005Ool00`00Oomoo`03Ool30005
Ool00`00Oomoo`03Ool30002Ool30005Ool30003Ool00`00Ool00005Ool00`00Oomoo`020002Ool0
1000Ool000002goo00<007oo00002goo00@007ooOol000]oo`03001oogoo00ioo`00;Woo00<007oo
Ool047oo00<007ooOol0>goo1`001Goo00<007ooOol00goo100017oo00<007ooOol00goo1P000Woo
00<007ooOol017oo00<007ooOol00P000Woo00<007ooOol017oo00<007ooOol00P000goo0`0000=o
o`0000000Woo0P000Woo00L007ooOomoo`00Ool00005Ool30004Ool20002Ool2000<Ool00`00Oomo
o`0;Ool00`00Oomoo`06Ool5000?Ool002eoo`03001oogoo019oo`03001oogoo03Qoo`800004Ool0
00000002Ool00`00Oomoo`04Ool00`00Oomoo`0300000goo001oo`05Ool00`00Oomoo`02000017oo
000000001Goo00<007ooOol00Woo100017oo00<007ooOol017oo0`000Woo0`001Goo0`0017oo00<0
07ooOol00Woo00@007oo001oo`<0009oo`03001oogoo0080009oo`80009oo`03001oogoo00Uoo`03
001oogoo009oo`@000=oo`8000Uoo`04001oogoo000@Ool002eoo`03001oogoo01=oo`03001oogoo
03Aoo`H000Eoo`03001oogoo00Aoo`<000Aoo`03001oogoo00Aoo`L00003Ool007oo00Aoo`D0009o
o`03001oogoo00Aoo`03001oogoo00<0009oo`<0009oo`03001oogoo00<00003Ool007oo00Eoo`03
001oogoo009oo`<000Eoo`<00004Ool000000005Ool00`00Oomoo`09Ool00`00Oomoo`08Ool00`00
Oomoo`09Ool00`00Ool0000@Ool002eoo`<001Aoo`80039oo`800003Ool00000009oo`8000Eoo`03
001oogoo00<0009oo`03001oogoo00=oo`03001oogoo00P000Aoo`03001oogoo0080009oo`03001o
ogoo00=oo`03001oogoo00=oo`<000=oo`8000Eoo`<000Eoo`03001oogoo00=oo`03001oogoo0080
0003Ool00000009oo`@00003Ool00000009oo`03001oogoo00=oo`03001oogoo00Qoo`<000Uoo`<0
00Yoo`80011oo`00;Goo00<007ooOol05Woo00<007ooOol0;Goo1@001Goo0P000goo10001Goo00<0
07ooOol00goo2@001Woo0`0017oo00<007ooOol00goo00<007ooOol00`000goo0`0000=oo`000000
0Woo0P000Woo00<007ooOol00goo00<007ooOol00Woo10000goo2@0017oo00<007ooOol00Woo0`00
2goo00<007ooOol09Woo000IOol50002Ool20004Ool20005Ool00`00Oomoo`0GOol00`00Oomoo`0Z
Ool40002Ool00`00Oomoo`04Ool50002Ool00`00Oomoo`04Ool00`00Oomoo`020002Ool40002Ool2
0002Ool30002Ool00`00Oomoo`03Ool00`00Oomoo`02Ool30005Ool20004Ool30004Ool00`00Oomo
o`03Ool50003Ool00`00Oomoo`070002Ool00`00Oomoo`02Ool50003Ool00`00Oomoo`09Ool00`00
Oomoo`0UOol001]oo`05001oogooOol00002Ool01000Oomoo`000Woo00<007ooOol00Woo00<007oo
Ool067oo00<007ooOol09goo10001Goo00<007ooOol01P0017oo00<007ooOol017oo0`000Woo1000
1Goo0`001Goo00<007ooOol00goo00<007ooOol00P000Woo0P000Woo0`000Woo0`0000=oo`00Ool0
1Goo00<007ooOol00P000Woo00<007ooOol00goo1`001Goo00<007ooOol00P000Woo00<007ooOol0
0goo00<007ooOol02Woo00<007ooOol097oo000KOol00`00Oomoo`04Ool01000Oomoo`000Woo00<0
07ooOol00Woo00<007ooOol06Goo00<007ooOol08goo1@0000=oo`00Ool017oo1P000Woo00<007oo
Ool00goo00<007ooOol00`000goo0`000Woo0P000Woo0P000Woo00<007ooOol017oo00<007ooOol0
0Woo0`001Goo0`0000=oo`0000000P001Goo00<007ooOol00goo0`001Goo00<007ooOol01P000Woo
00<007ooOol00goo0`0017oo00<007ooOol017oo00<007ooOol027oo0P009Woo000KOol00`00Oomo
o`02Ool20003Ool01000Oomoo`000goo00<007ooOol06goo00<007ooOol087oo0P0000=oo`000000
17oo00<007ooOol01@001Woo00<007ooOol00Woo0`000goo100017oo100017oo00<007ooOol017oo
00<007ooOol00P000Woo0P000Woo1P000goo00<007ooOol00goo00<007ooOol00`000Woo00<007oo
Ool00goo1`001Goo00<007ooOol00P0000=oo`00Ool01Goo00@007ooOomoo`<0009oo`8000Qoo`03
001oogoo02=oo`006goo00<007ooOol00Woo00<007ooOol00Woo00@007ooOol000=oo`03001oogoo
01aoo`03001oogoo01eoo`D000Ioo`D0009oo`03001oogoo00Aoo`03001oogoo008000=oo`@000=o
o`@000=oo`03001oogoo00=oo`03001oogoo009oo`@000Eoo`D0009oo`03001oogoo00=oo`04001o
ogooOol40005Ool00`00Oomoo`070002Ool00`00Oomoo`02Ool40004Ool00`00Oomoo`03Ool40003
Ool200000goo00000008Ool00`00Oomoo`0ROol001Yoo`8000Aoo`<000=oo`8000Aoo`03001oogoo
01eoo`03001oogoo01Yoo`@0009oo`<0009oo`D000Eoo`03001oogoo00=oo`<000=oo`<000=oo`<0
009oo`03001oogoo009oo`03001oogoo00=oo`D000=oo`03001oogoo00H000Aoo`03001oogoo00=o
o`@000=oo`03001oogoo00=oo`P000Aoo`D0009oo`8000Aoo`03001oogoo00<0009oo`03001oogoo
00800003Ool00000009oo`03001oogoo00Eoo`03001oogoo029oo`00;7oo0P007goo00<007ooOol0
5goo10001Goo1@000goo00<007ooOol00goo00<007ooOol00`000goo10000Woo0`001Goo00<007oo
Ool00goo00<007ooOol00P000Woo00<007ooOol00goo0`0000Eoo`00001oo`0000Ioo`03001oogoo
0080009oo`03001oogoo00=oo`03001oogoo00800003Ool000000080009oo`03001oogoo0080009o
o`03001oogoo00=oo`03001oogoo009oo`<000Eoo`H000Aoo`03001oogoo00Eoo`8002=oo`00;7oo
00<007ooOol07goo00<007ooOol04goo10000goo1`0000=oo`00Ool017oo00<007ooOol00Woo0`00
1Goo0`000Woo0P000Woo00<007ooOol017oo00<007ooOol00Woo0`001Goo00<007ooOol01P0017oo
00<007ooOol00goo0`0017oo00<007ooOol00goo0`0000Aoo`00000000Eoo`<000Aoo`03001oogoo
00=oo`800003Ool00000009oo`03001oogoo00D000=oo`03001oogoo00=oo`03001oogoo00Ioo`03
001oogoo021oo`00;7oo00<007ooOol087oo00<007ooOol047oo10000goo0P0000Aoo`00000000Ao
o`03001oogoo00=oo`03001oogoo0080009oo`<00003Ool0000000D000Aoo`03001oogoo00Aoo`03
001oogoo0080009oo`03001oogoo00=oo`H0009oo`03001oogoo009oo`03001oogoo00<00003Ool0
07oo00Eoo`03001oogoo0080009oo`<00003Ool00000009oo`800003Ool007oo00Eoo`04001oogoo
Ool40004Ool400000goo001oo`04Ool01000Oomoogoo0`0027oo00<007ooOol087oo000[Ool00`00
Oomoo`0ROol00`00Oomoo`0<Ool40003Ool300000goo00000002Ool00`00Oomoo`03Ool01000Oomo
ogoo10001Woo0`0000=oo`0000000Woo00<007ooOol00goo00<007ooOol00Woo10001Goo00<007oo
Ool01@001Goo00@007ooOomoo`@000Eoo`03001oogoo00=oo`<0009oo`8000Aoo`@000Aoo`03001o
ogoo00=oo`@000=oo`800003Ool00000008000Aoo`03001oogoo009oo`@000=oo`03001oogoo00Io
o`03001oogoo01moo`00:goo00<007ooOol08goo00<007ooOol027oo1@000goo0P0000Aoo`000000
00Aoo`03001oogoo00=oo`@000=oo`80009oo`L000Aoo`03001oogoo00=oo`D000=oo`03001oogoo
00=oo`H00003Ool007oo00Aoo`@000=oo`03001oogoo00=oo`03001oogoo00<0009oo`<0009oo`@0
009oo`03001oogoo00=oo`03001oogoo00<0009oo`03001oogoo009oo`@0009oo`03001oogoo009o
o`03001oogoo00800003Ool007oo00Eoo`03001oogoo00Eoo`80021oo`000goo100097oo0P009Goo
0P001Woo10000Woo1`0000=oo`00Ool01Goo00<007ooOol00P000Woo00<007ooOol00goo100000=o
o`0000000Woo00<007ooOol00goo00<007ooOol00P000Woo00<007ooOol017oo00<007ooOol01@00
1Goo00<007ooOol00P000Woo00<007ooOol00Woo00<007ooOol00goo0`000goo0P000goo0P0000=o
o`00Ool017oo00<007ooOol00Woo0`001Goo00<007ooOol01@000goo00@007ooOomoo`@000Aoo`03
001oogoo00=oo`800003Ool0000000=oo`80021oo`0017oo00<007oo000097oo00<007ooOol09Woo
00@007ooOomoo`@000=oo`D000Eoo`03001oogoo00=oo`<000Eoo`03001oogoo00L000Aoo`03001o
ogoo00=oo`<000Aoo`03001oogoo00Aoo`D0009oo`8000=oo`@000Aoo`03001oogoo00=oo`03001o
ogoo008000=oo`<0009oo`<000Aoo`03001oogoo009oo`03001oogoo0080009oo`03001oogoo00=o
o`@0009oo`03001oo`00009oo`@000=oo`03001oogoo009oo`03001oogoo00<00004Ool00000Ool2
0004Ool00`00Oomoo`0MOol000=oo`03001oo`0002Eoo`03001oogoo02Moo``0009oo`03001oogoo
00=oo`03001oogoo00<0009oo`03001oogoo00=oo`@00003Ool00000009oo`03001oogoo00=oo`03
001oogoo0080009oo`03001oogoo00=oo`03001oogoo00H000Eoo`@000=oo`03001oogoo00=oo`04
001oogooOol40003Ool20002Ool300000goo001oo`04Ool01000Oomoogoo10001Goo00<007ooOol0
100017oo00<007ooOol00P0000=oo`00Ool017oo00@007ooOomoo`@0009oo`800003Ool00000009o
o`03001oogoo009oo`03001oogoo01aoo`000goo100097oo00<007ooOol09Woo10000Woo1@001Goo
00@007ooOomoo`@000Ioo`03001oogoo00L000Eoo`04001oogooOol40004Ool00`00Oomoo`03Ool6
0002Ool20002Ool20002Ool00`00Oomoo`02Ool00`00Oomoo`03Ool40004Ool70005Ool00`00Oomo
o`02Ool40003Ool20003Ool50002Ool01000Oomoogoo1@000goo00<007ooOol00Woo10000goo1P00
17oo00<007ooOol00Woo00<007ooOol02goo0P0017oo0P0017oo0P000goo000[Ool00`00Oomoo`0T
Ool90003Ool00`00Oomoo`03Ool40003Ool20005Ool400000goo00000002Ool00`00Oomoo`03Ool4
0003Ool00`00Oomoo`03Ool00`00Oomoo`02000017oo000000001Goo0`0017oo00<007ooOol00goo
00<007ooOol00P000Woo00<007ooOol01P000goo00<007ooOol00goo00<007ooOol00P000Woo00<0
07ooOol00goo1`0017oo10000goo00<007oo00000Woo00<007ooOol00P000Woo00<007ooOol01000
0goo00<007ooOol00Woo0P000Woo00<007oo00002goo00@007ooOol000Qoo`04001oogoo0002Ool0
02Yoo`03001oogoo029oo`@0009oo`<00003Ool007oo00Aoo`03001oogoo0080009oo`03001oogoo
00=oo`80009oo`L000Eoo`03001oogoo0080009oo`03001oogoo00=oo`04001oogooOol70002Ool2
0002Ool20002Ool00`00Oomoo`03Ool00`00Oomoo`02Ool30004Ool50002Ool00`00Oomoo`03Ool0
1000Oomoogoo100017oo00<007ooOol01`000Woo00<007ooOol00P0000=oo`00Ool017oo00<007oo
Ool01@0017oo100000=oo`00Ool00goo00<007ooOol00P000Woo00<007ooOol00Woo00<007ooOol0
2goo00<007ooOol027oo00<007ooOol00Goo000ZOol00`00Oomoo`0POol80005Ool01000Oomoogoo
10001Goo00<007ooOol00goo1`000Woo0P000goo100017oo00<007ooOol00goo10000Woo0`0017oo
100017oo00<007ooOol00goo00<007ooOol00P000Woo00<007ooOol01@0017oo00<007ooOol00goo
10000goo00<007ooOol00Woo20000goo100017oo00<007ooOol00Woo1@000Woo00<007ooOol01000
0goo00@007ooOomoo`@000Aoo`03001oogoo009oo`03001oogoo009oo`@000=oo`8000aoo`40005o
o`5oo`00:Woo00<007ooOol07Goo10000Woo0P000Woo00<007ooOol00goo10000goo00<007ooOol0
17oo00<007ooOol00P0000=oo`0000000P001Goo10000goo00<007ooOol00goo00<007ooOol00P00
0goo0`000goo10000goo00<007ooOol00goo00@007ooOomoo`@000Aoo`D0009oo`03001oogoo00=o
o`03001oogoo00800003Ool007oo00Aoo`03001oogoo00800003Ool000000080009oo`@000=oo`03
001oogoo009oo`03001oogoo0080009oo`04001oo`00Ool500000goo001oo`03Ool40003Ool00`00
Oomoo`03Ool01000Oomoogoo0P002Woo00<007ooOol027oo00@007ooOol0009oo`00:Woo0P0077oo
20001Goo00<007ooOol00P000Woo00<007ooOol00goo00<007ooOol00goo1`000Woo0P000Woo0P00
0Woo00<007ooOol00Woo00<007ooOol00goo0`000goo0`000goo0P000Woo00<007ooOol00Woo00<0
07ooOol00goo10000goo00<007ooOol01@001Goo00@007ooOomoo`@000Aoo`03001oogoo009oo`P0
009oo`800003Ool007oo00Aoo`04001oogooOol40004Ool70004Ool01P00Ool00000Ool000Eoo`04
001oogooOol8000:Ool30009Ool20003Ool002Yoo`03001oogoo01Qoo`P0009oo`03001oogoo00=o
o`<000Eoo`03001oogoo00=oo`03001oogoo00800003Ool00000008000Eoo`<000Aoo`03001oogoo
00=oo`03001oogoo008000=oo`<000=oo`<000Aoo`03001oogoo00=oo`03001oogoo0080009oo`03
001oogoo009oo`<00005Ool00000Ool00005Ool40003Ool00`00Oomoo`02Ool00`00Oomoo`020000
0goo0000000200000goo000000020004Ool00`00Oomoo`02Ool40003Ool00`00Oomoo`0700000goo
001oo`02Ool30004Ool00`00Oomoo`02Ool40003Ool300000goo001oo`0JOol001Ioo`D00004Ool0
00000004Ool20005Ool00`00Oomoo`0FOol200000goo000000030005Ool00`00Oomoo`020002Ool0
0`00Oomoo`04Ool01000Oomoogoo20000Woo0P000Woo0P000Woo00<007ooOol00goo00@007ooOomo
o`@000=oo`<000=oo`80009oo`03001oogoo00=oo`04001oogooOol40004Ool00`00Oomoo`060004
Ool00`00Oomoo`0200000goo001oo`04Ool01000Oomoogoo30000goo00<007ooOol00Woo00<007oo
Ool00P000Woo00<007ooOol00Woo1`000goo1@000Woo00<007ooOol00Woo00<007ooOol00P000Woo
00<007ooOol00`000goo00<007ooOol06Goo000HOol01@00Oomoogoo00000Woo00@007ooOol0009o
o`03001oogoo009oo`03001oogoo01=oo`P0009oo`8000=oo`@000Eoo`03001oogoo00=oo`@0009o
o`@000Aoo`@000Aoo`03001oogoo00=oo`@000Aoo`<0009oo`@000Aoo`03001oogoo00=oo`@000=o
o`03001oogoo009oo`<00003Ool00000009oo`04001oogooOol40004Ool00`00Oomoo`02Ool40002
Ool300001Goo000007oo00001Goo00@007ooOomoo`@000Aoo`03001oogoo00D000=oo`03001oogoo
00800003Ool007oo00=oo`04001oogooOol40003Ool40007Ool00`00Oomoo`0HOol001Qoo`05001o
ogooOol00002Ool01000Oomoo`000Woo00@007ooOomoo`8001=oo`800003Ool0000000<000Ioo`<0
00=oo`03001oogoo00Aoo`03001oogoo008000=oo`@000=oo`@000=oo`03001oogoo00=oo`03001o
ogoo0080009oo`80009oo`L000=oo`03001oogoo00=oo`03001oogoo0080009oo`03001oogoo009o
o`03001oogoo00H000Aoo`@000=oo`03001oogoo009oo`03001oogoo008000=oo`<00004Ool00000
0004Ool00`00Oomoo`02Ool40003Ool00`00Oomoo`02Ool500000goo001oo`02Ool40002Ool00`00
Ool00004Ool30003Ool60009Ool00`00Oomoo`0HOol001Qoo`04001oogooOol30003Ool01000Oomo
o`000goo00<007ooOol03goo20000Woo0`000Woo0P0000=oo`00Ool017oo00<007ooOol00goo0`00
0goo10000goo0P000Woo00<007ooOol00Woo00<007ooOol00goo0`001Goo0`0000=oo`0000000Woo
00<007ooOol00goo00@007ooOomoo`@000Aoo`04001oogooOol60003Ool00`00Oomoo`020002Ool0
0`00Oomoo`02Ool01000Oomoogoo10000goo0P000Woo0P000Woo00<007ooOol00Woo00<007ooOol0
0P000Woo00<007ooOol00Woo00<007ooOol010000goo1@000goo00@007ooOomoo`80009oo`800004
Ool007ooOol4000=Ool00`00Oomoo`0HOol001Qoo`03001oogoo009oo`03001oogoo009oo`04001o
ogoo0003Ool00`00Oomoo`0=Ool90004Ool40004Ool00`00Oomoo`03Ool00`00Oomoo`020003Ool4
0002Ool40004Ool00`00Oomoo`03Ool00`00Oomoo`020002Ool20002Ool60004Ool00`00Oomoo`03
Ool40003Ool00`00Oomoo`02Ool60002Ool01000Oomoogoo100017oo00<007ooOol00goo0`0017oo
0`0000Aoo`00000000Aoo`04001oogooOol40004Ool01000Oomoogoo1P0000=oo`0000000P000goo
00<007ooOol00Woo00@007ooOomoo`@000=oo`D000moo`03001oogoo01Qoo`005goo0P0017oo0`00
0goo0P0017oo00<007ooOol02Goo2@000goo1@000goo00<007ooOol00goo00@007ooOomoo`@000=o
o`P000=oo`03001oogoo00=oo`04001oogooOol40005Ool60002Ool00`00Oomoo`03Ool00`00Oomo
o`020002Ool00`00Oomoo`02Ool00`00Oomoo`02000017oo000000001Goo0`000goo00<007ooOol0
0goo00<007ooOol00P0000=oo`0000000Woo1P000Woo00<007ooOol00goo0`000goo00<007ooOol0
0goo0`0000=oo`0000000Woo0P0000=oo`00Ool017oo00@007ooOomoo`D00003Ool00000009oo`@0
019oo`03001oogoo01Qoo`00:Goo00<007ooOol01Goo100000=oo`0000001@000goo0P000Woo00<0
07ooOol00Woo00<007ooOol00goo100017oo10000Woo0P0000=oo`00Ool00goo00<007ooOol00goo
10000goo0P000Woo1P001Goo00@007ooOomoo`@000Aoo`03001oogoo00=oo`D0009oo`80009oo`80
0003Ool007oo00Aoo`04001oogooOol40004Ool50002Ool00`00Oomoo`02Ool00`00Oomoo`020000
0goo001oo`04Ool00`00Oomoo`0200001Goo00000000Ool00`0017oo00<007ooOol00Woo10000goo
00@007ooOomoo`@001Eoo`03001oogoo01Qoo`00:7oo0P001Goo0P000Woo1P000goo100017oo00<0
07ooOol00goo00<007ooOol00P000Woo0P000Woo200017oo00<007ooOol00Woo00<007ooOol00P00
0Woo00<007ooOol00goo1`0000=oo`0000000goo1@000goo00<007ooOol00goo00<007ooOol01@00
17oo10000goo00<007ooOol00goo10000goo00<007ooOol01@0017oo00@007ooOomoo`@000Aoo`04
001oogooOol40002Ool50002Ool00`00Oomoo`02Ool00`00Oomoo`020002Ool00`00Oomoo`02Ool6
000GOol00`00Oomoo`0HOol002Qoo`04001oogooOol<0003Ool20002Ool00`00Oomoo`03Ool01000
Oomoogoo10001Goo10000Woo0P000Woo00<007ooOol00goo00@007ooOomoo`<000Eoo`03001oogoo
00800003Ool00000008000Eoo`<000=oo`03001oogoo00=oo`04001oogooOol60003Ool40003Ool0
0`00Oomoo`02Ool00`00Oomoo`020002Ool00`00Oomoo`02Ool50002Ool01000Oomoogoo1@000goo
00<007ooOol00Woo10000Woo1@000Woo00D007ooOomoo`0000=oo`@000Aoo`03001oogoo00@000=o
o`03001oogoo01Ioo`03001oogoo01Moo`00:7oo00D007oo000007oo00L000=oo`@000Aoo`03001o
ogoo00=oo`@000=oo`80009oo`P000Eoo`04001oogooOol50002Ool00`00Oomoo`03Ool70002Ool2
0002Ool200000goo001oo`03Ool00`00Oomoo`03Ool40002Ool20003Ool200000goo001oo`04Ool0
1000Oomoogoo100017oo00<007ooOol01@0017oo10000goo00<007ooOol00Woo00<007ooOol00P00
0Woo1`0017oo00@007ooOomoo`@000=oo`03001oogoo009oo`D000Ioo`03001oogoo01Eoo`03001o
ogoo01Moo`00:7oo1@0000=oo`0000001`000goo0P0017oo00<007ooOol00P000Woo0P001Goo0`00
0Woo0P000Woo00<007ooOol00goo10000goo00<007ooOol00Woo00<007ooOol00P000Woo0`0017oo
10000goo00<007ooOol00goo00<007ooOol00P000goo0`0000=oo`0000000P0017oo00<007ooOol0
0Woo10000goo00@007ooOomoo`D000=oo`03001oogoo00800003Ool007oo00Aoo`04001oogooOol4
0002Ool50003Ool00`00Oomoo`02Ool40002Ool00`00Oomoo`02Ool00`00Oomoo`04000:Ool00`00
Oomoo`0DOol00`00Oomoo`0GOol002Qoo`800003Ool00000008000Ioo`03001oogoo00=oo`8000=o
o`<000Eoo`80009oo`L000Eoo`03001oogoo0080009oo`03001oogoo00=oo`04001oogooOol70003
Ool40002Ool00`00Oomoo`03Ool01000Oomoogoo10000goo1P000goo00<007ooOol00Woo00<007oo
Ool00P000Woo00<007ooOol00goo1@0000=oo`0000000goo0`0017oo00<007ooOol00Woo10000goo
1@0000=oo`00Ool00goo00<007ooOol00P0000=oo`00Ool00goo00@007ooOomoo`H000eoo`03001o
ogoo01=oo`03001oogoo01Moo`00:Goo00<007ooOol02Woo00<007ooOol00goo10000Woo0P001Goo
1P000goo0P000goo100017oo00<007ooOol00goo10000Woo10000Woo0P000Woo00<007ooOol00Woo
00<007ooOol00Woo100017oo0`0000Eoo`00001oo`0000Eoo`04001oogooOol30005Ool00`00Oomo
o`050003Ool50002Ool00`00Oomoo`02Ool00`00Oomoo`020002Ool00`00Oomoo`050003Ool01000
Oomoogoo100017oo00<007ooOol00Woo0`0000=oo`00000047oo00<007ooOol04goo00<007ooOol0
5goo000ZOol2000;Ool00`00Oomoo`020002Ool00`00Oomoo`02Ool20002Ool500000goo00000005
Ool30003Ool00`00Oomoo`03Ool00`00Oomoo`020002Ool40003Ool30004Ool00`00Oomoo`02Ool0
0`00Oomoo`020002Ool20002Ool60004Ool01000Oomoogoo1@000Woo00<007ooOol00goo1@000Woo
10000Woo00<007ooOol00goo00@007ooOomoo`<000Eoo`@0009oo`04001oogooOol40003Ool00`00
Oomoo`02Ool00`00Oomoo`02000017oo000000004goo00<007ooOol04Woo00<007ooOol05goo000/
Ool00`00Oomoo`09Ool20005Ool01000Oomoogoo1`0017oo0P000Woo0P0000=oo`00Ool00goo00<0
07ooOol00goo1`000goo10000Woo00<007ooOol00goo00@007ooOomoo`<000Eoo`D000=oo`03001o
ogoo009oo`@000=oo`03001oogoo009oo`03001oogoo00800003Ool0000000=oo`800003Ool007oo
00=oo`03001oogoo009oo`@0009oo`80009oo`@000Aoo`@0009oo`03001oogoo009oo`04001oogoo
Ool7000GOol00`00Oomoo`0AOol00`00Oomoo`0GOol002eoo`03001oogoo00Uoo`03001oogoo00=o
o`P0009oo`8000=oo`@000=oo`03001oogoo00=oo`03001oogoo00800003Ool00000008000=oo`80
009oo`03001oogoo009oo`04001oogooOol50002Ool20002Ool400000goo001oo`03Ool00`00Oomo
o`020002Ool00`00Oomoo`02Ool01000Oomoogoo1`0000=oo`0000000P000goo00<007ooOol00Woo
00<007ooOol00P000Woo00@007ooOomoo`@000=oo`03001oogoo00800003Ool007oo00Aoo`04001o
ogooOol4000017oo000000006goo00<007ooOol047oo00<007ooOol05goo000^Ool00`00Oomoo`09
Ool00`00Oomoo`020002Ool50004Ool40003Ool00`00Oomoo`02Ool01000Oomoogoo20000goo0`00
17oo00<007ooOol00goo0`000goo0P000goo1@0017oo00@007ooOomoo`<000Aoo`03001oogoo00=o
o`<0009oo`H000=oo`03001oogoo009oo`04001oogooOol30004Ool500000goo00000003Ool30004
Ool00`00Oomoo`02Ool40002Ool3000NOol00`00Oomoo`0AOol00`00Oomoo`0FOol002moo`8000Yo
o`L000Aoo`@0009oo`03001oogoo009oo`03001oogoo00=oo`<000=oo`P0009oo`03001oogoo00=o
o`03001oogoo00800003Ool007oo00Aoo`H0009oo`04001oogooOol50002Ool00`00Oomoo`03Ool0
0`00Oomoo`0200000goo000000040002Ool00`00Oomoo`02Ool01000Oomoogoo1@000Woo00<007oo
Ool00P0000=oo`0000000goo1@000Woo00<007ooOol00Woo00<007ooOol00P000Woo00<007ooOol0
0P008Goo00<007ooOol047oo00<007ooOol05Woo000aOol00`00Oomoo`0DOol01@00Oomoogoo0000
1Goo00<007ooOol00P000Woo1@0000=oo`0000000Woo00<007ooOol00Woo00@007ooOomoo`@000Ao
o`03001oogoo00800004Ool000000004Ool40003Ool00`00Oomoo`02Ool01000Oomoogoo1`0000Ao
o`00000000Aoo`04001oogooOol50003Ool01000Oomoogoo1P0000=oo`0000000P000goo00D007oo
Oomoo`0000=oo`@000=oo`@002Aoo`03001oogoo00moo`03001oogoo01Ioo`00<Woo00<007ooOol0
4goo00<007ooOol00Woo00@007ooOomoo`@000=oo`L000Aoo`04001oogooOol50003Ool00`00Oomo
o`02Ool60003Ool40002Ool00`00Oomoo`03Ool01000Oomoogoo10000Woo1P000Woo00<007ooOol0
0goo0`000goo00<007ooOol00Woo20000Woo0`001Goo00@007ooOomoo`@000=oo`D002Qoo`03001o
ogoo00ioo`03001oogoo01Ioo`00<goo00<007ooOol04goo00@007ooOomoo`@000Aoo`H000=oo`03
001oogoo00=oo`<000=oo`03001oogoo009oo`03001oogoo00D00004Ool007ooOol20002Ool01@00
Oomoogoo00001Goo10000goo1@000Woo00<007ooOol00Woo00<007ooOol00P0000=oo`00Ool00goo
00<007ooOol00P0000=oo`0000000goo10000goo00<007ooOol00Woo10000Woo00<007ooOol01000
:goo00<007ooOol03Woo00<007ooOol05Woo000dOol00`00Oomoo`0COol30002Ool30002Ool40000
0goo00000004Ool00`00Oomoo`0200000goo001oo`04Ool01000Oomoogoo1@0017oo0`0017oo00<0
07ooOol00Woo00<007ooOol00P000Woo00<007ooOol01@0017oo00@007ooOomoo`@000Aoo`04001o
ogooOol600000goo000000020003Ool01@00Oomoogoo00000Woo0P000Woo00@007ooOomoo`D002io
o`03001oogoo00eoo`03001oogoo01Ioo`00=Goo0P0057oo00<007ooOol00goo1@0017oo0P000goo
0`000goo00<007ooOol00goo1`000Woo10000Woo00<007ooOol00goo00@007ooOomoo`<000Eoo`@0
009oo`8000=oo`@000=oo`03001oogoo009oo`@00003Ool0000000@0009oo`03001oogoo009oo`04
001oogooOol30005Ool5000bOol00`00Oomoo`0<Ool00`00Oomoo`0FOol003Moo`03001oogoo019o
o`03001oogoo00@0009oo`8000=oo`D0009oo`03001oogoo009oo`03001oogoo00800003Ool00000
008000=oo`800003Ool007oo00=oo`04001oogooOol50002Ool20002Ool300000goo00000003Ool4
0002Ool00`00Oomoo`02Ool00`00Oomoo`020002Ool50002Ool01@00Oomoogoo00000goo1@000Woo
0P000Woo1000=Woo00<007ooOol037oo00<007ooOol05Goo000hOol00`00Oomoo`0BOol30007Ool3
0002Ool00`00Oomoo`03Ool01000Oomoogoo1`000Woo10000goo00<007ooOol00goo0`000goo00@0
07ooOomoo`D000Aoo`@0009oo`03001oogoo009oo`04001oogooOol30003Ool40004Ool01000Oomo
ogoo10000goo00@007ooOomoo`H003Qoo`03001oogoo00aoo`03001oogoo01Eoo`00>Goo00<007oo
Ool077oo00<007ooOol00Woo00<007ooOol00Woo10000Woo1`000goo00<007ooOol00Woo00<007oo
Ool00P0000=oo`00Ool017oo1@000Woo00<007ooOol00P0000=oo`00Ool00goo00@007ooOomoo`D0
009oo`D0009oo`8000=oo`@0009oo`03001oogoo00=oo`<00003Ool0000003aoo`03001oogoo00]o
o`03001oogoo01Eoo`00>Woo0P0077oo00<007ooOol00Woo00<007ooOol00P000goo1P000Woo00<0
07ooOol00Woo00@007ooOomoo`@000Aoo`03001oogoo00H000=oo`<000Aoo`03001oogoo009oo`<0
00=oo`H00003Ool007oo009oo`D0009oo`03001oogoo009oo`03001oogoo00800004Ool00000000o
Ool00`00Oomoo`0:Ool00`00Oomoo`0EOol003aoo`03001oogoo01Yoo`04001oogooOol40003Ool3
0002Ool00`00Oomoo`02Ool01000Oomoogoo10000goo00@007ooOomoo`D000Aoo`@0009oo`03001o
ogoo009oo`03001oogoo00800003Ool00000009oo`D000Aoo`<000=oo`05001oogooOol00003Ool7
0012Ool00`00Oomoo`0:Ool00`00Oomoo`0EOol003eoo`03001oogoo01Yoo`<000=oo`D000Eoo`04
001oogooOol40002Ool00`00Oomoo`03Ool600000goo000000030002Ool00`00Oomoo`02Ool01000
Oomoogoo0`0017oo10000goo00<007ooOol00P0000=oo`00Ool00goo00@007ooOomoo`@00003Ool0
000004Moo`03001oogoo00Uoo`03001oogoo01Eoo`00?Woo00<007ooOol06goo100017oo0P000goo
1@000Woo00<007ooOol00Woo00<007ooOol00P0000=oo`00000010000goo00D007ooOomoo`0000=o
o`D000=oo`D00003Ool0000000=oo`<000=oo`03001oogoo00=oo`<0009oo`<004Yoo`03001oogoo
00Qoo`03001oogoo01Eoo`00?goo0P009Goo0`000goo00<007ooOol00Woo00@007ooOomoo`H00005
Ool00000Ool00004Ool01000Oomoogoo10000goo00<007ooOol01@0017oo10000Woo00<007ooOol0
0Woo00<007ooOol00P0000=oo`0000000P00CWoo00<007ooOol027oo00<007ooOol057oo0011Ool0
0`00Oomoo`0SOol00`00Oomoo`02Ool01000Oomoogoo1`000Woo0`0017oo00@007ooOomoo`@0009o
o`8000=oo`D0009oo`D0009oo`05001oogooOol00003Ool40002Ool2001AOol00`00Oomoo`08Ool0
0`00Oomoo`0DOol0049oo`03001oogoo029oo`03001oogoo00=oo`<0009oo`L0009oo`03001oogoo
00=oo`<0009oo`03001oogoo00=oo`D00003Ool00000008000=oo`05001oogooOol00003Ool40003
Ool3001DOol00`00Oomoo`07Ool00`00Oomoo`0DOol004=oo`03001oogoo029oo`03001oogoo0080
009oo`H0009oo`03001oogoo009oo`03001oogoo00800003Ool007oo00=oo`03001oogoo00P0009o
o`03001oogoo009oo`04001oogooOol40003Ool4001HOol00`00Oomoo`06Ool00`00Oomoo`0DOol0
04Aoo`03001oogoo029oo`8000=oo`<0009oo`05001oogooOol00003Ool40003Ool01000Oomoogoo
1P0000Eoo`00001oo`0000Aoo`04001oogooOol40002Ool00`00Oomoo`02001MOol00`00Oomoo`05
Ool00`00Oomoo`0DOol004Eoo`8002Aoo`<000Eoo`04001oogooOol40003Ool01000Oomoogoo1`00
00Aoo`00000000=oo`04001oogooOol50002Ool01000Oomoogoo0`00Ggoo00<007ooOol01Goo00<0
07ooOol057oo0017Ool00`00Oomoo`0ZOol30002Ool00`00Oomoo`03Ool30002Ool60002Ool01000
Oomoogoo10000goo00<007ooOol00Woo0`00Hgoo00<007ooOol017oo00<007ooOol057oo0018Ool0
0`00Oomoo`0ZOol00`00Oomoo`02Ool00`00Oomoo`020002Ool50002Ool01000Oomoogoo10000Woo
00<007ooOol00Woo00<007ooOol00P00Igoo00<007ooOol00goo00<007ooOol057oo0019Ool00`00
Oomoo`0ZOol01000Oomoogoo2@0000=oo`00Ool00goo0`000Woo00<007ooOol00Woo00@007ooOomo
o`<006Yoo`03001oogoo00=oo`03001oogoo01=oo`00BWoo0P00:goo0`0000=oo`00000010000goo
00<007ooOol00P0000=oo`00Ool00goo00@007ooOomoo`@006eoo`03001oogoo00=oo`03001oogoo
01=oo`00C7oo00<007ooOol0:Woo0P001Goo00@007ooOomoo`<000Aoo`04001oogooOol4001bOol0
0`00Oomoo`02Ool00`00Oomoo`0COol004eoo`03001oogoo035oo`<0009oo`8000=oo`@007Moo`05
001oogooOol0000EOol004ioo`03001oogoo035oo`03001oogoo009oo`@007aoo`04001oogoo000E
Ool004moo`80035oo`03001oogoo0080081oo`04001oogoo000EOol0055oo`03001oogoo02moo`80
08=oo`03001oo`0001Eoo`00DWoo00<007ooOol0;goo00<007ooOol0PWoo00<007oo000057oo001C
Ool00`00Oomoo`0^Ool00`00Oomoo`23Ool2000DOol005Aoo`03001oogoo02eoo`03001oogoo08=o
o`8001Aoo`00EGoo0P00;Goo00<007ooOol0Pgoo0P0057oo001GOol00`00Oomoo`0YOol00`00Oomo
o`20Ool4000FOol005Qoo`03001oogoo02Qoo`03001oogoo07aoo`@001Yoo`00FGoo00<007ooOol0
9goo00<007ooOol0N7oo10007Woo001JOol2000WOol00`00Oomoo`1dOol4000ROol005aoo`03001o
ogoo02Aoo`03001oogoo075oo`<002Ioo`00GGoo00<007ooOol08goo00<007ooOol0KGoo1000:Goo
001NOol00`00Oomoo`0ROol00`00Oomoo`1YOol4000]Ool005moo`80029oo`03001oogoo06Eoo`@0
035oo`00HGoo00<007ooOol07goo00<007ooOol0HGoo1000=Goo001ROol00`00Oomoo`0NOol00`00
Oomoo`1NOol3000iOol006=oo`03001oogoo01eoo`03001oogoo05Yoo`@003aoo`00I7oo0P007Goo
00<007ooOol0EWoo1000@7oo001VOol00`00Oomoo`0JOol00`00Oomoo`1BOol40014Ool006Moo`03
001oogoo01Uoo`03001oogoo04ioo`@004Qoo`00J7oo00<007ooOol067oo00<007ooOol0Bgoo0`00
C7oo001YOol00`00Oomoo`0GOol00`00Oomoo`17Ool4001?Ool006Yoo`8001Moo`03001oogoo04=o
o`@005=oo`00K7oo00<007ooOol057oo00<007ooOol0?goo1000Egoo001]Ool00`00Oomoo`0COol0
0`00Oomoo`0kOol4001KOol006ioo`03001oogoo019oo`03001oogoo03Moo`@005moo`00Kgoo0P00
4Woo00<007ooOol0=7oo0`00Hgoo001aOol00`00Oomoo`0?Ool00`00Oomoo`0`Ool4001VOol0079o
o`03001oogoo00ioo`03001oogoo02aoo`@006Yoo`00Lgoo00<007ooOol03Goo00<007ooOol0:7oo
1000KWoo001dOol2000<Ool00`00Oomoo`0UOol4001bOol007Ioo`03001oogoo00Uoo`03001oogoo
029oo`<007Ioo`00Mgoo00<007ooOol027oo00<007ooOol07Woo1000NGoo001hOol00`00Oomoo`07
Ool00`00Oomoo`0JOol4001mOol007Uoo`03001oogoo00Ioo`03001oogoo01Ioo`@0085oo`00NWoo
0P001Woo00<007ooOol04Woo1000QGoo001lOol00`00Oomoo`03Ool00`00Oomoo`0?Ool30029Ool0
07eoo`03001oogoo009oo`03001oogoo00]oo`@008aoo`00OWoo00D007ooOomoo`0000Uoo`@0091o
o`00Ogoo0P0000=oo`00Ool017oo1000U7oo0021Ool200000goo00000002002HOol0089oo`8009ao
o`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003o
OolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5oo`00ogoo8Goo003oOolQOol00?moob5o
o`00ogoo8Goo003oOolQOol00?moob5oo`00\
\>"],
  ImageRangeCache->{{{78.5625, 307.938}, {287.188, 101.375}} -> {-0.582864, 
  0.471449, 0.00420652, 0.00420652}}]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(hesstoo\  = \ Outer[D, derivs, vars]\  /. \ critpointtoo\)], "Input"],

Cell[BoxData[
    \({{\(-577.2363283209577`\), \(-3102.1466733853977`\)}, 
      {\(-3102.1466733853977`\), \(-17859.35018924585`\)}}\)], "Output"]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \({evals2, evecs2}\  = \ Eigensystem[hesstoo]\)], "Input"],

Cell[BoxData[
    \({{\(-18399.31601546412`\), \(-37.270502102684986`\)}, 
      {{0.1714836060829197`, 0.985186973546036`}, 
        {0.985186973546036`, \(-0.17148360608291965`\)}}}\)], "Output"]
}, Open  ]],

Cell["\<\
It's a local maximum here, but the function is this region is too crinkly to \
be of much use.\
\>", "Text"]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"4.0 for Microsoft Windows",
ScreenRectangle->{{0, 640}, {0, 424}},
WindowSize->{496, 415},
WindowMargins->{{Automatic, 52}, {Automatic, 70}},
PrintingCopies->1,
PrintingPageRange->{Automatic, Automatic}
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1739, 51, 28, 0, 64, "Subtitle"],

Cell[CellGroupData[{
Cell[1792, 55, 165, 6, 43, "Subsubsection"],
Cell[1960, 63, 384, 8, 109, "Text"],
Cell[2347, 73, 191, 8, 138, "Input"],
Cell[2541, 83, 45, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[2611, 87, 65, 2, 30, "Input"],
Cell[2679, 91, 38, 1, 29, "Output"]
}, Open  ]],
Cell[2732, 95, 327, 5, 90, "Text"],
Cell[3062, 102, 58, 2, 30, "Input"],
Cell[3123, 106, 38, 0, 33, "Text"],
Cell[3164, 108, 40, 0, 30, "Input"],
Cell[3207, 110, 47, 0, 33, "Text"],
Cell[3257, 112, 126, 4, 66, "Input"],
Cell[3386, 118, 257, 6, 90, "Text"],

Cell[CellGroupData[{
Cell[3668, 128, 74, 2, 30, "Input"],
Cell[3745, 132, 57, 1, 29, "Output"]
}, Open  ]],
Cell[3817, 136, 72, 2, 33, "Text"],

Cell[CellGroupData[{
Cell[3914, 142, 76, 2, 30, "Input"],
Cell[3993, 146, 57, 1, 29, "Output"]
}, Open  ]],
Cell[4065, 150, 461, 12, 147, "Text"],
Cell[4529, 164, 665, 30, 128, "Text"],
Cell[5197, 196, 355, 6, 90, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[5589, 207, 147, 4, 60, "Subsubsection"],
Cell[5739, 213, 607, 9, 147, "Text"],
Cell[6349, 224, 194, 3, 70, "Input"],
Cell[6546, 229, 61, 0, 33, "Text"],

Cell[CellGroupData[{
Cell[6632, 233, 258, 5, 90, "Input"],
Cell[6893, 240, 310, 5, 110, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[7252, 251, 129, 3, 43, "Subsubsection"],
Cell[7384, 256, 469, 7, 109, "Text"],
Cell[7856, 265, 161, 3, 50, "Input"],
Cell[8020, 270, 166, 4, 52, "Text"],
Cell[8189, 276, 137, 2, 50, "Input"],

Cell[CellGroupData[{
Cell[8351, 282, 157, 3, 50, "Input"],
Cell[8511, 287, 111, 2, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[8659, 294, 71, 1, 30, "Input"],
Cell[8733, 297, 84, 1, 29, "Output"]
}, Open  ]],
Cell[8832, 301, 269, 5, 71, "Text"],

Cell[CellGroupData[{
Cell[9126, 310, 180, 3, 70, "Input"],
Cell[9309, 315, 112, 2, 29, "Output"]
}, Open  ]],
Cell[9436, 320, 187, 8, 33, "Text"],

Cell[CellGroupData[{
Cell[9648, 332, 193, 3, 50, "Input"],
Cell[9844, 337, 60763, 1454, 242, 23124, 985, "GraphicsData", 
"PostScript", "Graphics"]
}, Open  ]],

Cell[CellGroupData[{
Cell[70644, 1796, 83, 1, 30, "Input"],
Cell[70730, 1799, 131, 2, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[70898, 1806, 71, 1, 30, "Input"],
Cell[70972, 1809, 203, 3, 48, "Output"]
}, Open  ]],
Cell[71190, 1815, 2218, 58, 361, "Text"],
Cell[73411, 1875, 62, 0, 33, "Text"],
Cell[73476, 1877, 417, 7, 150, "Input"],

Cell[CellGroupData[{
Cell[73918, 1888, 87, 1, 30, "Input"],
Cell[74008, 1891, 322, 5, 48, "Output"]
}, Open  ]],
Cell[74345, 1899, 768, 20, 147, "Text"],

Cell[CellGroupData[{
Cell[75138, 1923, 914, 20, 330, "Input"],
Cell[76055, 1945, 129599, 5893, 242, 93088, 5438, "GraphicsData", 
"PostScript", "Graphics"]
}, Open  ]],
Cell[205669, 7841, 108, 3, 33, "Text"],

Cell[CellGroupData[{
Cell[205802, 7848, 242, 4, 70, "Input"],
Cell[206047, 7854, 98460, 2657, 242, 65762, 2249, "GraphicsData", 
"PostScript", "Graphics"]
}, Open  ]],

Cell[CellGroupData[{
Cell[304544, 10516, 89, 1, 30, "Input"],
Cell[304636, 10519, 147, 2, 29, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[304820, 10526, 76, 1, 30, "Input"],
Cell[304899, 10529, 197, 3, 48, "Output"]
}, Open  ]],
Cell[305111, 10535, 118, 3, 33, "Text"]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

